/*==========================================================================
//  x_client.c -- by various                    Last updated:  4-20-1999
//--------------------------------------------------------------------------
//  This file contains various player related code.
//========================================================================*/

#include "g_local.h"
#include "x_client.h"
#include "x_fire.h"
#include "x_rules.h"


/*=====================/  General Player Functions  /=====================*/

/*----------------------------------------------------/ New Code /--------//
//  This checks if the client is in a player model.
//------------------------------------------------------------------------*/
qboolean Coven_IsPlayer (edict_t *ent)
{
        if (!ent->client)
                return false;

        if (ent->client->resp.spectator)
                return false;

        if ((ent->s.modelindex != 255) && (ent->health <= ent->gib_health))
                return false;

        return true;
}

/*----------------------------------------------------/ New Code /--------//
//  This produces a combination of flesh and metal gibs based
//  on the player's battlesuit if any.  In addition, all players
//  leave a head behind.
//
//  * Light suits and PPMs are reduced to four gibs of meat.
//  * Medium suits are reduced to two pieces of flesh and metal each.
//  * Heavy suits explode and produce four scraps of bloody metal.
//------------------------------------------------------------------------*/
void Coven_SuitGib (edict_t *ent, int damage, int delta)
{
        int     n;

        gi.sound (ent, CHAN_BODY, gi.soundindex ("misc/udeath.wav"), 1, ATTN_NORM, 0);

        if ((ent->battlesuit == SUIT_GRAVYBURNER) || (ent->battlesuit == SUIT_WIRECUTTER))
        {       ThrowGib (ent, "models/objects/gibs/sm_metal/tris.md2", damage, GIB_METALLIC);
                ThrowGib (ent, "models/objects/gibs/sm_metal/tris.md2", damage, GIB_METALLIC);
                ThrowGib (ent, "models/objects/gibs/sm_meat/tris.md2", damage, GIB_ORGANIC);
                ThrowGib (ent, "models/objects/gibs/sm_meat/tris.md2", damage, GIB_ORGANIC);
        }
        else if ((ent->battlesuit == SUIT_RHINO) || (ent->battlesuit == SUIT_SUGARCONNY) || (ent->battlesuit == SUIT_BULKHEAD))
        {       gi.WriteByte (svc_temp_entity);
                if (gi.pointcontents(ent->s.origin) & MASK_WATER)
                        gi.WriteByte (TE_ROCKET_EXPLOSION_WATER);
                else
                        gi.WriteByte (TE_ROCKET_EXPLOSION);
                gi.WritePosition (ent->s.origin);
                gi.multicast (ent->s.origin, MULTICAST_PVS);

                for (n= 0; n < 4; n++)
                        ThrowGib (ent, "models/objects/gibs/sm_metal/tris.md2", damage, GIB_METALLIC);
        }
        else
        {       for (n= 0; n < 4; n++)
                        ThrowGib (ent, "models/objects/gibs/sm_meat/tris.md2", damage, GIB_ORGANIC);
        }

        if (delta)
                ent->s.origin[2] += delta;

        ThrowClientHead (ent, damage);

        ent->takedamage = DAMAGE_NO;
}


/*=========================/  Player Spawning  /=========================*/

/*
===============
SetAnimAliases
===============
*/
void Coven_SetAnimAliases (edict_t *ent)
{
/* PM:  Anim aliases should be set only when a player is spawned. */

        /* Ignore if not in a battlesuit. */
        if (!ent->battlesuit)
                return;

	/* LIGHT BATTLESUITS */
        if (ent->battlesuit == SUIT_MERCURY)
	{
                ent->swim_anim_start            = FRAME_Mswim1;
                ent->swim_anim_end              = FRAME_Mswim6;
                ent->swim_attack_anim_start     = FRAME_Mswimstrike1;
                ent->swim_attack_anim_end       = FRAME_Mswimstrike6;
                ent->swim_pain_anim_start       = FRAME_Mswimpain1;
                ent->swim_pain_anim_end         = FRAME_Mswimpain6;
                ent->swim_death_anim_start      = FRAME_Mswimdeath1;
                ent->swim_death_anim_end        = FRAME_Mswimdeath20;
                ent->crouch_anim_start          = FRAME_Mdodge1;
                ent->crouch_anim_end            = FRAME_Mdodge30;
                ent->crouch_attack_anim_start   = FRAME_Mdodgestrike1;
                ent->crouch_attack_anim_end     = FRAME_Mdodgestrike2;
                ent->crouch_move_anim_start     = FRAME_Mdodgewalk1;
                ent->crouch_move_anim_end       = FRAME_Mdodgewalk6;
                ent->crouch_pain_anim_start     = FRAME_Mdodgepain1;
                ent->crouch_pain_anim_end       = FRAME_Mdodgepain6;
                ent->crouch_death_anim_start    = FRAME_Mdodgedeath1;
                ent->crouch_death_anim_end      = FRAME_Mdodgedeath8;
                ent->stand_anim_start           = FRAME_Mstand1;
                ent->stand_anim_end             = FRAME_Mstand35;
                ent->attack_anim_start          = FRAME_Mstandfire1;
                ent->attack_anim_end            = FRAME_Mstandfire2;
                ent->run_anim_start             = FRAME_Mrun1;
                ent->run_anim_end               = FRAME_Mrun4;
                ent->run_attack_anim_start      = FRAME_Mrunfire1;
                ent->run_attack_anim_end        = FRAME_Mrunfire4;
                ent->jump_anim_start            = FRAME_Mjump1;
                ent->jump_anim_end              = FRAME_Mjump6;
                ent->pain_anim_start            = FRAME_Mpain1;
                ent->pain_anim_end              = FRAME_Mpain6;
                ent->death_anim_start           = FRAME_Mdeath1;
                ent->death_anim_end             = FRAME_Mdeath16;
                return;
	}
        if (ent->battlesuit == SUIT_ZEPPELIN)
	{
                ent->swim_anim_start            = FRAME_Zswim1;
                ent->swim_anim_end              = FRAME_Zswim6;
                ent->swim_attack_anim_start     = FRAME_Zswimstrike1;
                ent->swim_attack_anim_end       = FRAME_Zswimstrike6;
                ent->swim_pain_anim_start       = FRAME_Zswimpain1;
                ent->swim_pain_anim_end         = FRAME_Zswimpain6;
                ent->swim_death_anim_start      = FRAME_Zswimdeath1;
                ent->swim_death_anim_end        = FRAME_Zswimdeath19;
                ent->crouch_anim_start          = FRAME_Zdodge1;
                ent->crouch_anim_end            = FRAME_Zdodge20;
		ent->crouch_attack_anim_start	= FRAME_Zdodgestrike1;
                ent->crouch_attack_anim_end     = FRAME_Zdodgestrike2;
                ent->crouch_move_anim_start     = FRAME_Zdodgewalk1;
                ent->crouch_move_anim_end       = FRAME_Zdodgewalk6;
                ent->crouch_pain_anim_start     = FRAME_Zdodgepain1;
                ent->crouch_pain_anim_end       = FRAME_Zdodgepain6;
                ent->crouch_death_anim_start    = FRAME_Zdodgedeath1;
                ent->crouch_death_anim_end      = FRAME_Zdodgedeath8;
                ent->stand_anim_start           = FRAME_Zstand1;
                ent->stand_anim_end             = FRAME_Zstand20;
                ent->attack_anim_start          = FRAME_Zstandfire1;
                ent->attack_anim_end            = FRAME_Zstandfire2;
                ent->run_anim_start             = FRAME_Zrun1;
                ent->run_anim_end               = FRAME_Zrun4;
                ent->run_attack_anim_start      = FRAME_Zrunfire1;
                ent->run_attack_anim_end        = FRAME_Zrunfire4;
                ent->jump_anim_start            = FRAME_Zjump1;
                ent->jump_anim_end              = FRAME_Zjump6;
                ent->pain_anim_start            = FRAME_Zpain1;
                ent->pain_anim_end              = FRAME_Zpain6;
                ent->death_anim_start           = FRAME_Zdeath1;
                ent->death_anim_end             = FRAME_Zdeath25;
                return;
	}

	/* MEDIUM BATTLESUITS */
        if (ent->battlesuit == SUIT_GRAVYBURNER)
	{
                ent->swim_anim_start            = FRAME_GBswim1;
                ent->swim_anim_end              = FRAME_GBswim8;
                ent->swim_attack_anim_start     = FRAME_GBswimstrike1;
                ent->swim_attack_anim_end       = FRAME_GBswimstrike8;
                ent->swim_pain_anim_start       = FRAME_GBswimpain1;
                ent->swim_pain_anim_end         = FRAME_GBswimpain6;
                ent->swim_death_anim_start      = FRAME_GBswimdeath1;
                ent->swim_death_anim_end        = FRAME_GBswimdeath15;
                ent->crouch_anim_start          = FRAME_GBdodge1;
                ent->crouch_anim_end            = FRAME_GBdodge20;
                ent->crouch_attack_anim_start   = FRAME_GBdodgestrike1;
                ent->crouch_attack_anim_end     = FRAME_GBdodgestrike2;
                ent->crouch_move_anim_start     = FRAME_GBdodgewalk1;
                ent->crouch_move_anim_end       = FRAME_GBdodgewalk12;
                ent->crouch_pain_anim_start     = FRAME_GBdodgepain1;
                ent->crouch_pain_anim_end       = FRAME_GBdodgepain6;
                ent->crouch_death_anim_start    = FRAME_GBdodgedeath1;
                ent->crouch_death_anim_end      = FRAME_GBdodgedeath13;
                ent->stand_anim_start           = FRAME_GBstand1;
                ent->stand_anim_end             = FRAME_GBstand20;
                ent->attack_anim_start          = FRAME_GBstandfire1;
                ent->attack_anim_end            = FRAME_GBstandfire2;
                ent->run_anim_start             = FRAME_GBrun1;
                ent->run_anim_end               = FRAME_GBrun8;
                ent->run_attack_anim_start      = FRAME_GBrunfire1;
                ent->run_attack_anim_end        = FRAME_GBrunfire8;
                ent->jump_anim_start            = FRAME_GBjump1;
                ent->jump_anim_end              = FRAME_GBjump6;
                ent->pain_anim_start            = FRAME_GBpain1;
                ent->pain_anim_end              = FRAME_GBpain6;
                ent->death_anim_start           = FRAME_GBdeath1;
                ent->death_anim_end             = FRAME_GBdeath25;
                return;
	}
        if (ent->battlesuit == SUIT_WIRECUTTER)
	{
                ent->swim_anim_start            = FRAME_WCswim1;
                ent->swim_anim_end              = FRAME_WCswim10;
                ent->swim_attack_anim_start     = FRAME_WCswimstrike1;
                ent->swim_attack_anim_end       = FRAME_WCswimstrike10;
                ent->swim_pain_anim_start       = FRAME_WCswimpain1;
                ent->swim_pain_anim_end         = FRAME_WCswimpain6;
                ent->swim_death_anim_start      = FRAME_WCswimdeath1;
                ent->swim_death_anim_end        = FRAME_WCswimdeath21;
                ent->crouch_anim_start          = FRAME_WCdodge1;
                ent->crouch_anim_end            = FRAME_WCdodge28;
                ent->crouch_attack_anim_start   = FRAME_WCdodgestrike1;
                ent->crouch_attack_anim_end     = FRAME_WCdodgestrike2;
                ent->crouch_move_anim_start     = FRAME_WCdodgewalk1;
                ent->crouch_move_anim_end       = FRAME_WCdodgewalk12;
                ent->crouch_pain_anim_start     = FRAME_WCdodgepain1;
                ent->crouch_pain_anim_end       = FRAME_WCdodgepain6;
                ent->crouch_death_anim_start    = FRAME_WCdodgedeath1;
                ent->crouch_death_anim_end      = FRAME_WCdodgedeath8;
                ent->stand_anim_start           = FRAME_WCstand1;
                ent->stand_anim_end             = FRAME_WCstand35;
                ent->attack_anim_start          = FRAME_WCstandfire1;
                ent->attack_anim_end            = FRAME_WCstandfire2;
                ent->run_anim_start             = FRAME_WCrun1;
                ent->run_anim_end               = FRAME_WCrun8;
                ent->run_attack_anim_start      = FRAME_WCrunfire1;
                ent->run_attack_anim_end        = FRAME_WCrunfire8;
                ent->jump_anim_start            = FRAME_WCjump1;
                ent->jump_anim_end              = FRAME_WCjump6;
                ent->pain_anim_start            = FRAME_WCpain1;
                ent->pain_anim_end              = FRAME_WCpain6;
                ent->death_anim_start           = FRAME_WCdeath1;
                ent->death_anim_end             = FRAME_WCdeath18;
                return;
	}

	/* HEAVY BATTLESUITS */
        if (ent->battlesuit == SUIT_RHINO)
	{
                ent->swim_anim_start            = FRAME_Rswim1;
                ent->swim_anim_end              = FRAME_Rswim6;
                ent->swim_attack_anim_start     = FRAME_Rswimfire1;
                ent->swim_attack_anim_end       = FRAME_Rswimfire6;
                ent->swim_pain_anim_start       = FRAME_Rswimpain1;
                ent->swim_pain_anim_end         = FRAME_Rswimpain6;
                ent->swim_death_anim_start      = FRAME_Rswimdie1;
                ent->swim_death_anim_end        = FRAME_Rswimdie20;
                ent->crouch_anim_start          = FRAME_Rcrstnd1;
                ent->crouch_anim_end            = FRAME_Rcrstnd19;
                ent->crouch_attack_anim_start   = FRAME_Rcrattak1;
                ent->crouch_attack_anim_end     = FRAME_Rcrattak9;
                ent->crouch_move_anim_start     = FRAME_Rcrwalk1;
                ent->crouch_move_anim_end       = FRAME_Rcrwalk6;
                ent->crouch_pain_anim_start     = FRAME_Rcrpain1;
                ent->crouch_pain_anim_end       = FRAME_Rcrpain4;
                ent->crouch_death_anim_start    = FRAME_Rcrdeath1;
                ent->crouch_death_anim_end      = FRAME_Rcrdeath5;
                ent->stand_anim_start           = FRAME_Rstand1;
                ent->stand_anim_end             = FRAME_Rstand40;
                ent->attack_anim_start          = FRAME_Rattack1;
                ent->attack_anim_end            = FRAME_Rattack8;
                ent->run_anim_start             = FRAME_Rrun1;
                ent->run_anim_end               = FRAME_Rrun6;
                ent->run_attack_anim_start      = FRAME_Rrunfire1;
                ent->run_attack_anim_end        = FRAME_Rrunfire6;
                ent->jump_anim_start            = FRAME_Rjump1;
                ent->jump_anim_end              = FRAME_Rjump6;
                ent->pain_anim_start            = FRAME_Rpain1;
                ent->pain_anim_end              = FRAME_Rpain4;
                ent->death_anim_start           = FRAME_Rdeath13;
                ent->death_anim_end             = FRAME_Rdeath20;
                return;
	}
        if (ent->battlesuit == SUIT_SUGARCONNY)
	{
                ent->swim_anim_start            = FRAME_SCswim1;
                ent->swim_anim_end              = FRAME_SCswim11;
                ent->swim_attack_anim_start     = FRAME_SCswimstrike1;
                ent->swim_attack_anim_end       = FRAME_SCswimstrike11;
                ent->swim_pain_anim_start       = FRAME_SCswimpain1;
                ent->swim_pain_anim_end         = FRAME_SCswimpain6;
                ent->swim_death_anim_start      = FRAME_SCswimdeath1;
                ent->swim_death_anim_end        = FRAME_SCswimdeath21;
                ent->crouch_anim_start          = FRAME_SCdodge1;
                ent->crouch_anim_end            = FRAME_SCdodge16;
                ent->crouch_attack_anim_start   = FRAME_SCdodgestrike1;
                ent->crouch_attack_anim_end     = FRAME_SCdodgestrike2;
                ent->crouch_move_anim_start     = FRAME_SCdodgewalk1;
                ent->crouch_move_anim_end       = FRAME_SCdodgewalk13;
                ent->crouch_pain_anim_start     = FRAME_SCdodgepain1;
                ent->crouch_pain_anim_end       = FRAME_SCdodgepain6;
                ent->crouch_death_anim_start    = FRAME_SCdodgedeath1;
                ent->crouch_death_anim_end      = FRAME_SCdodgedeath7;
                ent->stand_anim_start           = FRAME_SCstand1;
                ent->stand_anim_end             = FRAME_SCstand21;
                ent->attack_anim_start          = FRAME_SCstandfire1;
                ent->attack_anim_end            = FRAME_SCstandfire2;
                ent->run_anim_start             = FRAME_SCrun1;
                ent->run_anim_end               = FRAME_SCrun10;
                ent->run_attack_anim_start      = FRAME_SCrunfire1;
                ent->run_attack_anim_end        = FRAME_SCrunfire10;
                ent->jump_anim_start            = FRAME_SCjump1;
                ent->jump_anim_end              = FRAME_SCjump6;
                ent->pain_anim_start            = FRAME_SCpain1;
                ent->pain_anim_end              = FRAME_SCpain6;
                ent->death_anim_start           = FRAME_SCdeath1;
                ent->death_anim_end             = FRAME_SCdeath15;
                return;
	}

        /* HIDDEN BATTLESUIT */
        if (ent->battlesuit == SUIT_BULKHEAD)
	{
                ent->swim_anim_start            = FRAME_BHswim1;
                ent->swim_anim_end              = FRAME_BHswim7;
                ent->swim_attack_anim_start     = FRAME_BHstandfire1;
                ent->swim_attack_anim_end       = FRAME_BHstandfire2;
                ent->swim_pain_anim_start       = FRAME_BHpain1;
                ent->swim_pain_anim_end         = FRAME_BHpain6;
                ent->swim_death_anim_start      = FRAME_BHdeath1;
                ent->swim_death_anim_end        = FRAME_BHdeath10;
                ent->crouch_anim_start          = FRAME_BHdodge1;
                ent->crouch_anim_end            = FRAME_BHdodge1;
                ent->crouch_attack_anim_start   = FRAME_BHstandfire1;
                ent->crouch_attack_anim_end     = FRAME_BHstandfire2;
                ent->crouch_move_anim_start     = FRAME_BHdodge1;       // hmmmm, doesnt move feet in this anim
                ent->crouch_move_anim_end       = FRAME_BHdodge1;       // hmmmm, doesnt move feet in this anim
                ent->crouch_pain_anim_start     = FRAME_BHdodgepain1;
                ent->crouch_pain_anim_end       = FRAME_BHdodgepain6;
                ent->crouch_death_anim_start    = FRAME_BHdeath1;
                ent->crouch_death_anim_end      = FRAME_BHdeath10;
                ent->stand_anim_start           = FRAME_BHstand1;
                ent->stand_anim_end             = FRAME_BHstand10;
                ent->attack_anim_start          = FRAME_BHstandfire1;
                ent->attack_anim_end            = FRAME_BHstandfire2;
                ent->run_anim_start             = FRAME_BHrun1;
                ent->run_anim_end               = FRAME_BHrun2;
                ent->run_attack_anim_start      = FRAME_BHstandfire1;
                ent->run_attack_anim_end        = FRAME_BHstandfire2;
                ent->jump_anim_start            = FRAME_BHswim1;        // AWE - NOT CORRECT, BUT WILL DO
                ent->jump_anim_end              = FRAME_BHswim7;        // AWE - NOT CORRECT, BUT WILL DO
                ent->pain_anim_start            = FRAME_BHpain1;
                ent->pain_anim_end              = FRAME_BHpain6;
                ent->death_anim_start           = FRAME_BHdeath1;
                ent->death_anim_end             = FRAME_BHdeath10;
                return;
	}
}

/*----------------------------------------------------/ New Code /--------//
//  This assigns some player properties based on battlesuit.
//------------------------------------------------------------------------*/
void Coven_SuitBox (edict_t *ent)
{
        if (ent->battlesuit == SUIT_MERCURY)
        {
                ent->air_finished = level.time + 24;
                ent->air_max    = 24;
                ent->mass       = MASS_LIGHT;
                ent->model      = "models/armour/mercury/tris.md2";
                return;
        }
        if (ent->battlesuit == SUIT_ZEPPELIN)
        {
                ent->air_finished = level.time + 24;
                ent->air_max    = 24;
                ent->mass       = MASS_LIGHT;
                ent->model      = "models/armour/zeppelin/tris.md2";
                return;
        }
        if (ent->battlesuit == SUIT_GRAVYBURNER)
        {
                ent->gib_health = -61;
                ent->mass       = MASS_MEDIUM;
                ent->model      = "models/armour/gravyburner/tris.md2";
                return;
        }
        if (ent->battlesuit == SUIT_WIRECUTTER)
        {
                ent->gib_health = -61;
                ent->mass       = MASS_MEDIUM;
                ent->model      = "models/armour/wirecutter/tris.md2";
                return;
        }
        if (ent->battlesuit == SUIT_RHINO)
        {
                ent->gib_health = -81;
                ent->mass       = MASS_HEAVY;
                ent->model      = "models/armour/rhino/tris.md2";
                return;
        }
        if (ent->battlesuit == SUIT_SUGARCONNY)
        {
                ent->gib_health = -81;
                ent->mass       = MASS_HEAVY;
                ent->model      = "models/armour/sugarconny/tris.md2";
                return;
        }
        if (ent->battlesuit == SUIT_BULKHEAD)
        {
                ent->fireflags  &= ~FIREFLAG_IGNITE;
                ent->gib_health = -81;
                ent->mass       = MASS_HEAVY;
                ent->model      = "models/armour/bulkhead/tris.md2";
                return;
        }
}

/*----------------------------------------------------/ New Code /--------//
//  This checks if the player has exactly two guns.
//------------------------------------------------------------------------*/
qboolean Coven_CheckGuns (edict_t *ent)
{
        int     weapons = 0;

        if (!ent->client)
                return false;

        if (!ent->client->resp.gunmask)
                return false;

        if (ent->client->resp.gunmask & AB_GRAPPLE)             weapons++;
        if (ent->client->resp.gunmask & AB_ASSAULT_SHOTGUN)     weapons++;
        if (ent->client->resp.gunmask & AB_PARTICLE_BEAM)       weapons++;
        if (ent->client->resp.gunmask & AB_GRENADE_LAUNCHER)    weapons++;
        if (ent->client->resp.gunmask & AB_PROD_ROD)            weapons++;
        if (ent->client->resp.gunmask & AB_PHOTON_CANNON)       weapons++;
        if (ent->client->resp.gunmask & AB_MINIGUN)             weapons++;
        if (ent->client->resp.gunmask & AB_BAZOOKA)             weapons++;
        if (ent->client->resp.gunmask & AB_CARVER)              weapons++;
        if (ent->client->resp.gunmask & AB_FLAMETHROWER)        weapons++;
        if (ent->client->resp.gunmask & AB_AUTO_SHOTGUN)        weapons++;
        if (ent->client->resp.gunmask & AB_MIRV)                weapons++;

        if (ab_arms == ARMS_ALL)
        {       if (weapons == 4)
                        return true;
        }
        else
        {       if (weapons == 2)
                        return true;
        }

        return false;
}

/*----------------------------------------------------/ New Code /--------//
//  This checks if the player's battlesuit is ready to play.
//------------------------------------------------------------------------*/
qboolean Coven_SuitReady (edict_t *ent)
{
/* Entity must be a client. */
        if (!ent->client)
                return false;

/* Player must be a battlesuit. */
        if (!ent->battlesuit)
                return false;

/* Player must have a weapon in each arm. */
        if (!Coven_CheckGuns (ent))
                return false;

/* Player is ready to fight! */
        return true;
}

/*----------------------------------------------------/ New Code /--------//
//  This checks if the player may bypass the line and return to the
//  arena immediately.  This is desirable in case someone dies from
//  a telefrag meltdown between rounds, especially from mass spawning
//  in elimination.
//------------------------------------------------------------------------*/
qboolean Coven_BypassLine (edict_t *ent)
{
// Player can bypass the line only between rounds.
        if (!Coven_NoFire ())
                return false;

// Anyone respawning between rounds in elimination may return.
        if (ab_rules == RULES_ELIMINATION)
                return true;

// Let player return to the arena if a teammate is in.
        if (ab_teams >= 2)
        {
                edict_t *e;
                int     i;

                for (i=0; i<maxclients->value; i++)
                {
                        if (g_edicts[i+1].inuse && g_edicts[i+1].client)
                        {
                                e = &g_edicts[i+1];
                                if (e->client->resp.suitcolor == ent->client->resp.suitcolor)
                                        if (e->RA_playing)
                                                return true;
                        }
                }
        }

// Player must wait in line.
        return false;
}

/*----------------------------------------------------/ New Code /--------//
//  This checks if the player is allowed to fight.
//------------------------------------------------------------------------*/
qboolean Coven_CanFight (edict_t *ent)
{
/* This is irrelevant in single-player. */
        if (!deathmatch->value)
                return true;

/* Battlesuit must be ready. */
        if (!Coven_SuitReady (ent))
                return false;

/* Neutrality is not allowed in team games. */
        if (!ent->client->resp.suitcolor && (ab_teams >= 2))
                return false;

/* Don't let the player spawn just yet.  Make him a spectator first. */
        if (!ent->client->resp.mayplay)
                return false;

/* Is player allowed to fight in the arena yet? */
        if (Coven_ArenaOn() && !ent->RA_playing)
        {
                if (Coven_BypassLine (ent))
                {       // Player is allowed to fight immediately.
                        ent->RA_playing = true;
                        if (ab_rules == RULES_ELIMINATION)
                                Coven_ResetEliminationBonus ();
                        return true;
                }

                // Player must wait for his turn to fight.
                if (!ent->RA_quepos)
                {       // Make sure player is in line.
                        ent->stall_time = 0;
                        Coven_AddToPlayerQue (ent);
                }
                return false;
        }

/* Player is ready to fight! */
        return true;
}


/*=======================/  Power Armor Handling  /=======================*/

/*----------------------------------------------------/ New Code /--------//
//  This checks if the player is carrying power armor of any sort.
//------------------------------------------------------------------------*/
qboolean Coven_HasPowerArmor (edict_t *ent)
{
        if (ent->client->pers.inventory[ITEM_INDEX(FindItem("Power Shield"))])
                return true;

        if (ent->client->pers.inventory[ITEM_INDEX(FindItem("Power Screen"))])
                return true;

        return false;
}

/*----------------------------------------------------/ New Code /--------//
//  This checks if the player is carrying both types of power armor.
//------------------------------------------------------------------------*/
qboolean Coven_OnePowerArmor (edict_t *ent)
{
        if ( ((ent->client->pers.inventory[ITEM_INDEX(FindItem("Power Shield"))]) +
             (ent->client->pers.inventory[ITEM_INDEX(FindItem("Power Screen"))])) == 1 )
                return true;

        return false;
}

/*----------------------------------------------------/ New Code /--------//
//  This lets the player pickup cells.  This is called when player
//  picks up power armor.  If player could not pickup cells before,
//  he or she can now with power armor.
//------------------------------------------------------------------------*/
void Coven_EnableCells (edict_t *ent)
{
        if (ent->client->pers.max_cells)
                return;

        if (ent->client->pers.inventory[ITEM_INDEX(FindItem("Ammo Pack"))])
                ent->client->pers.max_cells = PACK_MAX_CELLS;
        else if (ent->client->pers.inventory[ITEM_INDEX(FindItem("Bandolier"))])
                ent->client->pers.max_cells = BAND_MAX_CELLS;
        else
                ent->client->pers.max_cells = NORMAL_MAX_CELLS;
}

/*----------------------------------------------------/ New Code /--------//
//  This stops the player from picking up cells.  This is called when
//  player drops all power armor.  If player could not pickup cells
//  without power armor, he or she will lose all cells.
//------------------------------------------------------------------------*/
void Coven_DisableCells (edict_t *ent)
{
        if (!ent->client->pers.max_cells)
                return;

        if (ent->client->pers.inventory[ITEM_INDEX(FindItem("Particle Beam"))])
                return;

        if (ent->client->pers.inventory[ITEM_INDEX(FindItem("Photon Cannon"))])
                return;

        if (Coven_HasPowerArmor (ent))
                return;

        ent->client->pers.max_cells = 0;
        ent->client->pers.inventory[ITEM_INDEX(FindItem("Cells"))] = 0;
}


/*=======================/  Equipment Inventory  /=======================*/

/*----------------------------------------------------/ New Code /--------//
//  This gives armor to suits if alternate armor is on.
//------------------------------------------------------------------------*/
void Coven_GiveAltArmor (edict_t *ent)
{
        if (!alt_armor)
                return;

        if ((ent->battlesuit == SUIT_MERCURY) || (ent->battlesuit == SUIT_ZEPPELIN))
        {       /* Light suits get green armor. */
                ent->client->pers.inventory[ITEM_INDEX(FindItem("Jacket Armor"))] = ALT_ARMOR_BASE;
        }
        else if ((ent->battlesuit == SUIT_GRAVYBURNER) || (ent->battlesuit == SUIT_WIRECUTTER))
        {       /* Medium suits get yellow armor. */
                ent->client->pers.inventory[ITEM_INDEX(FindItem("Combat Armor"))] = ALT_ARMOR_BASE;
        }
        else
        {       /* Heavy suits get red armor. */
                ent->client->pers.inventory[ITEM_INDEX(FindItem("Body Armor"))] = ALT_ARMOR_BASE;
        }
}

/*----------------------------------------------------/ New Code /--------//
//  This gives the jetpack plus some fuel to light suits.
//------------------------------------------------------------------------*/
void Coven_GiveJetpack (edict_t *ent)
{
        if (!jetpack_on)
                return;

        if ((ent->battlesuit == SUIT_MERCURY) || (ent->battlesuit == SUIT_ZEPPELIN))
        {       /* Put jetpack in inventory. */
                ent->client->pers.inventory[ITEM_INDEX(FindItem("Jetpack"))] = 1;

                /* Set max fuel capacity for jetpack. */
                if (ent->client->pers.max_slugs < 30)
                        ent->client->pers.max_slugs = 30;

                /* Give a little juice. */
                if (ent->client->pers.inventory[ITEM_INDEX(FindItem ("Fuel"))] < 10)
                        ent->client->pers.inventory[ITEM_INDEX(FindItem ("Fuel"))] = 10;
        }
}

/*----------------------------------------------------/ New Code /--------//
//  This gives back runes (and extra health) the player got while
//  dead or observing.
//------------------------------------------------------------------------*/
void Coven_RestoreRunes (edict_t *ent, int a, int d, int h)
{
        ent->client->pers.inventory[ITEM_INDEX(FindItem("Armour Rune"))] = a;
        ent->client->pers.inventory[ITEM_INDEX(FindItem("Damage Rune"))] = d;
        ent->client->pers.inventory[ITEM_INDEX(FindItem("Health Rune"))] = h;

        if (h)
        {       /* Set extra health. */
                ent->client->pers.max_health = h * rune_scale + DEFAULT_MAXHEALTH;
                ent->client->pers.health     = ent->client->pers.max_health;
        }
}


/*=========================/  Weapons and Ammo  /=========================*/

/*----------------------------------------------------/ New Code /--------//
//  This gives back weapons the player lost.
//------------------------------------------------------------------------*/
void Coven_RestoreGuns (edict_t *ent)
{
        gitem_t *item = NULL;
        gitem_t *last = NULL;

/* Entity must be a client. */
        if (!ent->client)
                return;

/* Abort if player had no weapons. */
        if (!ent->client->resp.gunmask)
                return;

/* Check all the gun bits and give back the weapons. */
        if (ent->client->resp.gunmask & AB_GRAPPLE)
        {       item = FindItem("Grapple");
                ent->client->pers.inventory[ITEM_INDEX(item)] = 1;
                if ((ent->client->resp.lastgun) && (item == ent->client->resp.lastgun))
                        last = item;
        }
        if (ent->client->resp.gunmask & AB_ASSAULT_SHOTGUN)
        {       item = FindItem("Assault Shotgun");
                ent->client->pers.inventory[ITEM_INDEX(item)] = 1;
                Coven_GiveAmmo (ent, item);
                if ((ent->client->resp.lastgun) && (item == ent->client->resp.lastgun))
                        last = item;
        }
        if (ent->client->resp.gunmask & AB_PARTICLE_BEAM)
        {       item = FindItem("Particle Beam");
                ent->client->pers.inventory[ITEM_INDEX(item)] = 1;
                Coven_GiveAmmo (ent, item);
                if ((ent->client->resp.lastgun) && (item == ent->client->resp.lastgun))
                        last = item;
        }
        if (ent->client->resp.gunmask & AB_GRENADE_LAUNCHER)
        {       item = FindItem("Grenade Launcher");
                ent->client->pers.inventory[ITEM_INDEX(item)] = 1;
                Coven_GiveAmmo (ent, item);
                if ((ent->client->resp.lastgun) && (item == ent->client->resp.lastgun))
                        last = item;
        }
        if (ent->client->resp.gunmask & AB_PROD_ROD)
        {       item = FindItem("Prod Rod");
                ent->client->pers.inventory[ITEM_INDEX(item)] = 1;
                if ((ent->client->resp.lastgun) && (item == ent->client->resp.lastgun))
                        last = item;
        }
        if (ent->client->resp.gunmask & AB_PHOTON_CANNON)
        {       item = FindItem("Photon Cannon");
                ent->client->pers.inventory[ITEM_INDEX(item)] = 1;
                Coven_GiveAmmo (ent, item);
                if ((ent->client->resp.lastgun) && (item == ent->client->resp.lastgun))
                        last = item;
        }
        if (ent->client->resp.gunmask & AB_MINIGUN)
        {       item = FindItem("Minigun");
                ent->client->pers.inventory[ITEM_INDEX(item)] = 1;
                Coven_GiveAmmo (ent, item);
                if ((ent->client->resp.lastgun) && (item == ent->client->resp.lastgun))
                        last = item;
        }
        if (ent->client->resp.gunmask & AB_BAZOOKA)
        {       item = FindItem("Bazooka");
                ent->client->pers.inventory[ITEM_INDEX(item)] = 1;
                Coven_GiveAmmo (ent, item);
                if ((ent->client->resp.lastgun) && (item == ent->client->resp.lastgun))
                        last = item;
        }
        if (ent->client->resp.gunmask & AB_CARVER)
        {       item = FindItem("Carver");
                ent->client->pers.inventory[ITEM_INDEX(item)] = 1;
                if ((ent->client->resp.lastgun) && (item == ent->client->resp.lastgun))
                        last = item;
        }
        if (ent->client->resp.gunmask & AB_FLAMETHROWER)
        {       item = FindItem("Flamethrower");
                ent->client->pers.inventory[ITEM_INDEX(item)] = 1;
                Coven_GiveAmmo (ent, item);
                if ((ent->client->resp.lastgun) && (item == ent->client->resp.lastgun))
                        last = item;
        }
        if (ent->client->resp.gunmask & AB_AUTO_SHOTGUN)
        {       item = FindItem("Auto Shotgun");
                ent->client->pers.inventory[ITEM_INDEX(item)] = 1;
                Coven_GiveAmmo (ent, item);
                if ((ent->client->resp.lastgun) && (item == ent->client->resp.lastgun))
                        last = item;
        }
        if (ent->client->resp.gunmask & AB_MIRV)
        {       item = FindItem("MIRV Launcher");
                ent->client->pers.inventory[ITEM_INDEX(item)] = 1;
                Coven_GiveAmmo (ent, item);
                if ((ent->client->resp.lastgun) && (item == ent->client->resp.lastgun))
                        last = item;
        }

/* Make the last weapon held (before death) the current weapon. */
        if (last)
                item = last;

        if (item)
                ent->client->pers.weapon = item;
}

/*----------------------------------------------------/ New Code /--------//
//  Give player some ammo for the weapon 'gun'.
//
//  NOTE:  In normal or joust, give two boxes worth of ammo.
//  In all weapons mode, give only one box of ammo for each weapon.
//------------------------------------------------------------------------*/
void Coven_GiveAmmo (edict_t *ent, gitem_t *gun)
{
/* Give shells for either shotgun. */
        if ( (!strcmp(gun->pickup_name, "Assault Shotgun")) ||
             (!strcmp(gun->pickup_name, "Auto Shotgun")) )
        {       ent->client->pers.max_shells = NORMAL_MAX_SHELLS;
                if (ab_arms == ARMS_ALL)
                        ent->client->pers.inventory[ITEM_INDEX(FindItem ("Shells"))] = 15;
                else
                        ent->client->pers.inventory[ITEM_INDEX(FindItem ("Shells"))] = 30;
                return;
        }

/* Give bullets for the minigun. */
        if (!strcmp(gun->pickup_name, "Minigun"))
        {       ent->client->pers.max_bullets = NORMAL_MAX_BULLETS;
                if (ab_arms == ARMS_ALL)
                        ent->client->pers.inventory[ITEM_INDEX(FindItem ("Bullets"))] = 100;
                else
                        ent->client->pers.inventory[ITEM_INDEX(FindItem ("Bullets"))] = 200;
                return;
        }

/* Give grenades for the grenade launcher. */
        if (!strcmp(gun->pickup_name, "Grenade Launcher"))
        {       ent->client->pers.max_grenades = NORMAL_MAX_GRENADES;
                if (ab_arms == ARMS_ALL)
                        ent->client->pers.inventory[ITEM_INDEX(FindItem ("Grenades"))] = 5;
                else
                        ent->client->pers.inventory[ITEM_INDEX(FindItem ("Grenades"))] = 10;
                return;
        }

/* Give rockets for the bazooka or MIRV. */
        if ( (!strcmp(gun->pickup_name, "Bazooka")) ||
             (!strcmp(gun->pickup_name, "MIRV Launcher")) )
        {       ent->client->pers.max_rockets = NORMAL_MAX_ROCKETS;
                if (ab_arms == ARMS_ALL)
                        ent->client->pers.inventory[ITEM_INDEX(FindItem ("Rockets"))] = 5;
                else
                        ent->client->pers.inventory[ITEM_INDEX(FindItem ("Rockets"))] = 10;
                return;
        }

/* Give cells for the particle beam or photon cannon. */
        if ( (!strcmp(gun->pickup_name, "Particle Beam")) ||
             (!strcmp(gun->pickup_name, "Photon Cannon")) )
        {       ent->client->pers.max_cells = NORMAL_MAX_CELLS;
                if (ab_arms == ARMS_ALL)
                        ent->client->pers.inventory[ITEM_INDEX(FindItem ("Cells"))] = 50;
                else
                        ent->client->pers.inventory[ITEM_INDEX(FindItem ("Cells"))] = 100;
                return;
        }

/* Give fuel/slugs for the flamethrower. */
        if (!strcmp(gun->pickup_name, "Flamethrower"))
        {       ent->client->pers.max_slugs = NORMAL_MAX_FUEL;
                if (ab_arms == ARMS_ALL)
                        ent->client->pers.inventory[ITEM_INDEX(FindItem ("Fuel"))] = 20;
                else
                        ent->client->pers.inventory[ITEM_INDEX(FindItem ("Fuel"))] = 40;
                return;
        }
}

/*----------------------------------------------------/ New Code /--------//
//  Set the mask used to give weapons to the player.
//------------------------------------------------------------------------*/
void Coven_SetGunmask (edict_t *ent, gitem_t *gun)
{
        if (!strcmp(gun->pickup_name, "Grapple"))
                ent->client->resp.gunmask |= AB_GRAPPLE;
        else if (!strcmp(gun->pickup_name, "Assault Shotgun"))
                ent->client->resp.gunmask |= AB_ASSAULT_SHOTGUN;
        else if (!strcmp(gun->pickup_name, "Particle Beam"))
                ent->client->resp.gunmask |= AB_PARTICLE_BEAM;
        else if (!strcmp(gun->pickup_name, "Grenade Launcher"))
                ent->client->resp.gunmask |= AB_GRENADE_LAUNCHER;
        else if (!strcmp(gun->pickup_name, "Prod Rod"))
                ent->client->resp.gunmask |= AB_PROD_ROD;
        else if (!strcmp(gun->pickup_name, "Photon Cannon"))
                ent->client->resp.gunmask |= AB_PHOTON_CANNON;
        else if (!strcmp(gun->pickup_name, "Minigun"))
                ent->client->resp.gunmask |= AB_MINIGUN;
        else if (!strcmp(gun->pickup_name, "Bazooka"))
                ent->client->resp.gunmask |= AB_BAZOOKA;
        else if (!strcmp(gun->pickup_name, "Carver"))
                ent->client->resp.gunmask |= AB_CARVER;
        else if (!strcmp(gun->pickup_name, "Flamethrower"))
                ent->client->resp.gunmask |= AB_FLAMETHROWER;
        else if (!strcmp(gun->pickup_name, "Auto Shotgun"))
                ent->client->resp.gunmask |= AB_AUTO_SHOTGUN;
        else if (!strcmp(gun->pickup_name, "MIRV Launcher"))
                ent->client->resp.gunmask |= AB_MIRV;
}

/*----------------------------------------------------/ New Code /--------//
//  This wipes out all guns and ammo from the player's inventory.
//------------------------------------------------------------------------*/
void Coven_RemoveGuns (edict_t *ent)
{
/* Take away jetpack. */
        ent->client->pers.inventory[ITEM_INDEX(FindItem ("Jetpack"))] = 0;

/* Take away all weapons. */
        ent->client->pers.inventory[ITEM_INDEX(FindItem ("Grapple"))] = 0;
        ent->client->pers.inventory[ITEM_INDEX(FindItem ("Assault Shotgun"))] = 0;
        ent->client->pers.inventory[ITEM_INDEX(FindItem ("Particle Beam"))] = 0;
        ent->client->pers.inventory[ITEM_INDEX(FindItem ("Grenade Launcher"))] = 0;
        ent->client->pers.inventory[ITEM_INDEX(FindItem ("Prod Rod"))] = 0;
        ent->client->pers.inventory[ITEM_INDEX(FindItem ("Photon Cannon"))] = 0;
        ent->client->pers.inventory[ITEM_INDEX(FindItem ("Minigun"))] = 0;
        ent->client->pers.inventory[ITEM_INDEX(FindItem ("Bazooka"))] = 0;
        ent->client->pers.inventory[ITEM_INDEX(FindItem ("Carver"))] = 0;
        ent->client->pers.inventory[ITEM_INDEX(FindItem ("Flamethrower"))] = 0;
        ent->client->pers.inventory[ITEM_INDEX(FindItem ("Auto Shotgun"))] = 0;
        ent->client->pers.inventory[ITEM_INDEX(FindItem ("MIRV Launcher"))] = 0;

/* Take away all the ammo too. */
        ent->client->pers.inventory[ITEM_INDEX(FindItem ("Shells"))] = 0;
        ent->client->pers.inventory[ITEM_INDEX(FindItem ("Bullets"))] = 0;
        ent->client->pers.inventory[ITEM_INDEX(FindItem ("Grenades"))] = 0;
        ent->client->pers.inventory[ITEM_INDEX(FindItem ("Rockets"))] = 0;
        ent->client->pers.inventory[ITEM_INDEX(FindItem ("Cells"))] = 0;
        ent->client->pers.inventory[ITEM_INDEX(FindItem ("Fuel"))] = 0;

/* Reset maximum ammo capacity. */
        ent->client->pers.max_bullets  = 0;
        ent->client->pers.max_shells   = 0;
        ent->client->pers.max_rockets  = 0;
        ent->client->pers.max_grenades = 0;
        ent->client->pers.max_cells    = 0;
        ent->client->pers.max_slugs    = 0;

/* The player had no weapons before. */
        ent->client->resp.gunmask = 0;
        ent->client->resp.lastgun = NULL;
}

/*----------------------------------------------------/ New Code /--------//
//  This automatically gives all four weapons to a suit.
//------------------------------------------------------------------------*/
void Coven_FourWeapons (edict_t *ent)
{
        if ((ent->battlesuit == SUIT_MERCURY) || (ent->battlesuit == SUIT_ZEPPELIN))
                ent->client->resp.gunmask = (AB_GRAPPLE | AB_ASSAULT_SHOTGUN | AB_PARTICLE_BEAM | AB_GRENADE_LAUNCHER);
        else if ((ent->battlesuit == SUIT_GRAVYBURNER) || (ent->battlesuit == SUIT_WIRECUTTER))
                ent->client->resp.gunmask = (AB_PROD_ROD | AB_PHOTON_CANNON | AB_MINIGUN | AB_BAZOOKA);
        else
                ent->client->resp.gunmask = (AB_CARVER | AB_FLAMETHROWER | AB_AUTO_SHOTGUN | AB_MIRV);
}


/*=======================/  Player Preparation  /=======================*/

/*----------------------------------------------------/ New Code /--------//
//  This starts the player with a weapon in hand.
//------------------------------------------------------------------------*/
void Coven_StartWeapon (edict_t *ent)
{
//--------------------//
        /* Copied from ChangeWeapon() in p_weapons.c. */
	ent->client->pers.lastweapon = ent->client->pers.weapon;
	ent->client->pers.weapon = ent->client->newweapon;
	ent->client->newweapon = NULL;
	ent->client->machinegun_shots = 0;

// Battlesuits do NOT support VWeps!!!

	if (ent->client->pers.weapon && ent->client->pers.weapon->ammo)
		ent->client->ammo_index = ITEM_INDEX(FindItem(ent->client->pers.weapon->ammo));
	else
		ent->client->ammo_index = 0;

	if (!ent->client->pers.weapon)
	{	// dead
		ent->client->ps.gunindex = 0;
		return;
	}

	ent->client->weaponstate = WEAPON_ACTIVATING;
	ent->client->ps.gunframe = 0;
	ent->client->ps.gunindex = gi.modelindex(ent->client->pers.weapon->view_model);
//--------------------//

        ent->client->pers.selected_item = ITEM_INDEX(ent->client->pers.weapon);

        /* Set client speed. */
        if (deathmatch->value && (ent->mass >= MASS_HEAVY))
        {       // Lower speed for heavy suits.
                gi.WriteByte (svc_stufftext);
                gi.WriteString ("cl_forwardspeed 100; cl_sidespeed 100; cl_upspeed 100");
                gi.unicast (ent, true);
        }
        else
        {       // Standard speed for lighter suits or in single-player.
                gi.WriteByte (svc_stufftext);
                gi.WriteString ("cl_forwardspeed 200; cl_sidespeed 200; cl_upspeed 200");
                gi.unicast (ent, true);
        }
}


/*==========================/  Dead Drop Ammo  /==========================*/

extern void Touch_Item
 (edict_t *ent, edict_t *other, cplane_t *plane, csurface_t *surf);

/*----------------------------------------------------/ New Code /--------//
//  This returns what kind of ammo the other second weapon uses.
//------------------------------------------------------------------------*/
gitem_t *Coven_FindOtherAmmo (edict_t *ent)
{
        char    *gunname;

/* Entity must be a client. */
        if (!ent->client)
                return NULL;

/* Abort if player has no weapons. */
        if (!ent->client->resp.gunmask)
                return NULL;

        if (!ent->client->pers.weapon)
                return NULL;

/* If all weapons are on, check for special case. */
        if (ab_arms == ARMS_ALL)
        {       /* Don't drop ammo for melee arm. */
                if (ent->client->pers.weapon->ammo)
                        return NULL;

                /* Drop ammo for weapon last used. */
                if (ent->client->pers.lastweapon && ent->client->pers.lastweapon->ammo)
                {
                        gitem_t         *item;

                        item = FindItem (ent->client->pers.lastweapon->ammo);
                        if (ent->client->pers.inventory[ITEM_INDEX(item)])
                                return item;
                }
        }

/* Check gun bits, find the second weapon, and return ammo it uses. */
        gunname = ent->client->pers.weapon->pickup_name;

        if ( (ent->client->resp.gunmask & AB_ASSAULT_SHOTGUN) && (stricmp(gunname, "Assault Shotgun")) )
                return FindItem("Shells");
        if ( (ent->client->resp.gunmask & AB_PARTICLE_BEAM) && (stricmp(gunname, "Particle Beam")) )
                return FindItem("Cells");
        if ( (ent->client->resp.gunmask & AB_GRENADE_LAUNCHER) && (stricmp(gunname, "Grenade Launcher")) )
                return FindItem("Grenades");

        if ( (ent->client->resp.gunmask & AB_PHOTON_CANNON) && (stricmp(gunname, "Photon Cannon")) )
                return FindItem("Cells");
        if ( (ent->client->resp.gunmask & AB_MINIGUN) && (stricmp(gunname, "Minigun")) )
                return FindItem("Bullets");
        if ( (ent->client->resp.gunmask & AB_BAZOOKA) && (stricmp(gunname, "Bazooka")) )
                return FindItem("Rockets");

        if ( (ent->client->resp.gunmask & AB_FLAMETHROWER) && (stricmp(gunname, "Flamethrower")) )
                return FindItem("Fuel");
        if ( (ent->client->resp.gunmask & AB_AUTO_SHOTGUN) && (stricmp(gunname, "Auto Shotgun")) )
                return FindItem("Shells");
        if ( (ent->client->resp.gunmask & AB_MIRV) && (stricmp(gunname, "MIRV Launcher")) )
                return FindItem("Rockets");

        return NULL;
}

/*----------------------------------------------------/ New Code /--------//
//  This causes the player to drop fuel if player had a jetpack,
//  or cells if player had power armor.  Fuel is checked first
//  then cells.  All such ammo is dropped to the rear of the player.
//------------------------------------------------------------------------*/
static void Coven_DropThirdAmmo
(edict_t *ent, qboolean fuel, qboolean cells)
{
	gitem_t		*item;
	edict_t		*drop;
        int             ammocount;
        int             ammoindex;

        // Check for third ammo to drop.
        item = NULL;

        // Check for Jetpack.
        if (!fuel && ent->client->pers.inventory[ITEM_INDEX(FindItem ("Jetpack"))])
        {
                item = FindItem ("Fuel");
                ammoindex = ITEM_INDEX(item);
                ammocount = ent->client->pers.inventory[ammoindex];
                if (ammocount < 1)
                        item = NULL;
        }

        // Fuel
        if (item)
        {       // Throw fuel behind player.
                ent->client->v_angle[YAW] += 180;
                drop = Drop_Item (ent, item);
                ent->client->v_angle[YAW] -= 180;
                drop->spawnflags = DROPPED_PLAYER_ITEM;

                if (ammocount >= item->quantity)
                        drop->count = item->quantity;
                else
                        drop->count = ammocount;
                ent->client->pers.inventory[ammoindex] -= drop->count;

                return;
        }

        // Check for Power armor.
        if (!cells && Coven_HasPowerArmor (ent))
        {
                item = FindItem ("Cells");
                ammoindex = ITEM_INDEX(item);
                ammocount = ent->client->pers.inventory[ammoindex];
                if (ammocount < 1)
                        item = NULL;
        }

        // Cells
        if (item)
        {       // Throw cells behind player.
                ent->client->v_angle[YAW] += 180;
                drop = Drop_Item (ent, item);
                ent->client->v_angle[YAW] -= 180;
                drop->spawnflags = DROPPED_PLAYER_ITEM;

                if (ammocount >= item->quantity)
                        drop->count = item->quantity;
                else
                        drop->count = ammocount;
                ent->client->pers.inventory[ammoindex] -= drop->count;

                return;
        }
}

/*----------------------------------------------------/ New Code /--------//
//  This causes a slain player to drop ammo.  Up to three distinct
//  ammo boxes may be dropped.
//------------------------------------------------------------------------*/
void Coven_DeadDropAmmo (edict_t *ent)
{
	gitem_t		*item;
	edict_t		*drop;
	float		spread;
        int             ammocount;
        int             ammoindex;
        qboolean        cells = false;
        qboolean        fuel = false;

        if (!ent->client->pers.weapon)
        {       // No weapon?  Just check if other items use ammo.
                Coven_DropThirdAmmo (ent, false, false);
                return;
        }

        // Initialize some info.
        spread = 30;
        if (ent->client->pers.hand == LEFT_HANDED)
                spread = 0 - spread;

        // Get ammo from current weapon.
        if (ent->client->pers.weapon->ammo)
        {
                item = FindItem (ent->client->pers.weapon->ammo);
                ammoindex = ent->client->ammo_index;
                ammocount = ent->client->pers.inventory[ammoindex];
                if (ammocount < 1)
                        item = NULL;
        }
        else
                item = NULL;

        if (item)
        {       // Drop ammo from current weapon.
                ent->client->v_angle[YAW] -= spread;
                drop = Drop_Item (ent, item);
                ent->client->v_angle[YAW] += spread;
                drop->spawnflags = DROPPED_PLAYER_ITEM;

                // Drop ammo and subtract from inventory.
                if (ammocount >= item->quantity)
                        drop->count = item->quantity;
                else
                        drop->count = ammocount;
                ent->client->pers.inventory[ammoindex] -= drop->count;

                // Was the ammo cells or fuel?  If so, remember that.
                if (!strcmp(item->classname, "ammo_cells"))
                        cells = true;
                else if (!strcmp(item->classname, "ammo_slugs"))
                        fuel = true;
        }

        // Check what ammo the other weapon uses.
        if (item = Coven_FindOtherAmmo (ent))
        {       // Check if player had ammo.
                ammoindex = ITEM_INDEX (item);
                ammocount = ent->client->pers.inventory[ammoindex];
                if (ammocount < 1)
                        item = NULL;
        }

        // Do we still have some ammo?
        if (item)
        {       // Drop other ammo.
                ent->client->v_angle[YAW] += spread;
                drop = Drop_Item (ent, item);
                ent->client->v_angle[YAW] -= spread;
                drop->spawnflags = DROPPED_PLAYER_ITEM;

                // Drop ammo and subtract from inventory.
                if (ammocount >= item->quantity)
                        drop->count = item->quantity;
                else
                        drop->count = ammocount;
                ent->client->pers.inventory[ammoindex] -= drop->count;

                // Was the ammo cells or fuel?  If so, remember that.
                if (!strcmp(item->classname, "ammo_cells"))
                        cells = true;
                else if (!strcmp(item->classname, "ammo_slugs"))
                        fuel = true;
        }

        // Check for ammo used by items.
        if (!fuel || !cells)
                Coven_DropThirdAmmo (ent, fuel, cells);
}


/*===========================/  END OF FILE  /===========================*/
