/*
    Copyright (C) 1999 by Seth Galbraith

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "sysdef.h"
#include "squawk.h"
#include "sqkview.h"

#include "csengine/world.h"
#include "csengine/sector.h"
#include "csengine/camera.h"
#include "csengine/csview.h"
#include "csengine/light.h"
#include "csengine/polygon.h"
#include "csengine/cssprite.h"

#include "csobject/nameobj.h"
#include "csparser/crossbld.h"
#include "csparser/csloader.h"
#include "cssys/common/system.h"
#include "cstools/simpcons.h"
#include "csutil/impexp.h"
#include "csutil/inifile.h"
#include "csutil/vfs.h"

#include "igraph2d.h"
#include "igraph3d.h"
#include "itxtmgr.h"

// increment version number by one with every public release
#define SQK_VERSION "Squawk game version 16"

// size of the sky box
// @@@ the sky box should be practically infinite in size ...
// @@@ orthographically projected with maximum z values
#define SQK_SKYBOX_SIZE 50

sqkSystem* Sys = NULL;

void cleanup ()
{
  pprintf ("Cleaning up...\n");
  delete Sys;
  pprintf_close();
}

void message_cleanup_and_exit (char * s)
{
    Sys->Printf (MSG_FATAL_ERROR, s);
    cleanup ();
    exit (1);
}

void debug_dump ()
{
}

sqkSystem::sqkSystem ()
{
  frameTime = 0;
  world = NULL;
}

sqkSystem::~sqkSystem ()
{
  delete world;
}

void sqkSystem::CreateSkyBox ()
{
  skybox = world->NewSector ();
  csNameObject::AddName (*skybox, "skybox");
  csTextureHandle* tm_up    = csLoader::LoadTexture (world, "up",    "sqk_u.jpg");
  csTextureHandle* tm_down  = csLoader::LoadTexture (world, "down",  "sqk_d.jpg");
  csTextureHandle* tm_left  = csLoader::LoadTexture (world, "left",  "sqk_l.jpg");
  csTextureHandle* tm_right = csLoader::LoadTexture (world, "right", "sqk_r.jpg");
  csTextureHandle* tm_front = csLoader::LoadTexture (world, "front", "sqk_f.jpg");
  csTextureHandle* tm_back  = csLoader::LoadTexture (world, "back",  "sqk_b.jpg");
  csPolygon3D* p;

  // (up) ceiling
  p = skybox->NewPolygon (tm_up);
  p->AddVertex (-SQK_SKYBOX_SIZE,  SQK_SKYBOX_SIZE, -SQK_SKYBOX_SIZE);
  p->AddVertex ( SQK_SKYBOX_SIZE,  SQK_SKYBOX_SIZE, -SQK_SKYBOX_SIZE);
  p->AddVertex ( SQK_SKYBOX_SIZE,  SQK_SKYBOX_SIZE,  SQK_SKYBOX_SIZE);
  p->AddVertex (-SQK_SKYBOX_SIZE,  SQK_SKYBOX_SIZE,  SQK_SKYBOX_SIZE);
  p->SetTextureSpace (p->Vobj (0), p->Vobj (1), SQK_SKYBOX_SIZE * 2);
  p->SetFlags (CS_POLY_MIPMAP | CS_POLY_LIGHTING, 0);

  // (down) floor
  p = skybox->NewPolygon (tm_down);
  p->AddVertex (-SQK_SKYBOX_SIZE, -SQK_SKYBOX_SIZE,  SQK_SKYBOX_SIZE);
  p->AddVertex ( SQK_SKYBOX_SIZE, -SQK_SKYBOX_SIZE,  SQK_SKYBOX_SIZE);
  p->AddVertex ( SQK_SKYBOX_SIZE, -SQK_SKYBOX_SIZE, -SQK_SKYBOX_SIZE);
  p->AddVertex (-SQK_SKYBOX_SIZE, -SQK_SKYBOX_SIZE, -SQK_SKYBOX_SIZE);
  p->SetTextureSpace (p->Vobj (0), p->Vobj (1), SQK_SKYBOX_SIZE * 2);
  p->SetFlags (CS_POLY_MIPMAP | CS_POLY_LIGHTING, 0);

  // (left) negative x wall
  p = skybox->NewPolygon (tm_left);
  p->AddVertex (-SQK_SKYBOX_SIZE,  SQK_SKYBOX_SIZE, -SQK_SKYBOX_SIZE);
  p->AddVertex (-SQK_SKYBOX_SIZE,  SQK_SKYBOX_SIZE,  SQK_SKYBOX_SIZE);
  p->AddVertex (-SQK_SKYBOX_SIZE, -SQK_SKYBOX_SIZE,  SQK_SKYBOX_SIZE);
  p->AddVertex (-SQK_SKYBOX_SIZE, -SQK_SKYBOX_SIZE, -SQK_SKYBOX_SIZE);
  p->SetTextureSpace (p->Vobj (0), p->Vobj (1), SQK_SKYBOX_SIZE * 2);
  p->SetFlags (CS_POLY_MIPMAP | CS_POLY_LIGHTING, 0);

  // (right) positive x wall 
  p = skybox->NewPolygon (tm_right);
  p->AddVertex ( SQK_SKYBOX_SIZE,  SQK_SKYBOX_SIZE,  SQK_SKYBOX_SIZE);
  p->AddVertex ( SQK_SKYBOX_SIZE,  SQK_SKYBOX_SIZE, -SQK_SKYBOX_SIZE);
  p->AddVertex ( SQK_SKYBOX_SIZE, -SQK_SKYBOX_SIZE, -SQK_SKYBOX_SIZE);
  p->AddVertex ( SQK_SKYBOX_SIZE, -SQK_SKYBOX_SIZE,  SQK_SKYBOX_SIZE);
  p->SetTextureSpace (p->Vobj (0), p->Vobj (1), SQK_SKYBOX_SIZE * 2);
  p->SetFlags (CS_POLY_MIPMAP | CS_POLY_LIGHTING, 0);

  // (front) positive z wall
  p = skybox->NewPolygon (tm_front);
  p->AddVertex (-SQK_SKYBOX_SIZE,  SQK_SKYBOX_SIZE,  SQK_SKYBOX_SIZE);
  p->AddVertex ( SQK_SKYBOX_SIZE,  SQK_SKYBOX_SIZE,  SQK_SKYBOX_SIZE);
  p->AddVertex ( SQK_SKYBOX_SIZE, -SQK_SKYBOX_SIZE,  SQK_SKYBOX_SIZE);
  p->AddVertex (-SQK_SKYBOX_SIZE, -SQK_SKYBOX_SIZE,  SQK_SKYBOX_SIZE);
  p->SetTextureSpace (p->Vobj (0), p->Vobj (1), SQK_SKYBOX_SIZE * 2);
  p->SetFlags (CS_POLY_MIPMAP | CS_POLY_LIGHTING, 0);

  // (back) negative z wall
  p = skybox->NewPolygon (tm_back);
  p->AddVertex ( SQK_SKYBOX_SIZE,  SQK_SKYBOX_SIZE, -SQK_SKYBOX_SIZE);
  p->AddVertex (-SQK_SKYBOX_SIZE,  SQK_SKYBOX_SIZE, -SQK_SKYBOX_SIZE);
  p->AddVertex (-SQK_SKYBOX_SIZE, -SQK_SKYBOX_SIZE, -SQK_SKYBOX_SIZE);
  p->AddVertex ( SQK_SKYBOX_SIZE, -SQK_SKYBOX_SIZE, -SQK_SKYBOX_SIZE);
  p->SetTextureSpace (p->Vobj (0), p->Vobj (1), SQK_SKYBOX_SIZE * 2);
  p->SetFlags (CS_POLY_MIPMAP | CS_POLY_LIGHTING, 0);
}

void sqkSystem::AddSprite (char* mesh, char* skin, char* skinname,
  float x, float y, float z, float scale, bool lit)
{
  csCrossBuild_SpriteTemplateFactory builder;
  csMatrix3 m;

  csLoader::LoadTexture (world, skinname, skin);
  converter * filedata = new converter;
  filedata->ivcon (mesh);
  csSpriteTemplate * t = (csSpriteTemplate*)builder.CrossBuild(*filedata);
  CHK (delete filedata);
  t->SetTexture (world->GetTextures (), skinname);
  world->sprite_templates.Push (t);

  csSpriteAction * action = t->AddAction ();
  for (int i=0; i < t->GetNumFrames (); i++)
    action->AddFrame (t->GetFrame (i), 100);
  action->SetName ("all");

  csSprite3D * s = t->NewSprite ();
  s->MoveToSector (skybox);  s->MoveTo (x, y, z);
  s->SetAction ("all");      s->SetFrame (0);
  m.Identity();  m=m*scale;  s->SetTransform (m);
  if (lit) s->DeferUpdateLighting (CS_NLIGHT_STATIC|CS_NLIGHT_DYNAMIC, 10);
  sprites.Push (s);
}

void sqkSystem::CreateViews ()
{
  sqkView * v;

  // upper view
  v = new sqkView ();
  v->SetSize (FrameWidth, FrameHeight);
  v->AddView (new csView (world, piG3D));
  v->AddVertex (0, 0.006, 0.992);  v->AddVertex (0, 0.594, 0.992);
  v->AddVertex (0, 0.594, 0.708);  v->AddVertex (0, 0.444, 0.508);
  v->AddVertex (0, 0.006, 0.508);
  v->AddView (new csView (world, piG3D));
  v->AddVertex (1, 0.594, 0.992);  v->AddVertex (1, 0.994, 0.992);
  v->AddVertex (1, 0.994, 0.708);  v->AddVertex (1, 0.594, 0.708);
  v->SetSector (skybox);
  v->SetPosition (csVector3 (0, 0, 0));
  v->SetCenter (0.5, 0.8);
  v->SetFOV (1.0);
  views.Push (v);

  // lower left view
  v = new sqkView ();
  v->SetSize (FrameWidth, FrameHeight);
  v->AddView (new csView (world, piG3D));
  v->AddVertex (0, 0.006, 0.492);  v->AddVertex (0, 0.444, 0.492);
  v->AddVertex (0, 0.444, 0.008);  v->AddVertex (0, 0.006, 0.008);
  v->SetSector (skybox);
  v->SetPosition (csVector3 (0, 0, 0));
  v->SetCenter (0.22, 0.25);
  // @@@ when I set the FOV to 0.5 and look at the sprite ...
  // @@@ I get an FPE crash. (djgpp, nasm, optimize)
  // v->SetFOV (0.5);
  v->SetFOV (1.0);
  views.Push (v);

  // lower right view
  v = new sqkView ();
  v->SetSize (FrameWidth, FrameHeight);
  v->AddView (new csView (world, piG3D));
  v->AddVertex (0, 0.602, 0.692);  v->AddVertex (0, 0.994, 0.692);
  v->AddVertex (0, 0.995, 0.008);  v->AddVertex (0, 0.456, 0.008);
  v->AddVertex (0, 0.456, 0.495);
  v->SetSector (skybox);
  v->SetPosition (csVector3 (0, 0, 0));
  v->SetCenter (0.75, 0.35);
  // @@@ when I set the FOV to 0.5 and look at the sprite ...
  // @@@ I get an FPE crash. (djgpp, nasm, optimize)
  // v->SetFOV (0.5);
  v->SetFOV (1.0);
  v->gravity = false;
  views.Push (v);
}

void sqkSystem::InitApp ()
{
  if (!Open (SQK_VERSION))
    message_cleanup_and_exit ("Error opening system!\n");

  ITextureManager* txtmgr;
  piG3D->GetTextureManager (&txtmgr);
  txtmgr->Initialize ();
  txtmgr->Prepare ();
  txtmgr->AllocPalette ();

  csSimpleConsole* con = new csSimpleConsole (Sys->Config);
  con->SetupColors (txtmgr);
  con->SetTransparent (0);
  con->SetMaxLines (1000);       // Some arbitrarily high value.
  Sys->Console = con;

  Sys->DemoReady = true;

  Printf (MSG_INITIALIZATION, "%s\n", SQK_VERSION);

  world->EnableLightingCache (false);
  world->Initialize (GetISystemFromSystem (this), piG3D, NULL, Sys->Vfs);

  // Change to virtual Squawk's directory
  Sys->Vfs->ChDir (Sys->Config->GetStr ("Squawk", "DATA", "/data/squawk"));

  CreateSkyBox ();
  skybox->AddLight (new csStatLight (0,0,0, SQK_SKYBOX_SIZE*2, 1,1,1, false));

//  AddSprite ("ghost.mdl",    "ghost.bmp",    "ghost",    0,0,5,  1.0, false);
//  AddSprite ("hooded2.mdl",  "hooded2.png",  "hooded2",  0,0,-5, 0.5, true);
  AddSprite ("ghosty.mdl",   "ghosty.png",   "ghosty",   -5,0,5,  1.0, false);
  AddSprite ("santa.mdl",    "santa.png",    "santa",    -5,0,-5, 1.0, false);
  AddSprite ("elcape_o.mdl", "elcape_o.png", "elcape_o", -10,0,0, 0.5, false);

  world->Prepare (piG3D);
  txtmgr->AllocPalette ();
  con->SetupColors (txtmgr);
  if (SUCCEEDED (piG2D->BeginDraw()))
  {
    piG2D->ClearAll (0);
    piG2D->FinishDraw ();
  }

  CreateViews ();
}

void sqkSystem::eatkeypress (int key, bool shift, bool alt, bool ctrl)
{
  (void)shift; (void)alt; (void)ctrl;

  switch (key)
  {
    case CSKEY_ESC : Shutdown = true; break;

    case '1' : GetSprite(0)->SetAction("run");    break;
    case '2' : GetSprite(0)->SetAction("stand");  break;
    case '3' : GetSprite(0)->SetAction("pain");   break;
    case '4' : GetSprite(0)->SetAction("death");  break;
    case '5' : GetSprite(0)->SetAction("attack"); break;
    case '6' : GetSprite(0)->SetAction("jump");   break;

    case CSKEY_RIGHT : GetView(0)->Rotate (VEC_ROT_RIGHT,0.03); break;
    case CSKEY_LEFT  : GetView(0)->Rotate (VEC_ROT_LEFT, 0.03); break;
    case CSKEY_UP    : GetView(0)->Rotate (VEC_TILT_UP,  0.03); break;
    case CSKEY_DOWN  : GetView(0)->Rotate (VEC_TILT_DOWN,0.03); break;

    case CSKEY_INS   : GetView(0)->Rotate (VEC_TILT_LEFT, 0.03); break;
    case CSKEY_DEL   : GetView(0)->Rotate (VEC_TILT_RIGHT,0.03); break;

    case CSKEY_HOME  : GetView(0)->Rotate (csVector3(-1,-1,0),0.02); break;
    case CSKEY_END   : GetView(0)->Rotate (csVector3( 1,-1,0),0.02); break;
    case CSKEY_PGUP  : GetView(0)->Rotate (csVector3(-1, 1,0),0.02); break;
    case CSKEY_PGDN  : GetView(0)->Rotate (csVector3( 1, 1,0),0.02); break;

    case 'd' : GetView(1)->Rotate (VEC_ROT_RIGHT,0.03); break;
    case 'a' : GetView(1)->Rotate (VEC_ROT_LEFT, 0.03); break;
    case 'w' : GetView(1)->Rotate (VEC_TILT_UP,  0.03); break;
    case 's' : GetView(1)->Rotate (VEC_TILT_DOWN,0.03); break;

    case 'l' : GetView(2)->Rotate (VEC_ROT_RIGHT,0.03); break;
    case 'j' : GetView(2)->Rotate (VEC_ROT_LEFT, 0.03); break;
    case 'i' : GetView(2)->Rotate (VEC_TILT_UP,  0.03); break;
    case 'k' : GetView(2)->Rotate (VEC_TILT_DOWN,0.03); break;
  }
}

void sqkSystem::NextFrame (long elapsed_time, long current_time)
{
  int i;

  SysSystemDriver::NextFrame (elapsed_time, current_time);

  // Handle all events.

  csEvent *Event;
  while ((Event = EventQueue->Get ()))
  {
    switch (Event->Type)
    {
      case csevKeyDown:
        eatkeypress (Event->Key.Code,
          Event->Key.ShiftKeys & CSMASK_SHIFT,
          Event->Key.ShiftKeys & CSMASK_ALT,
          Event->Key.ShiftKeys & CSMASK_CTRL);
        break;
      case csevMouseMove: break;
      case csevMouseDown: break;
      case csevMouseUp:   break;
    }
    delete Event;
  }

  // animate sprite

  if (frameTime == 0)  frameTime = current_time + 50;
  if (current_time > frameTime)
  {
    frameTime = current_time + 50;
    for (i=0; i < sprites.Length(); i++)
    {
      int frame = GetSprite(i)->GetCurFrame();
      if (frame >= GetSprite(i)->GetNumFrames()-1) GetSprite(i)->SetFrame(0);
      else GetSprite(i)->SetFrame(frame + 1);
    }
  }

  // Draw the views

  if (piG3D->BeginDraw (CSDRAW_3DGRAPHICS) != S_OK) return;
  for (i=0; i < views.Length(); i++)
    GetView(i)->Draw ();
  piG3D->FinishDraw ();
  piG3D->Print (NULL);
}

int main (int argc, char* argv[])
{
  Sys = new sqkSystem ();
  Sys->world = new csWorld ();
  if (!Sys->Initialize(argc,argv,"squawk.cfg","VFS.cfg",Sys->world->GetEngineConfigCOM()))
  {
    Sys->Printf (MSG_FATAL_ERROR, "Error initializing system!\n");
    cleanup ();
    exit (1);
  }

  Sys->InitApp ();
  Sys->Loop ();
  cleanup ();
  return 0;
}
