/*
  Crystal Space QuakeMDL convertor : spr writer
  Copyright (C) 1998 by Jorrit Tyberghein
  Written by Nathaniel Saint Martin aka NooTe <noote@bigfoot.com>

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.

  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the Free
  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "mdl.h"

bool WritePCX (char *name, unsigned char *data, unsigned char *pal, int width, int height);

bool WriteSPR(char *spritename, Mdl mdl, float scaleMdl, int delayMdl, float positionMdlX, float positionMdlY, float positionMdlZ, bool actionNamingMdl)
{
  FILE *f;
  char *spritefilename;
  int i = 0, j = 0, k = 0, v = 0, vertex = 0;

  if(spritename==NULL || strlen(spritename)==0)
  {
    fprintf(stderr, "void file name\n");
    return false;
  }

  spritefilename = new char [strlen(spritename)+5];
  strcpy(spritefilename, spritename);
  strcat(spritefilename, ".spr");

  // generate skin texture
  if(mdl.nbskins>0)
  {
    fprintf(stdout, "Generate Skin texture file\n");
    fprintf(stdout, "\tSkin size = %dx%d (must resize to power of 2 width and height)\n", mdl.skinwidth, mdl.skinheight);
    for(i=0; i<mdl.nbskins; i++)
    {
      char *skinfilename;
      skinfilename = new char [strlen(spritename)+10];
      if(mdl.nbskins>1) sprintf(skinfilename, "%s%d.pcx", spritename, i);
      else sprintf(skinfilename, "%s.pcx", spritename);
      
      FILE *fs;
      unsigned char palette[768];
      
      // read quake palette
      if((fs=fopen("palette.lmp","rb"))!=NULL)
      {
        if(fread(&palette, 768, 1, fs) != 1)
          fprintf(stderr, "Error then reading palette.lmp file.\n");
        fclose(fs);
      }
      else // else generate a gray palette
      {
        fprintf(stderr, "Warning : don't find palette.lmp file !\nusing palette of gray...\n");
        unsigned char *p=&palette[0];
        for(i=0; i<256; i++)
        {
          *p++=i/4;
          *p++=i/4;
          *p++=i/4;
        }
      }
      if(!WritePCX (skinfilename, mdl.skins[i].texs[0], palette, mdl.skinwidth, mdl.skinheight))
      {
        fprintf(stderr, "Error then writing %s!\n", skinfilename);
      }
    }
  }
  else
  {
    fprintf(stderr, "Warning : no skin in this model !\n");
  }

  if((f=fopen(spritefilename, "w"))==NULL)
  {
    fprintf(stderr, "Cannot open file %s !\n", spritename);
    return false;
  }

  // begin hard work now
  fprintf(f, "SPRITE '%s' (\n", spritename);
  fprintf(f, "\tTEXNR ('%s')\n", spritename);
  fprintf(stdout, "Generate MDL/SPR vertex correspondence\n");

  // count back seam vertices
  int back_seam_verts = 0;
  int back_triangles = 0;

  long * BS_verts = new long [mdl.nbvertices];
  char * verts    = new char [mdl.nbvertices];
  memset (verts, 0, mdl.nbvertices * sizeof(char));

  for (i=0; i<mdl.nbtriangles; i++)
  {
    if (!mdl.triangles[i].facefront)
    {
      back_triangles++;
      for (j=0; j<3; j++)
      {
        vertex = mdl.triangles[i].vertice[j];
        if (mdl.vertices[vertex].onseam) verts[vertex] = 1;
      }
    }
  }

  for (i=0; i<mdl.nbvertices; i++)
    if (verts[i])
    {
      BS_verts[i] = mdl.nbvertices + back_seam_verts;
      back_seam_verts++;
    }

  fprintf(stdout, "\t%d back triangles\n", back_triangles);
  fprintf(stdout, "\t%d back seam vertices detected\n", back_seam_verts);

  // create sprite skin vertices
  triangle_t * SPR_triangles = new triangle_t [mdl.nbtriangles];
  vertice_t * SPR_vertices   = new vertice_t  [mdl.nbvertices+back_seam_verts];
  memset(SPR_triangles, 0, mdl.nbtriangles*sizeof(triangle_t));
  memset(SPR_vertices,  0, mdl.nbvertices*2*sizeof(vertice_t));

  // find corresponding mdl skin vertices
  for(i=0; i<mdl.nbtriangles; i++)
  {
    SPR_triangles[i].facefront = mdl.triangles[i].facefront;
//    SPR_triangles[i].vertice = new long [3];
    for(j=0; j<3; j++)
    {
      vertex = mdl.triangles[i].vertice[j];

      // copy mdl vertices to sprite
      SPR_vertices[vertex].s = mdl.vertices[vertex].s;
      SPR_vertices[vertex].t = mdl.vertices[vertex].t;
      SPR_triangles[i].vertice[j] = vertex;

      // create a duplicate vertex for back seam triangles
      if( (mdl.vertices[vertex].onseam) && (!mdl.triangles[i].facefront))
      {
        SPR_vertices[BS_verts[vertex]].s = mdl.vertices[vertex].s + (mdl.skinwidth / 2);
        SPR_vertices[BS_verts[vertex]].s = mdl.vertices[vertex].t;
        SPR_triangles[i].vertice[j] = BS_verts[vertex];
      }
    }
  }

  // create sprite frameset
  frameset_t * SPR_framesets = new frameset_t [mdl.nbframesets];
  for (i=0; i<mdl.nbframesets; i++)
  {
    long framecount = mdl.framesets[i].nbframes;
    SPR_framesets[i].nbframes = framecount;
    SPR_framesets[i].frames = new frame_t [framecount];
    for (j=0; j<framecount; j++)
    {
      SPR_framesets[i].frames[j].trivert = new trivertx_t [mdl.nbvertices+back_seam_verts];
    }
  }

  // copy corresponding mdl framesets
  for (i=0; i<mdl.nbframesets; i++)
  {
    for (j=0; j<mdl.framesets[i].nbframes; j++)
    {
      for (k=0; k<mdl.nbtriangles; k++)
      {
        for (v=0; v<3; v++)
        {
          long SPR_vertex = SPR_triangles[k].vertice[v];
          long MDL_vertex = mdl.triangles[k].vertice[v];
    
          SPR_framesets[i].frames[j].trivert[SPR_vertex].packedposition[0]=
          mdl.framesets[i].frames[j].trivert[MDL_vertex].packedposition[0];
          SPR_framesets[i].frames[j].trivert[SPR_vertex].packedposition[1]=
          mdl.framesets[i].frames[j].trivert[MDL_vertex].packedposition[1];
          SPR_framesets[i].frames[j].trivert[SPR_vertex].packedposition[2]=
          mdl.framesets[i].frames[j].trivert[MDL_vertex].packedposition[2];
        }
      }
    }
  }

  fprintf(stdout, "Generate Frames\n");
  for(i=0; i<mdl.nbframesets; i++)
  {
    for(j=0; j<mdl.framesets[i].nbframes; j++)
    {
      fprintf(f, "\tFRAME '%s' (", mdl.framesets[i].frames[j].name);
      for(k=0; k<mdl.nbvertices+back_seam_verts; k++)
      {
        // it seem y and z are switched
        float x = (float) SPR_framesets[i].frames[j].trivert[k].packedposition[0];
        float z = (float) SPR_framesets[i].frames[j].trivert[k].packedposition[1];
        float y = (float) SPR_framesets[i].frames[j].trivert[k].packedposition[2];
        float u = (float) SPR_vertices[k].s;
        float v = (float) SPR_vertices[k].t;
          
        x = ((x * mdl.scaleX) + mdl.originX) * scaleMdl;
        y = ((y * mdl.scaleZ) + mdl.originZ) * scaleMdl;
        z = ((z * mdl.scaleY) + mdl.originY) * scaleMdl;

        x += positionMdlX;
        y += positionMdlY;
        z += positionMdlZ;

        u = u / (float) mdl.skinwidth;
        v = v / (float) mdl.skinheight;

        fprintf(f, " V (%.3f,%.3f,%.3f:%.3f,%.3f)", x, y, z, u, v);
      }
      fprintf(f, " )\n");
    }
  }

  fprintf(stdout, "Generate Actions\n");
  for(i=0; i<mdl.nbframesets; i++)
  {
    if(mdl.framesets[i].group)
    {
      char name_action[64];
      memset(name_action, 0, 64);

      int base_action = strlen(mdl.framesets[i].frames[0].name);
      for(j=strlen(mdl.framesets[i].frames[0].name)-1; j>1; j--)
      {
        if(!isdigit(mdl.framesets[i].frames[0].name[j])) break;
      }
      base_action=j+1;
      strncpy(name_action, mdl.framesets[i].frames[0].name, base_action);
      
      fprintf(f, "\tACTION '%s' (", name_action);

      for(j=0; j<mdl.framesets[i].nbframes; j++)
      {
        float delay = mdl.framesets[i].delay[j];
        if(j>0) delay -= mdl.framesets[i].delay[j-1];
        delay *= 1000.0;
        fprintf(f, " F ('%s', %d)", mdl.framesets[i].frames[j].name, (int)delay);
      }
      fprintf(f, " )\n");
    }
    else if(actionNamingMdl)
    {
      if(!isdigit(mdl.framesets[i].frames[0].name[strlen(mdl.framesets[i].frames[0].name)-1]))
      {
        fprintf(f, "\tACTION '%s' ( F ('%s', %d) )\n", mdl.framesets[i].frames[0].name, mdl.framesets[i].frames[0].name, delayMdl);
      }
      else
      {
        char name_action[64];
        memset(name_action, 0, 64);

        int base_action;
        for(j=strlen(mdl.framesets[i].frames[0].name)-1; j>1; j--)
        {
          if(!isdigit(mdl.framesets[i].frames[0].name[j])) break;
        }
        base_action=j+1;
        strncpy(name_action, mdl.framesets[i].frames[0].name, base_action);

        fprintf(f, "\tACTION '%s' (", name_action);

        fprintf(f, " F(%s, %d)", mdl.framesets[i].frames[0].name, delayMdl);
        for(j=i+1; j<mdl.nbframesets; j++, i++)
        {
          if(mdl.framesets[j].group) break;

          char toy[64], toy2[64];
          bool scrash = false;
          strcpy(toy, name_action);
          strcat(toy, "%s");
          sscanf(mdl.framesets[j].frames[0].name, toy, &toy2);
          for(k=0; k<(int)strlen(toy2); k++)
          {
            if(!isdigit(toy2[k]))
            {
              scrash = true;
              break;
            }
          }
          if(scrash) break;

          if(strncmp(name_action, mdl.framesets[j].frames[0].name, base_action) == 0)
          {
            fprintf(f, " F(%s, %d)", mdl.framesets[j].frames[0].name, delayMdl);
          }
          else break;
        }
        fprintf(f, " )\n");
      }
    }
  }

  fprintf(stdout, "Generate Triangles\n");
  for(i=0; i<mdl.nbtriangles; i++)
  {
/*
    fprintf(f, "\tTRIANGLE (%ld,%ld,%ld)\n",
      mdl.triangles[i].vertice[0],
      mdl.triangles[i].vertice[1],
      mdl.triangles[i].vertice[2]);
*/
    fprintf(f, "\tTRIANGLE (%ld,%ld,%ld)\n",
      SPR_triangles[i].vertice[0],
      SPR_triangles[i].vertice[1],
      SPR_triangles[i].vertice[2]);
  }

  fprintf(f, ")\n");

  fclose(f);

  return true;
}
