/*
  Crystal Space QuakeMDL convertor : spr writer
  Copyright (C) 1998 by Jorrit Tyberghein
  Written by Nathaniel Saint Martin aka NooTe <noote@bigfoot.com>

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.

  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the Free
  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <math.h>

#include "mdl.h"

typedef struct {
  float x;
  float y;
  float z;
} v3d;

typedef struct {
  v3d * vertices;
} v3d_frame;

typedef struct {
  v3d_frame * frames;
} v3d_scene;

bool WritePCX (char *name, unsigned char *data, unsigned char *pal, int width, int height);

bool WriteSPR(char *spritename, Mdl mdl, float scaleMdl, int delayMdl,
  float positionMdlX, float positionMdlY, float positionMdlZ,
  bool actionNamingMdl, bool resizeSkin, bool altFormat)
{
  FILE *f;
  char *spritefilename;
  Mdl spr;
  int i = 0, j = 0, k = 0, v = 0, vertex = 0;
  float x, y;

  if(spritename==NULL || strlen(spritename)==0)
  {
    fprintf(stderr, "void file name\n");
    return false;
  }

  spritefilename = new char [strlen(spritename)+5];
  strcpy(spritefilename, spritename);
  strcat(spritefilename, ".spr");

  // generate skin texture
  if (mdl.nbskins < 1) fprintf(stderr, "Warning : no skin in this model !\n");
  else
  {
    fprintf(stdout, "Generate Skin texture file\n");

    if (resizeSkin)
    {
      fprintf(stdout, "\tOld Skin size = %dx%d\n", mdl.skinwidth, mdl.skinheight);

      x = mdl.skinwidth;
      y = mdl.skinheight;
      spr.skinwidth  = 1;
      spr.skinheight = 1;
      while (x > 1) {x /= 2;  spr.skinwidth  *= 2;}
      while (y > 1) {y /= 2;  spr.skinheight *= 2;}

      fprintf(stdout, "\tNew Skin size = %dx%d\n", spr.skinwidth, spr.skinheight);
    }
    else // resizeSkin == false
    {
      fprintf(stdout, "\tSkin size = %dx%d\n", mdl.skinwidth, mdl.skinheight);
      spr.skinwidth  = mdl.skinwidth;
      spr.skinheight = mdl.skinheight;
    }

    // read quake palette
    unsigned char palette[768];
    if((f=fopen("palette.lmp","rb"))!=NULL)
    {
      if(fread(&palette, 768, 1, f) != 1)
        fprintf(stderr, "Error when reading palette.lmp file.\n");
      fclose(f);
    }
    else // else generate a gray palette
    {
      fprintf(stderr, "Warning : didn't find palette.lmp file !\nusing gray palette...\n");
      unsigned char *p=&palette[0];
      for(i=0; i<256; i++)
      {
        *p++=k/4;
        *p++=k/4;
        *p++=k/4;
      }
    }

    spr.skins = new skin_t [mdl.nbskins];

    char *skinfilename;
    skinfilename = new char [strlen(spritename)+10];

    for(i=0; i<mdl.nbskins; i++)
    {
      spr.skins[i].texs = new unsigned char * [mdl.skins[i].nbtexs];
      for (j=0; j<mdl.skins[i].nbtexs; j++)
      {
        spr.skins[i].texs[j] = new unsigned char [spr.skinwidth * spr.skinheight];

        if (mdl.skins[i].nbtexs>1)
          sprintf(skinfilename, "%s%d%c.pcx", spritename, i, j+'a');
        else if (mdl.nbskins>1)
          sprintf(skinfilename, "%s%d.pcx", spritename, i);
        else sprintf(skinfilename, "%s.pcx", spritename);

        if (resizeSkin)
        {
          float yscale = (float)mdl.skinwidth/(float)spr.skinwidth;
          float xscale = (float)mdl.skinwidth/(float)spr.skinwidth;
          y = 0.5;
          for (int row=0; row<spr.skinheight; row++, y+=yscale)
          {
            x = 0.5;
            for (int col=0; col<spr.skinwidth; col++, x+=xscale)
              spr.skins[i].texs[j][col + row * spr.skinwidth]
              = mdl.skins[i].texs[j][((int)x) + ((int)y) * mdl.skinwidth];
          }
        }
        else // resizeSkin == false
          for (k=0; k < mdl.skinwidth*mdl.skinheight; k++)
            spr.skins[i].texs[j][k] = mdl.skins[i].texs[j][k];

        if(!WritePCX (skinfilename, spr.skins[i].texs[j], palette, spr.skinwidth, spr.skinheight))
          fprintf(stderr, "Error when writing %s!\n", skinfilename);
      }
    }
  }

  if((f=fopen(spritefilename, "w"))==NULL)
  {
    fprintf(stderr, "Cannot open file %s !\n", spritename);
    return false;
  }

  // begin hard work now
  fprintf(f, "SPRITE '%s' (\n", spritename);
  fprintf(f, "\tTEXNR ('%s')\n", spritename);
  fprintf(stdout, "Generate MDL/SPR vertex correspondence\n");

  // count back seam vertices
  long back_seam_verts = 0;
  long * BS_verts = new long [mdl.nbvertices];
  bool * verts    = new bool [mdl.nbvertices];
  memset (verts, false, mdl.nbvertices * sizeof(bool));

  // detect which vertices are back seam vertices
  for (i=0; i<mdl.nbtriangles; i++)
    if (!mdl.triangles[i].facefront)
      for (j=0; j<3; j++)
      {
        vertex = mdl.triangles[i].vertice[j];
        if (mdl.vertices[vertex].onseam) verts[vertex] = true;
      }

  // assign a new, unique vertex number to each back skin vertex
  for (i=0; i<mdl.nbvertices; i++)
    if (verts[i])
    {
      BS_verts[i] = mdl.nbvertices + back_seam_verts;
      back_seam_verts++;
    }

  fprintf(stdout, "\t%ld back seam vertices detected\n", back_seam_verts);

  // create sprite skin vertices
  spr.triangles = new triangle_t [mdl.nbtriangles];
  spr.vertices  = new vertice_t  [mdl.nbvertices+back_seam_verts];
  memset(spr.triangles, 0, mdl.nbtriangles*sizeof(triangle_t));
  memset(spr.vertices,  0, mdl.nbvertices*2*sizeof(vertice_t));

  // find corresponding mdl skin vertices
  for(i=0; i<mdl.nbtriangles; i++)
    for(j=0; j<3; j++)
    {
      vertex = mdl.triangles[i].vertice[j];

      // copy mdl vertices to sprite
      spr.vertices[vertex].s = mdl.vertices[vertex].s;
      spr.vertices[vertex].t = mdl.vertices[vertex].t;
      spr.triangles[i].vertice[j] = vertex;

      // create a duplicate vertex for back seam triangles
      if( (mdl.vertices[vertex].onseam) && (!mdl.triangles[i].facefront))
      {
        spr.vertices[BS_verts[vertex]].s = mdl.vertices[vertex].s + (mdl.skinwidth / 2);
        spr.vertices[BS_verts[vertex]].t = mdl.vertices[vertex].t;
        spr.triangles[i].vertice[j] = BS_verts[vertex];
      }
    }

  // create sprite frameset
  spr.framesets = new frameset_t [mdl.nbframesets];
  for (i=0; i<mdl.nbframesets; i++)
  {
    spr.framesets[i].nbframes = mdl.framesets[i].nbframes;
    spr.framesets[i].frames = new frame_t [spr.framesets[i].nbframes];
    for (j=0; j<mdl.framesets[i].nbframes; j++)
      spr.framesets[i].frames[j].trivert = new trivertx_t [mdl.nbvertices+back_seam_verts];
  }

  // copy corresponding mdl framesets
  for (i=0; i<mdl.nbframesets; i++)
    for (j=0; j<mdl.framesets[i].nbframes; j++)
      for (k=0; k<mdl.nbtriangles; k++)
        for (v=0; v<3; v++)
        {
          long SPR_vertex = spr.triangles[k].vertice[v];
          long MDL_vertex = mdl.triangles[k].vertice[v];

          spr.framesets[i].frames[j].trivert[SPR_vertex].packedposition[0]=
          mdl.framesets[i].frames[j].trivert[MDL_vertex].packedposition[0];
          spr.framesets[i].frames[j].trivert[SPR_vertex].packedposition[1]=
          mdl.framesets[i].frames[j].trivert[MDL_vertex].packedposition[1];
          spr.framesets[i].frames[j].trivert[SPR_vertex].packedposition[2]=
          mdl.framesets[i].frames[j].trivert[MDL_vertex].packedposition[2];
        }

  // generate vertex normals
  v3d_scene * vertexNormalScenes = new v3d_scene [mdl.nbframesets];
  if(vertexNormalScenes == NULL)
  {
    fprintf(stderr, "Cannot create vertex normal scenes!\n");
    return false;
  }
  v3d * triangleNormals = new v3d [mdl.nbtriangles];
  if(vertexNormalScenes == NULL)
  {
    fprintf(stderr, "Cannot create triangle normals!\n");
    return false;
  }
  v3d tv [3]; // triangle vertices
  v3d te [2]; // triangle edges
  if (altFormat)
  {
    fprintf(stdout, "Calculate vertex normals\n");
    for (i=0; i<mdl.nbframesets; i++)
    {
      // fprintf(stdout, "\tScene %d\n", i);
      vertexNormalScenes[i].frames = new v3d_frame [mdl.framesets[i].nbframes];
      if(vertexNormalScenes[i].frames == NULL)
      {
        fprintf(stderr, "Cannot create vertex normal frames!\n");
        return false;
      }
      for (j=0; j<mdl.framesets[i].nbframes; j++)
      {
        // fprintf(stdout, "\t\tFrame %d\n", j);
        vertexNormalScenes[i].frames[j].vertices = new v3d [mdl.nbvertices+back_seam_verts];
        if(vertexNormalScenes[i].frames[j].vertices == NULL)
        {
          fprintf(stderr, "Cannot create vertex normals!\n");
          return false;
        }
        for (k=0; k<mdl.nbtriangles; k++)
        {
          // fprintf(stdout, "\t\t\tTriangle %d\n", k);
          // get triangle vertices
          vertex = spr.triangles[k].vertice[0];
          tv[0].x = (float) spr.framesets[i].frames[j].trivert[vertex].packedposition[0];
          tv[0].z = (float) spr.framesets[i].frames[j].trivert[vertex].packedposition[1];
          tv[0].y = (float) spr.framesets[i].frames[j].trivert[vertex].packedposition[2];
          vertex = spr.triangles[k].vertice[1];
          tv[1].x = (float) spr.framesets[i].frames[j].trivert[vertex].packedposition[0];
          tv[1].z = (float) spr.framesets[i].frames[j].trivert[vertex].packedposition[1];
          tv[1].y = (float) spr.framesets[i].frames[j].trivert[vertex].packedposition[2];
          vertex = spr.triangles[k].vertice[2];
          tv[2].x = (float) spr.framesets[i].frames[j].trivert[vertex].packedposition[0];
          tv[2].z = (float) spr.framesets[i].frames[j].trivert[vertex].packedposition[1];
          tv[2].y = (float) spr.framesets[i].frames[j].trivert[vertex].packedposition[2];

          // calculate triangle edge vectors
          te[0].x = tv[1].x - tv[0].x;
          te[0].y = tv[1].y - tv[0].y;
          te[0].z = tv[1].z - tv[0].z;
      
          te[1].x = tv[2].x - tv[1].x;
          te[1].y = tv[2].y - tv[1].y;
          te[1].z = tv[2].z - tv[1].z;

          // normal = cross product of two edge vectors
          // triangleNormals[k].x = te[0].y * te[1].z - te[0].x * te[1].y;
          triangleNormals[k].x = te[0].y * te[1].z - te[0].z * te[1].y;
          triangleNormals[k].y = te[0].z * te[1].x - te[0].x * te[1].z;
          triangleNormals[k].z = te[0].x * te[1].y - te[0].y * te[1].x;

          // normalize the surface normal to a unit vector
          x = sqrt (
            triangleNormals[k].x * triangleNormals[k].x +
            triangleNormals[k].y * triangleNormals[k].y +
            triangleNormals[k].z * triangleNormals[k].z
          );
          if (x == 0)
            fprintf (stderr, "Warning: Zero length surface normal in scene %d frame %d triangle %d\n", i,j,k);
          else
          {
            triangleNormals[k].x /= x;
            triangleNormals[k].y /= x;
            triangleNormals[k].z /= x;
          }
        }
        for (k=0; k<mdl.nbvertices+back_seam_verts; k++)
        {
          // fprintf(stdout, "\t\t\tVertex %d\n", k);
          vertexNormalScenes[i].frames[j].vertices[k].x = 0;
          vertexNormalScenes[i].frames[j].vertices[k].y = 0;
          vertexNormalScenes[i].frames[j].vertices[k].z = 0;
          long count = 0;
          for (int t=0; t<mdl.nbtriangles; t++)
            if ( spr.triangles[t].vertice[0] == k
              || spr.triangles[t].vertice[1] == k
              || spr.triangles[t].vertice[2] == k)
            {
              // why do these lines generate a SIGFPE ???
              vertexNormalScenes[i].frames[j].vertices[k].x += triangleNormals[t].x;
              vertexNormalScenes[i].frames[j].vertices[k].z += triangleNormals[t].z;
              vertexNormalScenes[i].frames[j].vertices[k].y += triangleNormals[t].y;
              count++;
            }
          if (count == 0)
            fprintf (stderr, "Warning: No triangles attatched to vertex %d in scene %d frame %d\n", k,i,j);
          else
          {
            vertexNormalScenes[i].frames[j].vertices[k].x /= count;
            vertexNormalScenes[i].frames[j].vertices[k].y /= count;
            vertexNormalScenes[i].frames[j].vertices[k].z /= count;
          }
        }
      }
    }
  }

  // write skinmaps to file
  if (altFormat)
  {
    fprintf(stdout, "Generate Skinmaps\n");
    fprintf(f, "\tSKINMAP 'skinmap0' (");
    for(k=0; k<mdl.nbvertices+back_seam_verts; k++)
    {
      float u = (float) spr.vertices[k].s / (float) mdl.skinwidth;
      float v = (float) spr.vertices[k].t / (float) mdl.skinheight;
      fprintf(f, " V (%.3f,%.3f)", u, v);
    }
    fprintf(f, " )\n");
  }

  // write frames to file
  fprintf(stdout, "Generate Frames\n");
  for(i=0; i<mdl.nbframesets; i++)
  {
    for(j=0; j<mdl.framesets[i].nbframes; j++)
    {
      fprintf(f, "\tFRAME '%s' (", mdl.framesets[i].frames[j].name);
      for(k=0; k<mdl.nbvertices+back_seam_verts; k++)
      {
        // it seem y and z are switched
        float x = (float) spr.framesets[i].frames[j].trivert[k].packedposition[0];
        float z = (float) spr.framesets[i].frames[j].trivert[k].packedposition[1];
        float y = (float) spr.framesets[i].frames[j].trivert[k].packedposition[2];
          
        x = ((x * mdl.scaleX) + mdl.originX) * scaleMdl + positionMdlX;
        y = ((y * mdl.scaleZ) + mdl.originZ) * scaleMdl + positionMdlY;
        z = ((z * mdl.scaleY) + mdl.originY) * scaleMdl + positionMdlZ;

        if (altFormat)
        {
          fprintf(f, " V (%.3f,%.3f,%.3f:%.2f,%.2f,%.2f)", x, y, z,
            vertexNormalScenes[i].frames[j].vertices[k].x,
            vertexNormalScenes[i].frames[j].vertices[k].y,
            vertexNormalScenes[i].frames[j].vertices[k].z);
        }
        else
        {
          float u = (float) spr.vertices[k].s / (float) mdl.skinwidth;
          float v = (float) spr.vertices[k].t / (float) mdl.skinheight;
          fprintf(f, " V (%.3f,%.3f,%.3f:%.3f,%.3f)", x, y, z, u, v);
        }
      }
      fprintf(f, " )\n");
    }
  }

  fprintf(stdout, "Generate Actions\n");
  for(i=0; i<mdl.nbframesets; i++)
  {
    if(mdl.framesets[i].group)
    {
      char name_action[64];
      memset(name_action, 0, 64);

      int base_action = strlen(mdl.framesets[i].frames[0].name);
      for(j=strlen(mdl.framesets[i].frames[0].name)-1; j>1; j--)
      {
        if(!isdigit(mdl.framesets[i].frames[0].name[j])) break;
      }
      base_action=j+1;
      strncpy(name_action, mdl.framesets[i].frames[0].name, base_action);
      
      fprintf(f, "\tACTION '%s' (", name_action);

      for(j=0; j<mdl.framesets[i].nbframes; j++)
      {
        float delay = mdl.framesets[i].delay[j];
        if(j>0) delay -= mdl.framesets[i].delay[j-1];
        delay *= 1000.0;
        fprintf(f, " F ('%s', %d)", mdl.framesets[i].frames[j].name, (int)delay);
      }
      fprintf(f, " )\n");
    }
    else if(actionNamingMdl)
    {
      if(!isdigit(mdl.framesets[i].frames[0].name[strlen(mdl.framesets[i].frames[0].name)-1]))
      {
        fprintf(f, "\tACTION '%s' ( F ('%s', %d) )\n", mdl.framesets[i].frames[0].name, mdl.framesets[i].frames[0].name, delayMdl);
      }
      else
      {
        char name_action[64];
        memset(name_action, 0, 64);

        int base_action;
        for(j=strlen(mdl.framesets[i].frames[0].name)-1; j>1; j--)
        {
          if(!isdigit(mdl.framesets[i].frames[0].name[j])) break;
        }
        base_action=j+1;
        strncpy(name_action, mdl.framesets[i].frames[0].name, base_action);

        fprintf(f, "\tACTION '%s' (", name_action);

        fprintf(f, " F(%s, %d)", mdl.framesets[i].frames[0].name, delayMdl);
        for(j=i+1; j<mdl.nbframesets; j++, i++)
        {
          if(mdl.framesets[j].group) break;

          char toy[64], toy2[64];
          bool scrash = false;
          strcpy(toy, name_action);
          strcat(toy, "%s");
          sscanf(mdl.framesets[j].frames[0].name, toy, &toy2);
          for(k=0; k<(int)strlen(toy2); k++)
          {
            if(!isdigit(toy2[k]))
            {
              scrash = true;
              break;
            }
          }
          if(scrash) break;

          if(strncmp(name_action, mdl.framesets[j].frames[0].name, base_action) == 0)
          {
            fprintf(f, " F(%s, %d)", mdl.framesets[j].frames[0].name, delayMdl);
          }
          else break;
        }
        fprintf(f, " )\n");
      }
    }
  }

  fprintf(stdout, "Generate Triangles\n");
  for(i=0; i<mdl.nbtriangles; i++)
  {
    fprintf(f, "\tTRIANGLE (%ld,%ld,%ld)\n",
      spr.triangles[i].vertice[0],
      spr.triangles[i].vertice[1],
      spr.triangles[i].vertice[2]);
  }

  fprintf(f, ")\n");

  fclose(f);

  return true;
}
