/*
    Maze editor: 3D model class
    Copyright (C) 1999 by Andrew Zabolotny <bit@eltech.ru>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "sysdef.h"
#include "csengine/cssprite.h"
#include "csengine/triangle.h"

#include "me_app.h"
#include "me_cmds.h"
#include "me_draft.h"
#include "me_poly.h"
#include "me_sprit.h"
#include "me_util.h"

//---------------------------------------------------// 3D sprite class //----//

mz3DSprite::mz3DSprite () : mz3DModel ()
{
  static int sprite_count = 0;
  // Generate a name for model
  char tmp [16];
  sprintf (tmp, "sprite%d", sprite_count++);
  SetName (tmp);
  // Generate first frame
  FrameNo = 0;
  CHK (F.Push (new mzVertexVector (V)));
  SetFrameName (0, "frame0");
}

mz3DSprite::~mz3DSprite ()
{
  // Clear the "main" vertex array to avoid duplicate object deletion
  // since it is just a copy of the "real" vertex array from frame set (F)
  V.Clear ();
}

mzModelType mz3DSprite::GetType ()
{
  return mzmtSprite;
}

void mz3DSprite::SetFrameName (int iFrame, char *iName)
{
  if ((iFrame < 0)
   || (iFrame >= Frames ()))
    return;

  if (iFrame < FrameName.Length ())
    FrameName.FreeItem (FrameName [iFrame]);
  FrameName [iFrame] = strnew (iName);
}

bool mz3DSprite::SetFrame (int iFrame, char *iFrameName)
{
  if ((iFrame < 0) || (iFrame > Frames ()))
    return false;

  // Save current frame, if valid
  if ((FrameNo >= 0) && (FrameNo < F.Length ()))
    Frame (FrameNo) = V;

  // If iFrame is equal to F.Length(), a new frame is created
  int oldFrame = FrameNo;
  if ((FrameNo = iFrame) == F.Length ())
  {
    CHK (F.Push (new mzVertexVector (64, 64)));
    if (!iFrameName)
    {
      char tmp [16];
      sprintf (tmp, "frame%d", FrameNo);
      SetFrameName (FrameNo, tmp);
    } /* endif */

    // duplicate previous frame, if any
    if (oldFrame >= 0)
    {
      mzVertexVector *newV = (mzVertexVector *) F [FrameNo];
      for (int vertex = 0; vertex < Vertices (); vertex++)
        CHKB (newV->Push (new mz3DVertex (Vertex (vertex, oldFrame))));
    } /* endif */
  }

  if (iFrameName)
    SetFrameName (FrameNo, iFrameName);

  // Assign current frame to current vertex array
  V = Frame (FrameNo);

  // duplicate selection
  if ((oldFrame >= 0) && (oldFrame != FrameNo))
    for (int vertex = 0; vertex < Vertices (); vertex++)
      Vertex (vertex).Selected = Vertex (vertex, oldFrame).Selected;
  return true;
}

void mz3DSprite::DeleteFrame (int iFrame)
{
  if ((iFrame < 0) || (iFrame >= Frames ()))
    return;

  // Save old current frame
  Frame (FrameNo) = V;

  for (int i = A.Length () - 1; i >= 0; i--)
  {
    mzSpriteAction *sa = (mzSpriteAction *)A.Get (i);
    sa->NotifyFrameDeleted ((char *)FrameName [iFrame]);
  } /* endfor */
  F.Delete (iFrame);
  FrameName.Delete (iFrame);
  if (FrameNo >= F.Length ())
    FrameNo = F.Length () - 1;
  if (FrameNo < 0)
    FrameNo = 0;

  // Assign current frame to current vertex array
  V = Frame (FrameNo);
}

void mz3DSprite::InsertVertex (mz3DVertex *iVertex)
{
  for (int i = Frames () - 1; i >= 0; i--)
    Frame (i).Push (i == 0 ? iVertex : new mz3DVertex (*iVertex));
}

bool mz3DSprite::DeleteVertex (int iVertex, bool iRemapPolygons)
{
  if ((iVertex < 0)
   || (iVertex >= Vertices ()))
    return false;

  int i;
  for (i = Frames () - 1; i >= 0; i--)
    Frame (i).Delete (iVertex);

  if (iRemapPolygons)
    for (i = Polygons () - 1; i >= 0; i--)
      if (!Polygon (i).NotifyVertexDeleted (iVertex))
        DeletePolygon (i);
  return true;
}

void mz3DSprite::InsertAction (mzSpriteAction *iAction)
{
  A.Push (iAction);
}

bool mz3DSprite::DeleteAction (int iAction)
{
  if ((iAction < 0) || (iAction >= A.Length ()))
    return false;

  A.Delete (iAction);
  return true;
}

void mz3DSprite::Clear (unsigned int iFlags)
{
  int i;

  if (iFlags & MZF_CLEAR_FRAMES)
    // Do not clear current frame
    for (i = Frames () - 1; i >= 0; i--)
      if (i != FrameNo)
        F.Delete (i);

  if (iFlags & MZF_CLEAR_VERTICES)
  {
    for (i = Frames () - 1; i >= 0; i--)
      Frame (i).DeleteAll ();
    V = Frame (FrameNo);
  } /* endif */

  if (iFlags & MZF_CLEAR_ACTIONS)
    A.DeleteAll ();

  mz3DModel::Clear (iFlags);
}

void mz3DSprite::Copy (mz3DModel *&oClipboard)
{
  if (!oClipboard)
    oClipboard = new mz3DSprite ();
  else if (oClipboard->GetType () != mzmtSprite)
  {
    CHK (delete oClipboard);
    oClipboard = new mz3DSprite ();
  }
  mz3DModel::Copy (oClipboard);
}

void mz3DSprite::Draw (mzCameraView *Canvas)
{
  (void)Canvas;
}

void mz3DSprite::Draw (mzDraftEditor *Canvas, bool iModified, bool iCopy)
{
  mz3DModel::Draw (Canvas, iModified, iCopy);
}

bool mz3DSprite::Merge (MazeEditor *app, csSpriteTemplate *iSprite)
{
  SetName (iSprite->GetName ());

  if (!Texture)
    SetTexture (app->NewTexture (iSprite->GetTexture ()));

  for (int frame = 0; frame < iSprite->GetNumFrames (); frame++)
  {
    csFrame* f = iSprite->GetFrame (frame);
    SetFrame (frame, f->GetName ());
    int tex_idx = f->GetTexIndex();
    V.DeleteAll ();

    if (iSprite->VerticesAreCompressed ())
    {
      for (int vertex = 0; vertex < iSprite->GetNumTexels (); vertex++)
      {
        CHK (mz3DVertex *v = new mz3DVertex (iSprite->GetVertex (frame, vertex)));
        v->SetTexel (iSprite->GetTexel (tex_idx, vertex));
        V.Push (v);
      } /* endfor */
    }
    else
    {
      for (int vertex = 0; vertex < iSprite->GetNumTexels (); vertex++)
      {
        CHK (mz3DVertex *v = new mz3DVertex (iSprite->GetVertex (frame, vertex)));
        v->SetTexel (iSprite->GetTexel (tex_idx, vertex));
        V.Push (v);
      } /* endfor */
    }
  } /* endfor */

  csTriangle* triangles = iSprite->GetTriangles ();
  for (int triangle = 0; triangle < iSprite->GetNumTriangles (); triangle++)
  {
    CHK (mz3DPolygon *p = new mz3DPolygon (this));
    p->InsertVertex (triangles [triangle].a);
    p->InsertVertex (triangles [triangle].b);
    p->InsertVertex (triangles [triangle].c);
    InsertPolygon (p);
  } /* endfor */
  SetFrame (0);
  return true;
}

#define PUTS(s)				\
{					\
  size_t len = strlen (s);		\
  if (oStream->Write (s, len) != len)	\
    return false;			\
}

bool mz3DSprite::Save (iFile *oStream)
{
  int i, j;
  char buff [512];
  sprintf (buff, "  SPRITE '%s'\n  (\n", Name);
  PUTS (buff);

  if (Texture)
  {
    sprintf (buff, "    TEXNR ('%s')\n", Texture->GetName ());
    PUTS (buff);
  }

  for (i = 0; i < Frames (); i++)
  {
    sprintf (buff, "    FRAME '%s' (", GetFrameName (i));
    PUTS (buff);
    for (j = 0; j < Vertices (); j++)
    {
      if (j)
        strcpy (buff, " V(");
      else
        strcpy (buff, "V(");
      char *out = buff + strlen (buff);
      out += ftoah (out, Vertex (j, i).x);
      *out++ = ',';
      out += ftoah (out, Vertex (j, i).y);
      *out++ = ',';
      out += ftoah (out, Vertex (j, i).z);
      *out++ = ':';
      out += ftoah (out, Vertex (j, i).u);
      *out++ = ',';
      out += ftoah (out, Vertex (j, i).v);
      *out++ = ')';
      size_t sl = out - buff;
      if (oStream->Write (buff, sl) != sl)
        return false;
    } /* endfor */
    PUTS (")\n");
  } /* endfor */

  for (i = 0; i < Actions (); i++)
  {
    mzSpriteAction &a = Action (i);
    sprintf (buff, "    ACTION '%s' (", a.Name);
    PUTS (buff);
    for (j = 0; j < a.Frames (); j++)
    {
      sprintf (buff, "F(%s,%d)%s", a.FrameName (j), a.Delay (j),
        (j < a.Frames () - 1) ? " " : "");
      PUTS (buff);
    } /* endfor */
    PUTS (")\n");
  } /* endfor */

  for (i = 0; i < Polygons (); i++)
  {
    mz3DPolygon &p = Polygon (i);
    sprintf (buff, "    TRIANGLE (%d,%d,%d)\n", p [0], p [1], p [2]);
    PUTS (buff);
  } /* endfor */

  PUTS ("  )\n");
  return true;
}

bool mz3DSprite::HandleEvent (mzDraftEditor* Canvas, csEvent& Event)
{
#define app Canvas->app

  switch (Event.Type)
  {
    case csevCommand:
      switch (Event.Command.Code)
      {
        case cscmdMzNextModelFrame:
          if ((GetFrame () < Frames () - 1)
           && (SetFrame (GetFrame () + 1)))
          {
            GLOBAL_DRAFTCHANGED;
            GLOBAL_MODELINFO;
          } /* endif */
          return true;
        case cscmdMzPrevModelFrame:
          if ((GetFrame () > 0)
           && (SetFrame (GetFrame () - 1)))
          {
            GLOBAL_DRAFTCHANGED;
            GLOBAL_MODELINFO;
          } /* endif */
          return true;
        case cscmdMzInsertModelFrame:
          if (SetFrame (Frames ()))
          {
            GLOBAL_DRAFTCHANGED;
            GLOBAL_MODELINFO;
          }
          return true;
        case cscmdMzDeleteModelFrame:
          DeleteFrame (GetFrame ());
          GLOBAL_DRAFTCHANGED;
          GLOBAL_MODELINFO;
          return true;
      }
  }

  return mz3DModel::HandleEvent (Canvas, Event);

#undef app
}
