// Copyright (C) 1999-2000 Id Software, Inc.
//
/**********************************************************************
	UI_QMENU.C

	Quake's menu framework system.
**********************************************************************/
#include "ui_local.h"

#define ID_CHILDSCROLLBAR 1
#define ID_CHILDSCROLLUP 2
#define ID_CHILDSCROLLDOWN 3
#define ID_CHILDBITMAP 4


sfxHandle_t menu_in_sound;
sfxHandle_t menu_move_sound;
sfxHandle_t menu_out_sound;
sfxHandle_t menu_buzz_sound;
sfxHandle_t menu_null_sound;

static qhandle_t	sliderBar;
static qhandle_t	sliderButton_0;
static qhandle_t	sliderButton_1;

vec4_t menu_text_color	    = {1.0, 1.0, 1.0, 1.0};
vec4_t menu_dim_color       = {0.0, 0.0, 0.0, 0.75};
vec4_t color_black	        = {0.00, 0.00, 0.00, 1.00};
vec4_t color_white	        = {1.00, 1.00, 1.00, 1.00};
vec4_t color_yellow	        = {1.00, 1.00, 0.00, 1.00};
vec4_t color_blue	        = {0.00, 0.00, 1.00, 1.00};
vec4_t color_lightOrange    = {1.00, 0.68, 0.00, 1.00 };
vec4_t color_orange	        = {1.00, 0.43, 0.00, 1.00};
vec4_t color_red		    = {1.00, 0.00, 0.00, 1.00};
vec4_t color_dim		    = {0.00, 0.00, 0.00, 0.25};
vec4_t color_green		    = {0.00, 1.00, 0.00, 1.00};

// current color scheme
vec4_t pulse_color          = {1.00, 1.00, 1.00, 1.00};
vec4_t text_color_disabled  = {0.50, 0.50, 0.50, 1.00};	// light gray
vec4_t text_color_normal	= {1.00, 0.43, 0.00, 1.00};	// light orange
vec4_t text_color_highlight = {1.00, 1.00, 0.00, 1.00};	// bright yellow
vec4_t listbar_color        = {1.00, 0.43, 0.00, 0.30};	// transluscent orange
vec4_t text_color_status    = {1.00, 1.00, 1.00, 1.00};	// bright white	

// action widget
static void	Action_Init( menuaction_s *a );
static void	Action_Draw( menuaction_s *a );

// radio button widget
static void	RadioButton_Init( menuradiobutton_s *rb );
static void	RadioButton_Draw( menuradiobutton_s *rb );
static sfxHandle_t RadioButton_Key( menuradiobutton_s *rb, int key );

// slider widget
static void Slider_Init( menuslider_s *s );
static sfxHandle_t Slider_Key( menuslider_s *s, int key );
static void	Slider_Draw( menuslider_s *s );

// spin control widget
static void	SpinControl_Init( menulist_s *s );
static void	SpinControl_Draw( menulist_s *s );
static sfxHandle_t SpinControl_Key( menulist_s *l, int key );

// text widget
static void Text_Init( menutext_s *b );
static void Text_Draw( menutext_s *b );

// scrolllist widget
static void	ScrollList_Init( menulist_s *l );
sfxHandle_t ScrollList_Key( menulist_s *l, int key );

// proportional text widget
static void PText_Init( menutext_s *b );
static void PText_Draw( menutext_s *b );

// proportional banner text widget
static void BText_Init( menutext_s *b );
static void BText_Draw( menutext_s *b );

//scrollbar widget
static void ScrollBar_Init( menuscrollbar_s *t );
static void ScrollBar_Draw( menuscrollbar_s *t );
static sfxHandle_t ScrollBar_Key( menuscrollbar_s *s, int key );

//scrolllist with scrollbar widget
static void ScrollList2_Init( menuscrolllist_s *t );
static void ScrollList2_Draw( menuscrolllist_s *t );
//static sfxHandle_t TextBox_Key( menutextbox_s *t, int key );

//tiling bitmap
static void TileBitmap_Init( menutilebitmap_s *b );
static void TileBitmap_Draw( menutilebitmap_s *b );

static void Tab_Init(menutab_s *t); 
static void Tab_Draw(menutab_s *t);//we do the callback when any of the tabs in the group are clicked

/*
=================
Text_Init
=================
*/
static void Text_Init( menutext_s *t )
{
	t->generic.flags |= QMF_INACTIVE;
}

/*
=================
Text_Draw
=================
*/
static void Text_Draw( menutext_s *t )
{
	int		x;
	int		y;
	char	buff[512];	
	float*	color;

	x = t->generic.x;
	y = t->generic.y;

	buff[0] = '\0';

	// possible label
	if (t->generic.name)
		strcpy(buff,t->generic.name);

	// possible value
	if (t->string)
		strcat(buff,t->string);
		
	if (t->generic.flags & QMF_GRAYED)
		color = text_color_disabled;
	else
		color = t->color;

	UI_DrawString( x, y, buff, t->style, color );
}

static void ScrollList2_Event( void* ptr, int event ) 
{
	void *owner;
	int id;
	menuscrolllist_s *t;

	id = ((menucommon_s*)ptr)->id;
	owner = ((menucommon_s*)ptr)->owner;
	if (!owner) //ack... bad control..
		return;

	t = (menuscrolllist_s *)owner;
	switch( id ) {

	case ID_CHILDSCROLLUP:
	case ID_CHILDSCROLLDOWN:
		if (event != QM_ACTIVATED)
			break;
		if (id == ID_CHILDSCROLLUP)
			ScrollList_Key( &t->lines, K_UPARROW );
		else
			ScrollList_Key( &t->lines, K_DOWNARROW );
		t->scrollbar.curvalue = (t->lines.curvalue*100.0)/(float)(t->lines.numitems-1);
		break;

	case ID_CHILDSCROLLBAR:
		if (event == QM_LOSTFOCUS)		
		{
			t->scrollbar.hasmouse = qfalse;
		}
		else if (event == QM_ACTIVATED)
		{
			t->lines.curvalue = (int)((t->lines.numitems-1) * (t->scrollbar.curvalue/(float)100.0));
	
			ScrollList_MakeItemVisible(&t->lines);
		}
		break;
	}
}

/*
=================
ScrollList2
EXAMPLE
	g_arenaservers.list.generic.type			= MTYPE_SCROLLLIST2;
	g_arenaservers.list.lines.generic.flags			= QMF_HIGHLIGHT_IF_FOCUS;
	g_arenaservers.list.lines.generic.id				= ID_LIST;
	g_arenaservers.list.lines.generic.callback		= ArenaServers_Event;
	g_arenaservers.list.lines.generic.x				= 16;
	g_arenaservers.list.lines.generic.y				= 90;
	g_arenaservers.list.lines.width					= MAX_LISTBOXWIDTH;
	g_arenaservers.list.lines.height					= 21;
	g_arenaservers.list.lines.itemnames				= (const char **)g_arenaservers.items;
	g_arenaservers.list.scrollbar.width					= 26;
	g_arenaservers.list.scrollbar.generic.name			= ART_SCROLLBAR;
	g_arenaservers.list.scrollbar.focuspic				= ART_SCROLLBAR_WIGET;

	g_arenaservers.list.scrollbar.focusheight			= 32;
	g_arenaservers.list.scrollbar.focuswidth				= 32;
	
	g_arenaservers.list.scrollbar.up.width				= 32;
	g_arenaservers.list.scrollbar.up.height				= 32;
	g_arenaservers.list.scrollbar.up.focuspic			= ART_ARROWS_UP;
	g_arenaservers.list.scrollbar.up.generic.name		= ART_ARROWS_UP;

	g_arenaservers.list.scrollbar.down.width				= 32;
	g_arenaservers.list.scrollbar.down.height			= 32;
	g_arenaservers.list.scrollbar.down.focuspic			= ART_ARROWS_DOWN;
	g_arenaservers.list.scrollbar.down.generic.name		= ART_ARROWS_DOWN;

	for( i = 0; i < MAX_LISTBOXITEMS; i++ ) {
		g_arenaservers.items[i] = g_arenaservers.table[i].buff;
	}
=================
*/
void ScrollList2_SetVisible(menuscrolllist_s *t ,qboolean visible)
{
	if (!visible)
	{
		t->lines.generic.flags |= QMF_HIDDEN;
		t->scrollbar.generic.flags |= QMF_HIDDEN;
		t->scrollbar.up.generic.flags |= QMF_HIDDEN;
		t->scrollbar.down.generic.flags |= QMF_HIDDEN;
	} else
	{
		t->lines.generic.flags &= ~QMF_HIDDEN;
		t->scrollbar.generic.flags &= ~QMF_HIDDEN;
		t->scrollbar.up.generic.flags &= ~QMF_HIDDEN;
		t->scrollbar.down.generic.flags &= ~QMF_HIDDEN;
	}
}

static void ScrollList2_Init( menuscrolllist_s *t )
{
	t->lines.generic.type = MTYPE_SCROLLLIST;
	Menu_AddItem(t->generic.parent,&(t->lines));
	t->scrollbar.generic.type = MTYPE_SCROLLBAR;
	t->scrollbar.generic.id = ID_CHILDSCROLLBAR;
	t->scrollbar.generic.callback = ScrollList2_Event;
	t->scrollbar.generic.owner = t;
	t->scrollbar.generic.flags	= QMF_LEFT_JUSTIFY|QMF_MOUSEONLY;
	t->scrollbar.minvalue				= 0;
	t->scrollbar.maxvalue				= 100;
	t->scrollbar.curvalue				= 0;

	t->scrollbar.up.generic.id = ID_CHILDSCROLLUP;
	t->scrollbar.up.generic.callback = ScrollList2_Event;
	t->scrollbar.up.generic.owner = t;
	t->scrollbar.down.generic.id = ID_CHILDSCROLLDOWN;
	t->scrollbar.down.generic.callback = ScrollList2_Event;
	t->scrollbar.down.generic.owner = t;

	if (t->scrollbar.generic.x == 0) //use the defaults
		t->scrollbar.generic.x = t->lines.generic.right;
	if (t->scrollbar.generic.y == 0)
		t->scrollbar.generic.y = t->lines.generic.top;
	if (t->scrollbar.height == 0)
		t->scrollbar.height = t->lines.generic.bottom - t->lines.generic.top;
	//subtract off the width of the scrollbar
	Menu_AddItem(t->generic.parent,&(t->scrollbar));

	t->generic.left = 0;//t->generic.x;
	t->generic.right = 0;//t->generic.x + t->width;
	t->generic.top = 0;//t->generic.y;
	t->generic.bottom = 0;//t->generic.y + t->height;
	
}
static void ScrollList2_Draw( menuscrolllist_s *t )
{
	//we don't need to do anything special..
}


/*
=================
ScrollBar_Init
=================
EXAMPLE
	g_arenaservers.scrollbar.generic.type			= MTYPE_SCROLLBAR;
	g_arenaservers.scrollbar.generic.id				= ID_SCROLLBAR;
	g_arenaservers.scrollbar.generic.name			= ART_SCROLLBAR;
	g_arenaservers.scrollbar.generic.flags			= QMF_LEFT_JUSTIFY|QMF_MOUSEONLY;
	g_arenaservers.scrollbar.generic.callback		= ArenaServers_Event;
	g_arenaservers.scrollbar.generic.x				= 600;
	g_arenaservers.scrollbar.generic.y				= 16;
	g_arenaservers.scrollbar.width					= 26;
	g_arenaservers.scrollbar.height					= 470;
	g_arenaservers.scrollbar.focuspic				= ART_SCROLLBAR_WIGET;
	g_arenaservers.scrollbar.minvalue				= 0;
	g_arenaservers.scrollbar.maxvalue				= 100;
	g_arenaservers.scrollbar.curvalue				= 0;
	g_arenaservers.scrollbar.focusheight			= 32;
	g_arenaservers.scrollbar.focuswidth				= 32;
	
	g_arenaservers.scrollbar.up.generic.callback	= ArenaServers_Event;
	g_arenaservers.scrollbar.up.generic.id			= ID_SCROLL_UP;
	g_arenaservers.scrollbar.up.width				= 32;
	g_arenaservers.scrollbar.up.height				= 64;
	g_arenaservers.scrollbar.up.focuspic			= ART_ARROWS_UP;
	g_arenaservers.scrollbar.up.generic.name		= ART_ARROWS_UP;

	g_arenaservers.scrollbar.down.generic.callback	= ArenaServers_Event;
	g_arenaservers.scrollbar.down.generic.id		= ID_SCROLL_DOWN;
	g_arenaservers.scrollbar.down.width				= 32;
	g_arenaservers.scrollbar.down.height			= 64;
	g_arenaservers.scrollbar.down.focuspic			= ART_ARROWS_DOWN;
	g_arenaservers.scrollbar.down.generic.name		= ART_ARROWS_DOWN;
*/
static void ScrollBar_Init( menuscrollbar_s *t )
{
	int	x;
	int	y;
	int	w;
	int	h;

	t->up.generic.type = t->down.generic.type = MTYPE_BITMAP;
	t->up.generic.flags = t->down.generic.flags = QMF_LEFT_JUSTIFY|QMF_PULSEIFFOCUS|QMF_MOUSEONLY;
	if (t->up.generic.x == 0)
		t->up.generic.x = t->generic.x;
	if (t->up.generic.y == 0)
		t->up.generic.y = t->generic.y;
	if (t->down.generic.x == 0)
		t->down.generic.x = t->generic.x;
	if (t->down.generic.y == 0)
		t->down.generic.y = t->generic.y + t->height - t->down.height;

	Menu_AddItem(t->generic.parent,&(t->up));
	Menu_AddItem(t->generic.parent,&(t->down));

	t->generic.y += t->up.height;
	t->height -= (t->down.height + t->up.height);



	x = t->generic.x;
	y = t->generic.y;
	w = t->width;
	h =	t->height;
	if( w < 0 ) {
		w = -w;
	}
	if( h < 0 ) {
		h = -h;
	}

	if (t->generic.flags & QMF_RIGHT_JUSTIFY)
	{
		x = x - w;
	}
	else if (t->generic.flags & QMF_CENTER_JUSTIFY)
	{
		x = x - w/2;
	}



	t->generic.left   = x;
	t->generic.right  = x + w;
	t->generic.top    = y;
	t->generic.bottom = y + h;

	t->shader      = 0;
	t->focusshader = 0;
}

/*
=================
ScrollBar_Draw
=================
*/
static void ScrollBar_Draw( menuscrollbar_s *t )
{
	float	x;
	float	y;
	float	w;
	float	h;

	x = t->generic.x;
	y = t->generic.y;
	w = t->width;
	h =	t->height;

	if (t->generic.flags & QMF_HASMOUSEFOCUS && t->hasmouse) //update our position!
		ScrollBar_Key(t,K_BACKSPACE);
	if (t->generic.flags & QMF_RIGHT_JUSTIFY)
	{
		x = x - w;
	}
	else if (t->generic.flags & QMF_CENTER_JUSTIFY)
	{
		x = x - w/2;
	}

	// used to refresh shader
	if (t->generic.name && !t->shader)
	{
		t->shader = trap_R_RegisterShaderNoMip( t->generic.name );
		if (!t->shader && t->errorpic)
			t->shader = trap_R_RegisterShaderNoMip( t->errorpic );
	}

	if (t->focuspic && !t->focusshader)
		t->focusshader = trap_R_RegisterShaderNoMip( t->focuspic );

	if (t->generic.flags & QMF_GRAYED)
	{
		if (t->shader)
		{
			trap_R_SetColor( colorMdGrey );
			UI_DrawHandlePic( x, y, w, h, t->shader );
			trap_R_SetColor( NULL );
		}
	}
	else
	{
		if (t->shader)
			UI_DrawHandlePic( x, y, w, h, t->shader );

	/*	if ((t->generic.flags & QMF_PULSE) || (t->generic.flags & QMF_PULSEIFFOCUS) && (Menu_ItemAtCursor( t->generic.parent ) == t))
		{	
			if (t->focuscolor)			
			{
				tempcolor[0] = t->focuscolor[0];
				tempcolor[1] = t->focuscolor[1];
				tempcolor[2] = t->focuscolor[2];
				color        = tempcolor;	
			}
			else
				color = pulse_color;
			color[3] = 0.5+0.5*sin(uis.realtime/PULSE_DIVISOR);

			trap_R_SetColor( color );
			UI_DrawHandlePic( x, y, w, h, t->focusshader );
			trap_R_SetColor( NULL );
		}
		else if ((t->generic.flags & QMF_HIGHLIGHT) || ((t->generic.flags & QMF_HIGHLIGHT_IF_FOCUS) && (Menu_ItemAtCursor( t->generic.parent ) == t)))
		{	
			if (t->focuscolor)
			{
				trap_R_SetColor( t->focuscolor );
				UI_DrawHandlePic( x, y, w, h, t->focusshader );
				trap_R_SetColor( NULL );
			}
			else
				UI_DrawHandlePic( x, y, w, h, t->focusshader );
		}*/
		y = y + (h - t->focusheight) * ((t->curvalue -  t->minvalue)/((float)t->maxvalue - t->minvalue));
		if (t->focusshader)
			UI_DrawHandlePic(x + (w - t->focuswidth)/(float)2,y,t->focuswidth,t->focusheight,t->focusshader);
	}
}

static sfxHandle_t ScrollBar_Key( menuscrollbar_s *s, int key )
{
	sfxHandle_t	sound = 0;
	int			y;
	float			oldvalue;

	switch (key)
	{
		case K_BACKSPACE:
			y           = uis.cursory - s->generic.y;
			oldvalue    = s->curvalue;
			//we pad it a bit so that we can always get to bottom/top
			s->curvalue = s->minvalue + ((y-5)* ((s->maxvalue-s->minvalue))/(float)(s->height-10));
//			trap_Print(va("old value: %f new value: %f pos: %d\n",oldvalue,s->curvalue,y));

			if (s->curvalue < s->minvalue)
				s->curvalue = s->minvalue;
			else if (s->curvalue > s->maxvalue)
				s->curvalue = s->maxvalue;
			if (s->curvalue != oldvalue)
				sound = menu_move_sound;
			else
				sound = 0;
			break;
		case K_MOUSE1: //capture the mouse
				s->hasmouse = !s->hasmouse;

			break;
		case K_KP_LEFTARROW:
		case K_LEFTARROW:
			if (s->curvalue > s->minvalue)
			{
				s->curvalue--;
				sound = menu_move_sound;
			}
			else
				sound = menu_buzz_sound;
			break;			

		case K_KP_RIGHTARROW:
		case K_RIGHTARROW:
			if (s->curvalue < s->maxvalue)
			{
				s->curvalue++;
				sound = menu_move_sound;
			}
			else
				sound = menu_buzz_sound;
			break;			

		default:
			// key not handled
			sound = 0;
			break;
}

	if ( sound && s->generic.callback )
		s->generic.callback( s, QM_ACTIVATED );
	return sound;
}
/*
=================
BText_Init
=================
*/
static void BText_Init( menutext_s *t )
{
	t->generic.flags |= QMF_INACTIVE;
}

/*
=================
BText_Draw
=================
*/
static void BText_Draw( menutext_s *t )
{
	int		x;
	int		y;
	float*	color;

	x = t->generic.x;
	y = t->generic.y;

	if (t->generic.flags & QMF_GRAYED)
		color = text_color_disabled;
	else
		color = t->color;

	UI_DrawBannerString( x, y, t->string, t->style, color );
}

/*
=================
PText_Init
=================
*/
static void PText_Init( menutext_s *t )
{
	int	x;
	int	y;
	int	w;
	int	h;
	float	sizeScale;

	sizeScale = UI_ProportionalSizeScale( t->style );

	x = t->generic.x;
	y = t->generic.y;
	w = UI_ProportionalStringWidth( t->string ) * sizeScale;
	h =	PROP_HEIGHT * sizeScale;

	if( t->generic.flags & QMF_RIGHT_JUSTIFY ) {
		x -= w;
	}
	else if( t->generic.flags & QMF_CENTER_JUSTIFY ) {
		x -= w / 2;
	}

	t->generic.left   = x - PROP_GAP_WIDTH * sizeScale;
	t->generic.right  = x + w + PROP_GAP_WIDTH * sizeScale;
	t->generic.top    = y;
	t->generic.bottom = y + h;
}

/*
=================
PText_Draw
=================
*/
static void PText_Draw( menutext_s *t )
{
	int		x;
	int		y;
	float *	color;
	int		style;

	x = t->generic.x;
	y = t->generic.y;

	if (t->generic.flags & QMF_GRAYED)
		color = text_color_disabled;
	else
		color = t->color;

	style = t->style;
	if( t->generic.flags & QMF_PULSEIFFOCUS ) {
		if( Menu_ItemAtCursor( t->generic.parent ) == t ) {
			style |= UI_PULSE;
		}
		else {
			style |= UI_INVERSE;
		}
	}

	UI_DrawProportionalString( x, y, t->string, style, color );
}


void Tab_Select(menutab_s *t)
{
	int i;
	menucommon_s *m;
	menutab_s *c;
	//clear the prev selected one
	for (i= t->generic.parent->nitems - 1 ; i >=0; i--)
	{
		m = (menucommon_s*)t->generic.parent->items[i];
		
		if (m->type != MTYPE_TAB )
			continue;
		
		c = (menutab_s *)m;
		if (c->groupid != t->groupid)
			continue;
		if (!c->selected)
			continue;
		
		c->selected = qfalse;
		c->bitmap.generic.flags &= ~QMF_HIGHLIGHT;
	}
	t->selected = qtrue;
	t->bitmap.generic.flags |= QMF_HIGHLIGHT;
	if (t->generic.callback)
		t->generic.callback(t,QM_ACTIVATED);
}

static void Tab_Event( void* ptr, int event ) 
{
	void *owner;
	int id;
	menutab_s *t;

	id = ((menucommon_s*)ptr)->id;
	owner = ((menucommon_s*)ptr)->owner;
	if (!owner) //ack... bad control..
		return;

	t = (menutab_s *)owner;
	switch( id ) {

	case ID_CHILDBITMAP: 
		if (event != QM_ACTIVATED)
			break;
		if (t->selected)
			break;
		Tab_Select(t);

		break;

	}
}
/*
=================
Tab_Init
EXAMPLE
	g_arenaservers.allservers.generic.type = 
		g_arenaservers.dmservers.generic.type = 
		g_arenaservers.teamdmservers.generic.type = MTYPE_TAB;
	g_arenaservers.allservers.generic.y = 
		g_arenaservers.dmservers.generic.y = 
		g_arenaservers.teamdmservers.generic.y = 68;
	g_arenaservers.allservers.groupid = 
		g_arenaservers.dmservers.groupid = 
		g_arenaservers.teamdmservers.groupid = GROUP_TABS;
	g_arenaservers.allservers.generic.callback = 
		g_arenaservers.dmservers.generic.callback = 
		g_arenaservers.teamdmservers.generic.callback = ArenaServers_Event;
	g_arenaservers.allservers.generic.x				= 8;
	g_arenaservers.allservers.generic.id			= ID_ALLSERVERS;
	g_arenaservers.allservers.bitmap.generic.name	= ART_ALLSERVERS0;
	g_arenaservers.allservers.bitmap.focuspic		= ART_ALLSERVERS1;
	g_arenaservers.allservers.bitmap.height			= 17;
	g_arenaservers.allservers.bitmap.width			= 101;
	g_arenaservers.dmservers.generic.x				= 109;
	g_arenaservers.dmservers.generic.id			= ID_DMSERVERS;
	g_arenaservers.dmservers.bitmap.generic.name	= ART_DMSERVERS0;
	g_arenaservers.dmservers.bitmap.focuspic		= ART_DMSERVERS1;
	g_arenaservers.dmservers.bitmap.height			= 17;
	g_arenaservers.dmservers.bitmap.width			= 41;
	g_arenaservers.teamdmservers.generic.x			= 150;
	g_arenaservers.teamdmservers.generic.id			= ID_TEAMDMSERVERS;
	g_arenaservers.teamdmservers.bitmap.generic.name	= ART_TEAMDMSERVERS0;
	g_arenaservers.teamdmservers.bitmap.focuspic		= ART_TEAMDMSERVERS1;
	g_arenaservers.teamdmservers.bitmap.height			= 17;
	g_arenaservers.teamdmservers.bitmap.width			= 78;
...
	Menu_AddItem( &g_arenaservers.menu, (void*) &g_arenaservers.allservers );
	Menu_AddItem( &g_arenaservers.menu, (void*) &g_arenaservers.dmservers );
	Menu_AddItem( &g_arenaservers.menu, (void*) &g_arenaservers.teamdmservers );
...
	Tab_Select(&g_arenaservers.allservers); //select the first tab

=================
*/
static void Tab_Init(menutab_s *t)
{
	t->generic.flags |= QMF_HIDDEN;
	t->bitmap.generic.id = ID_CHILDBITMAP;
	t->bitmap.generic.callback = Tab_Event;
	t->bitmap.generic.x = t->generic.x;
	t->bitmap.generic.y = t->generic.y;
	t->bitmap.generic.type = MTYPE_BITMAP;
	t->bitmap.generic.flags =  QMF_LEFT_JUSTIFY|QMF_MOUSEONLY;
	t->bitmap.generic.owner = t;
	t->selected = qfalse;
	

	Menu_AddItem(t->generic.parent,&(t->bitmap));
}
/*
=================
Tab_Draw
=================
*/
static void Tab_Draw(menutab_s *t)//we do the callback when any of the tabs in the group are clicked
{
	//do we need to do anything? probably not
}
/*
=================
TileBitmap_Init
EXAMPLE
	g_arenaservers.motdbg.generic.type			= MTYPE_TILEBITMAP;
	g_arenaservers.motdbg.generic.flags =    QMF_SILENT|QMF_LEFT_JUSTIFY ; 
	g_arenaservers.motdbg.bitmap.generic.name			= ART_MOTDBACK;
	g_arenaservers.motdbg.bitmap.generic.flags			= QMF_LEFT_JUSTIFY| QMF_SILENT;
	g_arenaservers.motdbg.bitmap.width = 32;
	g_arenaservers.motdbg.bitmap.height = 32;

	g_arenaservers.motdbg.generic.x				= 25;
	g_arenaservers.motdbg.generic.y				= 55;
	g_arenaservers.motdbg.width					= 601;
	g_arenaservers.motdbg.height				= 367;

=================
*/
static void TileBitmap_Init( menutilebitmap_s *b)
{
	b->bitmap.generic.x = 0;
	b->bitmap.generic.y = 0;
	Bitmap_Init(&b->bitmap);
	b->generic.left = b->generic.x;
	b->generic.right = b->generic.x + b->width;
	b->generic.top = b->generic.y;
	b->generic.bottom = b->generic.y + b->height;
}
/*
=================
TileBitmap_Draw
=================
*/
static void TileBitmap_Draw( menutilebitmap_s *b )
{
	int i,j;
	int x,y;
	for (i = b->generic.left ; i < b->generic.right ; i += b->bitmap.width)
		for (j = b->generic.top ; j < b->generic.bottom ; j+= b->bitmap.height)
		{
			if (b->generic.right - i < b->bitmap.width) //need to draw from the right to left
			{
				x = b->generic.right - b->bitmap.width;
			} else
				x = i;
			if (b->generic.bottom - j < b->bitmap.height)
			{
				y = b->generic.bottom - b->bitmap.width;
			} else
				y = j;
			b->bitmap.generic.x = x;
			b->bitmap.generic.y = y;
			Bitmap_Draw(&b->bitmap);

		}
}

/*
=================
Bitmap_Init
=================
*/
void Bitmap_Init( menubitmap_s *b )
{
	int	x;
	int	y;
	int	w;
	int	h;

	x = b->generic.x;
	y = b->generic.y;
	w = b->width;
	h =	b->height;
	if( w < 0 ) {
		w = -w;
	}
	if( h < 0 ) {
		h = -h;
	}

	if (b->generic.flags & QMF_RIGHT_JUSTIFY)
	{
		x = x - w;
	}
	else if (b->generic.flags & QMF_CENTER_JUSTIFY)
	{
		x = x - w/2;
	}

	b->generic.left   = x;
	b->generic.right  = x + w;
	b->generic.top    = y;
	b->generic.bottom = y + h;

	b->shader      = 0;
	b->focusshader = 0;
}

/*
=================
Bitmap_Draw
=================
*/
void Bitmap_Draw( menubitmap_s *b )
{
	float	x;
	float	y;
	float	w;
	float	h;
	vec4_t	tempcolor;
	float*	color;

	x = b->generic.x;
	y = b->generic.y;
	w = b->width;
	h =	b->height;

	if (b->generic.flags & QMF_RIGHT_JUSTIFY)
	{
		x = x - w;
	}
	else if (b->generic.flags & QMF_CENTER_JUSTIFY)
	{
		x = x - w/2;
	}

	// used to refresh shader
	if (b->generic.name && !b->shader)
	{
		b->shader = trap_R_RegisterShaderNoMip( b->generic.name );
		if (!b->shader && b->errorpic)
			b->shader = trap_R_RegisterShaderNoMip( b->errorpic );
	}

	if (b->focuspic && !b->focusshader)
		b->focusshader = trap_R_RegisterShaderNoMip( b->focuspic );

	if (b->generic.flags & QMF_GRAYED)
	{
		if (b->shader)
		{
			trap_R_SetColor( colorMdGrey );
			UI_DrawHandlePic( x, y, w, h, b->shader );
			trap_R_SetColor( NULL );
		}
	}
	else
	{
		if (b->shader)
			UI_DrawHandlePic( x, y, w, h, b->shader );

		if ((b->generic.flags & QMF_PULSE) || (b->generic.flags & QMF_PULSEIFFOCUS) && (Menu_ItemAtCursor( b->generic.parent ) == b))
		{	
			if (b->focuscolor)			
			{
				tempcolor[0] = b->focuscolor[0];
				tempcolor[1] = b->focuscolor[1];
				tempcolor[2] = b->focuscolor[2];
				color        = tempcolor;	
			}
			else
				color = pulse_color;
			color[3] = 0.5+0.5*sin(uis.realtime/PULSE_DIVISOR);

			trap_R_SetColor( color );
			UI_DrawHandlePic( x, y, w, h, b->focusshader );
			trap_R_SetColor( NULL );
		}
		else if ((b->generic.flags & QMF_HIGHLIGHT) || ((b->generic.flags & QMF_HIGHLIGHT_IF_FOCUS) && (Menu_ItemAtCursor( b->generic.parent ) == b)))
		{	
			if (b->focuscolor)
			{
				trap_R_SetColor( b->focuscolor );
				UI_DrawHandlePic( x, y, w, h, b->focusshader );
				trap_R_SetColor( NULL );
			}
			else
				UI_DrawHandlePic( x, y, w, h, b->focusshader );
		}
	}
}

/*
=================
Action_Init
=================
*/
static void Action_Init( menuaction_s *a )
{
	int	len;

	// calculate bounds
	if (a->generic.name)
		len = strlen(a->generic.name);
	else
		len = 0;

	// left justify text
	a->generic.left   = a->generic.x; 
	a->generic.right  = a->generic.x + len*BIGCHAR_WIDTH;
	a->generic.top    = a->generic.y;
	a->generic.bottom = a->generic.y + BIGCHAR_HEIGHT;
}

/*
=================
Action_Draw
=================
*/
static void Action_Draw( menuaction_s *a )
{
	int		x, y;
	int		style;
	float*	color;

	style = 0;
	color = menu_text_color;
	if ( a->generic.flags & QMF_GRAYED )
	{
		color = text_color_disabled;
	}
	else if (( a->generic.flags & QMF_PULSEIFFOCUS ) && ( a->generic.parent->cursor == a->generic.menuPosition ))
	{
		color = text_color_highlight;
		style = UI_PULSE;
	}
	else if (( a->generic.flags & QMF_HIGHLIGHT_IF_FOCUS ) && ( a->generic.parent->cursor == a->generic.menuPosition ))
	{
		color = text_color_highlight;
	}
	else if ( a->generic.flags & QMF_BLINK )
	{
		style = UI_BLINK;
		color = text_color_highlight;
	}

	x = a->generic.x;
	y = a->generic.y;

	UI_DrawString( x, y, a->generic.name, UI_LEFT|style, color );

	if ( a->generic.parent->cursor == a->generic.menuPosition )
	{
		// draw cursor
		UI_DrawChar( x - BIGCHAR_WIDTH, y, 13, UI_LEFT|UI_BLINK, color);
	}
}

/*
=================
RadioButton_Init
=================
*/
static void RadioButton_Init( menuradiobutton_s *rb )
{
	int	len;

	// calculate bounds
	if (rb->generic.name)
		len = strlen(rb->generic.name);
	else
		len = 0;

	rb->generic.left   = rb->generic.x - (len+1)*SMALLCHAR_WIDTH;
	rb->generic.right  = rb->generic.x + 6*SMALLCHAR_WIDTH;
	rb->generic.top    = rb->generic.y;
	rb->generic.bottom = rb->generic.y + SMALLCHAR_HEIGHT;
}

/*
=================
RadioButton_Key
=================
*/
static sfxHandle_t RadioButton_Key( menuradiobutton_s *rb, int key )
{
	switch (key)
	{
		case K_MOUSE1:
			if (!(rb->generic.flags & QMF_HASMOUSEFOCUS))
				break;

		case K_JOY1:
		case K_JOY2:
		case K_JOY3:
		case K_JOY4:
		case K_ENTER:
		case K_KP_ENTER:
		case K_KP_LEFTARROW:
		case K_LEFTARROW:
		case K_KP_RIGHTARROW:
		case K_RIGHTARROW:
			rb->curvalue = !rb->curvalue;
			if ( rb->generic.callback )
				rb->generic.callback( rb, QM_ACTIVATED );

			return (menu_move_sound);
	}

	// key not handled
	return 0;
}

/*
=================
RadioButton_Draw
=================
*/
static void RadioButton_Draw( menuradiobutton_s *rb )
{
	int	x;
	int y;
	float *color;
	int	style;
	qboolean focus;

	x = rb->generic.x;
	y = rb->generic.y;

	focus = (rb->generic.parent->cursor == rb->generic.menuPosition);

	if ( rb->generic.flags & QMF_GRAYED )
	{
		color = text_color_disabled;
		style = UI_LEFT|UI_SMALLFONT;
	}
	else if ( focus )
	{
		color = text_color_highlight;
		style = UI_LEFT|UI_PULSE|UI_SMALLFONT;
	}
	else
	{
		color = text_color_normal;
		style = UI_LEFT|UI_SMALLFONT;
	}

	if ( focus )
	{
		// draw cursor
		UI_FillRect( rb->generic.left, rb->generic.top, rb->generic.right-rb->generic.left+1, rb->generic.bottom-rb->generic.top+1, listbar_color ); 
		UI_DrawChar( x, y, 13, UI_CENTER|UI_BLINK|UI_SMALLFONT, color);
	}

	if ( rb->generic.name )
		UI_DrawString( x - SMALLCHAR_WIDTH, y, rb->generic.name, UI_RIGHT|UI_SMALLFONT, color );

	if ( !rb->curvalue )
	{
		UI_DrawHandlePic( x + SMALLCHAR_WIDTH, y + 2, 16, 16, uis.rb_off);
		UI_DrawString( x + SMALLCHAR_WIDTH + 16, y, "off", style, color );
	}
	else
	{
		UI_DrawHandlePic( x + SMALLCHAR_WIDTH, y + 2, 16, 16, uis.rb_on );
		UI_DrawString( x + SMALLCHAR_WIDTH + 16, y, "on", style, color );
	}
}

/*
=================
Slider_Init
=================
*/
static void Slider_Init( menuslider_s *s )
{
	int len;

	// calculate bounds
	if (s->generic.name)
		len = strlen(s->generic.name);
	else
		len = 0;

	s->generic.left   = s->generic.x - (len+1)*SMALLCHAR_WIDTH; 
	s->generic.right  = s->generic.x + (SLIDER_RANGE+2+1)*SMALLCHAR_WIDTH;
	s->generic.top    = s->generic.y;
	s->generic.bottom = s->generic.y + SMALLCHAR_HEIGHT;
}

/*
=================
Slider_Key
=================
*/
static sfxHandle_t Slider_Key( menuslider_s *s, int key )
{
	sfxHandle_t	sound;
	int			x;
	int			oldvalue;

	switch (key)
	{
		case K_MOUSE1:
			x           = uis.cursorx - s->generic.x - 2*SMALLCHAR_WIDTH;
			oldvalue    = s->curvalue;
			s->curvalue = (x/(float)(SLIDER_RANGE*SMALLCHAR_WIDTH)) * (s->maxvalue-s->minvalue) + s->minvalue;

			if (s->curvalue < s->minvalue)
				s->curvalue = s->minvalue;
			else if (s->curvalue > s->maxvalue)
				s->curvalue = s->maxvalue;
			if (s->curvalue != oldvalue)
				sound = menu_move_sound;
			else
				sound = 0;
			break;

		case K_KP_LEFTARROW:
		case K_LEFTARROW:
			if (s->curvalue > s->minvalue)
			{
				s->curvalue--;
				sound = menu_move_sound;
			}
			else
				sound = menu_buzz_sound;
			break;			

		case K_KP_RIGHTARROW:
		case K_RIGHTARROW:
			if (s->curvalue < s->maxvalue)
			{
				s->curvalue++;
				sound = menu_move_sound;
			}
			else
				sound = menu_buzz_sound;
			break;			

		default:
			// key not handled
			sound = 0;
			break;
	}

	if ( sound && s->generic.callback )
		s->generic.callback( s, QM_ACTIVATED );

	return (sound);
}

#if 1
/*
=================
Slider_Draw
=================
*/
static void Slider_Draw( menuslider_s *s ) {
	int			x;
	int			y;
	int			style;
	float		*color;
	int			button;
	qboolean	focus;
	
	x =	s->generic.x;
	y = s->generic.y;
	focus = (s->generic.parent->cursor == s->generic.menuPosition);

	if( s->generic.flags & QMF_GRAYED ) {
		color = text_color_disabled;
		style = UI_SMALLFONT;
	}
	else if( focus ) {
		color  = text_color_highlight;
		style = UI_SMALLFONT | UI_PULSE;
	}
	else {
		color = text_color_normal;
		style = UI_SMALLFONT;
	}

	// draw label
	UI_DrawString( x - SMALLCHAR_WIDTH, y, s->generic.name, UI_RIGHT|style, color );

	// draw slider
	UI_SetColor( color );
	UI_DrawHandlePic( x + SMALLCHAR_WIDTH, y, 96, 16, sliderBar );
	UI_SetColor( NULL );

	// clamp thumb
	if( s->maxvalue > s->minvalue )	{
		s->range = ( s->curvalue - s->minvalue ) / ( float ) ( s->maxvalue - s->minvalue );
		if( s->range < 0 ) {
			s->range = 0;
		}
		else if( s->range > 1) {
			s->range = 1;
		}
	}
	else {
		s->range = 0;
	}

	// draw thumb
	if( style & UI_PULSE) {
		button = sliderButton_1;
	}
	else {
		button = sliderButton_0;
	}

	UI_DrawHandlePic( (int)( x + 2*SMALLCHAR_WIDTH + (SLIDER_RANGE-1)*SMALLCHAR_WIDTH* s->range ) - 2, y - 2, 12, 20, button );
}
#else
/*
=================
Slider_Draw
=================
*/
static void Slider_Draw( menuslider_s *s )
{
	float *color;
	int	style;
	int	i;
	int x;
	int y;
	qboolean focus;
	
	x =	s->generic.x;
	y = s->generic.y;
	focus = (s->generic.parent->cursor == s->generic.menuPosition);

	style = UI_SMALLFONT;
	if ( s->generic.flags & QMF_GRAYED )
	{
		color = text_color_disabled;
	}
	else if (focus)
	{
		color  = text_color_highlight;
		style |= UI_PULSE;
	}
	else
	{
		color = text_color_normal;
	}

	if ( focus )
	{
		// draw cursor
		UI_FillRect( s->generic.left, s->generic.top, s->generic.right-s->generic.left+1, s->generic.bottom-s->generic.top+1, listbar_color ); 
		UI_DrawChar( x, y, 13, UI_CENTER|UI_BLINK|UI_SMALLFONT, color);
	}

	// draw label
	UI_DrawString( x - SMALLCHAR_WIDTH, y, s->generic.name, UI_RIGHT|style, color );

	// draw slider
	UI_DrawChar( x + SMALLCHAR_WIDTH, y, 128, UI_LEFT|style, color);
	for ( i = 0; i < SLIDER_RANGE; i++ )
		UI_DrawChar( x + (i+2)*SMALLCHAR_WIDTH, y, 129, UI_LEFT|style, color);
	UI_DrawChar( x + (i+2)*SMALLCHAR_WIDTH, y, 130, UI_LEFT|style, color);

	// clamp thumb
	if (s->maxvalue > s->minvalue)
	{
		s->range = ( s->curvalue - s->minvalue ) / ( float ) ( s->maxvalue - s->minvalue );
		if ( s->range < 0)
			s->range = 0;
		else if ( s->range > 1)
			s->range = 1;
	}
	else
		s->range = 0;

	// draw thumb
	if (style & UI_PULSE) {
		style &= ~UI_PULSE;
		style |= UI_BLINK;
	}
	UI_DrawChar( (int)( x + 2*SMALLCHAR_WIDTH + (SLIDER_RANGE-1)*SMALLCHAR_WIDTH* s->range ), y, 131, UI_LEFT|style, color);
}
#endif

/*
=================
SpinControl_Init
=================
*/
static void SpinControl_Init( menulist_s *s ) {
	int	len;
	int	l;
	const char* str;

	if (s->generic.name)
		len = strlen(s->generic.name) * SMALLCHAR_WIDTH;
	else
		len = 0;

	s->generic.left	= s->generic.x - SMALLCHAR_WIDTH - len;

	len = s->numitems = 0;
	while ( (str = s->itemnames[s->numitems]) != 0 )
	{
		l = strlen(str);
		if (l > len)
			len = l;

		s->numitems++;
	}		

	s->generic.top	  =	s->generic.y;
	s->generic.right  =	s->generic.x + (len+1)*SMALLCHAR_WIDTH;
	s->generic.bottom =	s->generic.y + SMALLCHAR_HEIGHT;
}

/*
=================
SpinControl_Key
=================
*/
static sfxHandle_t SpinControl_Key( menulist_s *s, int key )
{
	sfxHandle_t	sound;

	sound = 0;
	switch (key)
	{
		case K_MOUSE1:
			s->curvalue++;
			if (s->curvalue >= s->numitems)
				s->curvalue = 0;
			sound = menu_move_sound;
			break;
		
		case K_KP_LEFTARROW:
		case K_LEFTARROW:
			if (s->curvalue > 0)
			{
				s->curvalue--;
				sound = menu_move_sound;
			}
			else
				sound = menu_buzz_sound;
			break;

		case K_KP_RIGHTARROW:
		case K_RIGHTARROW:
			if (s->curvalue < s->numitems-1)
			{
				s->curvalue++;
				sound = menu_move_sound;
			}
			else
				sound = menu_buzz_sound;
			break;
	}

	if ( sound && s->generic.callback )
		s->generic.callback( s, QM_ACTIVATED );

	return (sound);
}

/*
=================
SpinControl_Draw
=================
*/
static void SpinControl_Draw( menulist_s *s )
{
	float *color;
	int	x,y;
	int	style;
	qboolean focus;

	x = s->generic.x;
	y =	s->generic.y;

	style = UI_SMALLFONT;
	focus = (s->generic.parent->cursor == s->generic.menuPosition);

	if ( s->generic.flags & QMF_GRAYED )
		color = text_color_disabled;
	else if ( focus )
	{
		color = text_color_highlight;
		style |= UI_PULSE;
	}
	else if ( s->generic.flags & QMF_BLINK )
	{
		color = text_color_highlight;
		style |= UI_BLINK;
	}
	else
		color = text_color_normal;

	if ( focus )
	{
		// draw cursor
		UI_FillRect( s->generic.left, s->generic.top, s->generic.right-s->generic.left+1, s->generic.bottom-s->generic.top+1, listbar_color ); 
		UI_DrawChar( x, y, 13, UI_CENTER|UI_BLINK|UI_SMALLFONT, color);
	}

	UI_DrawString( x - SMALLCHAR_WIDTH, y, s->generic.name, style|UI_RIGHT, color );
	UI_DrawString( x + SMALLCHAR_WIDTH, y, s->itemnames[s->curvalue], style|UI_LEFT, color );
}

/*
=================
ScrollList_Init
=================
*/
static void ScrollList_Init( menulist_s *l )
{
	l->oldvalue = 0;
	l->curvalue = 0;
	l->top      = 0;

	if( !l->columns ) {
		l->columns = 1;
		l->seperation = 0;
	}
	else if( !l->seperation ) {
		l->seperation = 3;
	}

	l->generic.left   =	l->generic.x;
	l->generic.top    = l->generic.y;	
	l->generic.right  =	l->generic.x + ( (l->width + l->seperation) * l->columns - l->seperation) * SMALLCHAR_WIDTH;
	l->generic.bottom =	l->generic.y + l->height * SMALLCHAR_HEIGHT;
}

/*
=================
ScrollList_Key
=================
*/
sfxHandle_t ScrollList_Key( menulist_s *l, int key )
{
	int	x;
	int	y;
	int	i;
	int	j;	
	int	c;
	int	cursorx;
	int	cursory;
	int	column;
	int	index;

	switch (key)
	{
		case K_MOUSE1:
			if (l->generic.flags & QMF_HASMOUSEFOCUS)
			{
				// check scroll region
				x = l->generic.x;
				y = l->generic.y;
				if (UI_CursorInRect( x, y, ( (l->width + l->seperation) * l->columns - l->seperation) * SMALLCHAR_WIDTH, l->height*SMALLCHAR_HEIGHT ))
				{
					cursorx = (uis.cursorx - x)/SMALLCHAR_WIDTH;
					column = cursorx / (l->width + l->seperation);
					cursory = (uis.cursory - y)/SMALLCHAR_HEIGHT;
					index = column * l->height + cursory;
					if (l->top + index < l->numitems)
					{
						l->oldvalue = l->curvalue;
						l->curvalue = l->top + index;

						i = trap_Milliseconds();
						if (l->oldvalue != l->curvalue && l->generic.callback)
						{
							l->generic.callback( l, QM_GOTFOCUS );
							l->lastclick = i;
							if ((l->generic.flags & (QMF_HIGHLIGHT_IF_FOCUS|QMF_PULSEIFFOCUS)) != 0) //crt -- only do the highlighting if the mode is right
							return (menu_move_sound);
							else
								return (menu_null_sound);
						}
						if (l->oldvalue == l->curvalue && (i - l->lastclick > 0) && (i - l->lastclick <= 500) && l->generic.callback)
						{
							l->generic.callback(l, QM_DOUBLECLICK);
						}
						l->lastclick = i;
						
					}
				}
			
				// absorbed, silent sound effect
				return (menu_null_sound);
			}
			break;

		case K_KP_HOME:
		case K_HOME:
			l->oldvalue = l->curvalue;
			l->curvalue = 0;
			l->top      = 0;

			if (l->oldvalue != l->curvalue && l->generic.callback)
			{
				l->generic.callback( l, QM_GOTFOCUS );
				return (menu_move_sound);
			}
			return (menu_buzz_sound);

		case K_KP_END:
		case K_END:
			l->oldvalue = l->curvalue;
			l->curvalue = l->numitems-1;
			if( l->columns > 1 ) {
				c = (l->curvalue / l->height + 1) * l->height;
				l->top = c - (l->columns * l->height);
			}
			else {
				l->top = l->curvalue - (l->height - 1);
			}
			if (l->top < 0)
				l->top = 0;			

			if (l->oldvalue != l->curvalue && l->generic.callback)
			{
				l->generic.callback( l, QM_GOTFOCUS );
				return (menu_move_sound);
			}
			return (menu_buzz_sound);

		case K_MWHEELUP:
		case K_PGUP:
		case K_KP_PGUP:
			if( l->columns > 1 ) {
				return menu_null_sound;
			}

			if (l->curvalue > 0)
			{
				l->oldvalue = l->curvalue;
				l->curvalue -= l->height-1;
				if (l->curvalue < 0)
					l->curvalue = 0;
				l->top = l->curvalue;
				if (l->top < 0)
					l->top = 0;

				if (l->generic.callback)
					l->generic.callback( l, QM_GOTFOCUS );

				return (menu_move_sound);
			}
			return (menu_buzz_sound);

		case K_MWHEELDOWN:
		case K_PGDN:
		case K_KP_PGDN:
			if( l->columns > 1 ) {
				return menu_null_sound;
			}

			if (l->curvalue < l->numitems-1)
			{
				l->oldvalue = l->curvalue;
				l->curvalue += l->height-1;
				if (l->curvalue > l->numitems-1)
					l->curvalue = l->numitems-1;
				l->top = l->curvalue - (l->height-1);
				if (l->top < 0)
					l->top = 0;

				if (l->generic.callback)
					l->generic.callback( l, QM_GOTFOCUS );

				return (menu_move_sound);
			}
			return (menu_buzz_sound);

		case K_KP_UPARROW:
		case K_UPARROW:
			if( l->curvalue == 0 ) {
				return menu_buzz_sound;
			}

			l->oldvalue = l->curvalue;
			l->curvalue--;

			if( l->curvalue < l->top ) {
				if( l->columns == 1 ) {
					l->top--;
				}
				else {
					l->top -= l->height;
				}
			}

			if( l->generic.callback ) {
				l->generic.callback( l, QM_GOTFOCUS );
			}

			return (menu_move_sound);

		case K_KP_DOWNARROW:
		case K_DOWNARROW:
			if( l->curvalue == l->numitems - 1 ) {
				return menu_buzz_sound;
			}

			l->oldvalue = l->curvalue;
			l->curvalue++;

			if( l->curvalue >= l->top + l->columns * l->height ) {
				if( l->columns == 1 ) {
					l->top++;
				}
				else {
					l->top += l->height;
				}
			}

			if( l->generic.callback ) {
				l->generic.callback( l, QM_GOTFOCUS );
			}

			return menu_move_sound;

		case K_KP_LEFTARROW:
		case K_LEFTARROW:
			if( l->columns == 1 ) {
				return menu_null_sound;
			}

			if( l->curvalue < l->height ) {
				return menu_buzz_sound;
			}

			l->oldvalue = l->curvalue;
			l->curvalue -= l->height;

			if( l->curvalue < l->top ) {
				l->top -= l->height;
			}

			if( l->generic.callback ) {
				l->generic.callback( l, QM_GOTFOCUS );
			}

			return menu_move_sound;

		case K_KP_RIGHTARROW:
		case K_RIGHTARROW:
			if( l->columns == 1 ) {
				return menu_null_sound;
			}

			c = l->curvalue + l->height;

			if( c >= l->numitems ) {
				return menu_buzz_sound;
			}

			l->oldvalue = l->curvalue;
			l->curvalue = c;

			if( l->curvalue > l->top + l->columns * l->height - 1 ) {
				l->top += l->height;
			}

			if( l->generic.callback ) {
				l->generic.callback( l, QM_GOTFOCUS );
			}

			return menu_move_sound;
	}

	// cycle look for ascii key inside list items
	if ( !Q_isprint( key ) )
		return (0);

	// force to lower for case insensitive compare
	if ( Q_isupper( key ) )
	{
		key -= 'A' - 'a';
	}

	// iterate list items
	for (i=1; i<=l->numitems; i++)
	{
		j = (l->curvalue + i) % l->numitems;
		c = l->itemnames[j][0];
		if ( Q_isupper( c ) )
		{
			c -= 'A' - 'a';
		}

		if (c == key)
		{
		//crt
			if (l->curvalue != j)
			{
				l->oldvalue = l->curvalue;
				l->curvalue = j;
				ScrollList_MakeItemVisible(l);
				if (l->generic.callback)
					l->generic.callback( l, QM_GOTFOCUS );
				return ( menu_move_sound );			
			}
			return (menu_buzz_sound);
		}
	}

	return (menu_buzz_sound);
}

void ScrollList_MakeItemVisible(menulist_s *l )
{
				// set current item, mimic windows listbox scroll behavior
			if (l->curvalue < l->top)
			{
				// behind top most item, set this as new top
				l->top = l->curvalue;
			}
			else if (l->curvalue > l->top+l->height-1)
			{
				// past end of list box, do page down
				l->top = (l->curvalue+1) - l->height;
			}
}

/*
=================
ScrollList_Draw
=================
*/
void ScrollList_Draw( menulist_s *l )
{
	int			x;
	int			y;
	int			i;
	int			base;
	int			column;
	float*		color;
	qboolean	hasfocus;
	int			style;

	hasfocus = (l->generic.parent->cursor == l->generic.menuPosition);

	x =	l->generic.x;
	for( column = 0; column < l->columns; column++ ) {
		y =	l->generic.y;
		base = l->top + column * l->height;
		for( i = base; i < base + l->height; i++) {
			if (i >= l->numitems)
				break;

			if (i == l->curvalue && (l->generic.flags & (QMF_HIGHLIGHT_IF_FOCUS|QMF_PULSEIFFOCUS)) != 0) //crt -- only do the highlighting if the mode is right
			{
				UI_FillRect(x-2,y,l->width*SMALLCHAR_WIDTH,SMALLCHAR_HEIGHT+2,listbar_color);
				color = text_color_highlight;

				if (hasfocus)
					style = UI_PULSE|UI_LEFT|UI_SMALLFONT;
				else
					style = UI_LEFT|UI_SMALLFONT;
			}
			else
			{
				color = text_color_normal;
				style = UI_LEFT|UI_SMALLFONT;
			}

			UI_DrawString(
				x,
				y,
				l->itemnames[i],
				style,
				color);

			y += SMALLCHAR_HEIGHT;
		}
		x += (l->width + l->seperation) * SMALLCHAR_WIDTH;
	}
}

/*
=================
Menu_AddItem
=================
*/
void Menu_AddItem( menuframework_s *menu, void *item )
{
	menucommon_s	*itemptr;

	if (menu->nitems >= MAX_MENUITEMS)
		trap_Error ("Menu_AddItem: excessive items");

	menu->items[menu->nitems] = item;
	((menucommon_s*)menu->items[menu->nitems])->parent        = menu;
	((menucommon_s*)menu->items[menu->nitems])->menuPosition  = menu->nitems;
	((menucommon_s*)menu->items[menu->nitems])->flags        &= ~QMF_HASMOUSEFOCUS;

	// perform any item specific initializations
	itemptr = (menucommon_s*)item;
	menu->nitems++;
	if (!(itemptr->flags & QMF_NODEFAULTINIT))
	{
		switch (itemptr->type)
		{
			case MTYPE_ACTION:
				Action_Init((menuaction_s*)item);
				break;

			case MTYPE_FIELD:
				MenuField_Init((menufield_s*)item);
				break;

			case MTYPE_SPINCONTROL:
				SpinControl_Init((menulist_s*)item);
				break;

			case MTYPE_RADIOBUTTON:
				RadioButton_Init((menuradiobutton_s*)item);
				break;

			case MTYPE_SLIDER:
				Slider_Init((menuslider_s*)item);
				break;

			case MTYPE_BITMAP:
				Bitmap_Init((menubitmap_s*)item);
				break;

			case MTYPE_TEXT:
				Text_Init((menutext_s*)item);
				break;

			case MTYPE_SCROLLLIST:
				ScrollList_Init((menulist_s*)item);
				break;

			case MTYPE_PTEXT:
				PText_Init((menutext_s*)item);
				break;

			case MTYPE_BTEXT:
				BText_Init((menutext_s*)item);
				break;
			case MTYPE_SCROLLBAR:
				ScrollBar_Init((menuscrollbar_s*)item);
				break;
			case MTYPE_SCROLLLIST2:
				ScrollList2_Init((menuscrolllist_s *)item);
				break;
			case MTYPE_TILEDBITMAP:
				TileBitmap_Init((menutilebitmap_s *)item);
				break;
			case MTYPE_TAB:
				Tab_Init((menutab_s *)item);
				break;
			default:
				trap_Error( va("Menu_Init: unknown type %d", itemptr->type) );
		}
	}


}

/*
=================
Menu_CursorMoved
=================
*/
void Menu_CursorMoved( menuframework_s *m )
{
	void (*callback)( void *self, int notification );
	
	if (m->cursor_prev == m->cursor)
		return;

	if (m->cursor_prev >= 0 && m->cursor_prev < m->nitems)
	{
		callback = ((menucommon_s*)(m->items[m->cursor_prev]))->callback;
		if (callback)
			callback(m->items[m->cursor_prev],QM_LOSTFOCUS);
	}
	
	if (m->cursor >= 0 && m->cursor < m->nitems)
	{
		callback = ((menucommon_s*)(m->items[m->cursor]))->callback;
		if (callback)
			callback(m->items[m->cursor],QM_GOTFOCUS);
	}
}

/*
=================
Menu_SetCursor
=================
*/
void Menu_SetCursor( menuframework_s *m, int cursor )
{
	if (((menucommon_s*)(m->items[cursor]))->flags & (QMF_GRAYED|QMF_INACTIVE))
	{
		// cursor can't go there
		return;
	}

	m->cursor_prev = m->cursor;
	m->cursor      = cursor;

	Menu_CursorMoved( m );
}

/*
=================
Menu_SetCursorToItem
=================
*/
void Menu_SetCursorToItem( menuframework_s *m, void* ptr )
{
	int	i;

	for (i=0; i<m->nitems; i++)
	{
		if (m->items[i] == ptr)
		{
			Menu_SetCursor( m, i );
			return;
		}
	}
}

/*
** Menu_AdjustCursor
**
** This function takes the given menu, the direction, and attempts
** to adjust the menu's cursor so that it's at the next available
** slot.
*/
void Menu_AdjustCursor( menuframework_s *m, int dir ) {
	menucommon_s	*item = NULL;
	qboolean		wrapped = qfalse;

wrap:
	while ( m->cursor >= 0 && m->cursor < m->nitems ) {
		item = ( menucommon_s * ) m->items[m->cursor];
		if (( item->flags & (QMF_GRAYED|QMF_MOUSEONLY|QMF_INACTIVE) ) ) {
			m->cursor += dir;
		}
		else {
			break;
		}
	}

	if ( dir == 1 ) {
		if ( m->cursor >= m->nitems ) {
			if ( m->wrapAround ) {
				if ( wrapped ) {
					m->cursor = m->cursor_prev;
					return;
				}
				m->cursor = 0;
				wrapped = qtrue;
				goto wrap;
			}
			m->cursor = m->cursor_prev;
		}
	}
	else {
		if ( m->cursor < 0 ) {
			if ( m->wrapAround ) {
				if ( wrapped ) {
					m->cursor = m->cursor_prev;
					return;
				}
				m->cursor = m->nitems - 1;
				wrapped = qtrue;
				goto wrap;
			}
			m->cursor = m->cursor_prev;
		}
	}
}

/*
=================
Menu_Draw
=================
*/
void Menu_Draw( menuframework_s *menu )
{
	int				i;
	menucommon_s	*itemptr;

	// draw menu
	for (i=0; i<menu->nitems; i++)
	{
		itemptr = (menucommon_s*)menu->items[i];

		if (itemptr->flags & QMF_HIDDEN)
			continue;

		if (itemptr->ownerdraw)
		{
			// total subclassing, owner draws everything
			itemptr->ownerdraw( itemptr );
		}	
		else 
		{
			switch (itemptr->type)
			{	
				case MTYPE_RADIOBUTTON:
					RadioButton_Draw( (menuradiobutton_s*)itemptr );
					break;

				case MTYPE_FIELD:
					MenuField_Draw( (menufield_s*)itemptr );
					break;
		
				case MTYPE_SLIDER:
					Slider_Draw( (menuslider_s*)itemptr );
					break;
 
				case MTYPE_SPINCONTROL:
					SpinControl_Draw( (menulist_s*)itemptr );
					break;
		
				case MTYPE_ACTION:
					Action_Draw( (menuaction_s*)itemptr );
					break;
		
				case MTYPE_BITMAP:
					Bitmap_Draw( (menubitmap_s*)itemptr );
					break;

				case MTYPE_TEXT:
					Text_Draw( (menutext_s*)itemptr );
					break;

				case MTYPE_SCROLLLIST:
					ScrollList_Draw( (menulist_s*)itemptr );
					break;
				
				case MTYPE_PTEXT:
					PText_Draw( (menutext_s*)itemptr );
					break;

				case MTYPE_BTEXT:
					BText_Draw( (menutext_s*)itemptr );
					break;
				case MTYPE_SCROLLBAR:
					ScrollBar_Draw((menuscrollbar_s*)itemptr);
					break;
				case MTYPE_SCROLLLIST2:
					ScrollList2_Draw((menuscrolllist_s *)itemptr);
					break;
				case MTYPE_TILEDBITMAP:
					TileBitmap_Draw((menutilebitmap_s *)itemptr);
					break;
				case MTYPE_TAB:
					Tab_Draw((menutab_s *)itemptr);
					break;
                default:
					trap_Error( va("Menu_Draw: unknown type %d", itemptr->type) );
			}
		}
#ifndef NDEBUG
		if( uis.debug ) {
			int	x;
			int	y;
			int	w;
			int	h;

			if( !( itemptr->flags & QMF_INACTIVE ) ) {
				x = itemptr->left;
				y = itemptr->top;
				w = itemptr->right - itemptr->left + 1;
				h =	itemptr->bottom - itemptr->top + 1;

				if (itemptr->flags & QMF_HASMOUSEFOCUS) {
					UI_DrawRect(x, y, w, h, colorYellow );
				}
				else {
					UI_DrawRect(x, y, w, h, colorWhite );
				}
			}
		}
#endif
	}

	itemptr = Menu_ItemAtCursor( menu );
	if ( itemptr && itemptr->statusbar)
		itemptr->statusbar( ( void * ) itemptr );
}

/*
=================
Menu_ItemAtCursor
=================
*/
void *Menu_ItemAtCursor( menuframework_s *m )
{
	if ( m->cursor < 0 || m->cursor >= m->nitems )
		return 0;

	return m->items[m->cursor];
}

/*
=================
Menu_ActivateItem
=================
*/
sfxHandle_t Menu_ActivateItem( menuframework_s *s, menucommon_s* item ) {
	if ( item->callback ) {
		item->callback( item, QM_ACTIVATED );
		if( !( item->flags & QMF_SILENT ) ) {
			return menu_move_sound;
		}
	}

	return 0;
}

/*
=================
Menu_DefaultKey
=================
*/
sfxHandle_t Menu_DefaultKey( menuframework_s *m, int key )
{
	sfxHandle_t		sound = 0;
	menucommon_s	*item;
	int				cursor_prev;

	// menu system keys
	switch ( key )
	{
		case K_MOUSE2:
		case K_ESCAPE:
			UI_PopMenu();
			return menu_out_sound;
	}

	if (!m || !m->nitems)
		return 0;

	// route key stimulus to widget
	item = Menu_ItemAtCursor( m );
	if (item && !(item->flags & (QMF_GRAYED|QMF_INACTIVE)))
	{
		switch (item->type)
		{
			case MTYPE_SPINCONTROL:
				sound = SpinControl_Key( (menulist_s*)item, key );
				break;

			case MTYPE_RADIOBUTTON:
				sound = RadioButton_Key( (menuradiobutton_s*)item, key );
				break;

			case MTYPE_SLIDER:
				sound = Slider_Key( (menuslider_s*)item, key );
				break;

			case MTYPE_SCROLLLIST:
				sound = ScrollList_Key( (menulist_s*)item, key );
				break;

			case MTYPE_FIELD:
				sound = MenuField_Key( (menufield_s*)item, &key );
				break;
			case MTYPE_SCROLLBAR:
				sound = ScrollBar_Key( (menuscrollbar_s*)item, key );
				break;
		}

		if (sound) {
			// key was handled
			return sound;		
		}
	}

	// default handling
	switch ( key )
	{
#ifndef NDEBUG
		case K_F11:
			uis.debug ^= 1;
			break;

		case K_F12:
			trap_Cmd_ExecuteText(EXEC_APPEND, "screenshot\n");
			break;
#endif
		case K_KP_UPARROW:
		case K_UPARROW:
			cursor_prev    = m->cursor;
			m->cursor_prev = m->cursor;
			m->cursor--;
			Menu_AdjustCursor( m, -1 );
			if ( cursor_prev != m->cursor ) {
				Menu_CursorMoved( m );
				sound = menu_move_sound;
			}
			break;

		case K_TAB:
		case K_KP_DOWNARROW:
		case K_DOWNARROW:
			cursor_prev    = m->cursor;
			m->cursor_prev = m->cursor;
			m->cursor++;
			Menu_AdjustCursor( m, 1 );
			if ( cursor_prev != m->cursor ) {
				Menu_CursorMoved( m );
				sound = menu_move_sound;
			}
			break;

		case K_MOUSE1:
		case K_MOUSE3:
			if (item)
				if ((item->flags & QMF_HASMOUSEFOCUS) && !(item->flags & (QMF_GRAYED|QMF_INACTIVE)))
					return (Menu_ActivateItem( m, item ));
			break;

		case K_JOY1:
		case K_JOY2:
		case K_JOY3:
		case K_JOY4:
		case K_AUX1:
		case K_AUX2:
		case K_AUX3:
		case K_AUX4:
		case K_AUX5:
		case K_AUX6:
		case K_AUX7:
		case K_AUX8:
		case K_AUX9:
		case K_AUX10:
		case K_AUX11:
		case K_AUX12:
		case K_AUX13:
		case K_AUX14:
		case K_AUX15:
		case K_AUX16:
		case K_KP_ENTER:
		case K_ENTER:
			if (item)
				if (!(item->flags & (QMF_MOUSEONLY|QMF_GRAYED|QMF_INACTIVE)))
					return (Menu_ActivateItem( m, item ));
			break;
	}

	return sound;
}

/*
=================
Menu_Cache
=================
*/
void Menu_Cache( void )
{
	uis.charset			= trap_R_RegisterShaderNoMip( "gfx/2d/bigchars" );
	uis.charsetProp		= trap_R_RegisterShaderNoMip( "menu/art/font1_prop.tga" );
	uis.charsetPropGlow	= trap_R_RegisterShaderNoMip( "menu/art/font1_prop_glo.tga" );
	uis.charsetPropB	= trap_R_RegisterShaderNoMip( "menu/art/font2_prop.tga" );
	uis.cursor          = trap_R_RegisterShaderNoMip( "menu/art/3_cursor2" );
	uis.rb_on           = trap_R_RegisterShaderNoMip( "menu/art/switch_on" );
	uis.rb_off          = trap_R_RegisterShaderNoMip( "menu/art/switch_off" );

	uis.whiteShader = trap_R_RegisterShaderNoMip( "white" );
	if ( uis.glconfig.hardwareType == GLHW_RAGEPRO ) {
		// the blend effect turns to shit with the normal 
		uis.menuBackShader	= trap_R_RegisterShaderNoMip( "menubackRagePro" );
	} else {
		uis.menuBackShader	= trap_R_RegisterShaderNoMip( "menuback" );
	}
	uis.menuBackNoLogoShader = trap_R_RegisterShaderNoMip( "menubacknologo" );

	menu_in_sound	= trap_S_RegisterSound( "sound/misc/menu1.wav" );
	menu_move_sound	= trap_S_RegisterSound( "sound/misc/menu2.wav" );
	menu_out_sound	= trap_S_RegisterSound( "sound/misc/menu3.wav" );
	menu_buzz_sound	= trap_S_RegisterSound( "sound/misc/menu4.wav" );

	// need a nonzero sound, make an empty sound for this
	menu_null_sound = -1;

	sliderBar = trap_R_RegisterShaderNoMip( "menu/art/slider2" );
	sliderButton_0 = trap_R_RegisterShaderNoMip( "menu/art/sliderbutt_0" );
	sliderButton_1 = trap_R_RegisterShaderNoMip( "menu/art/sliderbutt_1" );
}
	
