#!usr/common/bin/perl -w

# ReMaic (previously known as CamDem)
# Utility to change the camera view within (.ls format) Quake demos

$pi_under_180 = 45 / atan2(1,1);
$PLAYER = 1;
$CAMERA = 449;

&get_LMPC();
&get_arguments();
&read_camera_instructions();
&process_header_blocks();
while (not eof IN) { &write_block(&process_block(&read_block())) };
print STDOUT "blocks filmed\n";
close(IN);
close(OUT);
close(LOG);
&make_dem($output);

sub error {
    my($message) = @_;
    print " -- problem:\n";
    die $message;
}

sub read_camera_instructions {
    open(LOG, ">remaic.log") or die "Couldn't write remaic.log: $!";
    %history = %noted = @texts = @target = @camera = ();
    $cam_com = '';
    $noted{$PLAYER} = 1;
    foreach $line (<CAMERA>) {
        $line =~ s/#.*$//;            # remove comments
        $line =~ s/^\s*//;
        $line =~ s/\s*$//;            # remove bounding whitespace
        unless ($line eq '') {
            if ($line =~ /^[\d\.]*\s+((eyes)|(stay)|(cut)|(move)|(end move))/i) {
                push (@camera,$line)
            } elsif ($line =~ /^([\d\.]*)\s+text\S*\s(\w+).*(".*")/) {
                push (@texts,$1,$2,$3)
            } elsif ($line =~ /^[\d\.]*\s+((turn)|(view)|(pan)|(end pan)|(end turn))/i) {
                push (@target,$line)
            } else {
                &error("Couldn't process command in camera file:\n$line")
            };
        };
        while ($line =~ s/e(\d+)//i) {
            if ($1 == $CAMERA) {
                &error("Entity $CAMERA is the camera! Yuk! Recursive mess!")
            };
            $noted{$1} = 1;  # note any entities used
        };
    };
    print LOG ("Camera instructions read as:\n",join("\n",@camera));
    print LOG ("Text instructions read as:\n",join("\n",@texts));
    print LOG ("Target instructions read as:\n",join("\n",@target));
    close(CAMERA);
    push (@camera, '999999 s end');
    push (@target, '999999 v end');
    foreach $entity (keys %noted) {
        $history{$entity} = { 'x' => [ ],
                              'y' => [ ],
                              'z' => [ ] };
    };
    push(@texts,99999,'never','ever');
    &get_next_text();
}

sub process_header_blocks {
    $b = 0;
    if ($input =~ /start/) { &write_block(&read_block()) };
 # copy cd block and first block, but also give view to our camera
    @block = &read_block();
    splice(@block,-3,1," setview $CAMERA;\n");
    &write_block(@block);
 # copy second block, but also spawn the camera (with eyes as model)
    @block = &read_block();
   # note origins of all important entities
    foreach $entity (keys %history) {
        %stats = &seek_entity(1,@block);
        ($x,$y,$z) = split(/ /,$stats{'default_origin'});
        record_history($entity,0,$x,$y,$z);
        $coords{$entity} = "$x $y $z";
        if ($entity == $PLAYER) {
            $player_model = $stats{'default_modelindex'};
            $player_model =~ s/^(\d+(?:\.\d)?).*$/$1/;
            $eyes_model = $player_model + 1;
        };
    };
    @spawndata = (
 " spawnbaseline {\n",
 "  entity $CAMERA;\n",
 "  default_modelindex $eyes_model; // progs/eyes.mdl\n",
 "  default_frame 0;\n",
 "  default_colormap 1;\n",
 "  default_skin 0;\n",
 "  default_origin 0.000000000 0.000000000 0.000000000;\n",
 "  default_angles 0.000000000 0.000000000 0.000000000;\n",
 " }\n"
    );
    splice(@block,-2,1,@spawndata);
    &write_block(@block);
 # copy third block
    &write_block(&read_block());
}

sub record_history {
    my($entity,$time,$x,$y,$z) = @_;
    push(@{ $history{$entity}{'x'} }, "$time $x");
    push(@{ $history{$entity}{'y'} }, "$time $y");
    push(@{ $history{$entity}{'z'} }, "$time $z");
    print LOG "Recorded entity $entity at ($x $y $z) at $time\n";
}

sub get_next_text {
    $next_text_time = shift(@texts);
    $next_text_command = shift(@texts);
    $next_text = shift(@texts);
}

sub initialise_camera {
    $old_c_x = $old_c_y = $old_c_z = 999999;
    $c_x = $c_y = $c_z = $intermission = 0;
    ($x,$y,$z) = &coords($PLAYER);
    $x += 30; $y += 30; $z += 10;
    $next_cam = "stay $x $y $z";
    $t_x = $t_y = $t_z = 999999;
    $next_targ = 'view p p p';
    $next_targ_time = $next_cam_time = $time = 1;
}

sub coords {
    my($entity) = @_;
    return (split (/ /,$coords{$entity}));
};

sub process_block {
 # note any time and camera view information
    my(@block) = &get_time(@_);
 # update coords and history of all important entities
    foreach $entity (keys %history) {
        %stats = &seek_entity($entity,@block);
        ($x,$y,$z) = &coords($entity);
        $x = ($stats{'origin_x'}) || $x;
        $y = ($stats{'origin_y'}) || $y;
        $z = ($stats{'origin_z'}) || $z;
        $coords{$entity} = "$x $y $z";
        record_history($entity,$time,$x,$y,$z);
    };
    unless (defined $next_cam) { &initialise_camera() }; 
 # process current camera commands and place it in position
    &update_camera();
    &position_camera();
    if ($intermission && ($setview ne $PLAYER)) {
        if ($auto_intermission) {
            $setview = $PLAYER;
        } else {
            print <<INTERMISSION;
Intermission found: you may want to switch back to the player's
 view with the command 'eyes' just before time of $time.
INTERMISSION
            $intermission = 0;
        };
    };
    splice(@block,-1,0," setview $setview;\n");
    ($old_c_x, $old_c_y, $old_c_z) = ($c_x, $c_y, $c_z);
    splice(@block,-1,0,&camera_text($c_x,$c_y,$c_z));
    if ($time >= $next_text_time) {
        splice(@block,1,0," $next_text_command $next_text;\n");
        &get_next_text();
    };
 # process current target commands and point view at target
    &update_target();
    &target_camera();
    splice(@block,1,0,$camera_line);
 # note final figures in the log
    print LOG "time: $time ($next_cam_time $next_targ_time)\n";
    print LOG "camera: ($c_x $c_y $c_z)\n";
    print LOG "target: ($t_x $t_y $t_z)\n";
    return(@block);
}

sub update_camera {
    $time_left = $next_cam_time - $time;
    while ((-5 < $time_left) && ($time_left <= 0)) {
        print LOG "*activate* $next_cam\n";
 # "cut" is a synonym for "stay"
        $next_cam =~ s/cut/stay/;
 # "end move" is a synonym for "stay" at current co-ordinates
        if ($next_cam =~ /end move/) { $next_cam = "stay $cam_coords" };
 # "eyes" takes no co-ordinates
        if ($next_cam =~ /eyes/) { $next_cam .= ' n/a' };
        ($cam_com, $cam_coords) = $next_cam =~ /^([esm])\S*\s+(.*)$/;
        if ($cam_coords =~ /(\S+)\s+(\S+)\s+(\S+)\s+(\S+)/) {
            &error("Too many coords in camera command:\n$next_cam\n");
        };
        $line = shift(@camera);
        ($next_cam_time, $next_cam) = $line =~ /^(\S+)\s+(\S.*)/;
        $time_left = $next_cam_time - $time;
    };
}

sub update_target {
    $time_left = $next_targ_time - $time;
    while ((-3 < $time_left) && ($time_left <= 0)) {
        print LOG "*activate* $next_targ\n";
 # "end pan" and "end turn" are synonyms to "view" current target
        if ($next_targ =~ /end [(pan)|(turn)]/) { $next_targ = "view $targ_coords" };
        ($targ_com, $targ_coords) = $next_targ =~ /^([vpt])\S*\s+(.*)$/;
        if ($targ_coords =~ /(\S+)\s+(\S+)\s+(\S+)\s+(\S+)/) {
            &error("Too many arguments in camera command:\n$next_targ\n");
        };
        $line = shift(@target);
        ($next_targ_time, $next_targ) = $line =~ /^(\S+)\s+(\S.*)/;
        $time_left = $next_targ_time - $time;
    };
}

sub position_camera {
    if ($cam_com eq 'e') {
        $setview = $PLAYER;
        print LOG "Looking through player's eyes\n";
    } else {
        $setview = $CAMERA;
        ($x,$y,$z,$cam_coords)
         = &eval_coords($cam_coords,$old_c_x,$old_c_y,$old_c_z);
        if ($cam_com eq 'm') {
            print LOG "Moving to ($cam_coords) = ($x $y $z)\n";
	    $proportion = ($time - $old_time) / $time_left;
	    if ($proportion > 1) { $proportion = 1 };
            $c_x += $proportion * ($x - $c_x);
            $c_y += $proportion * ($y - $c_y);
            $c_z += $proportion * ($z - $c_z);
        } elsif ($cam_com eq 's') {
            print LOG "Staying at ($cam_coords) = ($x $y $z)\n";
            ($c_x, $c_y, $c_z) = ($x, $y, $z);
	};
    };
}

sub target_camera {
    if ($cam_com eq 'e') {
        $camera_line = $old_view;
    } else {
        ($x,$y,$z,$targ_coords)
          = &eval_coords($targ_coords,$t_x,$t_y,$t_z);
        if ($targ_com eq 'p') {
            print LOG "Panning view to ($targ_coords) = ($x $y $z)\n";
            $proportion = ($time - $old_time) / $time_left;
            if ($proportion > 1) { $proportion = 1 };
            $t_x += $proportion * ($x - $t_x);
            $t_y += $proportion * ($y - $t_y);
            $t_z += $proportion * ($z - $t_z);
            $camera_angles = &angles($t_x-$c_x, $t_y-$c_y, $t_z-$c_z);
        } elsif ($targ_com eq 't') {
            print LOG "Turning view to ($targ_coords) = ($x $y $z)\n";
            $proportion = ($time - $old_time) / $time_left;
            if ($proportion > 1) { $proportion = 1 };
            my ($angle1, $angle2) = split(/ /,$camera_angles);
            $turning_to = &angles($x - $c_x, $y - $c_y, $z - $c_z);
            print LOG "i.e. angles from $camera_angles\nto $turning_to\n";
            my ($aim_angle1, $aim_angle2) = split(/ /,$turning_to);
            $diff_angle1 = $aim_angle1 - $angle1;
            if ($diff_angle1 > 180) { $diff_angle1 -= 360 }
             elsif ($diff_angle1 < -180) { $diff_angle1 += 360 };
            $angle1 += $proportion * $diff_angle1;
            if ($angle1 > 360) { $angle1 -= 360 };
            $diff_angle2 = $aim_angle2 - $angle2;
            if ($diff_angle2 > 180) { $diff_angle2 -= 360 }
             elsif ($diff_angle2 < -180) { $diff_angle2 += 360 };
            $angle2 += $proportion * $diff_angle2;
            if ($angle2 > 360) { $angle2 -= 360 };
            if ($proportion == 1) { ($t_x, $t_y, $t_z) = ($x, $y, $z) };
            $camera_angles = sprintf "%.9f %.9f 0.000000000", ($angle1, $angle2);
        } elsif ($targ_com eq 'v') {
            print LOG "Holding view at ($targ_coords) = ($x $y $z)\n";
            ($t_x, $t_y, $t_z) = ($x, $y, $z);
            $camera_angles = &angles($t_x-$c_x, $t_y-$c_y, $t_z-$c_z);
        };
        $camera_line = " camera $camera_angles;\n";
    };
}

sub eval_coords {
    my($coords,$now_x,$now_y,$now_z) = @_;
    if ($coords =~ /^\S*$/) { $coords = "$coords $coords $coords" };
    my($x,$y,$z) = split(/\s+/,$coords);
    $calc_x = &eval_coord($x,'x',$now_x);
    $calc_y = &eval_coord($y,'y',$now_y);
    $calc_z = &eval_coord($z,'z',$now_z);
    if ($coords =~ /now/i) {
         if ($x =~ /now/i) { $x = $calc_x };
         if ($y =~ /now/i) { $y = $calc_y };
         if ($z =~ /now/i) { $z = $calc_z };
         $coords = "$x $y $z";
    };
    return($calc_x,$calc_y,$calc_z,$coords);
}

sub eval_coord {
    my($coord,$name,$now) = @_;
    my($expr,$offset) = ($coord,0);
    if ($expr =~ /^([+\-]?\d+(?:\.\d+)?)$/) { return $1 };
    if ($expr =~ s/([+\-]\d+(?:\.\d+)?)$//) { $offset = $1 };
    if ($expr =~ s/now//i) { $offset += $now };
    if ($expr eq '') { return($offset) };
    $expr =~ s/p/e1/i;
    if ($expr =~ /^e(\d+)~(\d+(?:\.\d+)?)$/i) {
        return(&look_up($2,$1,$name) + $offset)
    } elsif ($expr =~ /^e(\d+)$/) {
        ($x,$y,$z) = &coords($1);
        if ($name eq 'x') { return $x + $offset }
        elsif ($name eq 'y') { return $y + $offset }
        elsif ($name eq 'z') { return $z + $offset }
    };
    &error("Bad coordinate expression: $coord");
}

sub look_up {
    my($ago,$entity,$axis) = @_;
    do {
        $event = shift(@{ $history{$entity}{$axis} });
        print LOG "reading event: $event\n";
	($when,$what) = split(/ /,$event);
    } until ($when > $time - $ago);
    unshift(@{ $history{$entity}{$axis} },$event);
    return($what);
}
 
sub camera_text {
    my($x,$y,$z) = @_;
    return (" updateentity {\n",
            "  entity $CAMERA;\n",
            (sprintf "  origin_x %.9f;\n", $x),
            (sprintf "  origin_y %.9f;\n", $y),
            (sprintf "  origin_z %.9f;\n", $z),
            " }\n");
}
  
sub angles {
    my($x,$y,$z) = @_;
    unless ($x || $y || $z) {
        print STDERR "Warning: camera and target coincided\n";
        print LOG "*warning* camera position coincided with viewed target\n";
    };
    my $angle2 = ($x == 0) ? ( ($y >= 0) ? 90 : 270 )
                           : ( ($y == 0) ? (($x >= 0) ? 0 : 180 )
                                         : ( $pi_under_180 * atan2($y,$x) ) );
    if ($angle2 < 0) { $angle2 += 360 };
    $xxyy = $x * $x + $y * $y;
    my $angle1 = ($xxyy == 0) ? ( $z >= 0 ? -90 : 90 )
                              : ( $pi_under_180 * atan2(-$z,sqrt($xxyy)) );
    sprintf "%.9f %.9f 0.000000000", ($angle1, $angle2);
}

sub get_time {
    my(@b) = @_;
    $old_time = $time;
    if ($b[1] =~ /camera/) { ($old_view) = splice(@b,1,1) };
    if ($b[1] =~ /time (.*)s;/) {
        $time = $1;
    } elsif ($b[1] =~ /time (.*):(.*)m;/) {
        $time = $1 * 60 + $2;
    };
    if ($print_time) {
        ($printed_time = $time) =~ s/(\....).*$/$1/;
        splice(@b,1,0," centerprint \"$printed_time\\n\";\n");
    };
    return @b;
}

sub seek_entity {
    my($e,@b) = @_;
    my(%answer) = ();
    do { $line = shift(@b) } until (($line =~ /entity $e/) or ($line =~ /^\}/));
    if ($line =~ /entity $e/) {
	$line = shift(@b);
        until ($line =~ /^ \}/) {
	    ($key, $data) = $line =~ /^  (\w+)[ ;](.*?);?$/;
            if (defined $key) { $answer{$key} = $data };
            $line = shift(@b);
	};
    };
    return %answer;
}

sub read_block {
# reads one block on IN up to the next end of block
# end of file into @block.
    my(@block) = ();
    my($line);
    do {
        $line = <IN>;
	if (defined $line) {
            if (($cam_com eq 'e') or
                (($line !~ / vel_/) && ($line !~ /punchangle/)   
                                    && ($line !~ /weaponmodel/))) {
		push(@block,$line);
	    };
            if ($line =~ /intermission/) { $intermission = 1 };
	};
    } until ((not $line) || ($line =~ /^\}/));
    return @block;
}

sub write_block {
# writes @block to OUT
    my(@block) = @_;
    print OUT @block;
    print LOG ("Wrote block ", $b++, "\n\n");
    print "\b\b\b\b\b$b ";
    if ($debug) { $_ = <STDIN> };
    return @block;
}

sub get_arguments {
    unless (defined $ARGV[0]) {
        print <<HELP;
ReMaic v3 by Anthony Bailey <URL:mailto:baileya\@cs.man.ac.uk>
 Calling syntax: remaic [/t][/i] <input> [<screenplay> [<output>]]
  /t flag prints running clock times in output.
  /i flag switches to eyes automatically during intermissions.
  Default input and output file extensions are .ls, camera is .cam
  Default name for camera file is as input, with 'out' as default output
  If camera file specified, default name for output is the same
 Camera file syntax: a number of lines, where:
  <line> = <time> <instruct> [#<commentary>]
  <time> = <decimal number in seconds>
  <instruct> = eyes | text <console command> "<text>"
                | <positioning_command> <position> | end (move|pan|turn)
  <positioning_command> = stay | cut | move | view | pan | turn
  <position> = <coord> <coord> <coord> | <coord>
  <coord> = now | <abs> | <offset expression> | <expression>
  <abs> = <absolute decimal number in pixel units>
  <offset expression> = <expression>(+|-)<abs>
  <expression> = <entity>[~<time>]
  <entity> = p | e<integer>
HELP
        exit;
    };
    $debug = $print_time = $auto_intermission = 0;
    $input = $ARGV[0];
    while ($input =~ s!^/!!) {
        if ($input eq 'debug') {
            print "Debug mode on\n"; $debug = 1;
        } elsif ($input eq 't') {
            print "Write times on finished film\n"; $print_time = 1;
        } elsif ($input eq 'i') {
            print "Use normal intermission views\n"; $auto_intermission = 1;
        } else { die "Unknown command line option: $input" };
        shift @ARGV;
        $input = $ARGV[0];
    };
    my $camera = $ARGV[1] || $input;
    $output = $ARGV[2] || $ARGV[1] || 'out';
    $input = &get_ls($input);
    open(IN, "<$input")
     or die "Couldn't read expected input file $input:\n $!";
    unless ($camera =~ /\./) { $camera .= '.cam' };
    open(CAMERA, "<$camera")
     or die "Couldn't read expected camera file $camera:\n $!";
    unless ($output =~ /\./) { $output .= '.ls' };
    open(OUT, ">$output")
     or die "Couldn't write expected output file $output:\n $!";
    print "ReMaic v3: converting $input to $output using $camera.\n";
    print "0";
}

sub get_ls {
    my($file) = @_;
    if (-e "$file.ls") { $file .= '.ls' }
    elsif (-e "$file.dem") { $file .= '.dem' };
    if ($file =~ /\.dem$/) {
        ($other = $file) =~ s/\.dem$/\.ls/;
        system("$lmpc -s $file $other");
        $file = $other;
    };
    return $file;
}

sub make_dem {
    my($file) = @_;
    ($file_prefix = $file) =~ s/\..*$//;
    system("$lmpc -l $file $file_prefix.dem");
}

sub get_LMPC {
    $lmpc = 'lmpc.exe';
    if (-e 'remaic.cfg') {
        open(CFG,'<remaic.cfg') or die "Couldn't read config file remaic.cfg";
        chomp($lmpc = <CFG>);
        close(CFG);
    };
    while (! -x $lmpc) {
        print STDOUT <<LMPC;
ReMaic didn't find a copy of LMPC as '$lmpc'.
You will need this program to make the *.ls files that ReMaic works with.
If you do not have it, you should stop now and download it from
<URL:http://www.physik.uni-leipzig.de/~girlich/games/>.
Where is your copy? (enter e.g. 'c:\\quake\\utils\\lmpc.exe')
LMPC
        print STDOUT "? ";
        chomp ($lmpc = <STDIN>);
        &write_config();
    };
}

sub write_config {
    open(CFG,'>remaic.cfg') or die "Couldn't write config file remaic.cfg";
    print CFG "$lmpc\n";
    close(CFG);
}
