/*
    Relay -- a tool to record and play Quake2 demos
    Copyright (C) 2000 Conor Davis

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

    Conor Davis
    cedavis@planetquake.com
*/

#include <string.h>

#include "rp_local.h"
#include "dm2.h"
#include "q2utils.h"

/*
=============
SV_AddBlend
=============
*/
void SV_AddBlend (float r, float g, float b, float a, float *v_blend)
{
    float   a2, a3;

    if (a <= 0)
        return;
    a2 = v_blend[3] + (1-v_blend[3])*a; // new total alpha
    a3 = v_blend[3]/a2;     // fraction of color from old

    v_blend[0] = v_blend[0]*a3 + r*(1-a3);
    v_blend[1] = v_blend[1]*a3 + g*(1-a3);
    v_blend[2] = v_blend[2]*a3 + b*(1-a3);
    v_blend[3] = a2;
}

/*
=============
SV_CalcBlend
=============
*/
void SV_CalcBlend (edict_t *ent)
{
    int     contents;
    vec3_t  vieworg;

    ent->client->ps.blend[0] = ent->client->ps.blend[1] = 
        ent->client->ps.blend[2] = ent->client->ps.blend[3] = 0;

    // add for contents
    VectorAdd (ent->s.origin, ent->client->ps.viewoffset, vieworg);
    contents = gi.pointcontents (vieworg);
    if (contents & (CONTENTS_LAVA|CONTENTS_SLIME|CONTENTS_WATER) )
        ent->client->ps.rdflags |= RDF_UNDERWATER;
    else
        ent->client->ps.rdflags &= ~RDF_UNDERWATER;

    if (contents & (CONTENTS_SOLID|CONTENTS_LAVA))
        SV_AddBlend (1.0, 0.3, 0.0, 0.6, ent->client->ps.blend);
    else if (contents & CONTENTS_SLIME)
        SV_AddBlend (0.0, 0.1, 0.05, 0.6, ent->client->ps.blend);
    else if (contents & CONTENTS_WATER)
        SV_AddBlend (0.5, 0.3, 0.2, 0.4, ent->client->ps.blend);
}

/*
=================
ClientEndServerFrame

Called for each player at the end of the server frame
and right after spawning
=================
*/
void ClientEndServerFrame (edict_t *ent)
{
    int             i, relayflags;
    player_state_t  *ps;
    vec3_t          vec;

    relayflags = ent->client->relayflags;

    if (game.player != -1)
    {
        ent->client->ps = dm2in.players[game.player].ps[dm2in.current_frame & UPDATE_MASK];
        ent->client->ps.pmove.pm_type = PM_FREEZE;
        return;
    }

    // check for a disconnected player
    if (dm2in.svd.isdemo == RECORD_RELAY && ent->client->player != -1 && !ISBITSET(current_connected, ent->client->player))
        ChangePlayer(ent, -1);

    if (ent->client->player != -1)
    {
        UpdatePlayerOrigin(ent);

        ps = &dm2in.players[ent->client->player].ps[dm2in.current_frame & UPDATE_MASK];

        if (relayflags & RC_LOCKPOS)
        {
            VectorCopy(ps->pmove.velocity, ent->client->ps.pmove.velocity);
            VectorCopy(ps->viewoffset, ent->client->ps.viewoffset);
            ent->client->ps.pmove.pm_flags = ps->pmove.pm_flags;
            ent->client->ps.pmove.pm_time = ps->pmove.pm_time;
            ent->client->ps.pmove.gravity = ps->pmove.gravity;
            ent->client->ps.rdflags = ps->rdflags;
        }
        else
        {
            VectorClear(ent->client->ps.viewoffset);
            VectorClear(ent->client->ps.kick_angles);
            VectorClear(ent->client->ps.pmove.velocity);
            ent->client->ps.pmove.gravity = 0;
            ent->client->ps.pmove.pm_flags = 0;
            ent->client->ps.pmove.pm_time = 0;
            ent->client->ps.rdflags = 0;
        }

        if (relayflags & RC_LOCKVIEW)
        {
            if (relayflags & RC_LOCKPOS)
            {
                VectorCopy(ps->viewangles, ent->client->ps.viewangles);
                VectorCopy(ps->kick_angles, ent->client->ps.kick_angles);
                ent->client->ps.fov = ps->fov;
            }
            else
            {
                for (i = 0; i < 3; i++)
                    vec[i] = ps->pmove.origin[i] * 0.125;

                VectorSubtract(vec, ent->s.origin, vec);
                VectorNormalize(vec);

                vectoangles(vec, ent->client->ps.viewangles);
            }
            ent->client->ps.pmove.pm_type = PM_FREEZE;
        }
        else if (relayflags & RC_LOCKPOS)
            ent->client->ps.pmove.pm_type = PM_SPECTATOR;

        if (relayflags & RC_STATUSBAR)
        {
            ent->client->ps.gunindex = ps->gunindex;
            ent->client->ps.gunframe = ps->gunframe;
            VectorCopy(ps->gunoffset, ent->client->ps.gunoffset);
            VectorCopy(ps->gunangles, ent->client->ps.gunangles);
            for (i = 0; i < MAX_STATS; i++)
            {
                if (i != STAT_LAYOUTS)
                    ent->client->ps.stats[i] = ps->stats[i];
            }
        }
        else
        {
            ent->client->ps.gunindex = 0;
            ent->client->ps.gunframe = 0;
            VectorClear(ent->client->ps.gunoffset);
            VectorClear(ent->client->ps.gunangles);
            for (i = 0; i < MAX_STATS; i++)
            {
                if (i != STAT_LAYOUTS)
                    ent->client->ps.stats[i] = 0;
            }
        }

        if (relayflags & RC_LAYOUT)
        {
            if (ent->client->show_layout)
                ent->client->ps.stats[STAT_LAYOUTS] = ps->stats[STAT_LAYOUTS] & 1;
        }
    }

    if (relayflags & RC_NOTINT)
    {
        ent->client->ps.stats[STAT_FLASHES] = 0;
        memset(ent->client->ps.blend, 0, sizeof(ent->client->ps.blend));
    }
    else
    {
        if (ent->client->player != -1)
        {
            ent->client->ps.stats[STAT_FLASHES] = ps->stats[STAT_FLASHES];
            memcpy(ent->client->ps.blend, ps->blend, sizeof(ent->client->ps.blend));
        }
        else
            SV_CalcBlend(ent);
    }

    if (ent->client->ps.fov < 1)
        ent->client->ps.fov = 1;
    else if (ent->client->ps.fov > 160)
        ent->client->ps.fov = 90;

    for (i = 0; i < 3; i++)
        ent->client->ps.pmove.origin[i] = ent->s.origin[i] * 8;
    if (ent->client->ps.pmove.pm_type == PM_FREEZE)
    {
        for (i = 0; i < 3; i++)
            ent->client->ps.pmove.delta_angles[i] = ANGLE2SHORT(ent->client->ps.viewangles[i] - ent->client->cmd_angles[i]);
    }

    if (ent->client->bigupdate_time <= level.time)
    {
        if (strcmp(ent->client->oldlayout, ent->client->layout))
        {
            strcpy(ent->client->oldlayout, ent->client->layout);
            gi.WriteByte(SVC_LAYOUT);
            gi.WriteString(ent->client->layout);
            gi.unicast(ent, false);
            
            ent->client->bigupdate_time = level.time + 0.2;
        }
        else if (memcmp(ent->client->oldinventory, ent->client->inventory, sizeof(ent->client->inventory)))
        {
            memcpy(ent->client->oldinventory, ent->client->inventory, sizeof(ent->client->inventory));
            gi.WriteByte(SVC_INVENTORY);
            for (i = 0; i < MAX_ITEMS; i++)
                gi.WriteShort(ent->client->inventory[i]);
            gi.unicast(ent, false);

            ent->client->bigupdate_time = level.time + 0.2;
        }
    }

    if (ent->client->curmenu)
        ent->client->ps.stats[STAT_LAYOUTS] = 1;
}

