/*
    Relay -- a tool to record and play Quake2 demos
    Copyright (C) 2000 Conor Davis

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

    Conor Davis
    cedavis@planetquake.com
*/

#include <string.h>

#include "rp_local.h"
#include "dm2.h"
#include "q2utils.h"

// PlayerName
// Takes a player index and returns the name from the CS_PLAYERSKINS configstring area.
// The configstring is in name\model/skin format, so stop at the \ to get the name only
char *PlayerName(int num)
{
    static char str[8][64];
    static int  index;
    char        *s, *o;

    index = (index + 1) & 7;

    s = dm2in.configstrings[CS_PLAYERSKINS + num];
    o = str[index];
    
    while (*s && *s != '\\')
        *o++ = *s++;
    *o = 0;

    return str[index];
}

int ChangePlayer(edict_t *ent, int index)
{
    if (index == -1)
    {
        ent->client->player = -1;
        gi.WriteByte(SVC_CONFIGSTRING);
        gi.WriteShort(CS_STATUSBAR);
        gi.WriteString("");
        gi.unicast(ent, false);

        ent->client->ps.pmove.pm_flags = 0;
        ent->client->ps.pmove.pm_time = 0;
        ent->client->ps.pmove.gravity = 0;
        ent->client->ps.gunindex = 0;
        ent->client->ps.gunframe = 0;
        VectorClear(ent->client->ps.gunangles);
        VectorClear(ent->client->ps.gunoffset);
        VectorClear(ent->client->ps.viewoffset);
        VectorClear(ent->client->ps.kick_angles);
        ent->client->ps.stats[STAT_LAYOUTS] = 0;
        return 0;
    }

    if (dm2in.svd.isdemo == RECORD_RELAY)
    {
        if (index < 0 || index >= dm2in.maxclients || !ISBITSET(current_connected, index))
            return -1;
        ent->client->player = index;
    }
    else
    {
        ent->client->player = 0;
    }

    if (ent->client->relayflags & RC_STATUSBAR)
    {
        gi.WriteByte(SVC_CONFIGSTRING);
        gi.WriteShort(CS_STATUSBAR);
        gi.WriteString(dm2in.configstrings[CS_STATUSBAR]);
        gi.unicast(ent, false);
    }

    if (ent->client->relayflags & RC_LAYOUT)
    {
        if (!ent->client->curmenu)
            strcpy(ent->client->layout, dm2in.players[ent->client->player].layout);
    }

    if (ent->client->relayflags & RC_INVENTORY)
    {
        memcpy(ent->client->inventory, dm2in.players[ent->client->player].inventory, sizeof(ent->client->inventory));
    }

    return 0;
}

// Updates the client's origin if it is tracking a player
void UpdatePlayerOrigin(edict_t *ent)
{
    int             i, relayflags;
    player_state_t  *ps;
    vec3_t          start, dir, end;
    vec3_t          mins = {-16, -16, -16}, maxs = {16, 16, 16};
    trace_t         tr;

    relayflags = ent->client->relayflags;

    if (ent->client->player == -1)
        return;

    ps = &dm2in.players[ent->client->player].ps[dm2in.current_frame & UPDATE_MASK];

    if (relayflags & RC_LOCKPOS)
    {
        for (i = 0; i < 3; i++)
            ent->s.origin[i] = ps->pmove.origin[i] * 0.125;
    }
    else if (relayflags & RC_LOCKVIEW)
    {
        for (i = 0; i < 3; i++)
            start[i] = ps->pmove.origin[i] * 0.125;
        
        VectorAdd(start, ps->viewoffset, start);
        
        if (relayflags & RC_CHASEVIEW)
            VectorAdd(ent->client->cmd_angles, ps->viewangles, dir);
        else
            VectorCopy(ent->client->cmd_angles, dir);
        AngleVectors(dir, dir, NULL, NULL);
        
        VectorMA(start, -ent->client->dist, dir, end);
        tr = gi.trace(start, mins, maxs, end, NULL, MASK_SOLID);

        VectorSubtract(start, tr.endpos, dir);
        if (relayflags & RC_TRUNCDIST)
        {
            ent->client->dist = VectorLength(dir);
            if (ent->client->dist < 32)
                ent->client->dist = 32;
        }
        
        VectorCopy(tr.endpos, ent->s.origin);
    }
}

