// g_utils.c -- misc utility functions for game module

#include <math.h>
#include <string.h>

#include "rp_local.h"
#include "q2utils.h"

void G_ProjectSource (vec3_t point, vec3_t distance, vec3_t forward, vec3_t right, vec3_t result)
{
    result[0] = point[0] + forward[0] * distance[0] + right[0] * distance[1];
    result[1] = point[1] + forward[1] * distance[0] + right[1] * distance[1];
    result[2] = point[2] + forward[2] * distance[0] + right[2] * distance[1] + distance[2];
}


/*
=============
G_Find

Searches all active entities for the next one that holds
the matching string at fieldofs (use the FOFS() macro) in the structure.

Searches beginning at the edict after from, or the beginning if NULL
NULL will be returned if the end of the list is reached.

=============
*/
edict_t *G_Find (edict_t *from, int fieldofs, char *match)
{
    char    *s;

    if (!from)
        from = g_edicts;
    else
        from++;

    for ( ; from < &g_edicts[globals.num_edicts] ; from++)
    {
        if (!from->inuse)
            continue;
        s = *(char **) ((byte *)from + fieldofs);
        if (!s)
            continue;
        if (!Q_stricmp (s, match))
            return from;
    }

    return NULL;
}


/*
=================
findradius

Returns entities that have origins within a spherical area

findradius (origin, radius)
=================
*/
edict_t *findradius (edict_t *from, vec3_t org, float rad)
{
    vec3_t  eorg;
    int     j;

    if (!from)
        from = g_edicts;
    else
        from++;
    for ( ; from < &g_edicts[globals.num_edicts]; from++)
    {
        if (!from->inuse)
            continue;
        if (from->solid == SOLID_NOT)
            continue;
        for (j=0 ; j<3 ; j++)
            eorg[j] = org[j] - (from->s.origin[j] + (from->mins[j] + from->maxs[j])*0.5);
        if (VectorLength(eorg) > rad)
            continue;
        return from;
    }

    return NULL;
}

/*
=============
TempVector

This is just a convenience function
for making temporary vectors for function calls
=============
*/
float   *tv (float x, float y, float z)
{
    static  int     index;
    static  vec3_t  vecs[8];
    float   *v;

    // use an array so that multiple tempvectors won't collide
    // for a while
    v = vecs[index];
    index = (index + 1)&7;

    v[0] = x;
    v[1] = y;
    v[2] = z;

    return v;
}


/*
=============
VectorToString

This is just a convenience function
for printing vectors
=============
*/
char    *vtos (vec3_t v)
{
    static  int     index;
    static  char    str[8][32];
    char    *s;

    // use an array so that multiple vtos won't collide
    s = str[index];
    index = (index + 1)&7;

    Com_sprintf (s, 32, "(%i %i %i)", (int)v[0], (int)v[1], (int)v[2]);

    return s;
}


vec3_t VEC_UP       = {0, -1, 0};
vec3_t MOVEDIR_UP   = {0, 0, 1};
vec3_t VEC_DOWN     = {0, -2, 0};
vec3_t MOVEDIR_DOWN = {0, 0, -1};

void G_SetMovedir (vec3_t angles, vec3_t movedir)
{
    if (VectorCompare (angles, VEC_UP))
    {
        VectorCopy (MOVEDIR_UP, movedir);
    }
    else if (VectorCompare (angles, VEC_DOWN))
    {
        VectorCopy (MOVEDIR_DOWN, movedir);
    }
    else
    {
        AngleVectors (angles, movedir, NULL, NULL);
    }

    VectorClear (angles);
}


float vectoyaw (vec3_t vec)
{
    float   yaw;
    
    if (/*vec[YAW] == 0 &&*/ vec[PITCH] == 0) 
    {
        yaw = 0;
        if (vec[YAW] > 0)
            yaw = 90;
        else if (vec[YAW] < 0)
            yaw = -90;
    } 
    else
    {
        yaw = (int) (atan2(vec[YAW], vec[PITCH]) * 180 / M_PI);
        if (yaw < 0)
            yaw += 360;
    }

    return yaw;
}


void vectoangles (vec3_t value1, vec3_t angles)
{
    float   forward;
    float   yaw, pitch;
    
    if (value1[1] == 0 && value1[0] == 0)
    {
        yaw = 0;
        if (value1[2] > 0)
            pitch = 90;
        else
            pitch = 270;
    }
    else
    {
        if (value1[0])
            yaw = (int) (atan2(value1[1], value1[0]) * 180 / M_PI);
        else if (value1[1] > 0)
            yaw = 90;
        else
            yaw = -90;
        if (yaw < 0)
            yaw += 360;

        forward = sqrt (value1[0]*value1[0] + value1[1]*value1[1]);
        pitch = (int) (atan2(value1[2], forward) * 180 / M_PI);
        if (pitch < 0)
            pitch += 360;
    }

    angles[PITCH] = -pitch;
    angles[YAW] = yaw;
    angles[ROLL] = 0;
}

char *G_CopyString (char *in)
{
    char    *out;
    
    out = gi.TagMalloc (strlen(in)+1, TAG_LEVEL);
    strcpy (out, in);
    return out;
}


void G_InitEdict (edict_t *e)
{
    e->inuse = true;
    e->classname = "noclass";
    e->s.number = e - g_edicts;
}

/*
=================
G_Spawn

Either finds a free edict, or allocates a new one.
Try to avoid reusing an entity that was recently freed, because it
can cause the client to think the entity morphed into something else
instead of being removed and recreated, which can cause interpolated
angles and bad trails.
=================
*/
edict_t *G_Spawn (void)
{
    int         i;
    edict_t     *e;

    e = &g_edicts[(int)maxclients->value+1];
    for ( i=maxclients->value+1 ; i<globals.num_edicts ; i++, e++)
    {
        // the first couple seconds of server time can involve a lot of
        // freeing and allocating, so relax the replacement policy
        if (!e->inuse && ( e->freetime < 2 || level.time - e->freetime > 0.5 ) )
        {
            G_InitEdict (e);
            return e;
        }
    }
    
    if (i == game.maxentities)
        gi.error ("ED_Alloc: no free edicts");
        
    globals.num_edicts++;
    G_InitEdict (e);
    return e;
}

/*
=================
G_FreeEdict

Marks the edict as free
=================
*/
void G_FreeEdict (edict_t *ed)
{
    gi.unlinkentity (ed);       // unlink from world

    if (ed - g_edicts <= maxclients->value)
        return;

    memset (ed, 0, sizeof(*ed));
    ed->classname = "freed";
    ed->freetime = level.time;
    ed->inuse = false;
}
