/*
    Relay -- a tool to record and play Quake2 demos
    Copyright (C) 2000 Conor Davis

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

    Conor Davis
    cedavis@planetquake.com
*/

#include <string.h>

#include "rp_local.h"
#include "dm2.h"
#include "q2utils.h"

//
// Demo Menu
//

void DemoMenu_Show(edict_t *ent, menu_t *menu)
{
    char    buf[256];
    int     i, count;

    Menu_Start(menu);

    count = 0;
    for (i = 0; i < dm2in.maxclients; i++)
    {
        if (ISBITSET(current_connected, i))
            count++;
    }

    switch(dm2in.svd.isdemo)
    {
    case RECORD_NETWORK:    sprintf(buf, "Type:           Network");    break;
    case RECORD_CLIENT:     sprintf(buf, "Type:           Client");     break;
    case RECORD_SERVER:     sprintf(buf, "Type:           Server");     break;
    case RECORD_RELAY:      sprintf(buf, "Type:           Relay");      break;
    default:                sprintf(buf, "Type:           Unknown");
    }
    Menu_AddItem(buf, "");
    if (dm2in.svd.game[0])
        sprintf(buf, "Game:           %s", dm2in.svd.game);
    else
        strcpy(buf,  "Game:           baseq2");
    Menu_AddItem(buf, "");
    COM_FileBase(dm2in.configstrings[CS_MODELS+1], buf);
    Menu_AddItem(va("Map:            %s", buf), "");
    Menu_AddItem(va("Desc:           %.32s", dm2in.svd.mapname), "");
    Menu_AddItem(va("Maxclients:     %d", dm2in.maxclients), "");
    Menu_AddItem(va("Active players: %d", count), "");

    Menu_Finish(menu);
}

//
// Players Menu
//

void PlayersMenu_Select(edict_t *ent, menuitem_t *item, int key)
{
    if (ChangePlayer(ent, item->param[0]))
        gi.cprintf(ent, PRINT_HIGH, "Unable to switch to player %s\n", item->text);
}

void PlayersMenu_Show(edict_t *ent, menu_t *menu)
{
    int         i, num, index;
 
    // if menu is newly opened (cur == -1), put cursor on currently
    // tracked player, otherwise try to keep the cursor on the same
    // player after the menu changes
    if (menu->cur == -1 || !menu->items)
        num = ent->client->player + 1;
    else
    {
//        if (menu->items[menu->cur])
            num = menu->items[menu->cur].param[0];
//        else
//            num = -1;
    }

    Menu_Start(menu);

    Menu_AddItem("No player", "F0", PlayersMenu_Select, -1);
    if (dm2in.svd.isdemo == RECORD_RELAY)
    {
        for (i = 0; i < dm2in.maxclients; i++)
        {
            if (ISBITSET(current_connected, i))
            {
                index = Menu_AddItem(PlayerName(i), "F0", PlayersMenu_Select, i);
                if (index == num)
                    menu->cur = i;
            }
        }
    }
    else
        Menu_AddItem(PlayerName(dm2in.svd.player), "F0", PlayersMenu_Select, 0);

    Menu_Finish(menu);
}

//
// Settings Menu
//

void SettingsMenu_Select(edict_t *ent, menuitem_t *item, int key)
{
    ent->client->relayflags ^= item->param[0];
    if (item->param[0] & RC_STATUSBAR)
    {
        if (ent->client->relayflags & RC_STATUSBAR)
        {
            if (ent->client->player != -1)
            {
                gi.WriteByte(SVC_CONFIGSTRING);
                gi.WriteShort(CS_STATUSBAR);
                gi.WriteString(dm2in.configstrings[CS_STATUSBAR]);
                gi.unicast(ent, false);
            }
        }
        else
        {
            gi.WriteByte(SVC_CONFIGSTRING);
            gi.WriteShort(CS_STATUSBAR);
            gi.WriteString("");
            gi.unicast(ent, false);
        }
    }

    Menu_Update(ent, MENU_SETTINGS);
}

void SettingsMenu_Show(edict_t *ent, menu_t *menu)
{
    int     relayflags;

    relayflags = ent->client->relayflags;

    Menu_Start(menu);

    if (relayflags & RC_LOCKPOS)
        Menu_AddItem("Lock position: On", "F0", SettingsMenu_Select, RC_LOCKPOS);
    else
        Menu_AddItem("Lock position: Off", "F0", SettingsMenu_Select, RC_LOCKPOS);

    if (relayflags & RC_LOCKVIEW)
        Menu_AddItem("Lock view:     On", "F0", SettingsMenu_Select, RC_LOCKVIEW);
    else
        Menu_AddItem("Lock view:     Off", "F0", SettingsMenu_Select, RC_LOCKVIEW);

    if (relayflags & RC_CHASEVIEW)
        Menu_AddItem("Chase view:    On", "F0", SettingsMenu_Select, RC_CHASEVIEW);
    else
        Menu_AddItem("Chase view:    Off", "F0", SettingsMenu_Select, RC_CHASEVIEW);

    if (relayflags & RC_STATUSBAR)
        Menu_AddItem("Statusbar:     On", "F0", SettingsMenu_Select, RC_STATUSBAR);
    else
        Menu_AddItem("Statusbar:     Off", "F0", SettingsMenu_Select, RC_STATUSBAR);

    if (relayflags & RC_LAYOUT)
        Menu_AddItem("Layout:        On", "F0", SettingsMenu_Select, RC_LAYOUT);
    else
        Menu_AddItem("Layout:        Off", "F0", SettingsMenu_Select, RC_LAYOUT);

    if (relayflags & RC_NOTINT)
        Menu_AddItem("Tinting:       Off", "F0", SettingsMenu_Select, RC_NOTINT);
    else
        Menu_AddItem("Tinting:       On", "F0", SettingsMenu_Select, RC_NOTINT);

    Menu_Finish(menu);
}

//
// Main Menu (static)
//

void MainMenu_Show(edict_t *ent, menu_t *menu)
{
    Menu_Start(menu);

    Menu_AddItem("Demo info", "F0123", Menu_SelectOpen, "TIS", "Demo Info", MENU_DEMO, DemoMenu_Show);
    Menu_AddItem("Show Players", "F0123", Menu_SelectOpen, "TIS", "Players", MENU_PLAYERS, PlayersMenu_Show);
    Menu_AddItem("Settings", "F0123", Menu_SelectOpen, "TIS", "Settings", MENU_SETTINGS, SettingsMenu_Show);

    Menu_Finish(menu);
}
