/*
    Relay -- a tool to record and play Quake2 demos
    Copyright (C) 2000 Conor Davis

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

    Conor Davis
    cedavis@planetquake.com
*/

#include <assert.h>
#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>

#include "rp_local.h"
#include "block.h"
#include "dm2.h"
#include "pak.h"
#include "q2utils.h"
#include "utils.h"

game_import_t   gi;
game_import_t   real_gi;
game_export_t   globals;
game_locals_t   game;
level_locals_t  level;

edict_t     *g_edicts;

void SpawnEntities (char *mapname, char *entities, char *spawnpoint);
void ClientThink (edict_t *ent, usercmd_t *cmd);
qboolean ClientConnect (edict_t *ent, char *userinfo);
void ClientUserinfoChanged (edict_t *ent, char *userinfo);
void ClientDisconnect (edict_t *ent);
void ClientBegin (edict_t *ent);
void ClientCommand (edict_t *ent);
void RunEntity (edict_t *ent);
void WriteGame (char *filename, qboolean autosave);
void ReadGame (char *filename);
void WriteLevel (char *filename);
void ReadLevel (char *filename);
void InitGame (void);
void G_RunFrame (void);
void ServerCommand (void);

cvar_t  *deathmatch;
cvar_t  *password;
cvar_t  *spectator_password;
cvar_t  *needpass;
cvar_t  *maxclients;
cvar_t  *maxspectators;
cvar_t  *dedicated;

cvar_t  *filterban;

cvar_t  *run_pitch;

cvar_t  *flood_msgs;
cvar_t  *flood_persecond;
cvar_t  *flood_waitdelay;

dm2_t   dm2in;
short   edict_table[MAX_EDICTS];
byte    current_connected[MAX_CLIENTS/8];
byte    old_connected[MAX_CLIENTS/8];
PFILE   *infile;
cvar_t  *relayflags;
vec3_t  spawnpoints;
int     numportals;
cvar_t  *demospeed;

//===================================================================


void ShutdownGame (void)
{
    gi.dprintf ("==== ShutdownGame ====\n");

    if (infile)
    {
        pfclose(infile);
        infile = NULL;
    }
    if (dm2in.players)
    {
        gi.TagFree(dm2in.players);
        dm2in.players = NULL;
    }
    RemoveAllPackDirs();

    gi.FreeTags (TAG_LEVEL);
    gi.FreeTags (TAG_GAME);
}

/*
=================
GetGameAPI

Returns a pointer to the structure with all entry points
and global variables
=================
*/
game_export_t *GetGameAPI (game_import_t *import)
{
    real_gi = gi = *import;

    globals.apiversion = GAME_API_VERSION;
    globals.Init = InitGame;
    globals.Shutdown = ShutdownGame;
    globals.SpawnEntities = SpawnEntities;

    globals.WriteGame = WriteGame;
    globals.ReadGame = ReadGame;
    globals.WriteLevel = WriteLevel;
    globals.ReadLevel = ReadLevel;

    globals.ClientThink = ClientThink;
    globals.ClientConnect = ClientConnect;
    globals.ClientUserinfoChanged = ClientUserinfoChanged;
    globals.ClientDisconnect = ClientDisconnect;
    globals.ClientBegin = ClientBegin;
    globals.ClientCommand = ClientCommand;

    globals.RunFrame = G_RunFrame;

    globals.ServerCommand = ServerCommand;

    globals.edict_size = sizeof(edict_t);

    return &globals;
}

#ifndef GAME_HARD_LINKED
// this is only here so the functions in q_shared.c and q_shwin.c can link
void Sys_Error (char *error, ...)
{
    va_list     argptr;
    char        text[1024];

    va_start (argptr, error);
    vsprintf (text, error, argptr);
    va_end (argptr);

    gi.error (ERR_FATAL, "%s", text);
}

void Com_Printf (char *msg, ...)
{
    va_list     argptr;
    char        text[1024];

    va_start (argptr, msg);
    vsprintf (text, msg, argptr);
    va_end (argptr);

    gi.dprintf ("%s", text);
}

#endif

//======================================================================


/*
=================
ClientEndServerFrames
=================
*/
void ClientEndServerFrames (void)
{
    int     i;
    edict_t *ent;

    // calc the player views now that all pushing
    // and damage has been added
    for (i=0 ; i<maxclients->value ; i++)
    {
        ent = g_edicts + 1 + i;
        if (!ent->inuse || !ent->client)
            continue;
        ClientEndServerFrame (ent);
    }

}

/*
=================
CheckNeedPass
=================
*/
void CheckNeedPass (void)
{
    int need;

    // if password or spectator_password has changed, update needpass
    // as needed
    if (password->modified || spectator_password->modified) 
    {
        password->modified = spectator_password->modified = false;

        need = 0;

        if (*password->string && Q_stricmp(password->string, "none"))
            need |= 1;
        if (*spectator_password->string && Q_stricmp(spectator_password->string, "none"))
            need |= 2;

        gi.cvar_set("needpass", va("%d", need));
    }
}

int AdvanceFrame()
{
    block_t in;
    char    in_buffer[MAX_SVSLEN];

    BlockInit(&in, in_buffer, sizeof(in_buffer));

    if (DM2_ReadBlock(&in, infile) < 0)
    {
        gi.error("AdvanceFrame: Error reading dm2 block\n");
        return -1;
    }
    
    if (in.writeoffset == 0xffffffff)
    {
        pfclose(infile);
        infile = NULL;
        if (game.player != -1)
            gi.AddCommandString("killserver\n");
        else
            gi.bprintf(PRINT_HIGH, "End of demo reached\n");
    }
    else if (ParseBlock(&in) < 0)
        return -1;

    return 0;
}

/*
================
G_RunFrame

Advances the world by 0.1 seconds
================
*/
float nextframe_time = 0;

void G_RunFrame (void)
{
    int     i;
    edict_t *ent;

    level.framenum++;
    level.time = level.framenum*FRAMETIME;

    // I'm pretty sure quake2 doesn't calculate baselines until
    // the third frame
    if (level.framenum < 3)
        return;

    if (level.framenum == 3)
    {   // remove ents created by SpawnEntities
        for (i = game.maxentities+1; i < MAX_EDICTS; i++)
        {
            if (g_edicts[i].inuse)
                G_FreeEdict(&g_edicts[i]);
        }
    }

    if (infile)
    {
        for (i = 0; i < game.maxclients; i++)
        {   // only run the demo if someone is connected
            if (g_edicts[i+1].inuse)
            {
                // this prevents a big jump when the demo is unpaused
                if (demospeed->value <= 0)
                    nextframe_time = level.time;
                
                while (infile && demospeed->value > 0 && nextframe_time < level.time)
                {
                    if (AdvanceFrame())
                        return;

                    nextframe_time += (1/demospeed->value)*FRAMETIME;
                }
                break;
            }
        }

        if (dm2in.svd.isdemo == RECORD_RELAY)
        {
            // check if any players connected or disconnected,
            // and update menus accordingly
            if (memcmp(old_connected, current_connected, sizeof(current_connected)))
                Menu_UpdateAll(MENU_PLAYERS);
            memcpy(old_connected, current_connected, sizeof(current_connected));
        }
    }

    //
    // treat each object in turn
    // even the world gets a chance to think
    //
    ent = &g_edicts[0];
    for (i = 0; i < globals.num_edicts; i++, ent++)
    {
        if (!ent->inuse)
            continue;

        if (i > 0 && i <= maxclients->value)
        {
            ClientBeginServerFrame (ent);
            continue;
        }
    }

    // see if needpass needs updated
    CheckNeedPass ();

    // build the playerstate_t structures for all players
    ClientEndServerFrames ();
}

