/*
    Relay -- a tool to record and play Quake2 demos
    Copyright (C) 2000 Conor Davis

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

    Conor Davis
    cedavis@planetquake.com
*/

#include <assert.h>
#include <string.h>

#include "rp_local.h"
#include "q2utils.h"
#include "utils.h"

void CastMessage(char *buffer, size_t len, int id, int recipient, vec3_t origin, multicast_t type, qboolean reliable)
{
    int     i;
    size_t  j;
    edict_t *ent;

    for (i = 0; i < game.maxclients; i++)
    {
        ent = g_edicts + i + 1;
        if (!ent->inuse || !ent->client)
            continue;

        // ignore messages that are out of the viewable space (even unicast messages)
        if (type == MULTICAST_PVS && !gi.inPVS(ent->s.origin, origin))
            continue;
        if (type == MULTICAST_PHS && !gi.inPHS(ent->s.origin, origin))
            continue;

        if (recipient != -1 && recipient != ent->client->player)
            continue;

        gi.WriteByte(id);
        for (j = 0; j < len; j++)
            gi.WriteByte(buffer[j]);
        gi.unicast(ent, reliable);
    }
}

void UpdateEntity(dm2entity_t *entity)
{
    edict_t *ent;

    if (entity->active)
    {
        if (edict_table[entity->s.number])
            ent = g_edicts + edict_table[entity->s.number];
        else
        {
            gi.dprintf("ParseBlock: no entity space available, try reducing maxclients\n");
            return;
        }

        ent->inuse = true;
        ent->s = entity->s;
	    ent->s.number = ent - g_edicts;

        // needed for BSP models to show up (other way?)
        if (ent->s.solid == 31)
            gi.setmodel(ent, dm2in.configstrings[CS_MODELS+ent->s.modelindex]);
        
        if (ent->s.modelindex == 255)
        {
            assert(edict_table[ent->s.skinnum & 0xff] == ent - g_edicts - 1);
        }

        ent->svflags &= ~SVF_NOCLIENT;
        gi.linkentity(ent);
    }
    else if (edict_table[entity->s.number])
    {
        ent = g_edicts + edict_table[entity->s.number];
        ent->svflags |= SVF_NOCLIENT;
    }
}

//
// ParseBlock
//

int ParseBlock(block_t *block)
{
    int         id, result, recipient_index, current_index, nbytes, i;
    player_t    *recipient;
    edict_t     *ent;
    block_t     temp;
    char        temp_buffer[MAX_MSGLEN], *start;
    state_t     *current;

    BlockInit(&temp, temp_buffer, sizeof(temp_buffer));
    result = 0;

    while (block->readoffset < block->writeoffset)
    {
        BlockRewind(&temp);

        id = ReadByte(block);
        if (id & MSG_UNICAST)
        {   // unicast message
            recipient_index = ReadByte(block);
            recipient = &dm2in.players[recipient_index];
            id &= ~MSG_UNICAST;
        }
        else
        {   // multicast message
            recipient_index = -1;
            recipient = &dm2in.players[0];
        }

        if (ReadOverflow(block))
            gi.error("ParseBlock: Overflow reading command id\n");

        //gi.dprintf("%02x ", id);

        start = block->buffer + block->readoffset;

        switch(id)
        {
        case SVC_MUZZLEFLASH:
        case SVC_MUZZLEFLASH2:
            {
                int entity, value;

                nbytes = ReadMuzzleflash(block, &entity, &value);
                if (nbytes < 0)
                    gi.error("ParseBlock: Error reading SVC_MUZZLEFLASH message\n");

                // error checking
                if ((unsigned)entity >= MAX_EDICTS)
                    gi.error("ParseBlock: SVC_MUZZLEFLASH: invalid entity %u\n", entity);

                entity = edict_table[entity];

                WriteMuzzleflash(&temp, entity, value);

                CastMessage(temp.buffer, temp.writeoffset, id, recipient_index, g_edicts[entity].s.origin, MULTICAST_PVS, false);
            }
            break;
        case SVC_TEMP_ENTITY:
            {
                temp_entity_t   m;

                nbytes = ReadTempEntity(block, &dm2in, &m);
                if (nbytes < 0)
                    gi.error("ParseBlock: Error reading SVC_TEMP_ENTITY message\n");

                // error checking
                if ((unsigned)m.entity >= MAX_EDICTS)
                    gi.error("ParseBlock: SVC_TEMP_ENTITY: invalid entity %u\n", m.entity);
                if ((unsigned)m.dest_entity >= MAX_EDICTS)
                    gi.error("ParseBlock: SVC_TEMP_ENTITY: invalid dest_entity %u\n", m.dest_entity);

                m.entity = edict_table[m.entity];
                m.dest_entity = edict_table[m.entity];

                WriteTempEntity(&temp, &dm2in, &m);

                CastMessage(temp.buffer, temp.writeoffset, SVC_TEMP_ENTITY, recipient_index, m.origin, MULTICAST_PVS, false);
            }
            break;
        case SVC_LAYOUT:
            {
                char    string[MAX_MSGLEN];

                nbytes = ReadLayout(block, string, sizeof(string));
                if (nbytes < 0)
                    gi.error("ParseBlock: Error reading SVC_LAYOUT message\n");

                strcpy(recipient->layout, string);

                for (i = 0; i < game.maxclients; i++)
                {
                    ent = g_edicts + i + 1;
                    if (!ent->inuse || !ent->client)
                        continue;

                    if (recipient_index != -1 && ent->client->player != recipient_index)
                        continue;
                    if (ent->client->curmenu)
                        continue;

                    strcpy(ent->client->layout, string);
                    ent->client->show_layout = true;
                }
            }
            break;
        case SVC_INVENTORY:
            {
                short   items[MAX_ITEMS];
                
                nbytes = ReadInventory(block, items);
                if (nbytes < 0)
                    gi.error("ParseBlock: Error reading SVC_INVENTORY message\n");

                memcpy(recipient->inventory, items, sizeof(items));
                for (i = 0; i < game.maxclients; i++)
                {
                    ent = g_edicts + i + 1;
                    if (!ent->inuse || !ent->client)
                        continue;

                    if (recipient_index != -1 && ent->client->player != recipient_index)
                        continue;

                    memcpy(ent->client->inventory, recipient->inventory, sizeof(ent->client->inventory));
                }
            }
            break;
        case SVC_NOP:
            break;
        case SVC_DISCONNECT:
            break;
        case SVC_RECONNECT:
            break;
        case SVC_SOUND:
            {
                int         soundindex, entity, channel;
                float       volume, attenuation, timeofs;
                vec3_t      origin;
                qboolean    positioned;

                nbytes = ReadSound(block, &soundindex, &volume, &attenuation, &timeofs, &entity, &channel, origin, &positioned);
                if (nbytes < 0)
                    gi.error("ParseBlock: Error reading SVC_SOUND message\n");

                // error checking
                if ((unsigned)entity >= MAX_EDICTS)
                    gi.error("ParseBlock: SVC_SOUND: invalid entity %u\n", entity);

                if (!positioned)
                {
                    if (entity)
                        VectorCopy(current->entities[entity].s.origin, origin);
                    else if (dm2in.svd.isdemo == RECORD_NETWORK || dm2in.svd.isdemo == RECORD_CLIENT)
                    {   // "local sound" (no origin info provided)
                        for (i = 0; i < 3; i++)
                            origin[i] = dm2in.players[0].ps[current_index].pmove.origin[i] * 0.125F;
                    }
                }

                entity = edict_table[entity];

                WriteSound(&temp, soundindex, volume, attenuation, timeofs, entity, channel, origin, positioned);

                CastMessage(temp.buffer, temp.writeoffset, SVC_SOUND, recipient_index, origin, MULTICAST_PHS, false);
            }
            break;
        case SVC_PRINT:
            {
                nbytes = ReadPrint(block, NULL, NULL, 0);
                if (nbytes < 0)
                    gi.error("ParseBlock: Error reading SVC_PRINT message\n");

                CastMessage(start, nbytes, SVC_PRINT, recipient_index, NULL, MULTICAST_ALL, false);
            }
            break;
        case SVC_STUFFTEXT:
            {
                char    string[MAX_MSGLEN], *cur, *next;

                nbytes = ReadStufftext(block, string, sizeof(string));
                if (nbytes < 0)
                    gi.error("ParseBlock: Error reading SVC_STUFFTEXT message\n");

                cur = string;
                while (cur)
                {
                    next = SplitCommandString(cur, ';');
                    if (*cur)
                    {
                        if (!strcmp(Argv(cur, 0), "precache"))
                            result |= 0x01;
                    }
                    cur = next;
                }
            }
            break;
        case SVC_SERVERDATA:
            {
                serverdata_t    m;

                nbytes = ReadServerdata(block, &m);
                if (nbytes < 0)
                    gi.error("ParseBlock: Error reading SVC_SERVERDATA message\n");

                dm2in.svd = m;
            }
            break;
        case SVC_CONFIGSTRING:
            {
                int     index;
                char    string[MAX_MSGLEN];

                nbytes = ReadConfigstring(block, &index, string);
                if (nbytes < 0)
                    gi.error("ParseBlock: Error reading SVC_CONFIGSTRING message\n");

                strcpy(dm2in.configstrings[index], string);
                gi.configstring(index, string);
            }
            break;
        case SVC_SPAWNBASELINE:
            {
                int entity;

                entity = ReadBaselineEntity(block, &dm2in.baselines);
                if (entity < 0)
                    gi.error("ParseBlock: Error reading SVC_SPAWNBASELINE message\n");
            }
            break;
        case SVC_CENTERPRINT:
            {
                nbytes = ReadCenterprint(block, NULL, 0);
                if (nbytes < 0)
                    gi.error("ParseBlock: Error reading SVC_CENTERPRINT message\n");

                CastMessage(start, nbytes, SVC_CENTERPRINT, recipient_index, NULL, MULTICAST_ALL, false);
            }
            break;
        case SVC_PLAYERINFO:
            {
                player_state_t  *ps;
                int             num;

                // old PS should have been copied over via SVC_FRAME message
                if (dm2in.svd.isdemo == RECORD_RELAY)
                    ps = &dm2in.players[recipient_index].ps[current_index];
                else
                    ps = &dm2in.players[0].ps[current_index];

                nbytes = ReadPS(block, ps);
                if (nbytes < 0)
                    gi.error("ParseBlock: Error reading SVC_PLAYERINFO message\n");

                // update client origins, just in case a muzzleflash or something happens
                // in the same frame to make sure the client will be in the PVS
                for (i = 0; i < game.maxclients; i++)
                {
                    ent = g_edicts + i + 1;
                    if (!ent->inuse || !ent->client)
                        continue;
                    
                    if (ent->client->player != num)
                        continue;
                    
                    UpdatePlayerOrigin(ent);
                }
            }
            break;
        case SVC_PACKETENTITIES:
            {
                int entity;

                for (;;)
                {
                    entity = ReadPacketEntity(block, current, &dm2in.baselines);

                    if (entity < 0)
                        gi.error("ParseBlock: Error reading SVC_PACKETENTITIES message\n");
                    
                    if (!entity)
                        break;
                    
                    UpdateEntity(&current->entities[entity]);
                }
            }
            break;
        case SVC_FRAME:
            {
                int     seq1, seq2, area_count, connected_count;
                byte    areas[MAX_MAP_AREAPORTALS/8], connected[MAX_CLIENTS/8];

                nbytes = ReadFrame(block, &dm2in, &seq1, &seq2, &area_count, areas, &connected_count, connected);
                if (nbytes < 0)
                    gi.error("ParseBlock: Error reading SVC_FRAME message\n");

                if (dm2in.svd.isdemo != RECORD_SERVER)
                {
                    dm2in.current_frame = seq1;
                    dm2in.delta_frame = seq2;
                    current_index = dm2in.current_frame & UPDATE_MASK;

                    current = &dm2in.states[current_index];
                    // copy data from delta frame to current frame
                    if (dm2in.delta_frame == BASELINES_FRAME)
                    {
                        *current = dm2in.baselines;
                        for (i = 0; i < dm2in.maxclients; i++)
                            memset(&dm2in.players[i].ps[current_index], 0, sizeof(player_state_t));
                    }
                    else
                    {
                        *current = dm2in.states[dm2in.delta_frame & UPDATE_MASK];
                        for (i = 0; i < dm2in.maxclients; i++)
                        {
                            dm2in.players[i].ps[current_index] = dm2in.players[i].ps[dm2in.delta_frame & UPDATE_MASK];
                            dm2in.players[i].ps[current_index].stats[STAT_FLASHES] = 0;
                        }
                    }

                    current->frame = dm2in.current_frame;
                    memcpy(current->areas, areas, area_count);
                    if (dm2in.svd.isdemo == RECORD_RELAY)
                    {
                        for (i = 1; i <= numportals; i++)
                            gi.SetAreaPortalState(i, ISBITSET(current->areas, i));

                        memcpy(current->connected, connected, connected_count);
                    }
                    else
                    {
                        memset(current->connected, 0, sizeof(current->connected));
                        SETBIT(current->connected, 0, true);
                    }
                    memcpy(current_connected, current->connected, sizeof(connected));

                    for (i = 1; i < MAX_EDICTS; i++)
                    {
                        current->entities[i].s.event = 0;
                        VectorCopy(current->entities[i].s.origin, current->entities[i].s.old_origin);
                        UpdateEntity(&current->entities[i]);
                    }
                }
            }
            break;
        default:
            gi.error("ParseBlock: Unknown server command id %d\n", id);
            break;
        }
    }

    //gi.dprintf("\n");
    return result;
}
