/*
    Relay -- a tool to record and play Quake2 demos
    Copyright (C) 2000 Conor Davis

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

    Conor Davis
    cedavis@planetquake.com
*/

#include <stdlib.h>
#include <string.h>

#include "rp_local.h"
#include "dm2.h"
#include "q2utils.h"

void Cmd_Putaway_f(edict_t *ent)
{
    if (ent->client->curmenu)
        Menu_CloseAll(ent, true);
    else
        ent->client->show_layout = false;
}

void Cmd_Inven_f(edict_t *ent)
{
    if (ent->client->curmenu)
        Menu_CloseAll(ent, true);
    else
        Menu_Open(ent, "TIS", "Main Menu", MENU_MAIN, MainMenu_Show);
}

void Cmd_Player_f(edict_t *ent)
{
    int     num;

    if (game.player != -1)
        return;

    if (dm2in.svd.isdemo == RECORD_SERVER)
    {
        gi.cprintf(ent, PRINT_HIGH, "Cannot track players in serverrecord demos\n");
        return;
    }

    if (gi.argc() < 2)
    {
        if (ent->client->player == -1)
            gi.cprintf(ent, PRINT_HIGH, "Not tracking any player\n");
        else
            gi.cprintf(ent, PRINT_HIGH, "Tracking player %d\n", ent->client->player + 1);
        return;
    }

    num = atoi(gi.argv(1)) - 1;
    if (num == -1)
    {
        ent->client->player = -1;
        gi.WriteByte(SVC_CONFIGSTRING);
        gi.WriteShort(CS_STATUSBAR);
        gi.WriteString("");
        gi.unicast(ent, false);

        ent->client->ps.stats[STAT_LAYOUTS] = 0;
        return;
    }

    if (dm2in.svd.isdemo == RECORD_RELAY)
    {
        if (num < 0 || num >= dm2in.maxclients || !ISBITSET(current_connected, num))
        {
            gi.cprintf(ent, PRINT_HIGH, "%d is not a valid player index\n", num + 1);
            return;
        }
        ent->client->player = num;
    }
    else
    {
        ent->client->player = 0;
    }

    if (ent->client->relayflags & RC_STATUSBAR)
    {
        gi.WriteByte(SVC_CONFIGSTRING);
        gi.WriteShort(CS_STATUSBAR);
        gi.WriteString(dm2in.configstrings[CS_STATUSBAR]);
        gi.unicast(ent, false);

        strcpy(ent->client->layout, dm2in.players[ent->client->player].layout);
        memcpy(ent->client->inventory, dm2in.players[ent->client->player].inventory, sizeof(ent->client->inventory));
    }
}

void Cmd_Menu_f(edict_t *ent)
{
    char    *arg;

    if (gi.argc() < 2)
        return;

    arg = gi.argv(1);

    if (!Q_stricmp(arg, "close"))
        Menu_Close(ent, true);
    else if (!Q_stricmp(arg, "main"))
    {
        Menu_CloseAll(ent, false);
        Menu_Open(ent, "TIS", "Main Menu", MENU_MAIN, MainMenu_Show);
    }
    else if (!Q_stricmp(arg, "players"))
    {
        Menu_CloseAll(ent, false);
        Menu_Open(ent, "TIS", "Players", MENU_PLAYERS, PlayersMenu_Show);
    }
    else if (!Q_stricmp(arg, "settings"))
    {
        Menu_CloseAll(ent, false);
        Menu_Open(ent, "TIS", "Settings", MENU_SETTINGS, SettingsMenu_Show);
    }
    else
        gi.dprintf("Unknown menu: %s\n", arg);
}

void Cmd_Use_f(edict_t *ent)
{
    menu_t      *menu;
    int         index, i;
    char        *s;

    s = gi.args();
    index = 0;
    if (!Q_stricmp(s, "Blaster"))               index = 1;
    else if (!Q_stricmp(s, "Shotgun"))          index = 2;
    else if (!Q_stricmp(s, "Super Shotgun"))    index = 3;
    else if (!Q_stricmp(s, "Machinegun"))       index = 4;
    else if (!Q_stricmp(s, "Chaingun"))         index = 5;
    else if (!Q_stricmp(s, "Grenade Launcher")) index = 6;
    else if (!Q_stricmp(s, "Rocket Launcher"))  index = 7;
    else if (!Q_stricmp(s, "HyperBlaster"))     index = 8;
    else if (!Q_stricmp(s, "Railgun"))          index = 9;
    else if (!Q_stricmp(s, "BFG10K"))           index = 10;

    if (!index)
        return;

    menu = ent->client->curmenu;
    if (!menu)
    {
        if (index == 10)
        {
            ChangePlayer(ent, -1);
        }
        else
        {
            if(ChangePlayer(ent, index - 1))
                gi.cprintf(ent, PRINT_HIGH, "%d is not a valid player\n", index);
            else
                gi.cprintf(ent, PRINT_HIGH, "Tracking player: %s\n", PlayerName(ent->client->player));
        }
        return;
    }
    
    for (i = 0; i < 10; i++)
    {
        if (menu->top + i >= menu->num)
            return;

        if (menu->items[i].Select)
        {
            index--;
            if (!index)
            {
                menu->cur = menu->top + i;
                Menu_Display(ent);
                Menu_Select(ent, 0);
                return;
            }
        }
    }
}

/*
=================
ClientCommand
=================
*/
void ClientCommand (edict_t *ent)
{
    char    *cmd;

    if (!ent->client)
        return;     // not fully in game yet

    cmd = gi.argv(0);

    if (!Q_stricmp(cmd, "player"))
        Cmd_Player_f(ent);
    else if (!Q_stricmp(cmd, "lockpos"))
    {
        ent->client->relayflags ^= RC_LOCKPOS;
        if (ent->client->relayflags & RC_LOCKPOS)
            gi.cprintf(ent, PRINT_LOW, "lockpos on\n");
        else
            gi.cprintf(ent, PRINT_LOW, "lockpos off\n");
    }
    else if (!Q_stricmp(cmd, "lockview"))
    {
        ent->client->relayflags ^= RC_LOCKVIEW;
        if (ent->client->relayflags & RC_LOCKVIEW)
            gi.cprintf(ent, PRINT_LOW, "lockview on\n");
        else
            gi.cprintf(ent, PRINT_LOW, "lockview off\n");
    }
    else if (!Q_stricmp(cmd, "chaseview"))
    {
        ent->client->relayflags ^= RC_CHASEVIEW;
        if (ent->client->relayflags & RC_CHASEVIEW)
            gi.cprintf(ent, PRINT_LOW, "chaseview on\n");
        else
            gi.cprintf(ent, PRINT_LOW, "chaseview off\n");
    }
    else if (!Q_stricmp(cmd, "statusbar"))
    {
        ent->client->relayflags ^= RC_STATUSBAR;
        if (ent->client->relayflags & RC_STATUSBAR)
        {
            gi.cprintf(ent, PRINT_LOW, "statusbar on\n");
            if (ent->client->player != -1)
            {
                gi.WriteByte(SVC_CONFIGSTRING);
                gi.WriteShort(CS_STATUSBAR);
                gi.WriteString(dm2in.configstrings[CS_STATUSBAR]);
                gi.unicast(ent, false);
            }
        }
        else
        {
            gi.cprintf(ent, PRINT_LOW, "statusbar off\n");
            gi.WriteByte(SVC_CONFIGSTRING);
            gi.WriteShort(CS_STATUSBAR);
            gi.WriteString("");
            gi.unicast(ent, false);
        }
    }
    else if (!Q_stricmp(cmd, "notint"))
    {
        ent->client->relayflags ^= RC_NOTINT;
        if (ent->client->relayflags & RC_NOTINT)
            gi.cprintf(ent, PRINT_LOW, "notint on\n");
        else
            gi.cprintf(ent, PRINT_LOW, "notint off\n");
    }
    else if (!Q_stricmp(cmd, "menu"))
        Cmd_Menu_f(ent);
    else if (!Q_stricmp(cmd, "inven"))
        Cmd_Inven_f(ent);
    else if (!Q_stricmp(cmd, "invprev"))
        Menu_Prev(ent);
    else if (!Q_stricmp(cmd, "invnext"))
        Menu_Next(ent);
    else if (!Q_stricmp(cmd, "invuse"))
        Menu_Select(ent, 0);
    else if (!Q_stricmp(cmd, "help"))
        Menu_Select(ent, 1);
    else if (!Q_stricmp(cmd, "invdrop"))
        Menu_Close(ent, true);
    else if (!Q_stricmp(cmd, "use"))
        Cmd_Use_f(ent);
    else if (!Q_stricmp(cmd, "putaway"))
        Cmd_Putaway_f(ent);
}
