/*
    Relay -- a tool to record and play Quake2 demos
    Copyright (C) 2000 Conor Davis

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

    Conor Davis
    cedavis@planetquake.com
*/

#include <stdlib.h>
#include <string.h>

#include "rp_local.h"
#include "dm2.h"
#include "q2utils.h"

void ClientUserinfoChanged (edict_t *ent, char *userinfo);

/*
===========
ClientBegin

called when a client has finished connecting, and is ready
to be placed into the game.  This will happen every level load.
============
*/
void ClientBegin (edict_t *ent)
{
    int             i, j;
    player_state_t  *ps;

    ent->client = game.clients + (ent - g_edicts - 1);
    G_InitEdict(ent);

    if (game.player != -1)
    {
        ent->client->player = game.player;

        // make sure the player being watched is in the game
        if (dm2in.current_frame == 0)
        {
            while (infile && !ISBITSET(current_connected, game.player))
                AdvanceFrame();
        }

        ClientEndServerFrame(ent);
        return;
    }

    if (dm2in.svd.isdemo == RECORD_NETWORK || dm2in.svd.isdemo == RECORD_CLIENT)
    {
        ent->client->player = 0;
        if (ent->client->relayflags & RC_STATUSBAR)
        {
            gi.WriteByte(SVC_CONFIGSTRING);
            gi.WriteShort(CS_STATUSBAR);
            gi.WriteString(dm2in.configstrings[CS_STATUSBAR]);
            gi.unicast(ent, true);
        }
    }
    else
    {
        ent->client->player = -1;
        if (ent->client->relayflags & RC_STATUSBAR)
        {
            gi.WriteByte(SVC_CONFIGSTRING);
            gi.WriteShort(CS_STATUSBAR);
            gi.WriteString("");
            gi.unicast(ent, true);
        }
    }

    // find a good spot to put the client
    for (i = 0; i <= dm2in.maxclients; i++)
    {
        if (i == dm2in.maxclients)
        {   // couldn't find a player, try a spawnpoint
            VectorCopy(spawnpoints, ent->s.origin);
            break;
        }
        
        if (!ISBITSET(current_connected, i))
            continue;
        
        ps = &dm2in.players[i].ps[dm2in.current_frame & UPDATE_MASK];
        
        for (j = 0; j < 3; j++)
            ent->s.origin[j] = ps->pmove.origin[j] * 0.125;
        VectorAdd(ent->s.origin, ps->viewoffset, ent->s.origin);
        VectorAdd(ent->s.origin, tv(0, 0, 32), ent->s.origin);
        VectorCopy(ps->viewangles, ent->client->ps.viewangles);
        for (j = 0; j < 3; j++)
        {
            ent->client->ps.pmove.delta_angles[j] = ANGLE2SHORT(ent->client->ps.viewangles[j] - ent->client->cmd_angles[j]);
            ent->client->ps.pmove.origin[j] = ent->s.origin[j] * 8;
        }
        break;
    }
    
    // make sure all view stuff is valid
    ClientEndServerFrame (ent);
}

/*
===========
ClientUserInfoChanged

called whenever the player updates a userinfo variable.

The game can override any of the settings in place
(forcing skins or names, etc) before copying it off.
============
*/
void ClientUserinfoChanged (edict_t *ent, char *userinfo)
{
    char    *s;

    // check for malformed or illegal info strings
    if (!Info_Validate(userinfo))
    {
        strcpy (userinfo, "\\name\\badinfo\\skin\\male/grunt");
    }

    // set name
    s = Info_ValueForKey (userinfo, "name");
    strncpy (ent->client->netname, s, sizeof(ent->client->netname)-1);

    // fov
    ent->client->ps.fov = atoi(Info_ValueForKey(userinfo, "fov"));
    if (ent->client->ps.fov < 1)
        ent->client->ps.fov = 90;
    else if (ent->client->ps.fov > 160)
        ent->client->ps.fov = 160;

    // save off the userinfo in case we want to check something later
    strncpy (ent->client->userinfo, userinfo, sizeof(ent->client->userinfo)-1);
}

qboolean ClientConnect (edict_t *ent, char *userinfo)
{
    char    *value;

    // check to see if they are on the banned IP list
    value = Info_ValueForKey (userinfo, "ip");
    if (SV_FilterPacket(value))
    {
        Info_SetValueForKey(userinfo, "rejmsg", "Banned.");
        return false;
    }

    // check for a password
    value = Info_ValueForKey (userinfo, "password");
    if (*password->string && strcmp(password->string, "none") && 
        strcmp(password->string, value)) {
        Info_SetValueForKey(userinfo, "rejmsg", "Password required or incorrect.");
        return false;
    }

    // they can connect
    ent->client = game.clients + (ent - g_edicts - 1);
    memset(ent->client, 0, sizeof(*ent->client));

    ent->client->relayflags = RC_LOCKPOS|RC_LOCKVIEW|RC_STATUSBAR|RC_LAYOUT;
    ent->client->dist = 100;

    ClientUserinfoChanged (ent, userinfo);

    if (game.maxclients > 1)
        gi.dprintf ("%s connected\n", ent->client->netname);

    ent->svflags = 0; // make sure we start with known default
    return true;
}

void ClientDisconnect (edict_t *ent)
{
    if (!ent->client)
        return;

    gi.dprintf ("%s disconnected\n", ent->client->netname);

    gi.unlinkentity (ent);
    ent->s.modelindex = 0;
    ent->solid = SOLID_NOT;
    ent->inuse = false;
    ent->classname = "disconnected";
}


//==============================================================


edict_t *pm_passent;

// pmove doesn't need to know about passent and contentmask
trace_t PM_trace (vec3_t start, vec3_t mins, vec3_t maxs, vec3_t end)
{
    return gi.trace (start, mins, maxs, end, pm_passent, MASK_PLAYERSOLID);
}

void ClientThink (edict_t *ent, usercmd_t *ucmd)
{
    gclient_t   *client;
    int     i;
    pmove_t pm;

    client = ent->client;

    for (i = 0; i < 3; i++)
        client->cmd_angles[i] = SHORT2ANGLE(ucmd->angles[i]);

    if (client->player != -1)
    {
        if (client->relayflags & RC_LOCKPOS)
            return;
        if (client->relayflags & RC_LOCKVIEW)
        {
            client->dist -= ucmd->forwardmove * ucmd->msec * 0.0005;
            if (client->dist < 32)
                client->dist = 32;
            return;
        }
    }

    pm_passent = ent;

    // set up for pmove
    memset (&pm, 0, sizeof(pm));

    client->ps.pmove.pm_type = PM_SPECTATOR;
    pm.s = client->ps.pmove;

    for (i=0 ; i<3 ; i++)
    {
        pm.s.origin[i] = ent->s.origin[i]*8;
        pm.s.velocity[i] = ent->velocity[i]*8;
    }
    
    if (memcmp(&client->old_pmove, &pm.s, sizeof(pm.s)))
    {
        pm.snapinitial = true;
        //      gi.dprintf ("pmove changed!\n");
    }
    
    pm.cmd = *ucmd;
    
    pm.trace = PM_trace;    // adds default parms
    pm.pointcontents = gi.pointcontents;
    
    // perform a pmove
    gi.Pmove (&pm);
    
    // save results of pmove
    client->ps.pmove = pm.s;
    client->old_pmove = pm.s;
    
    for (i=0 ; i<3 ; i++)
    {
        ent->s.origin[i] = pm.s.origin[i]*0.125;
        ent->velocity[i] = pm.s.velocity[i]*0.125;
    }
    
    client->oldbuttons = client->buttons;
    client->buttons = ucmd->buttons;
    client->latched_buttons |= client->buttons & ~client->oldbuttons;

    // save light level the player is standing on for
    // monster sighting AI
    ent->light_level = ucmd->lightlevel;
}

void ClientBeginServerFrame (edict_t *ent)
{
    ent->client->latched_buttons = 0;
}
