/*
    Relay -- a tool to record and play Quake2 demos
    Copyright (C) 2000 Conor Davis

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

    Conor Davis
    cedavis@planetquake.com
*/

#include <ctype.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>

#include "rp_local.h"
#include "dm2.h"

#define MAX_MENU_ITEMS  128

static menuitem_t temp_items[MAX_MENU_ITEMS];
static int current_item;
  
void Menu_Close(edict_t *ent, qboolean update)
{
    menu_t  *menu;
    int     i;

    menu = ent->client->curmenu;
    if (!menu)
        return;

    if (menu->Close)
        menu->Close(ent, menu);

    // remove menu from list
    ent->client->curmenu = menu->next;

    // free contents
    if (menu->items)
    {
        for (i = 0; i < menu->num; i++)
        {
            if (menu->items[i].text)
                gi.TagFree(menu->items[i].text);
        }
        gi.TagFree(menu->items);
    }
    gi.TagFree(menu);

    if (update)
    {
        if (ent->client->curmenu)
            Menu_Display(ent);
        else
        {
            ent->client->layout[0] = 0;
            ent->client->ps.stats[STAT_LAYOUTS] = 0;
        }
    }
}

void Menu_CloseAll(edict_t *ent, qboolean update)
{
    while (ent->client->curmenu)
        Menu_Close(ent, false);

    if (update)
    {
        ent->client->layout[0] = 0;
        ent->client->ps.stats[STAT_LAYOUTS] = 0;
    }
}

int Menu_AddItem(char *text, char *fmt, ...)
{
    menuitem_t  *item;
    va_list     argptr;

    item = &temp_items[current_item++];
    memset(item, 0, sizeof(menuitem_t));
    item->text = G_CopyString(text);

    if (!fmt)
        return current_item - 1;

    va_start(argptr, fmt);
    while (*fmt)
    {
        switch(toupper(*fmt))
        {
        case 'A':
            item->align = va_arg(argptr, int);
            break;
        case 'O':
            item->indent = va_arg(argptr, int);
            break;
        case 'F':
            item->Select = va_arg(argptr, void *);
            break;
        case '0':
        case '1':
        case '2':
        case '3':
        case '4':
        case '5':
        case '6':
        case '7':
            item->param[*fmt - '0'] = va_arg(argptr, int);
            break;
        default:
            gi.dprintf("Menu_AddItem: unknown option '%c'\n", *fmt);
        }

        fmt++;
    }
    va_end(argptr);

    return current_item - 1;
}

void Menu_Display(edict_t *ent)
{
    char        *pos;
    int         y, i, num;
    menuitem_t  *item;
    menu_t      *menu;

    menu = ent->client->curmenu;
    if (!menu)
    {
        gi.dprintf("DisplayMenu: no menu for %s\n", ent->client->netname);
        return;
    }

    pos = ent->client->layout;
    // title
    if (menu->title && menu->title[0])
        pos += sprintf(pos, "yv 32 xv %d string2 \"%s\" ", 144 - strlen(menu->title)*4, menu->title);

    
    if (menu->cur != -1)
    {
        if (menu->top > menu->cur)
            menu->top = menu->cur;
        if (menu->top < menu->cur - 9)
            menu->top = menu->cur - 9;

        if (menu->top > menu->num - 10)
            menu->top = menu->num - 10;
        if (menu->top < 0)
            menu->top = 0;
    }

    // last chance to stop messups
    if (menu->cur != -1)
    {
        if (menu->cur >= menu->num || !menu->items[menu->cur].Select)
        {
            gi.dprintf("Menu had invalid cur: %d\n", menu->cur);
            menu->cur = -1;
        }
    }

    if (menu->top > 0)
        pos += sprintf(pos, "yv 48 xv 0 string2 \"(Up)\" ");

    num = 0;
    y = 56;
    for (i = 0; i < 10; i++)
    {
        if (menu->top + i >= menu->num)
            break;

        item = &menu->items[menu->top + i];

        if (item->Select)
        {
            num++;
            if (menu->cur == -1)
                menu->cur = menu->top + i;
            if (menu->cur == menu->top + i)
                pos += sprintf(pos, "yv %d xv 0 string2 \"%d %s\" ", y, num % 10, item->text);
            else
                pos += sprintf(pos, "yv %d xv 0 string \"%d %s\" ", y, num % 10, item->text);
        }
        else
            pos += sprintf(pos, "yv %d xv 16 string \"%s\" ", y, item->text);

        y += 8;
    }
    
    if (menu->top + 10 < menu->num)
        pos += sprintf(pos, "yv 136 xv 0 string2 \"(Down)\" ");

    pos += sprintf(pos, "yv 152 xv 0 string2 \"[ ] move cursor up/down\" ");
    pos += sprintf(pos, "yv 160 string2 \"Enter to select; ' to close\" ");
    pos += sprintf(pos, "yv 168 string2 \"F1 for help\" ");
}

void Menu_Update(edict_t *ent, int id)
{
    menu_t  *menu;

    for (menu = ent->client->curmenu; menu; menu = menu->next)
    {
        if (menu->id == id)
        {
            menu->Show(ent, menu);

            // in case the menu was closed
            if (!ent->client->curmenu)
                return;

            if (menu == ent->client->curmenu)
                Menu_Display(ent);
        }
    }
}

void Menu_UpdateAll(int id)
{
    edict_t *ent;
    int     i;

    for (i = 0; i < game.maxclients; i++)
    {
        ent = g_edicts + i + 1;
        if (!ent->inuse || !ent->client)
            continue;

        Menu_Update(ent, id);
    }
}

void Menu_Prev(edict_t *ent)
{
    menu_t      *menu;
    int         i;

    menu = ent->client->curmenu;
    if (!menu)
        return;

    for (i = menu->cur - 1; i >= 0; i--)
    {
        if (menu->items[i].Select)
        {
            menu->cur = i;
            Menu_Display(ent);
            return;
        }
    }

    for (i = menu->num - 1; i > menu->cur; i--)
    {
        if (menu->items[i].Select)
        {
            menu->cur = i;
            Menu_Display(ent);
            return;
        }
    }
}

void Menu_Next(edict_t *ent)
{
    menu_t      *menu;
    int         i;

    menu = ent->client->curmenu;
    if (!menu)
        return;

    for (i = menu->cur + 1; i < menu->num; i++)
    {
        if (menu->items[i].Select)
        {
            menu->cur = i;
            Menu_Display(ent);
            return;
        }
    }

    for (i = 0; i < menu->cur; i++)
    {
        if (menu->items[i].Select)
        {
            menu->cur = i;
            Menu_Display(ent);
            return;
        }
    }
}

void Menu_Select(edict_t *ent, int key)
{
    menu_t      *menu;

    menu = ent->client->curmenu;
    if (!menu)
        return;
    
    if (menu->cur < 0 || menu->cur >= menu->num)
        return;
    if (!menu->items[menu->cur].Select)
        return;

    menu->items[menu->cur].Select(ent, &menu->items[menu->cur], key);
}

// Use SelectOpenMenu as an item's Select function to open a menu when it is selected
// use param[0] to supply the UpdateFunc
void Menu_SelectOpen(edict_t *ent, menuitem_t *item, int key)
{
    if (!item->param[0])
    {
        gi.dprintf("Menu_SelectOpen: param[0] is NULL for item \"%s\"\n", item->text);
        return;
    }

    Menu_Open(ent, (char *)item->param[0], 
        item->param[1],
        item->param[2],
        item->param[3],
        item->param[4],
        item->param[5],
        item->param[6],
        item->param[7]);
}

void Menu_Start(menu_t *menu)
{
    current_item = 0;
    if (menu->items)
    {
        gi.TagFree(menu->items);
        menu->items = NULL;
    }
}

void Menu_Finish(menu_t *menu)
{
    menu->items = gi.TagMalloc(current_item*sizeof(menuitem_t), TAG_LEVEL);
    memcpy(menu->items, temp_items, current_item*sizeof(menuitem_t));
    menu->num = current_item;
}

void Menu_Open(edict_t *ent, char *fmt, ...)
{
    menu_t  *menu;
    va_list argptr;

    if (!fmt)
        return;

    menu = gi.TagMalloc(sizeof(menu_t), TAG_LEVEL);
    memset(menu, 0, sizeof(menu_t));
    menu->cur = -1;

    va_start(argptr, fmt);

    while (*fmt)
    {
        switch(toupper(*fmt))
        {
        case 'T':
            menu->title = G_CopyString(va_arg(argptr, char *));
            break;
        case 'I':
            menu->id = va_arg(argptr, int);
            break;
        case 'S':
            menu->Show = va_arg(argptr, void *);
            break;
        case 'C':
            menu->Close = va_arg(argptr, void *);
            break;
        case '0':
        case '1':
        case '2':
        case '3':
        case '4':
        case '5':
        case '6':
        case '7':
            menu->param[*fmt - '0'] = va_arg(argptr, int);
            break;
        default:
            gi.dprintf("Menu_Open: unknown option '%c'\n", *fmt);
        }

        fmt++;
    }

    va_end(argptr);

    menu->next = ent->client->curmenu;
    ent->client->curmenu = menu;

    if (menu->Show)
        menu->Show(ent, menu);
    else
        return;

    Menu_Display(ent);
}
