/*
    Relay -- a tool to record and play Quake2 demos
    Copyright (C) 2000 Conor Davis

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

    Conor Davis
    cedavis@planetquake.com
*/

#include <stdarg.h>

#include "r_local.h"
#include "block.h"
#include "dm2.h"
#include "q2utils.h"

void import_bprintf(int printlevel, char *fmt, ...)
{
    char    string[MAX_MSGLEN];
    va_list argptr;

    va_start(argptr, fmt);
    vsprintf(string, fmt, argptr);
    va_end(argptr);

    gi.bprintf(printlevel, "%s", string);

    WriteByte(&prefix, SVC_PRINT);
    WritePrint(&prefix, printlevel, string);
}

void import_cprintf(edict_t *ent, int printlevel, char *fmt, ...)
{
    char    string[MAX_MSGLEN];
    va_list argptr;

    va_start(argptr, fmt);
    vsprintf(string, fmt, argptr);
    va_end(argptr);

    gi.cprintf(ent, printlevel, "%s", string);

    // NULL ent means console only
    if (ent)
    {
        WriteByte(&prefix, SVC_PRINT | MSG_UNICAST);
        WriteByte(&prefix, (byte)(EDICT2NUM(ent) - 1));
        WritePrint(&prefix, printlevel, string);
    }
}

void import_centerprintf(edict_t *ent, char *fmt, ...)
{
    char    string[MAX_MSGLEN];
    va_list argptr;

    va_start(argptr, fmt);
    vsprintf(string, fmt, argptr);
    va_end(argptr);

    gi.centerprintf(ent, "%s", string);

    WriteByte(&prefix, SVC_CENTERPRINT | MSG_UNICAST);
    WriteByte(&prefix, (byte)(EDICT2NUM(ent) - 1));
    WriteCenterprint(&prefix, string);
}

void import_sound(edict_t *ent, int channel, int soundindex, float volume, float attenuation, float timeofs)
{
    gi.sound(ent, channel, soundindex, volume, attenuation, timeofs);

    WriteByte(&suffix, SVC_SOUND);
    WriteSound(&suffix, soundindex, volume, attenuation, timeofs, EDICT2NUM(ent), channel, NULL, false);
}

void import_positioned_sound(vec3_t origin, edict_t *ent, int channel, int soundindex, float volume, float attenuation, float timeofs)
{
    gi.sound(ent, channel, soundindex, volume, attenuation, timeofs);

    WriteByte(&suffix, SVC_SOUND);
    WriteSound(&suffix, soundindex, volume, attenuation, timeofs, EDICT2NUM(ent), channel, origin, true);
}

void import_configstring(int num, char *string)
{
    gi.configstring(num, string);
    if (string)
        strcpy(dm2out.configstrings[num], string);
    else
        dm2out.configstrings[num][0] = 0;

    if (dm2out.current_frame != 0)
    {
        WriteByte(&prefix, SVC_CONFIGSTRING);
        WriteConfigstring(&prefix, num, string);
    }
}

int import_modelindex(char *name)
{
    int index;

    index = gi.modelindex(name);
    if (index && !dm2out.configstrings[CS_MODELS+index][0])
    {
        strcpy(dm2out.configstrings[CS_MODELS+index], name);

        if (dm2out.current_frame != 0)
        {
            WriteByte(&prefix, SVC_CONFIGSTRING);
            WriteConfigstring(&prefix, CS_MODELS+index, name);
        }
    }

    return index;
}

int import_soundindex(char *name)
{
    int index;

    index = gi.soundindex(name);
    if (index && !dm2out.configstrings[CS_SOUNDS+index][0])
    {
        strcpy(dm2out.configstrings[CS_SOUNDS+index], name);

        if (dm2out.current_frame != 0)
        {
            WriteByte(&prefix, SVC_CONFIGSTRING);
            WriteConfigstring(&prefix, CS_SOUNDS+index, name);
        }
    }

    return index;
}

int import_imageindex(char *name)
{
    int index;

    index = gi.imageindex(name);
    if (index && !dm2out.configstrings[CS_IMAGES+index][0])
    {
        strcpy(dm2out.configstrings[CS_IMAGES+index], name);

        if (dm2out.current_frame != 0)
        {
            WriteByte(&prefix, SVC_CONFIGSTRING);
            WriteConfigstring(&prefix, CS_IMAGES+index, name);
        }
    }
    
    return index;
}

void import_setmodel(edict_t *ent, char *name)
{
    gi.setmodel(ent, name);
    if (ent->s.modelindex && !dm2out.configstrings[CS_MODELS+ent->s.modelindex][0])
    {
        strcpy(dm2out.configstrings[CS_MODELS+ent->s.modelindex], name);
        
        if (dm2out.current_frame != 0)
        {
            WriteByte(&prefix, SVC_CONFIGSTRING);
            WriteConfigstring(&prefix, CS_MODELS+ent->s.modelindex, name);
        }
    }
}

void import_SetAreaPortalState(int portalnum, qboolean connected)
{
    gi.SetAreaPortalState(portalnum, connected);
    SETBIT(areaportals, portalnum, connected);
}

void import_linkentity(edict_t *ent)
{
    gi.linkentity(ent);
}

void import_unlinkentity(edict_t *ent)
{
    gi.unlinkentity(ent);
}

void import_multicast(vec3_t origin, multicast_t to)
{
    int     id;

    gi.multicast(origin, to);

    // hope that only one message is sent at a time
    id = tempblock.buffer[0];
    switch(id)
    {
    case SVC_PRINT:
    case SVC_DISCONNECT:
    case SVC_RECONNECT:
    case SVC_SERVERDATA:
    case SVC_STUFFTEXT:
    case SVC_CONFIGSTRING:
    case SVC_CENTERPRINT:
    case SVC_SPAWNBASELINE:
        BlockWrite(&prefix, tempblock.buffer, tempblock.writeoffset);
        break;
    case SVC_MUZZLEFLASH:
    case SVC_MUZZLEFLASH2:
    case SVC_TEMP_ENTITY:
    case SVC_LAYOUT:
    case SVC_INVENTORY:
    case SVC_SOUND:
        BlockWrite(&suffix, tempblock.buffer, tempblock.writeoffset);
        break;
    default:
        gi.dprintf("RELAY: game has sent unsupported multicast message type: %d\n", id);
        break;
    }
    BlockRewind(&tempblock);
}

void import_unicast(edict_t *ent, qboolean reliable)
{
    byte    id;

    gi.unicast(ent, reliable);

    id = tempblock.buffer[0];
    switch(id)
    {
    case SVC_PRINT:
    case SVC_DISCONNECT:
    case SVC_RECONNECT:
    case SVC_SERVERDATA:
    case SVC_STUFFTEXT:
    case SVC_CONFIGSTRING:
    case SVC_CENTERPRINT:
    case SVC_SPAWNBASELINE:
        WriteByte(&prefix, (byte)(id | MSG_UNICAST));
        WriteByte(&prefix, (byte)(EDICT2NUM(ent) - 1));
        BlockWrite(&prefix, tempblock.buffer+1, tempblock.writeoffset-1);
        break;
    case SVC_MUZZLEFLASH:
    case SVC_MUZZLEFLASH2:
    case SVC_TEMP_ENTITY:
    case SVC_LAYOUT:
    case SVC_INVENTORY:
    case SVC_SOUND:
        WriteByte(&suffix, (byte)(id | MSG_UNICAST));
        WriteByte(&suffix, (byte)(EDICT2NUM(ent) - 1));
        BlockWrite(&suffix, tempblock.buffer+1, tempblock.writeoffset-1);
        break;
    default:
        gi.dprintf("RELAY: game has sent unsupported unicast message type: %d\n", id);
        break;
    }

    BlockRewind(&tempblock);
}

void import_WriteChar(int c)
{
    gi.WriteChar(c);

    WriteChar(&tempblock, (char)c);
}

void import_WriteByte(int c)
{
    gi.WriteByte(c);

    WriteByte(&tempblock, (byte)c);
}

void import_WriteShort(int c)
{
    gi.WriteShort(c);

    WriteShort(&tempblock, (short)c);
}

void import_WriteLong(int c)
{
    gi.WriteLong(c);

    WriteLong(&tempblock, (long)c);
}

void import_WriteFloat(float f)
{
    gi.WriteFloat(f);
}

void import_WriteString(char *s)
{
    gi.WriteString(s);

    if (s)
        WriteString(&tempblock, s);
    else
        WriteString(&tempblock, "");
}

void import_WritePosition(vec3_t pos)
{
    gi.WritePosition(pos);

    if (pos)
        WritePosition(&tempblock, pos);
    else
        WritePosition(&tempblock, vec3_origin);
}

void import_WriteDir(vec3_t pos)
{
    gi.WriteDir(pos);

    if (pos)
        WriteDir(&tempblock, pos);
    else
        WriteDir(&tempblock, vec3_origin);
}

void import_WriteAngle(float f)
{
    gi.WriteAngle(f);

    WriteAngle(&tempblock, f);
}
