/*
    Relay -- a tool to record and play Quake2 demos
    Copyright (C) 2000 Conor Davis

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

    Conor Davis
    cedavis@planetquake.com
*/

#include <assert.h>
#include <errno.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "shared.h"
#include "block.h"
#include "bsp.h"
#include "democonv.h"
#include "dm2.h"
#include "endian.h"
#include "getopt.h"
#include "pak.h"
#include "q2defines.h"
#include "q2utils.h"
#include "utils.h"

bsp_t   map;
dm2_t   dm2in;
dm2_t   dm2out;
int     playernum;
size_t  startframe;
size_t  endframe;

// things I don't want to recalculate all the time
int     player_cluster;

block_t prefix;
char    prefix_buffer[MAX_MSGLEN];
block_t suffix;
char    suffix_buffer[MAX_MSGLEN];

extern int ParseBlock(block_t *src);

void syntax()
{
    fprintf(stderr, "Demo Converter (dm2conv) " __DATE__ "\n");
    fprintf(stderr, "Copyright (C) 2000 Conor Davis\n");
    fprintf(stderr, "Version " RELAY_VERSION " " RELAY_RELEASE "\n");
    fprintf(stderr, "democonv [option ...] filename filename\n");
    fprintf(stderr, "-h, --help                    display this help and exit\n");
    fprintf(stderr, "-V, --version                 print version information and exit\n");
    fprintf(stderr, "-b, --basedir DIR             uses DIR as the quake2 directory\n");
    fprintf(stderr, "                              (. is default)\n");
    fprintf(stderr, "-p, --player PLAYER           sets the player perspective to use (0-255)\n");
    fprintf(stderr, "                              (0 is default)\n");
    fprintf(stderr, "-s, --start FRAME             beginning frame in source demo to record\n");
    fprintf(stderr, "-e, --end FRAME               end frame in source demo to record\n");
    exit(1);
}

void version()
{
    fprintf(stderr, "democonv " RELAY_VERSION " " RELAY_RELEASE " (" __DATE__ ")\n");
    exit(0);
}

//
// Main Program
//

int main(int argc, char *argv[])
{
    int             c, i, len, option_index, area_count;
    int             current_index, delta_index, cluster1, cluster2;
    block_t         in, out, bspfile;
    char            in_buffer[MAX_SVSLEN], out_buffer[MAX_MSGLEN];
    char            *infilename, *outfilename;
    char            basedir[MAX_OSPATH], path[MAX_OSPATH];
    byte            areas[MAX_MAP_AREAPORTALS/8];
    PFILE           *infile, *outfile;
    dm2entity_t     *entity;
    size_t          start_time, elapsed_time, numframes;
    vec3_t          alt_origin, delta = {0.125F, 0.125F, 0.125F};

    struct option long_options[] =
    {
        {"help",    no_argument,        0,  'h'},
        {"version", no_argument,        0,  'V'},
        {"player",  required_argument,  0,  'p'},
        {"start",   required_argument,  0,  's'},
        {"end",     required_argument,  0,  'e'},
        {"basedir", required_argument,  0,  'b'},
        {NULL,      0,                  0,  0}
    };

    DM2_Init(&dm2in);
    DM2_Init(&dm2out);
    BlockInit(&in, in_buffer, sizeof(in_buffer));
    BlockInit(&out, out_buffer, sizeof(out_buffer));
    BlockInit(&prefix, prefix_buffer, sizeof(prefix_buffer));
    BlockInit(&suffix, suffix_buffer, sizeof(suffix_buffer));
    memset(areas, 0xff, sizeof(areas));
    area_count = MAX_MAP_AREAS/8;
    
    // default settings
    strcpy(basedir, ".");
    playernum = 0;
    startframe = 0;
    endframe = 0;
    numframes = 0;

    option_index = 0;
    for (;;)
    {
        c = getopt_long(argc, argv, "hVp:s:e:", long_options, &option_index);
        if (c == EOF)
            break;

        switch(c)
        {
        case 'V':
            version();
            break;
        case 'b':
            strcpy(basedir, optarg);
            break;
        case 'e':
            endframe = (unsigned)atoi(optarg);
            break;
        case 'h':
            syntax();
            break;
        case 'p':
            playernum = atoi(optarg);
            break;
        case 's':
            startframe = (unsigned)atoi(optarg);
            break;
        default:
            break;
        }
    }

    if (argc - optind != 2)
    {
        fprintf(stderr, "Error: must supply a source and destination filename\n");
        syntax();
    }
    infilename = argv[optind];
    outfilename = argv[optind+1];

    printf("%s -> %s\n", infilename, outfilename);
    infile = pfopen(infilename, "rb");
    if (!infile)
        Sys_Error("Error: Unable to open file %s\n", infilename);

    // read pre-frame information
    for (;;)
    {
        BlockRewind(&prefix);
        BlockRewind(&suffix);

        if (DM2_ReadBlock(&in, infile) < 0)
            Sys_Error("Error reading from %s\n", infilename);

        if (in.writeoffset == 0xffffffff)
            Sys_Error("Premature end of demo\n");

         i = ParseBlock(&in);
         if (i < 0)
             Sys_Error("Error parsing %s\n", infilename);
         if (i & 0x01)
             break;
    }

    if (dm2in.svd.isdemo == RECORD_RELAY)
        dm2in.maxclients = atoi(dm2in.configstrings[CS_MAXCLIENTS]);
    else
        dm2in.maxclients = 1;
    dm2in.players = calloc(dm2in.maxclients, sizeof(player_t));

    if (playernum >= dm2in.maxclients)
        Sys_Error("Error: player out of range (%d >= %d)\n", playernum, dm2in.maxclients);

    // determine where to look for quake2 files
    sprintf(path, "%s/baseq2", basedir);
    AddPackDir(path, PACK_FILES|PACK_PACKS);
    if (dm2in.svd.game[0] && strcmp(dm2in.svd.game, "baseq2"))
    {
        sprintf(path, "%s/%s", basedir, dm2in.svd.game);
        AddPackDir(path, PACK_FILES|PACK_PACKS);
    }

    // get the map name and load the map info
    if (Pack_LoadFile(dm2in.configstrings[CS_MODELS+1], &bspfile) < 0)
        Sys_Error("Error loading bsp file %s\n", dm2in.configstrings[CS_MODELS+1]);
    if (ReadBSP(&map, &bspfile) < 0)
        Sys_Error("Error parsing bsp file %s\n", dm2in.configstrings[CS_MODELS+1]);
    free(bspfile.buffer);

    outfile = pfopen(outfilename, "wb");
    if (!outfile)
        Sys_Error("Error: Unable to write to file %s\n", outfilename);

    dm2out.svd = dm2in.svd;
    dm2out.svd.version = 34;
    dm2out.svd.isdemo = RECORD_CLIENT;
    dm2out.svd.player = playernum;
    dm2out.maxclients = 1;
    dm2out.players = calloc(dm2out.maxclients, sizeof(player_t));
    
    WritePreFrame(&dm2out.svd, NULL, dm2out.configstrings, &dm2out.baselines, outfile);

    // read and write frame information
    start_time = mstime();
    dm2out.delta_frame = BASELINES_FRAME;
    for (;;)
    {
        BlockRewind(&out);

        if (DM2_ReadBlock(&in, infile) < 0)
            Sys_Error("Error reading from %s: %s\n", infilename, strerror(errno));

        if (in.writeoffset == 0xffffffff)
            break;

        if (ParseBlock(&in) < 0)
            Sys_Error("Error parsing %s\n", infilename);

        if (startframe && dm2in.current_frame < startframe)
            continue;
        if (endframe && dm2in.current_frame > endframe)
            break;
        if (!ISBITSET(dm2in.states[dm2in.current_frame & UPDATE_MASK].connected, playernum))
            continue;

        dm2out.current_frame = dm2in.current_frame;

        current_index = dm2out.current_frame & UPDATE_MASK;
        delta_index = dm2out.delta_frame & UPDATE_MASK;

        dm2out.players[0].ps[current_index] = dm2in.players[playernum].ps[dm2in.current_frame & UPDATE_MASK];
        dm2out.states[current_index] = dm2in.states[dm2in.current_frame & UPDATE_MASK];

        // mark entities that are out of the PVS as not visible
        for (i = 1; i < MAX_EDICTS; i++)
        {
            entity = &dm2out.states[current_index].entities[i];

            if (!entity->active)
                continue;
            
            if (player_cluster == -1)
            {   // player is in a solid, mark everything as invisible
                entity->active = false;
                continue;
            }

            if (entity->s.solid == 31)
            {
                // mark all bsp models as visible until we find a way
                // to determine their origins
                continue;
            }
            else
            {
                // roundoff errors can put an entity in a wall (bad),
                // so test two points. This gets rid of most
                // entities incorrectly being marked as invisible.
                VectorAdd(entity->s.origin, delta, alt_origin);
                cluster1 = PointInCluster(&map, entity->s.origin);
                cluster2 = PointInCluster(&map, alt_origin);
                
                if (ClusterVisible(&map, player_cluster, cluster1, DVIS_PVS))
                    continue;
                if (ClusterVisible(&map, player_cluster, cluster2, DVIS_PVS))
                    continue;
                if (entity->s.sound)
                {
                    if (ClusterVisible(&map, player_cluster, cluster1, DVIS_PHS))
                        continue;
                    if (ClusterVisible(&map, player_cluster, cluster2, DVIS_PHS))
                        continue;
                }
            }
            
            entity->active = false;
        }

        // write prefix messages
        BlockWrite(&out, prefix.buffer, prefix.writeoffset);

        // write SVC_FRAME message
        WriteByte(&out, SVC_FRAME);
        WriteFrame(&out, &dm2out, dm2out.current_frame, dm2out.delta_frame, area_count, areas, 0, NULL);

        // write SVC_PLAYERINFO and SVC_PACKETENTITIES messages
        if (dm2out.delta_frame == BASELINES_FRAME)
        {
            player_state_t  dummy;
            memset(&dummy, 0, sizeof(player_state_t));

            WriteByte(&out, SVC_PLAYERINFO);
            WritePS(&out, &dm2out.players[0].ps[current_index], &dummy);

            WriteByte(&out, SVC_PACKETENTITIES);
            WritePacketEntities(&out, &dm2out.states[current_index], &dm2out.baselines, &dm2out.baselines);
        }
        else
        {
            if (dm2out.states[delta_index].frame != dm2out.delta_frame)
                Sys_Error("Unable to find delta frame %u for %u\n", dm2out.delta_frame, dm2out.current_frame);

            WriteByte(&out, SVC_PLAYERINFO);
            WritePS(&out, &dm2out.players[0].ps[current_index], &dm2out.players[0].ps[delta_index]);

            WriteByte(&out, SVC_PACKETENTITIES);
            WritePacketEntities(&out, &dm2out.states[current_index], &dm2out.states[delta_index], &dm2out.baselines);
        }

        // write suffix messages
        BlockWrite(&out, suffix.buffer, suffix.writeoffset);

        BlockRewind(&prefix);
        BlockRewind(&suffix);

        if (WriteOverflow(&out))
        {
            fprintf(stderr, "Warning: Write overflow (%d > %d bytes)\n", out.writeoffset, out.size);
            continue;
        }

        if (DM2_WriteBlock(&out, outfile) < 0)
            Sys_Error("Error writing to %s: %s\n", outfilename, strerror(errno));

        numframes++;
        dm2out.delta_frame = dm2out.current_frame;
    }

    // write end-of-demo marker
    len = LittleLong(0xffffffff);
    if (!pfwrite(&len, 4, 1, outfile))
        Sys_Error("Error writing to %s: %s\n", outfilename, strerror(errno));

    pfclose(infile);
    pfclose(outfile);
    if (dm2in.players)
        free(dm2in.players);
    if (dm2out.players)
        free(dm2out.players);
    RemoveAllPackDirs();

    elapsed_time = mstime() - start_time;
    printf("Seconds elapsed: %f\n", (float)elapsed_time / 1000);
    printf("Frames written:  %u\n", numframes);
    printf("Frames/sec:      %f\n", (float)numframes * 1000 / elapsed_time);

    return 0;
}

void Sys_Error(char *fmt, ...)
{
    va_list argptr;
    char    buffer[0x1000];

    va_start(argptr, fmt);
    vsprintf(buffer, fmt, argptr);
    va_end(argptr);

    fprintf(stderr, "%s", buffer);
    exit(1);
}

void Com_Printf(char *fmt, ...)
{
    va_list argptr;
    char    buffer[0x1000];

    va_start(argptr, fmt);
    vsprintf(buffer, fmt, argptr);
    va_end(argptr);

    printf("%s", buffer);
}
