/*
    Relay -- a tool to record and play Quake2 demos
    Copyright (C) 2000 Conor Davis

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

    Conor Davis
    cedavis@planetquake.com
*/

#ifndef __BSP_H
#define __BSP_H

#include "q2defines.h"
#include "block.h"

#define MAX_MAP_MODELS      1024
#define MAX_MAP_BRUSHES     8192
#define MAX_MAP_ENTITIES    2048
#define MAX_MAP_ENTSTRING   0x40000
#define MAX_MAP_TEXINFO     8192

#define MAX_MAP_AREAS       256
#define MAX_MAP_AREAPORTALS 1024
#define MAX_MAP_PLANES      65536
#define MAX_MAP_NODES       65536
#define MAX_MAP_BRUSHSIDES  65536
#define MAX_MAP_LEAFS       65536
#define MAX_MAP_VERTS       65536
#define MAX_MAP_FACES       65536
#define MAX_MAP_LEAFFACES   65536
#define MAX_MAP_LEAFBRUSHES 65536
#define MAX_MAP_PORTALS     65536
#define MAX_MAP_EDGES       128000
#define MAX_MAP_SURFEDGES   256000
#define MAX_MAP_LIGHTING    0x200000
#define MAX_MAP_VISIBILITY  0x100000

typedef struct
{
    int     fileofs, filelen;
} lump_t;

#define LUMP_ENTITIES       0
#define LUMP_PLANES         1
#define LUMP_VERTEXES       2
#define LUMP_VISIBILITY     3
#define LUMP_NODES          4
#define LUMP_TEXINFO        5
#define LUMP_FACES          6
#define LUMP_LIGHTING       7
#define LUMP_LEAFS          8
#define LUMP_LEAFFACES      9
#define LUMP_LEAFBRUSHES    10
#define LUMP_EDGES          11
#define LUMP_SURFEDGES      12
#define LUMP_MODELS         13
#define LUMP_BRUSHES        14
#define LUMP_BRUSHSIDES     15
#define LUMP_POP            16
#define LUMP_AREAS          17
#define LUMP_AREAPORTALS    18
#define HEADER_LUMPS        19

typedef struct
{
    float       mins[3], maxs[3];
    float       origin[3];      // for sounds or lights
    int         headnode;
    int         firstface, numfaces;    // submodels just draw faces
                                        // without walking the bsp tree
} model_t;

typedef struct
{
    float   point[3];
} vertex_t;

// 0-2 are axial planes
#define PLANE_X         0
#define PLANE_Y         1
#define PLANE_Z         2

// 3-5 are non-axial planes snapped to the nearest
#define PLANE_ANYX      3
#define PLANE_ANYY      4
#define PLANE_ANYZ      5

// planes (x&~1) and (x&~1)+1 are allways opposites

typedef struct
{
    float   normal[3];
    float   dist;
    int     type;       // PLANE_X - PLANE_ANYZ ?remove? trivial to regenerate
} plane_t;

typedef struct
{
    int         planenum;
    int         children[2];    // negative numbers are -(leafs+1), not nodes
    short       mins[3];        // for frustom culling
    short       maxs[3];
    unsigned short  firstface;
    unsigned short  numfaces;   // counting both sides
} node_t;

typedef struct texinfo_s
{
    float       vecs[2][4];     // [s/t][xyz offset]
    int         flags;          // miptex flags + overrides
    int         value;          // light emission, etc
    char        texture[32];    // texture name (textures/*.wal)
    int         nexttexinfo;    // for animations, -1 = end of chain
} texinfo_t;

// note that edge 0 is never used, because negative edge nums are used for
// counterclockwise use of the edge in a face
typedef struct
{
    unsigned short  v[2];       // vertex numbers
} edge_t;

#define MAXLIGHTMAPS    4
typedef struct
{
    unsigned short  planenum;
    short       side;

    int         firstedge;      // we must support > 64k edges
    short       numedges;   
    short       texinfo;

// lighting info
    byte        styles[MAXLIGHTMAPS];
    int         lightofs;       // start of [numstyles*surfsize] samples
} face_t;

typedef struct
{
    int             contents;           // OR of all brushes (not needed?)

    short           cluster;
    short           area;

    short           mins[3];            // for frustum culling
    short           maxs[3];

    unsigned short  firstleafface;
    unsigned short  numleaffaces;

    unsigned short  firstleafbrush;
    unsigned short  numleafbrushes;
} leaf_t;

typedef struct
{
    unsigned short  planenum;       // facing out of the leaf
    short   texinfo;
} brushside_t;

typedef struct
{
    int         firstside;
    int         numsides;
    int         contents;
} brush_t;

// the visibility lump consists of a header with a count, then
// byte offsets for the PVS and PHS of each cluster, then the raw
// compressed bit vectors
#define DVIS_PVS    0
#define DVIS_PHS    1
typedef struct
{
    int         numclusters;
    int         bitofs[8][2];   // bitofs[numclusters][2]
} vis_t;

// each area has a list of portals that lead into other areas
// when portals are closed, other areas may not be visible or
// hearable even if the vis info says that it should be
typedef struct
{
    int     portalnum;
    int     otherarea;
} areaportal_t;

typedef struct
{
    int     numareaportals;
    int     firstareaportal;
} area_t;

typedef struct
{
    int             version;
    int             num[HEADER_LUMPS];
    model_t         models[MAX_MAP_MODELS];
    byte            visdata[MAX_MAP_VISIBILITY];
    byte            lightdata[MAX_MAP_LIGHTING];
    char            entdata[MAX_MAP_ENTSTRING];
    leaf_t          leafs[MAX_MAP_LEAFS];
    plane_t         planes[MAX_MAP_PLANES];
    vertex_t        vertexes[MAX_MAP_VERTS];
    node_t          nodes[MAX_MAP_NODES];
    texinfo_t       texinfo[MAX_MAP_TEXINFO];
    face_t          faces[MAX_MAP_FACES];
    edge_t          edges[MAX_MAP_EDGES];
    unsigned short  leaffaces[MAX_MAP_LEAFFACES];
    unsigned short  leafbrushes[MAX_MAP_LEAFBRUSHES];
    int             surfedges[MAX_MAP_SURFEDGES];
    brush_t         brushes[MAX_MAP_BRUSHES];
    brushside_t     brushsides[MAX_MAP_BRUSHSIDES];
    area_t          areas[MAX_MAP_AREAS];
    areaportal_t    areaportals[MAX_MAP_AREAPORTALS];
} bsp_t;

extern int ReadBSP(bsp_t *bsp, block_t *block);
extern int PointInLeafnum(bsp_t *bsp, vec3_t point);
extern int PointInCluster(bsp_t *bsp, vec3_t point);

extern qboolean ClusterVisible(bsp_t *bsp, int cluster1, int cluster2, int viewset);
extern qboolean inPVS(bsp_t *bsp, vec3_t p1, vec3_t p2);
extern qboolean inPHS(bsp_t *bsp, vec3_t p1, vec3_t p2);

#endif  // __BSP_H