/*
    Relay -- a tool to record and play Quake2 demos
    Copyright (C) 2000 Conor Davis

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

    Conor Davis
    cedavis@planetquake.com
*/

#ifndef __BLOCK_H
#define __BLOCK_H

#include <stddef.h>

#include "q2defines.h"

typedef struct
{
    char        *buffer;        // where the data is kept
    size_t      size;           // number of bytes allocated for buffer
    size_t      readoffset;     // next Read* will read data from
                                // buffer + readoffset; <= writeoffset
    size_t      writeoffset;    // next Write* will write data to
                                // buffer + writeoffset; <= size
    size_t      writelen;       // number of bytes attempted to write
                                // (can be > size)
    qboolean    readoverflow;   // tried to read past writeoffset
    qboolean    writeoverflow;  // tried to write past size
} block_t;

extern void BlockInit(block_t *block, char *buffer, size_t maxsize);
extern void BlockRewind(block_t *block);
extern void BlockRewindRead(block_t *block);
extern qboolean ReadOverflow(block_t *block);
extern qboolean WriteOverflow(block_t *block);
extern const char *BlockRead(block_t *block, char *buffer, size_t len);
extern void BlockWrite(block_t *block, const char *buffer, size_t len);
extern const char *BlockCopy(block_t *dest, block_t *src, size_t len);

extern char ReadChar(block_t *block);
extern void WriteChar(block_t *block, char a);
extern unsigned char ReadByte(block_t *block);
extern void WriteByte(block_t *block, unsigned char a);
extern short ReadShort(block_t *block);
extern void WriteShort(block_t *block, short a);
extern long ReadLong(block_t *block);
extern void WriteLong(block_t *block, long a);
extern unsigned long ReadULong(block_t *block);
extern void WriteULong(block_t *block, unsigned long a);
extern float ReadFloat(block_t *block);
extern const char *ReadString(block_t *block);
extern void WriteString(block_t *block, const char *string);

extern float ReadAngle(block_t *block);
extern void WriteAngle(block_t *block, float a);
extern float ReadAngle16(block_t *block);
extern void WriteAngle16(block_t *block, float a);
extern float ReadCoord(block_t *block);
extern void WriteCoord(block_t *block, float a);
extern float *ReadPosition(block_t *block, float *vec);
extern void WritePosition(block_t *block, const float *vec);
extern short *ReadShortPosition(block_t *block, short *vec);
extern void WriteShortPosition(block_t *block, const short *vec);
extern float *ReadDir(block_t *block, float *a);
extern void WriteDir(block_t *block, const float *a);
extern float *ReadBlendVec(block_t *block, float *a);
extern void WriteBlendVec(block_t *block, const float *a);
extern float *ReadOffsetVec(block_t *block, float *a);
extern void WriteOffsetVec(block_t *block, const float *a);

#endif  // __BLOCK_H