/*
	Relay -- a tool to record and play Quake2 demos
	Copyright (C) 1999 Conor Davis

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

	Conor Davis
	cedavis@epid.org
*/

#include <stdarg.h>
#include <stdlib.h>
#include <string.h>

#include "g_local.h"
#include "dm2.h"

void Menu_FreeContents(menu_t *menu)
{
	menuitem_t	*item, *next;

	if (menu->title)
		gi.TagFree(menu->title);
	menu->title = NULL;

	for (item = menu->items; item; item = next)
	{
		next = item->next;
		gi.TagFree(item->text);
		gi.TagFree(item);
	}
	menu->items = NULL;
}

int Menu_CountItems(menu_t *menu)
{
	menuitem_t	*item;
	int			count;

	count = 0;
	for (item = menu->items; item; item = item->next)
		count++;

	return count;
}

int Menu_IndexFromItem(menu_t *menu, menuitem_t *item)
{
	menuitem_t	*cur;
	int			index;

	index = 0;
	for (cur = menu->items; cur != item; cur = cur->next)
	{
		if (!cur)
			return -1;

		index++;
	}

	return index;
}

menuitem_t *Menu_ItemFromIndex(menu_t *menu, int index)
{
	menuitem_t	*cur;

	cur = menu->items;
	while (index--)
	{
		if (!cur)
			return NULL;

		cur = cur->next;
	}
	return cur;
}

void Menu_Close(edict_t *ent, qboolean update)
{
	menu_t		*menu, *prev;

	menu = ent->client->curmenu;
	if (!menu)
		return;

	// remove menu from list
	if (ent->client->firstmenu == menu)
	{
		ent->client->firstmenu = NULL;
		ent->client->curmenu = NULL;
	}
	else
	{
		for (prev = ent->client->firstmenu; prev->next != menu; prev = prev->next)
			;

		ent->client->curmenu = prev;
		prev->next = NULL;
	}
	
	Menu_FreeContents(menu);
	gi.TagFree(menu);

	if (update)
	{
		if (ent->client->curmenu)
			Menu_Display(ent);
//		else if (ent->client->player != -1 && ent->client->relayflags & RC_LAYOUT)
//			strcpy(ent->client->layout, dm2in.players[ent->client->player].layout);
		else
		{
			ent->client->layout[0] = 0;
			ent->client->ps.stats[STAT_LAYOUTS] = 0;
		}
	}
}

void Menu_CloseAll(edict_t *ent, qboolean update)
{
	while (ent->client->curmenu)
		Menu_Close(ent, false);

	if (update)
	{
	//	if (ent->client->player != -1 && ent->client->relayflags & RC_LAYOUT)
	//		strcpy(ent->client->layout, dm2in.players[ent->client->player].layout);
	//	else
		{
			ent->client->layout[0] = 0;
			ent->client->ps.stats[STAT_LAYOUTS] = 0;
		}
	}
}

menu_t *Menu_Create(edict_t *ent)
{
	menu_t	*menu;

	menu = gi.TagMalloc(sizeof(menu_t), TAG_LEVEL);
	memset(menu, 0, sizeof(menu_t));

	if (ent->client->curmenu)
		ent->client->curmenu->next = menu;
	else
		ent->client->firstmenu = menu;
	ent->client->curmenu = menu;
	
	menu->cur = -1;
	return menu;
}

menuitem_t *Menu_AddItem(menu_t *menu, char *text, char *fmt, ...)
{
	menuitem_t	*item, *prev;
	va_list		argptr;

	item = gi.TagMalloc(sizeof(menuitem_t), TAG_LEVEL);
	if (menu->items)
	{
		for (prev = menu->items; prev->next; prev = prev->next)
			;

		prev->next = item;
	}
	else
		menu->items = item;

	item->text = G_CopyString(text);

	if (!fmt)
		return item;

	va_start(argptr, fmt);
	while (*fmt)
	{
		*fmt = toupper(*fmt);
		switch(*fmt)
		{
		case 'A':
			item->align = va_arg(argptr, int);
			break;
		case 'O':
			item->indent = va_arg(argptr, int);
			break;
		case 'F':
			item->SelectFunc = va_arg(argptr, void *);
			break;
		case '0':
		case '1':
		case '2':
		case '3':
		case '4':
		case '5':
		case '6':
		case '7':
			item->param[*fmt - '0'] = va_arg(argptr, int);
			break;
		default:
			gi.dprintf("AddMenuItem: unknown menu option %c\n", *fmt);
		}

		fmt++;
	}
	va_end(argptr);

	return item;
}

void Menu_Display(edict_t *ent)
{
	char		*pos;
	int			y, i, num;
	menuitem_t	*item;
	menu_t		*menu;

	menu = ent->client->curmenu;
	if (!menu)
	{
		gi.dprintf("DisplayMenu: no menu for %s\n", ent->client->netname);
		return;
	}

	pos = ent->client->layout;
	// title
	if (menu->title && menu->title[0])
		pos += sprintf(pos, "yv 32 xv %d string2 \"%s\" ", 144 - strlen(menu->title)*4, menu->title);

	
	if (menu->cur != -1)
	{
		if (menu->top > menu->cur)
			menu->top = menu->cur;
		if (menu->top < menu->cur - 9)
			menu->top = menu->cur - 9;

		num = Menu_CountItems(menu);
		if (menu->top > num - 10)
			menu->top = num - 10;
		if (menu->top < 0)
			menu->top = 0;
	}

	// last chance to stop messups
	if (menu->cur != -1)
	{
		item = Menu_ItemFromIndex(menu, menu->cur);
		if (!item || !item->SelectFunc)
		{
			gi.dprintf("Menu had invalid cur: %d\n", menu->cur);
			menu->cur = -1;
		}
	}

	if (menu->top > 0)
		pos += sprintf(pos, "yv 48 xv 0 string2 \"(Up)\" ");

	// skip non-visible menu items
	item = Menu_ItemFromIndex(menu, menu->top);

	num = 0;
	y = 56;
	for (i = 0; i < 10; i++)
	{
		if (!item)
			break;

		if (item->SelectFunc)
		{
			num++;
			if (menu->cur == -1)
				menu->cur = menu->top + i;
			if (menu->cur == menu->top + i)
				pos += sprintf(pos, "yv %d xv 0 string2 \"%d %s\" ", y, num % 10, item->text);
			else
				pos += sprintf(pos, "yv %d xv 0 string \"%d %s\" ", y, num % 10, item->text);
		}
		else
			pos += sprintf(pos, "yv %d xv 16 string \"%s\" ", y, item->text);

		item = item->next;
		y += 8;
	}
	
	if (item)
		pos += sprintf(pos, "yv 136 xv 0 string2 \"(Down)\" ");

	pos += sprintf(pos, "yv 152 xv 0 string2 \"[ ] move cursor up/down\" ");
	pos += sprintf(pos, "yv 160 string2 \"Enter to select; ' to close\" ");
	pos += sprintf(pos, "yv 168 string2 \"F1 for help\" ");
}

void Menu_Update(edict_t *ent, int id)
{
	menu_t	*menu;

	for (menu = ent->client->firstmenu; menu; menu = menu->next)
	{
		if (menu->id == id)
		{
			if (menu->UpdateFunc(ent, menu) == false)
			{
				Menu_CloseAll(ent, true);
				return;
			}
			if (menu == ent->client->curmenu)
				Menu_Display(ent);
		}
	}
}

void Menu_UpdateAll(int id)
{
	edict_t	*ent;
	int		i;

	for (i = 0; i < game.maxclients; i++)
	{
		ent = g_edicts + i + 1;
		if (!ent->inuse || !ent->client)
			continue;

		Menu_Update(ent, id);
	}
}

void Menu_Prev(edict_t *ent)
{
	menu_t		*menu;
	menuitem_t	*cur;
	int			i, last;

	menu = ent->client->curmenu;
	if (!menu)
		return;

	last = -1;
	cur = menu->items;
	for (i = 0; cur && i < menu->cur; i++)
	{
		if (cur->SelectFunc)
			last = i;

		cur = cur->next;
	}
	
	if (last != -1)
	{
		menu->cur = last;
		Menu_Display(ent);
		return;
	}

	if (!cur)
		return;

	while(1)
	{
		i++;
		cur = cur->next;
		if (!cur)
			break;

		if (cur->SelectFunc)
			menu->cur = i;
	}
	Menu_Display(ent);
}

void Menu_Next(edict_t *ent)
{
	menu_t		*menu;
	menuitem_t	*cur;
	int			i;

	menu = ent->client->curmenu;
	if (!menu)
		return;

	cur = menu->items;
	for (i = 0; cur && i < menu->cur; i++)
		cur = cur->next;

	if (!cur)
		return;

	do
	{
		i++;
		cur = cur->next;

		if (!cur)
		{
			if (menu->cur == -1)
				return; 
			i = 0;
			cur = menu->items;
		}

		if (cur->SelectFunc)
		{
			menu->cur = i;
			Menu_Display(ent);
			return;
		}
	} while (i != menu->cur);
}

void Menu_Select(edict_t *ent, int key)
{
	menu_t		*menu;
	menuitem_t	*item;

	menu = ent->client->curmenu;
	if (!menu)
		return;
	
	item = Menu_ItemFromIndex(menu, menu->cur);
	if (!item || !item->SelectFunc)
		return;

	item->SelectFunc(ent, item, key);
}

// Use SelectOpenMenu as an item's SelectFunc to open a menu when it is selected
// use param[0] to supply the UpdateFunc
void Menu_SelectOpen(edict_t *ent, menuitem_t *item, int key)
{
	menu_t	*menu;
	int		i;

	if (!item->param[0])
	{
		gi.dprintf("SelectOpenMenu: no UpdateFunc for item \"%s\"\n", item->text);
		return;
	}

	menu = Menu_Create(ent);
	menu->UpdateFunc = (qboolean (*)(edict_t *, menu_t *))item->param[0];
	for (i = 0; i < MAX_MENU_PARMS-1; i++)
		menu->param[i] = item->param[i+1];

	if (menu->UpdateFunc(ent, menu) == false)
	{
		Menu_CloseAll(ent, true);
		return;
	}
	Menu_Display(ent);
}

// Convenient way to open & show a menu
void Menu_Open(edict_t *ent, qboolean (*UpdateFunc)(edict_t *, menu_t *))
{
	menu_t	*menu;

	menu = Menu_Create(ent);
	menu->UpdateFunc = UpdateFunc;
	if (menu->UpdateFunc(ent, menu) == false)
	{
		Menu_CloseAll(ent, true);
		return;
	}
	Menu_Display(ent);
}