/*
	Relay -- a tool to record and play Quake2 demos
	Copyright (C) 1999 Conor Davis

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

	Conor Davis
	cedavis@epid.org
*/

// g_local.h -- local definitions for game module

#include <stdio.h>

// define GAME_INCLUDE so that game.h does not define the
// short, server-visible gclient_t and edict_t structures,
// because we define the full size ones in this file
#define	GAME_INCLUDE

#include "shared.h"
#include "dm2.h"

typedef struct
{
	int			edicts[MAX_EDICTS];
	int			configstrings[MAX_CONFIGSTRINGS];
	int			statusbar[MAX_STATS];
} trans_t;

// server relay options (game.relayflags)
#define R_SQUEEZE		0x01		// rearrange configstrings/edicts in a compact manner

// client relay options (ent->client->relayflags)
#define RC_LOCKPOS		0x01		// lock client's position to player
#define RC_LOCKVIEW		0x02		// lock client's view to/toward player
#define RC_CHASEVIEW	0x04		// w/ lockview, keep view relative to player's
#define RC_TRUNCDIST	0x08		// shorten lockview dist when blocked by walls
#define RC_STATUSBAR	0x10		// enable statusbar
#define RC_LAYOUT		0x20		// enable layouts
#define RC_INVENTORY	0x40		// enable inventory updates
#define RC_NOTINT		0x80		// disable screen blending


// the "gameversion" client command will print this plus compile date
#define	GAMEVERSION	"q2server"

#define	FRAMETIME		0.1

// memory tags to allow dynamic memory to be cleaned up
#define	TAG_GAME	765		// clear when unloading the dll
#define	TAG_LEVEL	766		// clear when loading a new level

// protocol bytes that can be directly added to messages
#define	svc_muzzleflash		1
#define	svc_muzzleflash2	2
#define	svc_temp_entity		3
#define	svc_layout			4
#define	svc_inventory		5
#define	svc_stufftext		11

// handedness values
#define RIGHT_HANDED			0
#define LEFT_HANDED				1
#define CENTER_HANDED			2

typedef struct
{
	gclient_t	*clients;		// [maxclients]

	// store latched cvars here that we want to get at often
	int			maxclients;
	int			maxentities;

	int			relayflags;
} game_locals_t;

typedef struct
{
	int			framenum;
	float		time;
} level_locals_t;

typedef enum {
	F_INT, 
	F_FLOAT,
	F_LSTRING,			// string on disk, pointer in memory, TAG_LEVEL
	F_GSTRING,			// string on disk, pointer in memory, TAG_GAME
	F_VECTOR,
	F_ANGLEHACK,
	F_EDICT,			// index on disk, pointer in memory
	F_ITEM,				// index on disk, pointer in memory
	F_CLIENT,			// index on disk, pointer in memory
	F_FUNCTION,
	F_MMOVE,
	F_IGNORE
} fieldtype_t;

typedef struct
{
	char	*name;
	int		ofs;
	fieldtype_t	type;
	int		flags;
} field_t;

extern	field_t fields[];

extern	game_import_t	gi;
extern	game_export_t	globals;
extern	game_locals_t	game;
extern	level_locals_t	level;

extern	edict_t			*g_edicts;

#define	FOFS(x) (int)&(((edict_t *)0)->x)

#define random()	((rand () & 0x7fff) / ((float)0x7fff))
#define crandom()	(2.0 * (random() - 0.5))

extern	cvar_t	*deathmatch;
extern	cvar_t	*password;
extern	cvar_t	*spectator_password;
extern	cvar_t	*needpass;
extern	cvar_t	*maxclients;
extern	cvar_t	*maxspectators;
extern	cvar_t	*dedicated;

extern	cvar_t	*filterban;

extern	cvar_t	*run_pitch;

extern	cvar_t	*flood_msgs;
extern	cvar_t	*flood_persecond;
extern	cvar_t	*flood_waitdelay;

//============================================================================

typedef struct menu_s menu_t;

struct gclient_s
{
	// known to server
	player_state_t	ps;				// communicated by server to clients
	int				ping;

	char			userinfo[MAX_INFO_STRING];
	char			netname[16];

	vec3_t			v_angle;		// remove?
	vec3_t			cmd_angles;
	int				buttons;
	int				oldbuttons;
	int				latched_buttons;
	pmove_state_t	old_pmove;

	int				relayflags;
	int				player;
	float			dist;

	short			oldinventory[MAX_ITEMS];
	short			inventory[MAX_ITEMS];

	char			oldlayout[MAX_MSGLEN];
	char			layout[MAX_MSGLEN];
	qboolean		layout_modified;				// so we don't need to strcmp the old and new menu's
	qboolean		show_layout;

	float			bigupdate_time;

	menu_t			*firstmenu;
	menu_t			*curmenu;
};

struct edict_s
{
	entity_state_t	s;
	struct gclient_s	*client;	// NULL if not a player
									// the server expects the first part
									// of gclient_s to be a player_state_t
									// but the rest of it is opaque

	qboolean	inuse;
	int			linkcount;

	// FIXME: move these fields to a server private sv_entity_t
	link_t		area;				// linked to a division node or leaf
	
	int			num_clusters;		// if -1, use headnode instead
	int			clusternums[MAX_ENT_CLUSTERS];
	int			headnode;			// unused if num_clusters != -1
	int			areanum, areanum2;

	//================================

	int			svflags;
	vec3_t		mins, maxs;
	vec3_t		absmin, absmax, size;
	solid_t		solid;
	int			clipmask;
	edict_t		*owner;


	// DO NOT MODIFY ANYTHING ABOVE THIS, THE SERVER
	// EXPECTS THE FIELDS IN THAT ORDER!

	//================================
	int			movetype;
	int			flags;

	char		*model;
	float		freetime;			// sv.time when the object was freed
	
	//
	// only used locally in game, not by server
	//
	char		*classname;
	vec3_t		velocity;
	int			light_level;
};

extern game_import_t	real_gi;
extern dm2_t	dm2in;
extern trans_t	dm2in_trans;
extern byte		connected[MAX_CLIENTS/8];
extern byte		old_connected[MAX_CLIENTS/8];
extern char		configstrings[MAX_CONFIGSTRINGS][64];
extern int		configstring_count[MAX_CONFIGSTRINGS];
extern FILE		*infile;
extern vec3_t	spawnpoints;
extern int		numportals;

// g_main.c
extern int ParseBlock(dm2_t *dm2, trans_t *trans, block_t *src);

// g_svcmds.c
extern qboolean SV_FilterPacket (char *from);

// p_view.c
extern void ClientBeginServerFrame (edict_t *ent);
extern void ClientEndServerFrame (edict_t *ent);

// g_utils.c
extern void G_ProjectSource (vec3_t point, vec3_t distance, vec3_t forward, vec3_t right, vec3_t result);
extern edict_t *G_Find (edict_t *from, int fieldofs, char *match);
extern edict_t *findradius (edict_t *from, vec3_t org, float rad);
extern float	*tv (float x, float y, float z);
extern char	*vtos (vec3_t v);
extern void G_SetMovedir (vec3_t angles, vec3_t movedir);
extern float vectoyaw (vec3_t vec);
extern void vectoangles (vec3_t value1, vec3_t angles);
extern char *G_CopyString (char *in);
extern void G_InitEdict (edict_t *e);
extern edict_t *G_Spawn (void);
extern void G_FreeEdict (edict_t *ed);

#include "menu.h"

// r_utils.h
extern int FindConfigstringSlot(int start, int end, char *string);
extern int relay_modelindex(char *string);
extern int relay_soundindex(char *string);
extern int relay_imageindex(char *string);
extern void relay_configstring(int index, char *string);
extern char *PlayerName(int num);
extern void UpdatePlayerOrigin(edict_t *ent);
extern int ChangePlayer(edict_t *ent, int index);

// r_menus.h
extern qboolean ShowPlayersMenu(edict_t *ent, menu_t *menu);
extern qboolean ShowSettingsMenu(edict_t *ent, menu_t *menu);
extern qboolean ShowMainMenu(edict_t *ent, menu_t *menu);
