/*
	Relay -- a tool to record and play Quake2 demos
	Copyright (C) 1999 Conor Davis

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

	Conor Davis
	cedavis@epid.org
*/

#include <direct.h>
#include <string.h>
#include <time.h>

#include "g_local.h"
#include "block.h"
#include "dm2.h"
#include "msg.h"
#include "q2utils.h"
#include "utils.h"

void SpawnEntities (char *mapname, char *entities, char *spawnpoint)
{
	int		i;
	edict_t	*ent;
	char	filename[MAX_QPATH];
	char	gamedir[MAX_OSPATH];
	char	path[MAX_OSPATH];
	cvar_t	*demofile;
	char	*c, t[3] = "% ";
	time_t	now;
	struct	tm *ltime;
	char	format[256][32] = {0};

	InitDM2(&dm2out);
	memset(areaportals, 0, sizeof(areaportals));

	globals.SpawnEntities(mapname, entities, spawnpoint);
	
	last_frame = -1;
	current_frame = 0;

	dm2out.svd.version = 34;
	dm2out.svd.key = 0;
	dm2out.svd.isdemo = RECORD_RELAY;
	strncpy(dm2out.svd.game, gi.cvar("game", "", CVAR_LATCH|CVAR_SERVERINFO)->string, MAX_QPATH-1);
	dm2out.svd.player = 0;
	strncpy(dm2out.svd.mapname, dm2out.configstrings[CS_NAME], MAX_QPATH-1);

	if (dm2out.svd.isdemo == RECORD_NETWORK || dm2out.svd.isdemo == RECORD_CLIENT)
		dm2out.maxclients = 1;
	else
		dm2out.maxclients = maxclients->value;

	dm2out.players = calloc(dm2out.maxclients, sizeof(player_t));

	for (i = 0; i < MAX_EDICTS; i++)
	{
		ent = NUM2EDICT(i);

		memcpy(&dm2out.baselines.entities[i], &ent->s, sizeof(entity_state_t));
	}

	// some configstrings are not supplied by the game dll
	sprintf(dm2out.configstrings[CS_MODELS+1], "maps/%s.bsp", mapname);

	demofile = gi.cvar("demofile", "", 0);
	if (demofile->string[0])
	{
		time(&now);
		ltime = localtime(&now);

		// get common escape codes from strftime()
		for (c = "AaBbdHIjMmSWwYy"; *c; c++)
		{
			t[1] = *c;
			if (strftime(format[*c], 32, t, ltime) == 0)
				format[*c][0] = 0;
		}

		// add our own escape codes here
		strncpy(format['F'], dm2out.configstrings[0], 31);
		strncpy(format['f'], mapname, 31);

		ExpandString(filename, sizeof(filename)-4, demofile->string, format);
		COM_DefaultExtension(filename, ".rla");

		GamePath(gamedir, gi.cvar("basedir", ".", 0)->string, gi.cvar("game", "", 0)->string);

		// create the demos/ directory in case it doesn't exist
		sprintf(path, "%s/demos", gamedir);
		_mkdir(path);

		sprintf(path, "%s/demos/%s", gamedir, filename);
		gi.dprintf("Writing demo file: %s\n", path);
		outfile = fopen(path, "wb");

		if (!outfile)
			gi.dprintf("RELAY: Unable to open relay outfile\n");
		else
			WritePreFrame(&dm2out.svd, NULL, dm2out.configstrings, &dm2out.baselines, outfile);
	}
	else
		outfile = NULL;
}

void ClientThink (edict_t *ent, usercmd_t *cmd)
{
	globals.ClientThink(ent, cmd);
}

qboolean ClientConnect (edict_t *ent, char *userinfo)
{
	return globals.ClientConnect(ent, userinfo);
}

void ClientUserinfoChanged (edict_t *ent, char *userinfo)
{
	globals.ClientUserinfoChanged(ent, userinfo);
}

void ClientDisconnect (edict_t *ent)
{
	globals.ClientDisconnect(ent);
}

void ClientBegin (edict_t *ent)
{
	globals.ClientBegin(ent);
}

void ClientCommand (edict_t *ent)
{
	globals.ClientCommand(ent);
}

void WriteGame (char *filename, qboolean autosave)
{
	globals.WriteGame(filename, autosave);
}

void ReadGame (char *filename)
{
	globals.ReadGame(filename);
}

void WriteLevel (char *filename)
{
	globals.WriteLevel(filename);
}

void ReadLevel (char *filename)
{
	globals.ReadLevel(filename);
}

void InitGame (void)
{
	globals.Init();
	Swap_Init();

	InitBlock(&tempblock, tempblock_buffer, sizeof(tempblock_buffer));
	InitBlock(&prefix, prefix_buffer, sizeof(prefix_buffer));
	InitBlock(&suffix, suffix_buffer, sizeof(suffix_buffer));

	maxclients = gi.cvar("maxclients", "1", CVAR_SERVERINFO|CVAR_LATCH);
}

void G_RunFrame (void)
{
	state_t			*to, *from;
	edict_t			*ent;
	block_t			out;
	char			buffer[MAX_SVSLEN];
	player_state_t	nullps = {0};
	player_t		*recipient;
	int				i, len, index1, index2;

	globals.RunFrame();

	current_frame++;

	index1 = current_frame & UPDATE_MASK;
	index2 = last_frame & UPDATE_MASK;

	to = &dm2out.states[index1];
	if (last_frame == -1)
		from = &dm2out.baselines;
	else
		from = &dm2out.states[index2];
	
	to->frame = current_frame;

	for (i = 1; i < MAX_EDICTS; i++)
	{
		ent = NUM2EDICT(i);

		// determine if entity is visible
		// FIXME: determine whether ent has been gi.linked?
		if (!ent->inuse || ent->svflags & SVF_NOCLIENT || (!ent->s.modelindex && !ent->s.effects && 
			!ent->s.sound && !ent->s.event))
		{
			SETBIT(to->visible, i, false);
		}
		else
		{
			SETBIT(to->visible, i, true);
			to->entities[i] = ent->s;
		}
	}

	if (dm2out.svd.isdemo == RECORD_RELAY)
	{
		memset(to->connected, 0, sizeof(to->connected));
		for (i = 0; i < dm2out.maxclients; i++)
		{
			ent = NUM2EDICT(i+1);
			if (ent->inuse && ent->client)
				SETBIT(to->connected, i, true);
		}

		memcpy(to->areas, areaportals, sizeof(areaportals));
	}

	InitBlock(&out, buffer, sizeof(buffer));
	
	WriteBin(&out, prefix.buffer, prefix.size);

	// write SVC_FRAME message
	WriteByte(&out, SVC_FRAME);
	WriteLong(&out, to->frame);
	WriteLong(&out, from->frame);
	WriteByte(&out, 0);		// ???
	if (dm2out.svd.isdemo == RECORD_RELAY)
	{
		len = 0;
		for (i = 0; i < MAX_MAP_AREAPORTALS/8; i++)
		{
			if (to->areas[i] != from->areas[i])
				len = i + 1;
		}
		WriteByte(&out, len);
		for (i = 0; i < len; i++)
			WriteByte(&out, to->areas[i]);

		len = 0;
		for (i = 0; i < (dm2out.maxclients+7)/8; i++)
		{
			if (to->connected[i] != from->connected[i])
				len = i + 1;
		}
		WriteByte(&out, len);
		for (i = 0; i < len; i++)
			WriteByte(&out, to->connected[i]);
	}
	else
	{
		WriteByte(&out, 0);
	}

	if (dm2out.svd.isdemo == RECORD_RELAY)
	{
		for (i = 0; i < dm2out.maxclients; i++)
		{
			ent = NUM2EDICT(i+1);
			if (!ent->inuse || !ent->client)
				continue;
			
			recipient = &dm2out.players[i];
			recipient->ps[index1] = ent->client->ps;
			
			WriteByte(&out, SVC_PLAYERINFO | MSG_UNICAST);
			WriteByte(&out, i);
			
			if (last_frame == -1)
				WritePS(&out, &recipient->ps[index1], &nullps);
			else
				WritePS(&out, &recipient->ps[index1], &recipient->ps[index2]);
		}
	}
	// client demo stuff goes here...
	
	// write SVC_PACKETENTITIES message
	WriteEntities(&out, to, from, &dm2out.baselines);

	WriteBin(&out, suffix.buffer, suffix.size);

	if (outfile)
	{
		len = LittleLong(out.size);
		fwrite(&len, 4, 1, outfile);
		fwrite(out.buffer, out.size, 1, outfile);
		fflush(outfile);
	}

	ResetBlock(&prefix);
	ResetBlock(&suffix);

	last_frame = current_frame;
}

void ShutdownGame (void)
{
	int	len;

	globals.Shutdown();
	UnloadGameModule(&proxydata);

	if (dm2out.players)
	{
		free(dm2out.players);
		dm2out.players = NULL;
	}

	if (outfile)
	{
		len = LittleLong(-1);
		fwrite(&len, 4, 1, outfile);
		fclose(outfile);
		outfile = NULL;
	}
}

void ServerCommand (void)
{
	globals.ServerCommand();
}
