/*
	Relay -- a tool to record and play Quake2 demos
	Copyright (C) 1999 Conor Davis

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

	Conor Davis
	cedavis@epid.org
*/

#include <string.h>
#include <assert.h>

#include "shared.h"
#include "block.h"
#include "endian.h"

char ReadChar(block_t *src)
{
	char	a;

	a = *(char *)src->p;
	src->p++;
	return a;
}

void WriteChar(block_t *dest, int a)
{
	*(char *)dest->p = a;
	dest->p++;
	dest->size++;
}

byte ReadByte(block_t *src)
{
	byte	a;

	a = *(byte *)src->p;
	src->p++;
	return a;
}

void WriteByte(block_t *dest, int a)
{
	*(byte *)dest->p = a;
	dest->p++;
	dest->size++;
}

short ReadShort(block_t *src)
{
	short	a;

	a = *(short *)src->p;
	src->p += 2;

	return LittleShort(a);
}

void WriteShort(block_t *dest, int a)
{
	*(short *)dest->p = LittleShort((short)a);
	dest->p += 2;
	dest->size += 2;
}

long ReadLong(block_t *src)
{
	long	a;

	a = *(long *)src->p;
	src->p += 4;

	return LittleLong(a);
}

void WriteLong(block_t *dest, int a)
{
	*(long *)dest->p = LittleLong(a);
	dest->p += 4;
	dest->size += 4;
}

float ReadFloat(block_t *src)
{
	union {int l; float f;} a;

	a.l = ReadLong(src);
	return a.f;
}

float ReadAngle(block_t *src)
{
	return (float)ReadChar(src) / 256 * 360;
}

void WriteAngle(block_t *dest, float a)
{
	WriteChar(dest, (char)(a * 256 / 360));
}

float ReadAngle16(block_t *src)
{
	return (float)ReadShort(src) / 65536 * 360;
}

void WriteAngle16(block_t *dest, float a)
{
	WriteShort(dest, (short)(a * 65535 / 360));
}

float ReadCoord(block_t *src)
{
	return (float)ReadShort(src) * 0.125F;
}

void WriteCoord(block_t *dest, float a)
{
	WriteShort(dest, (short)(a * 8));
}

float *ReadPosition(block_t *src, float *vec)
{
	vec[0] = ReadCoord(src);
	vec[1] = ReadCoord(src);
	vec[2] = ReadCoord(src);

	return vec;
}

void WritePosition(block_t *dest, float *vec)
{
	WriteCoord(dest, vec[0]);
	WriteCoord(dest, vec[1]);
	WriteCoord(dest, vec[2]);
}

short *ReadShortPosition(block_t *src, short *vec)
{
	vec[0] = ReadShort(src);
	vec[1] = ReadShort(src);
	vec[2] = ReadShort(src);

	return vec;
}

void WriteShortPosition(block_t *dest, short *vec)
{
	WriteShort(dest, vec[0]);
	WriteShort(dest, vec[1]);
	WriteShort(dest, vec[2]);
}

char *ReadString(block_t *src, char *buffer, int len)
{
	int		i;
	char	*b;

	for (i = 0, b = buffer; (i < len || len < 0) && *src->p; i++, src->p++)
		*b++ = *src->p;
	*b = 0;
	while (*src->p)
		src->p++;
	src->p++;

	return buffer;
}

void WriteString(block_t *dest, char *buffer)
{
	size_t	len;

	len = strlen(buffer) + 1;
	strcpy(dest->p, buffer);
	dest->p += len;
	dest->size += len;
}


#define DM2_NUMVERTEXNORMALS 162
float avertexnormals[3*DM2_NUMVERTEXNORMALS] =
{
	-0.525731027, 0, 0.850651026,
	-0.442862988, 0.238856003, 0.864188015,
	-0.295242012, 0, 0.955422997,
	-0.309017003, 0.5, 0.809017003,
	-0.162459999, 0.26286599, 0.951056004,
	0, 0, 1,
	0, 0.850651026, 0.525731027,
	-0.147621006, 0.71656698, 0.681717992,
	0.147621006, 0.71656698, 0.681717992,
	0, 0.525731027, 0.850651026,
	0.309017003, 0.5, 0.809017003,
	0.525731027, 0, 0.850651026,
	0.295242012, 0, 0.955422997,
	0.442862988, 0.238856003, 0.864188015,
	0.162459999, 0.26286599, 0.951056004,
	-0.681717992, 0.147621006, 0.71656698,
	-0.809017003, 0.309017003, 0.5,
	-0.587785006, 0.425325006, 0.688190997,
	-0.850651026, 0.525731027, 0,
	-0.864188015, 0.442862988, 0.238856003,
	-0.71656698, 0.681717992, 0.147621006,
	-0.688190997, 0.587785006, 0.425325006,
	-0.5, 0.809017003, 0.309017003,
	-0.238856003, 0.864188015, 0.442862988,
	-0.425325006, 0.688190997, 0.587785006,
	-0.71656698, 0.681717992, -0.147621006,
	-0.5, 0.809017003, -0.309017003,
	-0.525731027, 0.850651026, 0,
	0, 0.850651026, -0.525731027,
	-0.238856003, 0.864188015, -0.442862988,
	0, 0.955422997, -0.295242012,
	-0.26286599, 0.951056004, -0.162459999,
	0, 1, 0,
	0, 0.955422997, 0.295242012,
	-0.26286599, 0.951056004, 0.162459999,
	0.238856003, 0.864188015, 0.442862988,
	0.26286599, 0.951056004, 0.162459999,
	0.5, 0.809017003, 0.309017003,
	0.238856003, 0.864188015, -0.442862988,
	0.26286599, 0.951056004, -0.162459999,
	0.5, 0.809017003, -0.309017003,
	0.850651026, 0.525731027, 0,
	0.71656698, 0.681717992, 0.147621006,
	0.71656698, 0.681717992, -0.147621006,
	0.525731027, 0.850651026, 0,
	0.425325006, 0.688190997, 0.587785006,
	0.864188015, 0.442862988, 0.238856003,
	0.688190997, 0.587785006, 0.425325006,
	0.809017003, 0.309017003, 0.5,
	0.681717992, 0.147621006, 0.71656698,
	0.587785006, 0.425325006, 0.688190997,
	0.955422997, 0.295242012, 0,
	1, 0, 0,
	0.951056004, 0.162459999, 0.26286599,
	0.850651026, -0.525731027, 0,
	0.955422997, -0.295242012, 0,
	0.864188015, -0.442862988, 0.238856003,
	0.951056004, -0.162459999, 0.26286599,
	0.809017003, -0.309017003, 0.5,
	0.681717992, -0.147621006, 0.71656698,
	0.850651026, 0, 0.525731027,
	0.864188015, 0.442862988, -0.238856003,
	0.809017003, 0.309017003, -0.5,
	0.951056004, 0.162459999, -0.26286599,
	0.525731027, 0, -0.850651026,
	0.681717992, 0.147621006, -0.71656698,
	0.681717992, -0.147621006, -0.71656698,
	0.850651026, 0, -0.525731027,
	0.809017003, -0.309017003, -0.5,
	0.864188015, -0.442862988, -0.238856003,
	0.951056004, -0.162459999, -0.26286599,
	0.147621006, 0.71656698, -0.681717992,
	0.309017003, 0.5, -0.809017003,
	0.425325006, 0.688190997, -0.587785006,
	0.442862988, 0.238856003, -0.864188015,
	0.587785006, 0.425325006, -0.688190997,
	0.688190997, 0.587785006, -0.425325006,
	-0.147621006, 0.71656698, -0.681717992,
	-0.309017003, 0.5, -0.809017003,
	0, 0.525731027, -0.850651026,
	-0.525731027, 0, -0.850651026,
	-0.442862988, 0.238856003, -0.864188015,
	-0.295242012, 0, -0.955422997,
	-0.162459999, 0.26286599, -0.951056004,
	0, 0, -1,
	0.295242012, 0, -0.955422997,
	0.162459999, 0.26286599, -0.951056004,
	-0.442862988, -0.238856003, -0.864188015,
	-0.309017003, -0.5, -0.809017003,
	-0.162459999, -0.26286599, -0.951056004,
	0, -0.850651026, -0.525731027,
	-0.147621006, -0.71656698, -0.681717992,
	0.147621006, -0.71656698, -0.681717992,
	0, -0.525731027, -0.850651026,
	0.309017003, -0.5, -0.809017003,
	0.442862988, -0.238856003, -0.864188015,
	0.162459999, -0.26286599, -0.951056004,
	0.238856003, -0.864188015, -0.442862988,
	0.5, -0.809017003, -0.309017003,
	0.425325006, -0.688190997, -0.587785006,
	0.71656698, -0.681717992, -0.147621006,
	0.688190997, -0.587785006, -0.425325006,
	0.587785006, -0.425325006, -0.688190997,
	0, -0.955422997, -0.295242012,
	0, -1, 0,
	0.26286599, -0.951056004, -0.162459999,
	0, -0.850651026, 0.525731027,
	0, -0.955422997, 0.295242012,
	0.238856003, -0.864188015, 0.442862988,
	0.26286599, -0.951056004, 0.162459999,
	0.5, -0.809017003, 0.309017003,
	0.71656698, -0.681717992, 0.147621006,
	0.525731027, -0.850651026, 0,
	-0.238856003, -0.864188015, -0.442862988,
	-0.5, -0.809017003, -0.309017003,
	-0.26286599, -0.951056004, -0.162459999,
	-0.850651026, -0.525731027, 0,
	-0.71656698, -0.681717992, -0.147621006,
	-0.71656698, -0.681717992, 0.147621006,
	-0.525731027, -0.850651026, 0,
	-0.5, -0.809017003, 0.309017003,
	-0.238856003, -0.864188015, 0.442862988,
	-0.26286599, -0.951056004, 0.162459999,
	-0.864188015, -0.442862988, 0.238856003,
	-0.809017003, -0.309017003, 0.5,
	-0.688190997, -0.587785006, 0.425325006,
	-0.681717992, -0.147621006, 0.71656698,
	-0.442862988, -0.238856003, 0.864188015,
	-0.587785006, -0.425325006, 0.688190997,
	-0.309017003, -0.5, 0.809017003,
	-0.147621006, -0.71656698, 0.681717992,
	-0.425325006, -0.688190997, 0.587785006,
	-0.162459999, -0.26286599, 0.951056004,
	0.442862988, -0.238856003, 0.864188015,
	0.162459999, -0.26286599, 0.951056004,
	0.309017003, -0.5, 0.809017003,
	0.147621006, -0.71656698, 0.681717992,
	0, -0.525731027, 0.850651026,
	0.425325006, -0.688190997, 0.587785006,
	0.587785006, -0.425325006, 0.688190997,
	0.688190997, -0.587785006, 0.425325006,
	-0.955422997, 0.295242012, 0,
	-0.951056004, 0.162459999, 0.26286599,
	-1, 0, 0,
	-0.850651026, 0, 0.525731027,
	-0.955422997, -0.295242012, 0,
	-0.951056004, -0.162459999, 0.26286599,
	-0.864188015, 0.442862988, -0.238856003,
	-0.951056004, 0.162459999, -0.26286599,
	-0.809017003, 0.309017003, -0.5,
	-0.864188015, -0.442862988, -0.238856003,
	-0.951056004, -0.162459999, -0.26286599,
	-0.809017003, -0.309017003, -0.5,
	-0.681717992, 0.147621006, -0.71656698,
	-0.681717992, -0.147621006, -0.71656698,
	-0.850651026, 0, -0.525731027,
	-0.688190997, 0.587785006, -0.425325006,
	-0.587785006, 0.425325006, -0.688190997,
	-0.425325006, 0.688190997, -0.587785006,
	-0.425325006, -0.688190997, -0.587785006,
	-0.587785006, -0.425325006, -0.688190997,
	-0.688190997, -0.587785006, -0.425325006,
};

float *ReadDir(block_t *src, float *a)
{
	int code;
	
	code = ReadByte(src);
	if (code >= DM2_NUMVERTEXNORMALS)
		assert(0);

	a[0] = avertexnormals[3 * code + 0];
	a[1] = avertexnormals[3 * code + 1];
	a[2] = avertexnormals[3 * code + 2];
	
	return a;
}

void WriteDir(block_t *dest, float *a)
{
	int j;
	float dot, maxdot;
	int maxdotindex;
    
	maxdot = -999999.0;
	maxdotindex = -1;
	
	for (j = 0; j < DM2_NUMVERTEXNORMALS; j++)
	{
		dot = avertexnormals[3*j + 0] * a[0] +
			avertexnormals[3*j + 1] * a[1] +
			avertexnormals[3*j + 2] * a[2];

		if (dot > maxdot)
		{
			maxdot = dot;
			maxdotindex = j;
		}
	}
	
	WriteByte(dest, maxdotindex);
}

float ReadBlend(block_t *src)
{
	return (float)ReadByte(src) / 255;
}

void WriteBlend(block_t *dest, float a)
{
	WriteByte(dest, (byte)(a * 255));
}

float *ReadBlendVec(block_t *src, float *a)
{
	a[0] = ReadBlend(src);
	a[1] = ReadBlend(src);
	a[2] = ReadBlend(src);
	a[3] = ReadBlend(src);

	return a;
}

void WriteBlendVec(block_t *dest, float *a)
{
	WriteBlend(dest, a[0]);
	WriteBlend(dest, a[1]);
	WriteBlend(dest, a[2]);
	WriteBlend(dest, a[3]);
}

float ReadOffset(block_t *src)
{
	return (float)ReadChar(src) * 0.25F;
}

void WriteOffset(block_t *dest, float a)
{
	WriteChar(dest, (char)(a * 4));
}

float *ReadOffsetVec(block_t *src, float *a)
{
	a[0] = ReadOffset(src);
	a[1] = ReadOffset(src);
	a[2] = ReadOffset(src);

	return a;
}

void WriteOffsetVec(block_t *dest, float *a)
{
	WriteOffset(dest, a[0]);
	WriteOffset(dest, a[1]);
	WriteOffset(dest, a[2]);
}


char *ReadBin(block_t *src, void *buffer, size_t size)
{
	memcpy(buffer, src->p, size);
	src->p += size;

	return buffer;
}

void WriteBin(block_t *dest, void *buffer, size_t size)
{
	memcpy(dest->p, buffer, size);
	dest->p += size;
	dest->size += size;
}

char *CopyBlock(block_t *dest, block_t *src, size_t size)
{
	char	*a;

	a = dest->p;
	memcpy(dest->p, src->p, size);
	src->p += size;
	dest->p += size;
	dest->size += size;

	return a;
}

void ResetBlock(block_t *block)
{
	block->end = block->p = block->buffer;
	block->size = 0;
}

void InitBlock(block_t *block, char *buffer, size_t maxsize)
{
	block->end = block->p = block->buffer = buffer;
	block->size = 0;
	block->maxsize = maxsize;
}
