/*
	Relay -- a tool to record and play Quake2 demos
	Copyright (C) 1999 Conor Davis

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

	Conor Davis
	cedavis@epid.org
*/

#include <string.h>

#include "g_local.h"
#include "dm2.h"
#include "q2utils.h"

//
// Demo Menu
//
qboolean ShowDemoMenu(edict_t *ent, menu_t *menu)
{
	char	buf[256];
	int		i, count;

	Menu_FreeContents(menu);
	menu->title = G_CopyString("Demo Info");

	count = 0;
	for (i = 0; i < dm2in.maxclients; i++)
	{
		if (ISBITSET(connected, i))
			count++;
	}

	switch(dm2in.svd.isdemo)
	{
	case RECORD_NETWORK:	sprintf(buf, "Type:           Network");	break;
	case RECORD_CLIENT:		sprintf(buf, "Type:           Client");		break;
	case RECORD_SERVER:		sprintf(buf, "Type:           Server");		break;
	case RECORD_RELAY:		sprintf(buf, "Type:           Relay");		break;
	default:				sprintf(buf, "Type:           Unknown");
	}
	Menu_AddItem(menu, buf, "");
	if (dm2in.svd.game[0])
		sprintf(buf, "Game:           %s", dm2in.svd.game);
	else
		strcpy(buf,  "Game:           baseq2");
	Menu_AddItem(menu, buf, "");
	COM_FileBase(dm2in.configstrings[CS_MODELS+1], buf);
	Menu_AddItem(menu, va("Map:            %s", buf), "");
	Menu_AddItem(menu, va("Desc:           %.32s", dm2in.svd.mapname), "");
	Menu_AddItem(menu, va("Maxclients:     %d", dm2in.maxclients), "");
	Menu_AddItem(menu, va("Active players: %d", count), "");

	return true;
}

//
// Players Menu
//
void SelectPlayer(edict_t *ent, menuitem_t *item, int key)
{
	if (ChangePlayer(ent, item->param[0]))
		gi.cprintf(ent, PRINT_HIGH, "Unable to switch to player %s\n", item->text);
}

qboolean ShowPlayersMenu(edict_t *ent, menu_t *menu)
{
	int			i, num;
	menuitem_t	*item;

	// if menu is newly opened (cur == -1), put cursor on currently tracked player,
	// otherwise, try to keep the cursor on the same player after the menu changes
	if (menu->cur == -1)
		num = ent->client->player;
	else if (item = Menu_ItemFromIndex(menu, menu->cur))
		num = item->param[0];
	else
		num = -1;

	Menu_FreeContents(menu);
	menu->title = G_CopyString("Players");
	menu->id = ID_PLAYERS;

	Menu_AddItem(menu, "No player", "F0", SelectPlayer, -1);

	if (dm2in.svd.isdemo == RECORD_RELAY)
	{
		for (i = 0; i < dm2in.maxclients; i++)
		{
			if (ISBITSET(connected, i))
			{
				item = Menu_AddItem(menu, PlayerName(i), "F0", SelectPlayer, i);
				if (i == num)
					menu->cur = Menu_IndexFromItem(menu, item);
			}
		}
	}
	else
		Menu_AddItem(menu, PlayerName(dm2in.svd.player), "F0", SelectPlayer, 0);

	return true;
}

//
// Settings Menu
//
void SelectSetting(edict_t *ent, menuitem_t *item, int key)
{
	ent->client->relayflags ^= item->param[0];
	if (item->param[0] & RC_STATUSBAR)
	{
		if (ent->client->relayflags & RC_STATUSBAR)
		{
			if (ent->client->player != -1)
			{
				gi.WriteByte(SVC_CONFIGSTRING);
				gi.WriteShort(CS_STATUSBAR);
				gi.WriteString(dm2in.configstrings[CS_STATUSBAR]);
				gi.unicast(ent, false);
			}
		}
		else
		{
			gi.WriteByte(SVC_CONFIGSTRING);
			gi.WriteShort(CS_STATUSBAR);
			gi.WriteString("");
			gi.unicast(ent, false);
		}
	}

	Menu_Update(ent, ID_SETTINGS);
}

qboolean ShowSettingsMenu(edict_t *ent, menu_t *menu)
{
	int		relayflags;

	Menu_FreeContents(menu);
	menu->title = G_CopyString("Settings Menu");
	menu->id = ID_SETTINGS;

	relayflags = ent->client->relayflags;

	if (relayflags & RC_LOCKPOS)
		Menu_AddItem(menu, "Lock position: On", "F0", SelectSetting, RC_LOCKPOS);
	else
		Menu_AddItem(menu, "Lock position: Off", "F0", SelectSetting, RC_LOCKPOS);

	if (relayflags & RC_LOCKVIEW)
		Menu_AddItem(menu, "Lock view:     On", "F0", SelectSetting, RC_LOCKVIEW);
	else
		Menu_AddItem(menu, "Lock view:     Off", "F0", SelectSetting, RC_LOCKVIEW);

	if (relayflags & RC_CHASEVIEW)
		Menu_AddItem(menu, "Chase view:    On", "F0", SelectSetting, RC_CHASEVIEW);
	else
		Menu_AddItem(menu, "Chase view:    Off", "F0", SelectSetting, RC_CHASEVIEW);

	if (relayflags & RC_STATUSBAR)
		Menu_AddItem(menu, "Statusbar:     On", "F0", SelectSetting, RC_STATUSBAR);
	else
		Menu_AddItem(menu, "Statusbar:     Off", "F0", SelectSetting, RC_STATUSBAR);

	if (relayflags & RC_LAYOUT)
		Menu_AddItem(menu, "Layout:        On", "F0", SelectSetting, RC_LAYOUT);
	else
		Menu_AddItem(menu, "Layout:        Off", "F0", SelectSetting, RC_LAYOUT);

	if (relayflags & RC_NOTINT)
		Menu_AddItem(menu, "Tinting:       Off", "F0", SelectSetting, RC_NOTINT);
	else
		Menu_AddItem(menu, "Tinting:       On", "F0", SelectSetting, RC_NOTINT);

	return true;
}

//
// Main Menu
//
qboolean ShowMainMenu(edict_t *ent, menu_t *menu)
{
	Menu_FreeContents(menu);
	menu->title = G_CopyString("Main Menu");
	menu->id = ID_MAIN;

	Menu_AddItem(menu, "Demo info", "F0", Menu_SelectOpen, ShowDemoMenu);
	Menu_AddItem(menu, "Show Players", "F0", Menu_SelectOpen, ShowPlayersMenu);
	Menu_AddItem(menu, "Settings", "F0", Menu_SelectOpen, ShowSettingsMenu);

	return true;
}