/*
	Relay -- a tool to record and play Quake2 demos
	Copyright (C) 1999 Conor Davis

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

	Conor Davis
	cedavis@epid.org
*/

#include <string.h>

#include "g_local.h"
#include "q2utils.h"
#include "utils.h"

void CastMessage(block_t *block, int id, int recipient, vec3_t origin, multicast_t type, qboolean reliable)
{
	int			i, j;
	edict_t		*ent;

	if (!block->size)
		return;

	for (i = 0; i < game.maxclients; i++)
	{
		ent = g_edicts + i + 1;
		if (!ent->inuse || !ent->client)
			continue;

		// ignore messages that are out of the viewable space (even unicast messages)
		if (type == MULTICAST_PVS && !gi.inPVS(ent->s.origin, origin))
			continue;
		if (type == MULTICAST_PHS && !gi.inPHS(ent->s.origin, origin))
			continue;

		if (recipient != -1 && recipient != ent->client->player)
			continue;

		for (j = 0; j < block->size; j++)
			gi.WriteByte(block->buffer[j]);
		gi.unicast(ent, reliable);
	}
}

//
// ParseBlock
//

int ParseBlock(dm2_t *dm2, trans_t *trans, block_t *src)
{
	int			id, i, result, recipient_index;
	player_t	*recipient;
	edict_t		*ent;
	int			index1, index2;
	block_t		out;
	char		out_buffer[MAX_MSGLEN];
	vec3_t		origin;
	state_t		*current = NULL;

	InitBlock(&out, out_buffer, sizeof(out_buffer));

	result = 0;
	memcpy(old_connected, connected, sizeof(connected));

	while (src->p < src->end)
	{
		ResetBlock(&out);
		id = ReadByte(src);
		if (id & MSG_UNICAST)
		{
			recipient_index = ReadByte(src);
			recipient = &dm2->players[recipient_index];
			id &= ~MSG_UNICAST;
		}
		else
		{
			recipient_index = -1;
			recipient = &dm2->players[0];
		}

		switch (id)
		{
		case SVC_MUZZLEFLASH:
		case SVC_MUZZLEFLASH2:
			{
				int		entity;

				WriteByte(&out, id);
				entity = ReadShort(src);
				WriteShort(&out, trans->edicts[entity]);
				CopyBlock(&out, src, 1);					// value

				if (!entity || (unsigned)entity >= MAX_EDICTS)
					gi.error("ParseBlock: muzzleflash ent >= MAX_EDICTS\n");

				CastMessage(&out, id, recipient_index, current->entities[entity].origin, MULTICAST_PVS, false);
			}
			break;
		case SVC_TEMP_ENTITY:
			{
				int		type, nextid;

				type = ReadByte(src);
				WriteByte(&out, SVC_TEMP_ENTITY);
				WriteByte(&out, type);
				switch(type)
				{
				// case TE_PLASMATRAIL:			// old meaning
				case TE_GREENBLOOD:				// new meaning (3.15+)
					if (dm2->svd.version >= 32)
						goto impact_entity;
					else
						goto line_entity;
				// case TE_GREENBLOOD_old:		// old meaning
				case TE_BLUEHYPERBLASTER:		// new meaning (3.15+)
					if (dm2->svd.version >= 32)
						goto line_entity;
					else
						goto impact_entity;
				// point entity
				case TE_EXPLOSION1:
				case TE_EXPLOSION2:
				case TE_ROCKET_EXPLOSION:
				case TE_GRENADE_EXPLOSION:
				case TE_ROCKET_EXPLOSION_WATER:
				case TE_GRENADE_EXPLOSION_WATER:
				case TE_BFG_EXPLOSION:
				case TE_BFG_BIGEXPLOSION:
				case TE_BOSSTPORT:
				case TE_PLASMA_EXPLOSION:
				case TE_PLAIN_EXPLOSION:
				case TE_CHAINFIST_SMOKE:
				case TE_TRACKER_EXPLOSION:
				case TE_TELEPORT_EFFECT:
				case TE_DBALL_GOAL:
				case TE_NUKEBLAST:
				case TE_WIDOWSPLASH:
				case TE_EXPLOSION1_BIG:
				case TE_EXPLOSION1_NP:
					WritePosition(&out, ReadPosition(src, origin));
					break;
				// impact entity
				case TE_GUNSHOT:
				case TE_BLOOD:
				case TE_BLASTER:
				case TE_SHOTGUN:
				case TE_SPARKS:
				case TE_SCREEN_SPARKS:
				case TE_SHIELD_SPARKS:
				case TE_BULLET_SPARKS:
				// case TE_GREENBLOOD_new:
				// case TE_GREENBLOOD_old:
				case TE_BLASTER2:
				case TE_MOREBLOOD:
				case TE_HEATBEAM_SPARKS:
				case TE_HEATBEAM_STEAM:
				case TE_ELECTRIC_SPARKS:
				case TE_FLECHETTE:
impact_entity:
					WritePosition(&out, ReadPosition(src, origin));
					CopyBlock(&out, src, 1);				// movedir
					break;
				// line entity
				case TE_RAILTRAIL:
				case TE_BUBBLETRAIL:
				case TE_BFG_LASER:
				// case TE_PLASMATRAIL:
				// case TE_BLUEHYPERBLASTER:
				case TE_DEBUGTRAIL:
				case TE_BUBBLETRAIL2:
line_entity:
					WritePosition(&out, ReadPosition(src, origin));
					CopyBlock(&out, src, 6);				// endpos
					break;
				// special entity
				case TE_SPLASH:
				case TE_LASER_SPARKS:
				case TE_WELDING_SPARKS:
				case TE_TUNNEL_SPARKS:
					CopyBlock(&out, src, 1);				// count
					WritePosition(&out, ReadPosition(src, origin));
					CopyBlock(&out, src, 2);				// movedir, style
					break;
				case TE_PARASITE_ATTACK:
				case TE_MEDIC_CABLE_ATTACK:
				case TE_HEATBEAM:
				case TE_MONSTER_HEATBEAM:
					WriteShort(&out, trans->edicts[ReadShort(src)]);	// entity
					WritePosition(&out, ReadPosition(src, origin));
					CopyBlock(&out, src, 6);				// endpos
					break;
				case TE_GRAPPLE_CABLE:
					WriteShort(&out, trans->edicts[ReadShort(src)]);	// entity
					WritePosition(&out, ReadPosition(src, origin));
					CopyBlock(&out, src, 12);				// endpos, pos1
					break;
				case TE_FLAME:
					WriteShort(&out, trans->edicts[ReadShort(src)]);	// entity
					CopyBlock(&out, src, 8);				// count, endpos
					WritePosition(&out, ReadPosition(src, origin));
					CopyBlock(&out, src, 24);				// pos1, pos2, pos3, pos4
					break;
				case TE_LIGHTNING:
					WriteShort(&out, trans->edicts[ReadShort(src)]);	// dest_entity
					WriteShort(&out, trans->edicts[ReadShort(src)]);	// entity
					CopyBlock(&out, src, 6);				// endpos
					WritePosition(&out, ReadPosition(src, origin));
					break;
				case TE_FLASHLIGHT:
					WritePosition(&out, ReadPosition(src, origin));
					WriteShort(&out, trans->edicts[ReadShort(src)]);	// entity
					break;
				case TE_FORCEWALL:
					WritePosition(&out, ReadPosition(src, origin));
					CopyBlock(&out, src, 8);				// endpos, style
					break;
				case TE_STEAM:
					nextid = ReadShort(src);
					WriteShort(&out, nextid);
					CopyBlock(&out, src, 1);				// count
					WritePosition(&out, ReadPosition(src, origin));
					CopyBlock(&out, src, 4);				// movedir, style, plat2flags
					if (nextid != -1)
						CopyBlock(&out, src, 4);			// wait
					break;
				default:
					gi.error("ParseBlock: unknown TE type %d\n", type);
					break;
				}

				CastMessage(&out, SVC_TEMP_ENTITY, recipient_index, origin, MULTICAST_PVS, false);
			}
			break;
		case SVC_LAYOUT:
			{
				char	string[MAX_MSGLEN];

				ReadString(src, string, sizeof(string));
				strcpy(recipient->layout, string);

//				WriteByte(&out, SVC_LAYOUT);
//				WriteString(&out, string);
//				CastMessage(&out, SVC_LAYOUT, recipient_index, vec3_origin, MULTICAST_ALL, true);

				for (i = 0; i < game.maxclients; i++)
				{
					ent = g_edicts + i + 1;
					if (!ent->inuse || !ent->client)
						continue;

					if (recipient_index != -1 && ent->client->player != recipient_index)
						continue;
					if (ent->client->curmenu)
						continue;

					strcpy(ent->client->layout, string);
					ent->client->show_layout = true;
				}
			}
			break;
		case SVC_INVENTORY:
			{
				int		i;

				for (i = 0; i < MAX_ITEMS; i++)
					recipient->inventory[i] = ReadShort(src);
//				WriteByte(&out, SVC_INVENTORY);
//				for (i = 0; i < MAX_ITEMS; i++)
//					WriteShort(&out, recipient->inventory[i]);
//				CastMessage(&out, SVC_INVENTORY, recipient_index, vec3_origin, MULTICAST_ALL, true);

				for (i = 0; i < game.maxclients; i++)
				{
					ent = g_edicts + i + 1;
					if (!ent->inuse || !ent->client)
						continue;

					if (recipient_index != -1 && ent->client->player != recipient_index)
						continue;

					memcpy(ent->client->inventory, recipient->inventory, sizeof(ent->client->inventory));
				}
			}
			break;
		case SVC_NOP:
			{
			}
			break;
		case SVC_DISCONNECT:
			{
			}
			break;
		case SVC_RECONNECT:
			{
			}
			break;
		case SVC_SOUND:
			{
				int		mask, mix, entity;

				WriteByte(&out, SVC_SOUND);
				mask = ReadByte(src);
				WriteByte(&out, mask);
				WriteByte(&out, trans->configstrings[CS_SOUNDS+ReadByte(src)] - CS_SOUNDS);
				if (mask & SND_VOLUME)		CopyBlock(&out, src, 1);		// volume
				if (mask & SND_ATTENUATION)	CopyBlock(&out, src, 1);		// attenuation
				if (mask & SND_OFFSET)		CopyBlock(&out, src, 1);		// timeofs
				if (mask & SND_ENT)
				{
					mix = ReadShort(src);
					entity = mix >> 3;
					WriteShort(&out, (mix & 0x07) | (trans->edicts[entity] << 3));
				}
				if (mask & SND_POS)			// multicast from sound's origin
					WritePosition(&out, ReadPosition(src, origin));
				else if (mask & SND_ENT)	// multicast from entity's origin
					VectorCopy(current->entities[entity].origin, origin);
				else if (dm2->svd.isdemo == RECORD_NETWORK || dm2->svd.isdemo == RECORD_CLIENT)
				{	// "local sound", no entity or position info, so use player's origin
					for (i = 0; i < 3; i++)
						origin[i] = dm2->players[0].ps[index1].pmove.origin[i] * 0.125;
				}

				CastMessage(&out, SVC_SOUND, recipient_index, origin, MULTICAST_PHS, false);
			}
			break;
		case SVC_PRINT:
			{
				char	string[MAX_MSGLEN];

				WriteByte(&out, SVC_PRINT);
				CopyBlock(&out, src, 1);		// level
				ReadString(src, string, sizeof(string));
				WriteString(&out, string);

				CastMessage(&out, SVC_PRINT, recipient_index, vec3_origin, MULTICAST_ALL, false);
			}
			break;
		case SVC_STUFFTEXT:
			{
				char	string[MAX_MSGLEN], *cur, *next;

				ReadString(src, string, sizeof(string));

				cur = string;
				while (cur)
				{
					next = SplitCommandString(cur, ';');
					if (*cur)
					{
						if (!strcmp(Argv(cur, 0), "precache"))
						{
							result |= 0x01;
							break;
						}
					}
					cur = next;
				}
//				WriteByte(&out, SVC_STUFFTEXT);
//				WriteString(&out, string);

//				CastMessage(&out, SVC_STUFFTEXT, recipient_index, vec3_origin, MULTICAST_ALL, true);
			}
			break;
		case SVC_SERVERDATA:
			{
				dm2->svd.version = ReadLong(src);
				dm2->svd.key = ReadLong(src);
				dm2->svd.isdemo = ReadByte(src);
				ReadString(src, dm2->svd.game, sizeof(dm2->svd.game));
				dm2->svd.player = ReadShort(src);
				ReadString(src, dm2->svd.mapname, sizeof(dm2->svd.mapname));
			}
			break;
		case SVC_CONFIGSTRING:
			{
				int		index;
				char	string[MAX_MSGLEN];
				
				index = ReadShort(src);
				ReadString(src, string, sizeof(string));
				strcpy(dm2->configstrings[index], string);

				if (game.relayflags & R_SQUEEZE)
				{	// find first available configstring of each type
					if (index >= CS_MODELS)
					{
						if (index < CS_SOUNDS)				// models
							trans->configstrings[index] = FindConfigstringSlot(CS_MODELS+1, CS_SOUNDS, string);
						else if (index < CS_IMAGES)			// sounds
							trans->configstrings[index] = FindConfigstringSlot(CS_SOUNDS+1, CS_IMAGES, string);
						else if (index < CS_LIGHTS)			// images
							trans->configstrings[index] = FindConfigstringSlot(CS_IMAGES+1, CS_LIGHTS, string);
						else if (index < CS_ITEMS)			// lights
							trans->configstrings[index] = FindConfigstringSlot(CS_LIGHTS, CS_ITEMS, string);
						else if (index < CS_PLAYERSKINS)	// items
							trans->configstrings[index] = FindConfigstringSlot(CS_ITEMS, CS_PLAYERSKINS, string);
						else if (index < CS_GENERAL)		// playerskins
							trans->configstrings[index] = FindConfigstringSlot(CS_PLAYERSKINS, CS_GENERAL, string);
						else if (index < MAX_CONFIGSTRINGS)	// general
							trans->configstrings[index] = FindConfigstringSlot(CS_GENERAL, MAX_CONFIGSTRINGS, string);
						else
							gi.error("ParseBlock: configstring >= MAX_CONFIGSTRINGS (%d)\n", index);
					}
					else
					{
						gi.configstring(index, string);
						trans->configstrings[index] = index;
					}
				}
				else
				{	// preserve configstring order
					gi.configstring(index, string);
				}
			}
			break;
		case SVC_SPAWNBASELINE:
			{
				entity_state_t	es;
				int				mask;

				memset(&es, 0, sizeof(entity_state_t));

				mask = ReadByte(src);
				if (mask & U_MOREBITS1)	mask |= (ReadByte(src) <<  8);
				if (mask & U_MOREBITS2) mask |= (ReadByte(src) << 16);
				if (mask & U_MOREBITS3) mask |= (ReadByte(src) << 24);
				
				if (mask & U_NUMBER16)
					es.number = ReadShort(src);
				else
					es.number = ReadByte(src);
				
			
				if (mask & U_MODEL)				es.modelindex = ReadByte(src);
				if (mask & U_MODEL2)			es.modelindex2 = ReadByte(src);
				if (mask & U_MODEL3)			es.modelindex3 = ReadByte(src);
				if (mask & U_MODEL4)			es.modelindex4 = ReadByte(src);
				if (mask & U_FRAME8)			es.frame = ReadByte(src);
				if (mask & U_FRAME16)			es.frame = ReadShort(src);
				if (mask & U_SKIN8)
				{
					if (mask & U_SKIN16)		es.skinnum = ReadLong(src);
					else						es.skinnum = ReadByte(src);
				}
				else if (mask & U_SKIN16)		es.skinnum = ReadShort(src);
				if (mask & U_EFFECTS8)
				{
					if (mask & U_EFFECTS16)		es.effects = ReadLong(src);
					else						es.effects = ReadByte(src);
				}
				else if (mask & U_EFFECTS16)	es.effects = ReadShort(src);
				if (mask & U_RENDERFX8)
				{
					if (mask & U_RENDERFX16)	es.renderfx = ReadLong(src);
					else						es.renderfx = ReadByte(src);
				}
				else if (mask & U_RENDERFX16)	es.renderfx = ReadShort(src);
				if (mask & U_ORIGIN1)			es.origin[0] = ReadCoord(src);
				if (mask & U_ORIGIN2)			es.origin[1] = ReadCoord(src);
				if (mask & U_ORIGIN3)			es.origin[2] = ReadCoord(src);
				if (mask & U_ANGLE1)			es.angles[0] = ReadAngle(src);
				if (mask & U_ANGLE2)			es.angles[1] = ReadAngle(src);
				if (mask & U_ANGLE3)			es.angles[2] = ReadAngle(src);
				if (mask & U_OLDORIGIN)			ReadPosition(src, es.old_origin);
				if (mask & U_SOUND)				es.sound = ReadByte(src);
				if (mask & U_EVENT)				es.event = ReadByte(src);
				if (mask & U_SOLID)				es.solid = ReadShort(src);

				dm2->baselines.entities[es.number] = es;
			}
			break;
		case SVC_CENTERPRINT:
			{
				char	string[MAX_MSGLEN];

				ReadString(src, string, sizeof(string));
				
				WriteByte(&out, SVC_CENTERPRINT);
				WriteString(&out, string);

				CastMessage(&out, SVC_CENTERPRINT, recipient_index, vec3_origin, MULTICAST_ALL, false);
			}
			break;
		case SVC_PLAYERINFO:
			{
				player_state_t	*ps;
				int				num;

				// old PS should have been copied over via SVC_FRAME message
				if (dm2->svd.isdemo == RECORD_RELAY)
					ps = &dm2->players[recipient_index].ps[index1];
				else
					ps = &dm2->players[0].ps[index1];

				ReadPS(src, ps);
				// update client origins, just in case a muzzleflash or something happens
				// in the same frame to make sure the client will be in the PVS
				for (i = 0; i < game.maxclients; i++)
				{
					ent = g_edicts + i + 1;
					if (!ent->inuse || !ent->client)
						continue;

					if (ent->client->player != num)
						continue;

					UpdatePlayerOrigin(ent);
				}
			}
			break;
		case SVC_PACKETENTITIES:
			{	// old entities should have been copied over via SVC_FRAME message
				int				mask, entity;
				entity_state_t	*es;

				for (;;)
				{
					mask = ReadByte(src);
					if (mask & U_MOREBITS1)		mask |= ReadByte(src) <<  8;
					if (mask & U_MOREBITS2)		mask |= ReadByte(src) << 16;
					if (mask & U_MOREBITS3)		mask |= ReadByte(src) << 24;
					if (mask & U_NUMBER16)
						entity = ReadShort(src);
					else
						entity = ReadByte(src);

					if (!entity)
						break;

					if ((unsigned)entity >= MAX_EDICTS)
						gi.error("ParseBlock: Entity >= MAX_EDICTS (%u)\n", entity);

					es = &current->entities[entity];

					if (!ISBITSET(current->visible, entity))	// newly visible?
						*es = dm2->baselines.entities[entity];
					
					if (mask & U_MODEL)
						es->modelindex = ReadByte(src);
					if (mask & U_MODEL2)
						es->modelindex2 = ReadByte(src);
					if (mask & U_MODEL3)
						es->modelindex3 = ReadByte(src);
					if (mask & U_MODEL4)
						es->modelindex4 = ReadByte(src);
					if (mask & U_FRAME8)
						es->frame = ReadByte(src);
					if (mask & U_FRAME16)
						es->frame = ReadShort(src);
					if (mask & U_SKIN8)
					{
						if (mask & U_SKIN16)
							es->skinnum = ReadLong(src);
						else
							es->skinnum = ReadByte(src);
					}
					else if (mask & U_SKIN16)
						es->skinnum = ReadShort(src);
					if (mask & U_EFFECTS8)
					{
						if (mask & U_EFFECTS16)
							es->effects = ReadLong(src);
						else
							es->effects = ReadByte(src);
					}
					else if (mask & U_EFFECTS16)
						es->effects = ReadShort(src);
					if (mask & U_RENDERFX8)
					{
						if (mask & U_RENDERFX16)
							es->renderfx = ReadLong(src);
						else
							es->renderfx = ReadByte(src);
					}
					else if (mask & U_RENDERFX16)
						es->renderfx = ReadShort(src);
					if (mask & U_ORIGIN1)
						es->origin[0] = ReadCoord(src);
					if (mask & U_ORIGIN2)
						es->origin[1] = ReadCoord(src);
					if (mask & U_ORIGIN3)
						es->origin[2] = ReadCoord(src);
					if (mask & U_ANGLE1)
						es->angles[0] = ReadAngle(src);
					if (mask & U_ANGLE2)
						es->angles[1] = ReadAngle(src);
					if (mask & U_ANGLE3)
						es->angles[2] = ReadAngle(src);
					if (mask & U_OLDORIGIN)
						ReadPosition(src, es->old_origin);
					if (mask & U_SOUND)
						es->sound = ReadByte(src);
					if (mask & U_EVENT)
						es->event = ReadByte(src);
					if (mask & U_SOLID)
						es->solid = ReadShort(src);
					
					if (mask & U_REMOVE)
					{
						if (!ISBITSET(current->visible, entity))
							gi.error("ParseBlock: Tried to remove non-existing entity %d\n", entity);

						SETBIT(current->visible, entity, false);
						if (trans->edicts[entity])
						{
							ent = g_edicts + trans->edicts[entity];
							if (ent->inuse)
							{
								if (game.relayflags & R_SQUEEZE)
									trans->edicts[entity] = 0;
								G_FreeEdict(ent);
							}
						}
					}
					else
					{
						SETBIT(current->visible, entity, true);

						if (trans->edicts[entity])
							ent = g_edicts + trans->edicts[entity];
						else if (game.relayflags & R_SQUEEZE)
						{
							ent = G_Spawn();
							trans->edicts[entity] = ent - g_edicts;
						}
						else
							gi.error("ParseBlock: no entity space available, try reducing maxclients\n");

						ent->inuse = true;
						ent->s = *es;
						if (game.relayflags & R_SQUEEZE)
						{
							if (ent->s.modelindex && ent->s.modelindex != 255)
								ent->s.modelindex = trans->configstrings[CS_MODELS+ent->s.modelindex] - CS_MODELS;
							if (ent->s.modelindex2 && ent->s.modelindex2 != 255)
								ent->s.modelindex2 = trans->configstrings[CS_MODELS+ent->s.modelindex2] - CS_MODELS;
							if (ent->s.modelindex3 && ent->s.modelindex3 != 255)
								ent->s.modelindex3 = trans->configstrings[CS_MODELS+ent->s.modelindex3] - CS_MODELS;
							if (ent->s.modelindex4 && ent->s.modelindex4 != 255)
								ent->s.modelindex4 = trans->configstrings[CS_MODELS+ent->s.modelindex4] - CS_MODELS;
							if (ent->s.renderfx & RF_CUSTOMSKIN)
								ent->s.skinnum = trans->configstrings[CS_IMAGES+ent->s.skinnum] - CS_IMAGES;
							if (ent->s.sound)
								ent->s.sound = trans->configstrings[CS_SOUNDS+ent->s.sound] - CS_SOUNDS;
						}
						// needed for BSP models to show up (other way?)
						if (configstrings[CS_MODELS+ent->s.modelindex][0] == '*')
							gi.setmodel(ent, configstrings[CS_MODELS+ent->s.modelindex]);

						gi.linkentity(ent);
					}
				}
			}
			break;
		case SVC_FRAME:
			{
				int		seq1, seq2;
				int		count;
				state_t	*delta;

				if (dm2->svd.isdemo != RECORD_SERVER)
				{
					dm2->current_frame = seq1 = ReadLong(src);
					dm2->delta_frame = seq2 = ReadLong(src);

//					gi.dprintf(" %4d (%4d)", seq1, seq2);

					index1 = seq1 & UPDATE_MASK;
					index2 = seq2 & UPDATE_MASK;

					current = &dm2->states[index1];
					// copy data from delta frame to current frame
					if (seq2 == -1)
					{
						delta = &dm2->baselines;
						for (i = 0; i < dm2->maxclients; i++)
							memset(&dm2->players[i].ps[index1], 0, sizeof(player_state_t));
					}
					else
					{
						delta = &dm2->states[index2];
						for (i = 0; i < dm2->maxclients; i++)
						{
							dm2->players[i].ps[index1] = dm2->players[i].ps[index2];
							dm2->players[i].ps[index1].stats[STAT_FLASHES] = 0;
						}
					}

					*current = *delta;
					current->frame = seq1;
					if (dm2->svd.version != 26)
						ReadByte(src);		// ???
					count = ReadByte(src);
//					gi.dprintf(" Count: %d", count);

					for (i = 0; i < count; i++)
						current->areas[i] = ReadByte(src);

					if (dm2->svd.isdemo == RECORD_RELAY)
					{
						count = ReadByte(src);
						for (i = 0; i < count; i++)
							current->connected[i] = ReadByte(src);
						memcpy(connected, current->connected, sizeof(connected));
					}

					for (i = 1; i < MAX_EDICTS; i++)
					{
						current->entities[i].event = 0;
						if (ISBITSET(current->visible, i))
						{
							if (trans->edicts[i])
								ent = g_edicts + trans->edicts[i];
							else if (game.relayflags & R_SQUEEZE)
							{
								ent = G_Spawn();
								trans->edicts[i] = ent - g_edicts;
							}
							else
								gi.error("ParseBlock: no entity space available, try reducing maxclients\n");
							
							ent->inuse = true;
							ent->s = current->entities[i];
							if (game.relayflags & R_SQUEEZE)
							{
								if (ent->s.modelindex && ent->s.modelindex != 255)
									ent->s.modelindex = trans->configstrings[CS_MODELS+ent->s.modelindex] - CS_MODELS;
								if (ent->s.modelindex2 && ent->s.modelindex2 != 255)
									ent->s.modelindex2 = trans->configstrings[CS_MODELS+ent->s.modelindex2] - CS_MODELS;
								if (ent->s.modelindex3 && ent->s.modelindex3 != 255)
									ent->s.modelindex3 = trans->configstrings[CS_MODELS+ent->s.modelindex3] - CS_MODELS;
								if (ent->s.modelindex4 && ent->s.modelindex4 != 255)
									ent->s.modelindex4 = trans->configstrings[CS_MODELS+ent->s.modelindex4] - CS_MODELS;
								if (ent->s.renderfx & RF_CUSTOMSKIN)
									ent->s.skinnum = trans->configstrings[CS_IMAGES+ent->s.skinnum] - CS_IMAGES;
								if (ent->s.sound)
									ent->s.sound = trans->configstrings[CS_SOUNDS+ent->s.sound] - CS_SOUNDS;
							}
							// needed for BSP models to show up
							if (configstrings[CS_MODELS+ent->s.modelindex][0] == '*')
								gi.setmodel(ent, configstrings[CS_MODELS+ent->s.modelindex]);
							
							gi.linkentity(ent);
						}
						else if (trans->edicts[i])
						{
							ent = g_edicts + trans->edicts[i];
							if (ent->inuse)
							{
								if (game.relayflags & R_SQUEEZE)
									trans->edicts[i] = 0;
								G_FreeEdict(ent);
							}
						}
					}
				}
			}
			break;
		default:
			gi.error("ParseBlock: Unknown server command id %d\n", id);
			break;
		}
	}

	return result;
}
