/*
	Relay -- a tool to record and play Quake2 demos
	Copyright (C) 1999 Conor Davis

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

	Conor Davis
	cedavis@epid.org
*/

#include <stdlib.h>
#include <string.h>

#include "g_local.h"
#include "block.h"
#include "dm2.h"
#include "q2utils.h"
#include "utils.h"

/*
===============
ED_ParseField

Takes a key/value pair and sets the binary values
in an edict
===============
*/
void ED_ParseField (char *key, char *value, edict_t *ent)
{
	field_t	*f;
	byte	*b;
	float	v;
	vec3_t	vec;

	for (f=fields ; f->name ; f++)
	{
		if (!Q_stricmp(f->name, key))
		{	// found it
			b = (byte *)ent;

			switch (f->type)
			{
			case F_LSTRING:
				*(char **)(b+f->ofs) = G_CopyString (value);
				break;
			case F_VECTOR:
				sscanf (value, "%f %f %f", &vec[0], &vec[1], &vec[2]);
				((float *)(b+f->ofs))[0] = vec[0];
				((float *)(b+f->ofs))[1] = vec[1];
				((float *)(b+f->ofs))[2] = vec[2];
				break;
			case F_INT:
				*(int *)(b+f->ofs) = atoi(value);
				break;
			case F_FLOAT:
				*(float *)(b+f->ofs) = atof(value);
				break;
			case F_ANGLEHACK:
				v = atof(value);
				((float *)(b+f->ofs))[0] = 0;
				((float *)(b+f->ofs))[1] = v;
				((float *)(b+f->ofs))[2] = 0;
				break;
			case F_IGNORE:
				break;
			}
			return;
		}
	}
//	gi.dprintf ("%s is not a field\n", key);
}

/*
====================
ED_ParseEdict

====================
*/
char *ED_ParseEdict (char *data, edict_t *ent)
{
	char		keyname[256];
	char		*com_token;

// go through all the dictionary pairs
	while (1)
	{	
	// parse key
		com_token = COM_Parse (&data);
		if (com_token[0] == '}')
			break;
		if (!data)
			gi.error ("ED_ParseEntity: EOF without closing brace");

		strncpy (keyname, com_token, sizeof(keyname)-1);
		
	// parse value	
		com_token = COM_Parse (&data);
		if (!data)
			gi.error ("ED_ParseEntity: EOF without closing brace");

		if (com_token[0] == '}')
			gi.error ("ED_ParseEntity: closing brace without data");

		if (keyname[0] == '_')
			continue;

		ED_ParseField(keyname, com_token, ent);
	}

	return data;
}

void SpawnEntities (char *mapname, char *entities, char *spawnpoint)
{
	int			i, numportals;
	char		buf[MAX_MSGLEN], in_buffer[MAX_SVSLEN], *com_token;
	block_t		in;
	edict_t		*ent, dummy;
	char		demofile[MAX_QPATH];

	gi.FreeTags (TAG_LEVEL);

	gi.dprintf("=== SpawnEntities ===\n");
	memset (&level, 0, sizeof(level));
	memset (g_edicts, 0, game.maxentities * sizeof (g_edicts[0]));
	
	game.relayflags = gi.cvar("relayflags", "0", CVAR_LATCH)->value;

	// set client fields on player ents
	for (i=0 ; i<game.maxclients ; i++)
		g_edicts[i+1].client = game.clients + i;

	memset(configstrings, 0, sizeof(configstrings));
	InitDM2(&dm2in);
	memset(&dm2in_trans, 0, sizeof(trans_t));
	if ( !(game.relayflags & R_SQUEEZE) )
	{
		// leave room for client edicts
		for (i = game.maxclients + 1; i < MAX_EDICTS; i++)
			dm2in_trans.edicts[i - game.maxclients] = i;
		// leave configstrings as is
		for (i = 0; i < MAX_CONFIGSTRINGS; i++)
			dm2in_trans.configstrings[i] = i;
	}

	// load the demo file
	strncpy(demofile, gi.cvar("demofile", "", 0)->string, sizeof(demofile)-1);
	demofile[sizeof(demofile)-1] = 0;
	if (demofile[0])
	{
		COM_DefaultExtension(demofile, ".rla");

		GamePath(buf, gi.cvar("basedir", "", CVAR_NOSET)->string, gi.cvar("game", "", CVAR_SERVERINFO|CVAR_LATCH)->string);
		sprintf(buf+strlen(buf), "/demos/%s", demofile);

		gi.dprintf("Reading demo file: %s\n", buf);
		infile = fopen(buf, "rb");
		if (!infile)
			gi.error("Unable to open demo\n", buf);
	}
	else
		gi.error("Need to set demofile to run!\ne.g. \"set demofile demo1.dm2\"\n");

	InitBlock(&in, in_buffer, sizeof(in_buffer));
	while(1)
	{
		// shouldn't reach end of demo file while reading preframe info
		if (ReadBlock(&in, infile, dm2in.svd.isdemo) || in.size == -1)
			gi.error("SpawnEntities: Error reading demo file\n");

		if (ParseBlock(&dm2in, &dm2in_trans, &in) & 0x01)
			break;
	}
	
	// read in mapname from demo, and strip "maps/" prefix and ".bsp" suffix
	COM_FileBase(dm2in.configstrings[CS_MODELS+1], buf);

	// compare demo mapname with the map we are actually running
	// if they are different, restart the server with the demo's map
	if (Q_stricmp(mapname, buf))
	{
		fclose(infile);
		infile = NULL;
		gi.AddCommandString(va("gamemap %s\n", buf));
		// calling gi.error makes q2 stop the loading process for the current level and cleanup.
		// the commandstring will still take effect (better way?)
		gi.error("Changing map from %s to %s...\n", mapname, buf);
		return;
	}

	if (dm2in.svd.isdemo == RECORD_RELAY || dm2in.svd.version == RECORD_SERVER)
	{
		dm2in.maxclients = atoi(dm2in.configstrings[CS_MAXCLIENTS]);
		dm2in.players = gi.TagMalloc(dm2in.maxclients*sizeof(player_t), TAG_LEVEL);
	}
	else
	{
		dm2in.maxclients = 1;
		dm2in.players = gi.TagMalloc(sizeof(player_t), TAG_LEVEL);
	}
	sprintf(configstrings[CS_MODELS+1], "maps/%s.bsp", mapname);
	globals.num_edicts = game.maxentities;

// parse ents
	numportals = 0;
	while (1)
	{
		// parse the opening brace	
		com_token = COM_Parse (&entities);
		if (!entities)
			break;
		if (com_token[0] != '{')
			gi.error ("ED_LoadFromFile: found %s when expecting {",com_token);

		memset(&dummy, 0, sizeof(edict_t));
		entities = ED_ParseEdict (entities, &dummy);
		if (dummy.classname)
		{
			if (!Q_stricmp(dummy.classname, "func_areaportal"))
				numportals++;
			else if (!Q_strncasecmp(dummy.classname, "info_player_", 12))
				VectorCopy(dummy.s.origin, spawnpoints);

			gi.TagFree(dummy.classname);
		}
	}	

	// spawn entities so Quake2 properly sets the baselines
	for (i = 1; i < MAX_EDICTS; i++)
	{
		if (!dm2in.baselines.entities[i].modelindex)
			continue;

		if (dm2in_trans.edicts[i])
		{
			ent = g_edicts + dm2in_trans.edicts[i];
			ent->inuse = true;
		}
		else
		{
			ent = G_Spawn();
			dm2in_trans.edicts[i] = ent - g_edicts;
		}

		ent->s = dm2in.baselines.entities[i];

		if (ent->s.modelindex && ent->s.modelindex != 255)
			ent->s.modelindex = dm2in_trans.configstrings[CS_MODELS+ent->s.modelindex] - CS_MODELS;
		if (ent->s.modelindex2 && ent->s.modelindex2 != 255)
			ent->s.modelindex2 = dm2in_trans.configstrings[CS_MODELS+ent->s.modelindex2] - CS_MODELS;
		if (ent->s.modelindex3 && ent->s.modelindex3 != 255)
			ent->s.modelindex3 = dm2in_trans.configstrings[CS_MODELS+ent->s.modelindex3] - CS_MODELS;
		if (ent->s.modelindex4 && ent->s.modelindex4 != 255)
			ent->s.modelindex4 = dm2in_trans.configstrings[CS_MODELS+ent->s.modelindex4] - CS_MODELS;
		if (ent->s.renderfx & RF_CUSTOMSKIN)
			ent->s.skinnum = dm2in_trans.configstrings[CS_IMAGES+ent->s.skinnum] - CS_IMAGES;
		if (ent->s.sound)
			ent->s.sound = dm2in_trans.configstrings[CS_SOUNDS+ent->s.sound] - CS_SOUNDS;
		
		if (ent->s.modelindex && ent->s.modelindex < 255)
			gi.setmodel(ent, configstrings[CS_MODELS+ent->s.modelindex]);
		
		gi.linkentity(ent);
	}

/*	gi.dprintf("Version: %d\nKey:     %d\nIsdemo:  %d\nGame:    %s\nPlayer:  %d\nMap:     %s\n",
		dm2in.svd.version,
		dm2in.svd.key,
		dm2in.svd.isdemo,
			dm2in.svd.game,
		dm2in.svd.player,
		dm2in.svd.mapname);
*/

/*	for (i = 0; i < MAX_CONFIGSTRINGS; i++)
	{
		if (!dm2in.configstrings[i][0])
			continue;

		gi.dprintf("%-4d %s\n", i, dm2in.configstrings[i]);
	}
*/

	// set all areaportals as open
	for (i = 0; i <= numportals; i++)
		gi.SetAreaPortalState(i, true);
}


//===================================================================

#if 0
	// cursor positioning
	xl <value>
	xr <value>
	yb <value>
	yt <value>
	xv <value>
	yv <value>

	// drawing
	statpic <name>
	pic <stat>
	num <fieldwidth> <stat>
	string <stat>

	// control
	if <stat>
	ifeq <stat> <value>
	ifbit <stat> <value>
	endif

#endif
