/*
	Relay -- a tool to record and play Quake2 demos
	Copyright (C) 1999 Conor Davis

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

	Conor Davis
	cedavis@epid.org
*/

#include <stdarg.h>

#include "g_local.h"
#include "block.h"
#include "dm2.h"
#include "msg.h"
#include "q2utils.h"

void import_bprintf(int printlevel, char *fmt, ...)
{
	va_list	argptr;
	char	string[MAX_MSGLEN];

	va_start(argptr, fmt);
	vsprintf(string, fmt, argptr);
	va_end(argptr);

	gi.bprintf(printlevel, "%s", string);

	WriteByte(&prefix, SVC_PRINT);
	WriteByte(&prefix, printlevel);
	WriteString(&prefix, string);
}

void import_cprintf(edict_t *ent, int printlevel, char *fmt, ...)
{
	va_list	argptr;
	char	string[MAX_MSGLEN];

	va_start(argptr, fmt);
	vsprintf(string, fmt, argptr);
	va_end(argptr);

	gi.cprintf(ent, printlevel, "%s", string);

	// NULL ent means console only
	if (ent)
	{
		WriteByte(&prefix, SVC_PRINT | MSG_UNICAST);
		WriteByte(&prefix, EDICT2NUM(ent) - 1);
		WriteByte(&prefix, printlevel);
		WriteString(&prefix, string);
	}
}

void import_centerprintf(edict_t *ent, char *fmt, ...)
{
	char	string[MAX_MSGLEN];
	va_list	argptr;

	va_start(argptr, fmt);
	vsprintf(string, fmt, argptr);
	va_end(argptr);

	gi.centerprintf(ent, "%s", string);

	WriteByte(&prefix, SVC_CENTERPRINT | MSG_UNICAST);
	WriteByte(&prefix, EDICT2NUM(ent) - 1);
	WriteString(&prefix, string);
}

void import_sound(edict_t *ent, int channel, int soundindex, float volume, float attenuation, float timeofs)
{
	int		mask;

	gi.sound(ent, channel, soundindex, volume, attenuation, timeofs);

	mask = 0;
	if (volume != 1.0)				mask |= SND_VOLUME;
	if (attenuation != 1.0)			mask |= SND_ATTENUATION;
	if (timeofs)					mask |= SND_OFFSET;
	mask |= SND_ENT;
	
	WriteByte(&suffix, SVC_SOUND);
	WriteByte(&suffix, mask);
	WriteByte(&suffix, soundindex);
	if (mask & SND_VOLUME)			WriteByte(&suffix, volume * 255);
	if (mask & SND_ATTENUATION)		WriteByte(&suffix, attenuation * 64);
	if (mask & SND_OFFSET)			WriteByte(&suffix, timeofs * 1000);
	if (mask & SND_ENT)				WriteShort(&suffix, (EDICT2NUM(ent) << 3) | (channel & 0x07) );
}

void import_positioned_sound(vec3_t origin, edict_t *ent, int channel, int soundindex, float volume, float attenuation, float timeofs)
{
	int		mask;

	gi.sound(ent, channel, soundindex, volume, attenuation, timeofs);

	mask = 0;
	if (volume != 1.0)				mask |= SND_VOLUME;
	if (attenuation != 1.0)			mask |= SND_ATTENUATION;
	if (timeofs)					mask |= SND_OFFSET;
	mask |= SND_ENT;
	mask |= SND_POS;

	WriteByte(&suffix, SVC_SOUND);
	WriteByte(&suffix, mask);
	WriteByte(&suffix, soundindex);
	if (mask & SND_VOLUME)			WriteByte(&suffix, volume * 255);
	if (mask & SND_ATTENUATION)		WriteByte(&suffix, attenuation * 64);
	if (mask & SND_OFFSET)			WriteByte(&suffix, timeofs * 1000);
	if (mask & SND_ENT)				WriteShort(&suffix, (EDICT2NUM(ent) << 3) | (channel & 0x07) );
	if (mask & SND_POS)				WritePosition(&suffix, origin);
}

void import_configstring(int num, char *string)
{
	gi.configstring(num, string);
	if (string)
		strcpy(dm2out.configstrings[num], string);
	else
		dm2out.configstrings[num][0] = 0;

	if (current_frame != 0)
	{
		WriteByte(&prefix, SVC_CONFIGSTRING);
		WriteShort(&prefix, num);
		WriteString(&prefix, dm2out.configstrings[num]);
	}
}

int import_modelindex(char *name)
{
	int				index;

	index = gi.modelindex(name);
	if (index && !dm2out.configstrings[CS_MODELS+index][0])
	{
		strcpy(dm2out.configstrings[CS_MODELS+index], name);

		if (current_frame != 0)
		{
			WriteByte(&prefix, SVC_CONFIGSTRING);
			WriteShort(&prefix, CS_MODELS+index);
			WriteString(&prefix, name);
		}
	}

	return index;
}

int import_soundindex(char *name)
{
	int				index;

	index = gi.soundindex(name);
	if (index && !dm2out.configstrings[CS_SOUNDS+index][0])
	{
		strcpy(dm2out.configstrings[CS_SOUNDS+index], name);

		if (current_frame != 0)
		{
			WriteByte(&prefix, SVC_CONFIGSTRING);
			WriteShort(&prefix, CS_SOUNDS+index);
			WriteString(&prefix, name);
		}
	}

	return index;
}

int import_imageindex(char *name)
{
	int				index;

	index = gi.imageindex(name);
	if (index && !dm2out.configstrings[CS_IMAGES+index][0])
	{
		strcpy(dm2out.configstrings[CS_IMAGES+index], name);

		if (current_frame != 0)
		{
			WriteByte(&prefix, SVC_CONFIGSTRING);
			WriteShort(&prefix, CS_IMAGES+index);
			WriteString(&prefix, name);
		}
	}
	
	return index;
}

void import_setmodel(edict_t *ent, char *name)
{
	gi.setmodel(ent, name);
	if (ent->s.modelindex && !dm2out.configstrings[CS_MODELS+ent->s.modelindex][0])
	{
		strcpy(dm2out.configstrings[CS_MODELS+ent->s.modelindex], name);
		
		if (current_frame != 0)
		{
			WriteByte(&prefix, SVC_CONFIGSTRING);
			WriteShort(&prefix, CS_MODELS+ent->s.modelindex);
			WriteString(&prefix, name);
		}
	}
}

void import_SetAreaPortalState(int portalnum, qboolean connected)
{
	gi.SetAreaPortalState(portalnum, connected);
}

void import_linkentity(edict_t *ent)
{
	gi.linkentity(ent);
}

void import_unlinkentity(edict_t *ent)
{
	gi.unlinkentity(ent);
}

void import_multicast(vec3_t origin, multicast_t to)
{
	int		id;

	gi.multicast(origin, to);

	// hope that only one message is sent
	id = tempblock.buffer[0];
	switch(id)
	{
	case SVC_PRINT:
	case SVC_DISCONNECT:
	case SVC_RECONNECT:
	case SVC_SERVERDATA:
	case SVC_STUFFTEXT:
	case SVC_CONFIGSTRING:
	case SVC_CENTERPRINT:
	case SVC_SPAWNBASELINE:
		WriteBin(&prefix, tempblock.buffer, tempblock.size);
		break;
	case SVC_MUZZLEFLASH:
	case SVC_MUZZLEFLASH2:
	case SVC_TEMP_ENTITY:
	case SVC_LAYOUT:
	case SVC_INVENTORY:
	case SVC_SOUND:
		WriteBin(&suffix, tempblock.buffer, tempblock.size);
		break;
	default:
		gi.dprintf("RELAY: game has sent unsupported multicast message type: %d\n", id);
		break;
	}
	ResetBlock(&tempblock);
}

void import_unicast(edict_t *ent, qboolean reliable)
{
	int		id;

	gi.unicast(ent, reliable);

	id = tempblock.buffer[0];
	switch(id)
	{
	case SVC_PRINT:
	case SVC_DISCONNECT:
	case SVC_RECONNECT:
	case SVC_SERVERDATA:
	case SVC_STUFFTEXT:
	case SVC_CONFIGSTRING:
	case SVC_CENTERPRINT:
	case SVC_SPAWNBASELINE:
		WriteByte(&prefix, id | MSG_UNICAST);
		WriteByte(&prefix, EDICT2NUM(ent) - 1);
		WriteBin(&prefix, tempblock.buffer+1, tempblock.size-1);
		break;
	case SVC_MUZZLEFLASH:
	case SVC_MUZZLEFLASH2:
	case SVC_TEMP_ENTITY:
	case SVC_LAYOUT:
	case SVC_INVENTORY:
	case SVC_SOUND:
		WriteByte(&suffix, id | MSG_UNICAST);
		WriteByte(&suffix, EDICT2NUM(ent) - 1);
		WriteBin(&suffix, tempblock.buffer+1, tempblock.size-1);
		break;
	default:
		gi.dprintf("RELAY: game has sent unsupported multicast message type: %d\n", id);
		break;
	}

	ResetBlock(&tempblock);
}

void import_WriteChar(int c)
{
	gi.WriteChar(c);

	WriteChar(&tempblock, c);
}

void import_WriteByte(int c)
{
	gi.WriteByte(c);

	WriteByte(&tempblock, c);
}

void import_WriteShort(int c)
{
	gi.WriteShort(c);

	WriteShort(&tempblock, c);
}

void import_WriteLong(int c)
{
	gi.WriteLong(c);

	WriteShort(&tempblock, c);
}

void import_WriteFloat(float f)
{
	gi.WriteFloat(f);
}

void import_WriteString(char *s)
{
	gi.WriteString(s);

	if (s)
		WriteString(&tempblock, s);
	else
		WriteString(&tempblock, "");
}

void import_WritePosition(vec3_t pos)
{
	gi.WritePosition(pos);

	if (pos)
		WritePosition(&tempblock, pos);
	else
		WritePosition(&tempblock, vec3_origin);
}

void import_WriteDir(vec3_t pos)
{
	gi.WriteDir(pos);

	if (pos)
		WriteDir(&tempblock, pos);
	else
		WriteDir(&tempblock, vec3_origin);
}

void import_WriteAngle(float f)
{
	gi.WriteAngle(f);

	WriteAngle(&tempblock, f);
}
