/*
	Relay -- a tool to record and play Quake2 demos
	Copyright (C) 1999 Conor Davis

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

	Conor Davis
	cedavis@epid.org
*/

#include <stdio.h>

#include "shared.h"
#include "proxy.h"

#ifdef _WIN32
#define GAME_MODULE	"gamex86.dll"
#endif

typedef game_export_t *(*GetGameAPI_t) (game_import_t *);

void UnloadGameModule(proxy_t *proxydata)
{
	if (!proxydata->hModule)
		return;

	FreeLibrary(proxydata->hModule);
	proxydata->hModule = NULL;
}

qboolean LoadGameModule(proxy_t *proxydata, char *path, game_import_t *import)
{
	proxydata->hModule = LoadLibrary(path);
	proxydata->ge = NULL;
	if (!proxydata->hModule)
		return false;

	return true;
}

void LoadNextModule(proxy_t *proxydata, game_import_t *import)
{
	cvar_t	*basedir;
	cvar_t	*game;
	cvar_t	*proxy;
	cvar_t	*nextproxy;
	char	buf[1024], path[MAX_OSPATH];
	char	*gamedir, *current, *next;
	GetGameAPI_t GetGameAPI_f;

	basedir = import->cvar("basedir", ".", CVAR_NOSET);
	game = import->cvar("game", "", CVAR_SERVERINFO | CVAR_LATCH);
	proxy = import->cvar("proxy", "", CVAR_SERVERINFO | CVAR_LATCH);
	nextproxy = import->cvar("nextproxy", "", CVAR_NOSET);

	if (proxy->string[0] && !nextproxy->string[0])
		import->cvar_forceset("nextproxy", proxy->string);

	strncpy(buf, nextproxy->string, sizeof(buf)-1);
	buf[sizeof(buf)-1] = 0;

	if (game->string[0])
		gamedir = game->string;
	else
		gamedir = "baseq2";

	if (strcmp(buf, ":"))
	{
		// find the next proxy in the chain, cycle until we can successfully load one
		for (current = buf; current; current = next)
		{
			next = strchr(current, ':');
			
			if (next)
			{
				*next = 0;	// replace colon with NUL character
				next++;
				import->cvar_forceset("nextproxy", next);
			}
			else
				import->cvar_forceset("nextproxy", ":");
			
			if (current[0] && current[0] != ':')
			{
				sprintf(path, "%s/proxy/%s/" GAME_MODULE, basedir->string, current);
				import->dprintf("...loading proxy module \"%s\": ", path);
				if (LoadGameModule(proxydata, path, import))
				{
					import->dprintf("ok\n");
					break;
				}
				else
					import->dprintf("failed\n");
			}
		}
	}

	// load the default game dll if we haven't already loaded a proxy dll
	if (!proxydata->hModule)
	{
		sprintf(path, "%s/%s/" GAME_MODULE, basedir->string, gamedir);
		import->dprintf("...loading game module \"%s\": ", path);
		if (!LoadGameModule(proxydata, path, import))
		{
			import->dprintf("failed\n");
			sprintf(path, "%s/baseq2/" GAME_MODULE, basedir->string);
			import->dprintf("...loading default game module \"%s\": ", path);
			if (LoadGameModule(proxydata, path, import))
				import->dprintf("ok\n");
			else
				import->dprintf("failed\n");
		}
		else
			import->dprintf("ok\n");
	}

	if (!proxydata->hModule)
		return;

	GetGameAPI_f = (GetGameAPI_t) GetProcAddress(proxydata->hModule, "GetGameAPI");
	if (!GetGameAPI_f || !(proxydata->ge = GetGameAPI_f(import)))
	{
		UnloadGameModule(proxydata);
		proxydata->ge = NULL;
		return;
	}
}