/*
	Relay -- a tool to record and play Quake2 demos
	Copyright (C) 1999 Conor Davis

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

	Conor Davis
	cedavis@epid.org
*/

#ifndef	__DM2_H
#define	__DM2_H

#include <stdio.h>

#include "block.h"
#include "msg.h"

// most of these bits are from a serverrecord sample sent to Tom Vykruta
// http://www.stomped.com/demented2/dm2specs.htm

#define	RECORD_NETWORK		0
#define	RECORD_CLIENT		1
#define	RECORD_SERVER		2
// these are readable only by q2relay
#define RECORD_RELAY		0x80

// number of previous states to store for delta referencing
#define UPDATE_BACKUP	16
#define	UPDATE_MASK		(UPDATE_BACKUP-1)

// server-to-client commands
#define	SVC_BAD					0x00
#define	SVC_MUZZLEFLASH			0x01
#define	SVC_MUZZLEFLASH2		0x02
#define	SVC_TEMP_ENTITY			0x03
#define	SVC_LAYOUT				0x04
#define	SVC_INVENTORY			0x05
#define	SVC_NOP					0x06
#define	SVC_DISCONNECT			0x07
#define	SVC_RECONNECT			0x08
#define	SVC_SOUND				0x09
#define	SVC_PRINT				0x0A
#define	SVC_STUFFTEXT			0x0B
#define	SVC_SERVERDATA			0x0C
#define	SVC_CONFIGSTRING		0x0D
#define	SVC_SPAWNBASELINE		0x0E
#define	SVC_CENTERPRINT			0x0F
#define	SVC_DOWNLOAD			0x10
#define	SVC_PLAYERINFO			0x11
#define	SVC_PACKETENTITIES		0x12
#define SVC_DELTAPACKETENTITIES	0x13
#define	SVC_FRAME				0x14

// by default, messages are multicast
// if the id byte contains this bit, the message
// is unicast
#define	MSG_UNICAST				0x80

// client-to-server commands
#define	CLC_BAD				0x00
#define	CLC_NOP				0x01
#define	CLC_MOVE			0x02
#define	CLC_USERINFO		0x03
#define	CLC_STRINGCMD		0x04


// player_state_t mask bits
#define	PS_M_TYPE			(1<<0)
#define	PS_M_ORIGIN			(1<<1)
#define	PS_M_VELOCITY		(1<<2)
#define	PS_M_TIME			(1<<3)
#define	PS_M_FLAGS			(1<<4)
#define	PS_M_GRAVITY		(1<<5)
#define	PS_M_DELTA_ANGLES	(1<<6)
#define	PS_VIEWOFFSET		(1<<7)
#define	PS_VIEWANGLES		(1<<8)
#define	PS_KICKANGLES		(1<<9)
#define	PS_BLEND			(1<<10)
#define	PS_FOV				(1<<11)
#define	PS_WEAPONINDEX		(1<<12)
#define	PS_WEAPONFRAME		(1<<13)
#define	PS_RDFLAGS			(1<<14)

// usercmd_t mask bits
// ms and angle2 are allways sent, the others are optional
#define	CM_ANGLE1 	(1<<0)
#define	CM_ANGLE3 	(1<<1)
#define	CM_FORWARD	(1<<2)
#define	CM_SIDE		(1<<3)
#define	CM_UP		(1<<4)
#define	CM_BUTTONS	(1<<5)
#define	CM_IMPULSE	(1<<6)
#define CM_MSEC		(1<<7)

// sound mask bits
// a sound without an ent or pos will be a local only sound
#define	SND_VOLUME		(1<<0)		// a byte
#define	SND_ATTENUATION	(1<<1)		// a byte
#define	SND_POS			(1<<2)		// three coordinates
#define	SND_ENT			(1<<3)		// a short 0-2: channel, 3-12: entity
#define	SND_OFFSET		(1<<4)		// a byte, msec offset from frame start
// sound defaults
#define DEFAULT_SOUND_PACKET_VOLUME	1.0F
#define	DEFAULT_SOUND_PACKET_ATTENUATION 1.0F

// entity_state_t mask bits
// try to pack the common update flags into the first byte
#define	U_ORIGIN1	(1<<0)
#define	U_ORIGIN2	(1<<1)
#define	U_ANGLE2	(1<<2)
#define	U_ANGLE3	(1<<3)
#define	U_FRAME8	(1<<4)		// frame is a byte
#define	U_EVENT		(1<<5)
#define	U_REMOVE	(1<<6)		// REMOVE this entity, don't add it
#define	U_MOREBITS1	(1<<7)		// read one additional byte

// second byte
#define	U_NUMBER16	(1<<8)		// NUMBER8 is implicit if not set
#define	U_ORIGIN3	(1<<9)
#define	U_ANGLE1	(1<<10)
#define	U_MODEL		(1<<11)
#define U_RENDERFX8	(1<<12)		// fullbright, etc
#define	U_EFFECTS8	(1<<14)		// autorotate, trails, etc
#define	U_MOREBITS2	(1<<15)		// read one additional byte

// third byte
#define	U_SKIN8		(1<<16)
#define	U_FRAME16	(1<<17)		// frame is a short
#define	U_RENDERFX16 (1<<18)	// 8 + 16 = 32
#define	U_EFFECTS16	(1<<19)		// 8 + 16 = 32
#define	U_MODEL2	(1<<20)		// weapons, flags, etc
#define	U_MODEL3	(1<<21)
#define	U_MODEL4	(1<<22)
#define	U_MOREBITS3	(1<<23)		// read one additional byte

// fourth byte
#define	U_OLDORIGIN	(1<<24)		// FIXME: get rid of this
#define	U_SKIN16	(1<<25)
#define	U_SOUND		(1<<26)
#define	U_SOLID		(1<<27)

// message types
typedef struct
{
	short	entity;
	byte	value;
} muzzleflash_t;

typedef struct
{
	int		type;
	vec3_t	origin;
	vec3_t	movedir;
	vec3_t	endpos;
	int		count;
	int		style;
	int		entity;
	int		dest_entity;
	vec3_t	pos1;
	vec3_t	pos2;
	vec3_t	pos3;
	vec3_t	pos4;
	int		nextid;
	int		plat2flags;
	int		wait;
} temp_entity_t;

typedef struct
{
	byte	soundnum;
	float	volume;
	float	attenuation;
	float	timeofs;
	short	entity;
	short	channel;
	vec3_t	origin;
	qboolean positioned;
} sound_t;

typedef struct
{
	byte	level;
	char	string[MAX_MSGLEN];
} print_t;

typedef struct
{
	long	version;
	long	key;
	byte	isdemo;
	char	game[MAX_QPATH];
	short	player;
	char	mapname[MAX_QPATH];
} serverdata_t;

typedef struct
{
	short	index;
	char	string[MAX_MSGLEN];
} configstring_t;

typedef struct
{
	long	seq1;
	long	seq2;
	byte	areas[MAX_MAP_AREAS/8];
} frame_t;

// new, q2relay-specific message types
typedef struct
{
	int		maxclients;
} relayinfo_t;


typedef struct
{
	long			frame;
	block_t			block;
	char			block_buffer[MAX_MSGLEN];
} delta_t;

typedef struct
{
	long			frame;
	entity_state_t	entities[MAX_EDICTS];	
	byte			visible[MAX_EDICTS];	// visible entities
	byte			areas[MAX_MAP_AREAS/8];	// visible areas in client demos, open areaportals in relay demos
	// relay-specific data
	byte			connected[MAX_CLIENTS/8];
} state_t;

typedef struct
{
	char			layout[MAX_MSGLEN];
	short			inventory[MAX_ITEMS];
	player_state_t	ps[UPDATE_BACKUP];
} player_t;

typedef struct
{
	serverdata_t	svd;
	char			configstrings[MAX_CONFIGSTRINGS][64];
	state_t			baselines;
	state_t			states[UPDATE_BACKUP];
	player_t		*players;	// client demos will have one player, relay will have maxclients
	int				maxclients;	// determines size of players array. must be 1 for client demos!
	long			current_frame;
	long			delta_frame;
} dm2_t;


extern void InitDM2(dm2_t *dm2);

extern void ReadPS(block_t *src, player_state_t *ps);
extern void WritePS(block_t *dest, player_state_t *to, player_state_t *from);
extern void ReadEntities(block_t *src, state_t *to, state_t *baselines);
extern void WriteEntities(block_t *dest, state_t *to, state_t *from, state_t *baselines);

extern void WriteMessage(block_t *dest, int version, int id, void *void_data);
extern msg_t *ParseMessage(dm2_t *dm2, block_t *src);
extern int	ReadBlock(block_t *block, FILE *fd, int isdemo);
extern int ReadPreFrame(dm2_t *dm2, FILE *fd);
extern void WritePreFrame(serverdata_t *serverdata, relayinfo_t *relayinfo, char configstrings[MAX_CONFIGSTRINGS][64], state_t *baselines, FILE *fd);

#endif	// __DM2_H