/*
	Relay -- a tool to record and play Quake2 demos
	Copyright (C) 1999 Conor Davis

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

	Conor Davis
	cedavis@epid.org
*/

#include <assert.h>
#include <stdlib.h>
#include <string.h>

#include "shared.h"
#include "block.h"
#include "msg.h"
#include "dm2.h"
#include "q2utils.h"
#include "utils.h"

void InitDM2(dm2_t *dm2)
{
	int		i;

	memset(dm2, 0, sizeof(dm2_t));
	for (i = 0; i < MAX_EDICTS; i++)
		dm2->baselines.entities[i].number = i;
	dm2->baselines.frame = -1;
}

void ReadPS(block_t *src, player_state_t *ps)
{
	int	mask, i;

	mask = ReadShort(src);
	if (mask & PS_M_TYPE)			ps->pmove.pm_type = ReadByte(src);
	if (mask & PS_M_ORIGIN)			ReadShortPosition(src, ps->pmove.origin);
	if (mask & PS_M_VELOCITY)		ReadShortPosition(src, ps->pmove.velocity);
	if (mask & PS_M_TIME)			ps->pmove.pm_type = ReadByte(src);
	if (mask & PS_M_FLAGS)			ps->pmove.pm_flags = ReadByte(src);
	if (mask & PS_M_GRAVITY)		ps->pmove.gravity = ReadShort(src);
	if (mask & PS_M_DELTA_ANGLES)	ReadShortPosition(src, ps->pmove.delta_angles);

	if (mask & PS_VIEWOFFSET)		ReadOffsetVec(src, ps->viewoffset);
	if (mask & PS_VIEWANGLES)
	{
		ps->viewangles[0] = ReadAngle16(src);
		ps->viewangles[1] = ReadAngle16(src);
		ps->viewangles[2] = ReadAngle16(src);
	}
	if (mask & PS_KICKANGLES)		ReadOffsetVec(src, ps->kick_angles);
	if (mask & PS_WEAPONINDEX)		ps->gunindex = ReadByte(src);
	if (mask & PS_WEAPONFRAME)
	{
		ps->gunframe = ReadByte(src);
		ReadOffsetVec(src, ps->gunoffset);
		ReadOffsetVec(src, ps->gunangles);
	}
	if (mask & PS_BLEND)			ReadBlendVec(src, ps->blend);
	if (mask & PS_FOV)				ps->fov = ReadByte(src);
	if (mask & PS_RDFLAGS)			ps->rdflags = ReadByte(src);
	
	mask = ReadLong(src);
	for (i = 0; i < MAX_STATS; i++)
		if (mask & (1 << i))		ps->stats[i] = ReadShort(src);
}

void WritePS(block_t *dest, player_state_t *to, player_state_t *from)
{
	int		mask, mask2;
	int		i;

	// generate masks to save bandwidth
	mask = mask2 = 0;
	if (to->pmove.pm_type != from->pmove.pm_type)				mask |= PS_M_TYPE;
	if (!VectorCompare(to->pmove.origin, from->pmove.origin))	mask |= PS_M_ORIGIN;
	if (!VectorCompare(to->pmove.velocity, from->pmove.velocity))	mask |= PS_M_VELOCITY;
	if (to->pmove.pm_time != from->pmove.pm_time)				mask |= PS_M_TIME;
	if (to->pmove.pm_flags != from->pmove.pm_flags)				mask |= PS_M_FLAGS;
	if (to->pmove.gravity != from->pmove.gravity)				mask |= PS_M_GRAVITY;
	if (!VectorCompare(to->pmove.delta_angles, from->pmove.delta_angles))	mask |= PS_M_DELTA_ANGLES;
	if (!VectorCompare(to->viewoffset, from->viewoffset))		mask |= PS_VIEWOFFSET;
	if (!VectorCompare(to->viewangles, from->viewangles))		mask |= PS_VIEWANGLES;
	if (!VectorCompare(to->kick_angles, from->kick_angles))		mask |= PS_KICKANGLES;
	if (to->gunindex != from->gunindex)							mask |= PS_WEAPONINDEX;
	if (to->gunframe != from->gunframe ||
		!VectorCompare(to->gunoffset, from->gunoffset) ||
		!VectorCompare(to->gunangles, from->gunangles))			mask |= PS_WEAPONFRAME;
	if (to->blend[0] != from->blend[0] ||
		to->blend[1] != from->blend[1] ||
		to->blend[2] != from->blend[2] ||
		to->blend[3] != from->blend[3])							mask |= PS_BLEND;
	if (to->fov != from->fov)									mask |= PS_FOV;
	if (to->rdflags != from->rdflags)							mask |= PS_RDFLAGS;

	for (i = 0; i < MAX_STATS; i++)
	{
		if (i == STAT_FLASHES)
		{
			if (to->stats[i])									mask2 |= (1 << i);
		}
		else
		{
			if (to->stats[i] != from->stats[i])					mask2 |= (1 << i);
		}
	}

//	WriteByte(dest, SVC_PLAYERINFO);
	WriteShort(dest, mask);
	if (mask & PS_M_TYPE)			WriteByte(dest, to->pmove.pm_type);
	if (mask & PS_M_ORIGIN)			WriteShortPosition(dest, to->pmove.origin);
	if (mask & PS_M_VELOCITY)		WriteShortPosition(dest, to->pmove.velocity);
	if (mask & PS_M_TIME)			WriteByte(dest, to->pmove.pm_time);
	if (mask & PS_M_FLAGS)			WriteByte(dest, to->pmove.pm_flags);
	if (mask & PS_M_GRAVITY)		WriteShort(dest, to->pmove.gravity);
	if (mask & PS_M_DELTA_ANGLES)	WriteShortPosition(dest, to->pmove.delta_angles);
	if (mask & PS_VIEWOFFSET)		WriteOffsetVec(dest, to->viewoffset);
	if (mask & PS_VIEWANGLES)
	{
		WriteAngle16(dest, to->viewangles[0]);
		WriteAngle16(dest, to->viewangles[1]);
		WriteAngle16(dest, to->viewangles[2]);
	}
	if (mask & PS_KICKANGLES)		WriteOffsetVec(dest, to->kick_angles);
	if (mask & PS_WEAPONINDEX)		WriteByte(dest, to->gunindex);
	if (mask & PS_WEAPONFRAME)
	{
		WriteByte(dest, to->gunframe);
		WriteOffsetVec(dest, to->gunoffset);
		WriteOffsetVec(dest, to->gunangles);
	}
	if (mask & PS_BLEND)			WriteBlendVec(dest, to->blend);
	if (mask & PS_FOV)				WriteByte(dest, to->fov);
	if (mask & PS_RDFLAGS)			WriteByte(dest, to->rdflags);

	WriteLong(dest, mask2);
	for (i = 0; i < MAX_STATS; i++)
		if (mask2 & (1 << i))		WriteShort(dest, to->stats[i]);

}

void ReadEntities(block_t *src, state_t *to, state_t *baselines)
{
	int				mask, entity;
	entity_state_t	*es;

	while (1)
	{
		mask = ReadByte(src);
		if (mask & U_MOREBITS1)		mask |= ReadByte(src) <<  8;
		if (mask & U_MOREBITS2)		mask |= ReadByte(src) << 16;
		if (mask & U_MOREBITS3)		mask |= ReadByte(src) << 24;

		if (mask & U_NUMBER16)
			entity = ReadShort(src);
		else
			entity = ReadByte(src);

		if (entity == 0)
			break;

		es = &to->entities[entity];
		if (!ISBITSET(to->visible, entity))
		{
			memcpy(es, &baselines->entities[entity], sizeof(entity_state_t));
			es->event = 0;		// do this?
		}

		if (mask & U_MODEL)					es->modelindex = ReadByte(src);
		if (mask & U_MODEL2)				es->modelindex2 = ReadByte(src);
		if (mask & U_MODEL3)				es->modelindex3 = ReadByte(src);
		if (mask & U_MODEL4)				es->modelindex4 = ReadByte(src);
		if (mask & U_FRAME8)				es->frame = ReadByte(src);
		if (mask & U_FRAME16)				es->frame = ReadShort(src);
		if (mask & U_SKIN8)
		{
			if (mask & U_SKIN16)			es->skinnum = ReadLong(src);
			else							es->skinnum = ReadByte(src);
		}
		else if (mask & U_SKIN16)			es->skinnum = ReadShort(src);
		if (mask & U_EFFECTS8)
		{
			if (mask & U_EFFECTS16)			es->effects = ReadLong(src);
			else							es->effects = ReadByte(src);
		}
		else if (mask & U_EFFECTS16)		es->effects = ReadShort(src);
		if (mask & U_RENDERFX8)
		{
			if (mask & U_RENDERFX16)		es->renderfx = ReadLong(src);
			else							es->renderfx = ReadByte(src);
		}
		else if (mask & U_RENDERFX16)		es->renderfx = ReadShort(src);
		if (mask & U_ORIGIN1)				es->origin[0] = ReadCoord(src);
		if (mask & U_ORIGIN2)				es->origin[1] = ReadCoord(src);
		if (mask & U_ORIGIN3)				es->origin[2] = ReadCoord(src);
		if (mask & U_ANGLE1)				es->angles[0] = ReadAngle(src);
		if (mask & U_ANGLE2)				es->angles[1] = ReadAngle(src);
		if (mask & U_ANGLE3)				es->angles[2] = ReadAngle(src);
		if (mask & U_OLDORIGIN)				ReadPosition(src, es->old_origin);
		if (mask & U_SOUND)					es->sound = ReadByte(src);
		if (mask & U_EVENT)					es->event = ReadByte(src);
		if (mask & U_SOLID)					es->solid = ReadShort(src);

		if (mask & U_REMOVE)
			SETBIT(to->visible, entity, false);
		else
			SETBIT(to->visible, entity, true);
	}
}

void WriteEntities(block_t *dest, state_t *to, state_t *from, state_t *baselines)
{
	int				mask, i;
	qboolean		isvisible, wasvisible;
	entity_state_t	*to_es, *from_es;

	WriteByte(dest, SVC_PACKETENTITIES);
	for (i = 1; i < MAX_EDICTS; i++)
	{
		isvisible = ISBITSET(to->visible, i);
		wasvisible = ISBITSET(from->visible, i);

		if (!isvisible && !wasvisible)
			continue;

		to_es = &to->entities[i];

		mask = 0;
		if (!isvisible)
			mask |= U_REMOVE;
		else
		{
			if (!wasvisible)
				from_es = &baselines->entities[i];		// newly visible
			else
				from_es = &from->entities[i];

			if (to_es->modelindex != from_es->modelindex)	mask |= U_MODEL;
			if (to_es->modelindex2 != from_es->modelindex2)	mask |= U_MODEL2;
			if (to_es->modelindex3 != from_es->modelindex3)	mask |= U_MODEL3;
			if (to_es->modelindex4 != from_es->modelindex4)	mask |= U_MODEL4;
			if (to_es->origin[0] != from_es->origin[0])	mask |= U_ORIGIN1;
			if (to_es->origin[1] != from_es->origin[1])	mask |= U_ORIGIN2;
			if (to_es->origin[2] != from_es->origin[2])	mask |= U_ORIGIN3;
			if (to_es->angles[0] != from_es->angles[0])	mask |= U_ANGLE1;
			if (to_es->angles[1] != from_es->angles[1])	mask |= U_ANGLE2;
			if (to_es->angles[2] != from_es->angles[2])	mask |= U_ANGLE3;
			if (to_es->frame != from_es->frame)
			{
				if (to_es->frame > 0xFF)				mask |= U_FRAME16;
				else									mask |= U_FRAME8;
			}
			if (to_es->skinnum != from_es->skinnum)
			{
				if (to_es->skinnum > 0xFFFF)			mask |= U_SKIN16|U_SKIN8;
				else if (to_es->skinnum > 0xFF)			mask |= U_SKIN16;
				else									mask |= U_SKIN8;
			}
			if (to_es->effects != from_es->effects)
			{
				if (to_es->effects > 0xFFFF)			mask |= U_EFFECTS16|U_EFFECTS8;
				else if (to_es->effects > 0xFF)			mask |= U_EFFECTS16;
				else									mask |= U_EFFECTS8;
			}
			if (to_es->renderfx != from_es->renderfx)
			{
				if (to_es->renderfx > 0xFFFF)			mask |= U_RENDERFX16|U_RENDERFX8;
				else if (to_es->renderfx > 0xFF)		mask |= U_RENDERFX16;
				else									mask |= U_RENDERFX8;
			}
			if (!VectorCompare(to_es->old_origin, from_es->origin))
				mask |= U_OLDORIGIN;
			if (to_es->sound != from_es->sound)			mask |= U_SOUND;
			if (to_es->event)							mask |= U_EVENT;
			if (to_es->solid != from_es->solid)			mask |= U_SOLID;
		}

		// nothing has changed
		if (!mask && isvisible && wasvisible)
			continue;

		if (i > 0xFF)									mask |= U_NUMBER16;

		if (mask & 0xFF000000)							mask |= U_MOREBITS1|U_MOREBITS2|U_MOREBITS3;
		else if (mask & 0x00FF0000)						mask |= U_MOREBITS1|U_MOREBITS2;
		else if (mask & 0x0000FF00)						mask |= U_MOREBITS1;

		WriteByte(dest, mask & 0xFF);
		if (mask & U_MOREBITS1)			WriteByte(dest, (mask >>  8) & 0xFF);
		if (mask & U_MOREBITS2)			WriteByte(dest, (mask >> 16) & 0xFF);
		if (mask & U_MOREBITS3)			WriteByte(dest, (mask >> 24) & 0xFF);
		if (mask & U_NUMBER16)			WriteShort(dest, i);
		else							WriteByte(dest, i);
		if (mask & U_MODEL)				WriteByte(dest, to_es->modelindex);
		if (mask & U_MODEL2)			WriteByte(dest, to_es->modelindex2);
		if (mask & U_MODEL3)			WriteByte(dest, to_es->modelindex3);
		if (mask & U_MODEL4)			WriteByte(dest, to_es->modelindex4);
		if (mask & U_FRAME8)			WriteByte(dest, to_es->frame);
		if (mask & U_FRAME16)			WriteShort(dest, to_es->frame);
		if (mask & U_SKIN8)
		{
			if (mask & U_SKIN16)		WriteLong(dest, to_es->skinnum);
			else						WriteByte(dest, to_es->skinnum);
		}
		else if (mask & U_SKIN16)		WriteShort(dest, to_es->skinnum);
		if (mask & U_EFFECTS8)
		{
			if (mask & U_EFFECTS16)		WriteLong(dest, to_es->effects);
			else						WriteByte(dest, to_es->effects);
		}
		else if (mask & U_EFFECTS16)	WriteShort(dest, to_es->effects);
		if (mask & U_RENDERFX8)
		{
			if (mask & U_RENDERFX16)	WriteLong(dest, to_es->renderfx);
			else						WriteByte(dest, to_es->renderfx);
		}
		else if (mask & U_RENDERFX16)	WriteShort(dest, to_es->renderfx);
		if (mask & U_ORIGIN1)			WriteCoord(dest, to_es->origin[0]);
		if (mask & U_ORIGIN2)			WriteCoord(dest, to_es->origin[1]);
		if (mask & U_ORIGIN3)			WriteCoord(dest, to_es->origin[2]);
		if (mask & U_ANGLE1)			WriteAngle(dest, to_es->angles[0]);
		if (mask & U_ANGLE2)			WriteAngle(dest, to_es->angles[1]);
		if (mask & U_ANGLE3)			WriteAngle(dest, to_es->angles[2]);
		if (mask & U_OLDORIGIN)			WritePosition(dest, to_es->old_origin);
		if (mask & U_SOUND)				WriteByte(dest, to_es->sound);
		if (mask & U_EVENT)				WriteByte(dest, to_es->event);
		if (mask & U_SOLID)				WriteShort(dest, to_es->solid);
	}
	WriteShort(dest, 0);	// terminating packetentities message (mask=0, entity=0)
}

void WriteMessage(block_t *dest, int version, int id, void *void_data)
{
	switch(id)
	{
	case SVC_BAD:
		{
			WriteByte(dest, SVC_BAD);
		}
		break;
	case SVC_MUZZLEFLASH:
	case SVC_MUZZLEFLASH2:
		{
			muzzleflash_t *data = void_data;

			WriteByte(dest, id);
			WriteShort(dest, data->entity);
			WriteByte(dest, data->value);
		}
		break;
	case SVC_TEMP_ENTITY:
		{
			temp_entity_t *data = void_data;

			WriteByte(dest, SVC_TEMP_ENTITY);
			WriteByte(dest, data->type);

			switch(data->type)
			{
			// case TE_PLASMATRAIL:			// old meaning
			case TE_GREENBLOOD:				// new meaning (3.15+)
				if (version >= 32)
					goto impact_entity;
				else
					goto line_entity;
			// case TE_GREENBLOOD_old:		// old meaning
			case TE_BLUEHYPERBLASTER:		// new meaning (3.15+)
				if (version >= 32)
					goto line_entity;
				else
					goto impact_entity;
			// point entity
			case TE_EXPLOSION1:
			case TE_EXPLOSION2:
			case TE_ROCKET_EXPLOSION:
			case TE_GRENADE_EXPLOSION:
			case TE_ROCKET_EXPLOSION_WATER:
			case TE_GRENADE_EXPLOSION_WATER:
			case TE_BFG_EXPLOSION:
			case TE_BFG_BIGEXPLOSION:
			case TE_BOSSTPORT:
			case TE_PLASMA_EXPLOSION:
			case TE_PLAIN_EXPLOSION:
			case TE_CHAINFIST_SMOKE:
			case TE_TRACKER_EXPLOSION:
			case TE_TELEPORT_EFFECT:
			case TE_DBALL_GOAL:
			case TE_NUKEBLAST:
			case TE_WIDOWSPLASH:
			case TE_EXPLOSION1_BIG:
			case TE_EXPLOSION1_NP:
				WritePosition(dest, data->origin);
				break;
			// impact entity
			case TE_GUNSHOT:
			case TE_BLOOD:
			case TE_BLASTER:
			case TE_SHOTGUN:
			case TE_SPARKS:
			case TE_SCREEN_SPARKS:
			case TE_SHIELD_SPARKS:
			case TE_BULLET_SPARKS:
			// case TE_GREENBLOOD_new:
			// case TE_GREENBLOOD_old:
			case TE_BLASTER2:
			case TE_MOREBLOOD:
			case TE_HEATBEAM_SPARKS:
			case TE_HEATBEAM_STEAM:
			case TE_ELECTRIC_SPARKS:
			case TE_FLECHETTE:
impact_entity:
				WritePosition(dest, data->origin);
				WriteDir(dest, data->movedir);
				break;
			// line entity
			case TE_RAILTRAIL:
			case TE_BUBBLETRAIL:
			case TE_BFG_LASER:
			// case TE_PLASMATRAIL:
			// case TE_BLUEHYPERBLASTER:
			case TE_DEBUGTRAIL:
			case TE_BUBBLETRAIL2:
line_entity:
				WritePosition(dest, data->origin);
				WritePosition(dest, data->endpos);
				break;
			// special entity
			case TE_SPLASH:
			case TE_LASER_SPARKS:
			case TE_WELDING_SPARKS:
			case TE_TUNNEL_SPARKS:
				WriteByte(dest, data->count);
				WritePosition(dest, data->origin);
				WriteDir(dest, data->movedir);
				WriteByte(dest, data->style);
				break;
			case TE_PARASITE_ATTACK:
			case TE_MEDIC_CABLE_ATTACK:
			case TE_HEATBEAM:
			case TE_MONSTER_HEATBEAM:
				WriteShort(dest, data->entity);
				WritePosition(dest, data->origin);
				WritePosition(dest, data->endpos);
				break;
			case TE_GRAPPLE_CABLE:
				WriteShort(dest, data->entity);
				WritePosition(dest, data->origin);
				WritePosition(dest, data->endpos);
				WritePosition(dest, data->pos1);
				break;
			case TE_FLAME:
				WriteShort(dest, data->entity);
				WriteShort(dest, data->count);
				WritePosition(dest, data->endpos);
				WritePosition(dest, data->origin);
				WritePosition(dest, data->pos1);
				WritePosition(dest, data->pos2);
				WritePosition(dest, data->pos3);
				WritePosition(dest, data->pos4);
				break;
			case TE_LIGHTNING:
				WriteShort(dest, data->dest_entity);
				WriteShort(dest, data->entity);
				WritePosition(dest, data->endpos);
				WritePosition(dest, data->origin);
				break;
			case TE_FLASHLIGHT:
				WritePosition(dest, data->origin);
				WriteShort(dest, data->entity);
				break;
			case TE_FORCEWALL:
				WritePosition(dest, data->origin);
				WritePosition(dest, data->endpos);
				WriteShort(dest, data->style);
				break;
			case TE_STEAM:
				WriteShort(dest, data->nextid);
				WriteByte(dest, data->count);
				WritePosition(dest, data->origin);
				WriteDir(dest, data->movedir);
				WriteByte(dest, data->style);
				WriteShort(dest, data->plat2flags);
				if (data->nextid != -1)
					WriteLong(dest, data->wait);
				break;
			default:
				Com_Printf("WriteMessage: unknown temp entity type: %d\n", data->type);
				break;
			}
		}
		break;
	case SVC_LAYOUT:
		{
			WriteByte(dest, SVC_LAYOUT);
			WriteString(dest, (char *)void_data);
		}
		break;
	case SVC_INVENTORY:
		{
			short	*inventory = void_data;
			int		i;

			WriteByte(dest, SVC_INVENTORY);
			for (i = 0; i < MAX_ITEMS; i++)
				WriteShort(dest, inventory[i]);
		}
		break;
	case SVC_NOP:
		{
			WriteByte(dest, SVC_NOP);
		}
		break;
	case SVC_DISCONNECT:
		{
			WriteByte(dest, SVC_DISCONNECT);
		}
		break;
	case SVC_RECONNECT:
		{
			WriteByte(dest, SVC_RECONNECT);
		}
		break;
	case SVC_SOUND:
		{
			sound_t	*data = void_data;
			int		mask, mix;

			mask = 0;
			if (data->volume != DEFAULT_SOUND_PACKET_VOLUME)			mask |= SND_VOLUME;
			if (data->attenuation != DEFAULT_SOUND_PACKET_ATTENUATION)	mask |= SND_ATTENUATION;
			if (data->timeofs)											mask |= SND_OFFSET;
			if (data->entity || data->channel)							mask |= SND_ENT;
			if (data->positioned)										mask |= SND_POS;

			WriteByte(dest, SVC_SOUND);
			WriteByte(dest, mask);
			WriteByte(dest, data->soundnum);
			if (mask & SND_VOLUME)			WriteByte(dest, (byte)(data->volume * 255));
			if (mask & SND_ATTENUATION)		WriteByte(dest, (byte)(data->attenuation * 64));
			if (mask & SND_OFFSET)			WriteByte(dest, (byte)(data->timeofs * 1000));
			if (mask & SND_ENT)
			{
				mix = ((data->entity & 0x3FF) << 3) | (data->channel & 0x07);
				WriteShort(dest, mix);
			}
			if (mask & SND_POS)
				WritePosition(dest, data->origin);
		}
		break;
	case SVC_PRINT:
		{
			print_t	*data = void_data;

			WriteByte(dest, SVC_PRINT);
			WriteByte(dest, data->level);
			WriteString(dest, data->string);
		}
		break;
	case SVC_STUFFTEXT:
		{
			WriteByte(dest, SVC_STUFFTEXT);
			WriteString(dest, (char *)void_data);
		}
		break;
	case SVC_SERVERDATA:
		{
			serverdata_t	*data = void_data;

			WriteByte(dest, SVC_SERVERDATA);
			WriteLong(dest, data->version);
			WriteLong(dest, data->key);
			WriteByte(dest, data->isdemo);
			WriteString(dest, data->game);
			WriteShort(dest, data->player);
			WriteString(dest, data->mapname);
		}
		break;
	case SVC_CONFIGSTRING:
		{
			configstring_t	*data = void_data;

			WriteByte(dest, SVC_CONFIGSTRING);
			WriteShort(dest, data->index);
			WriteString(dest, data->string);
		}
		break;
	case SVC_SPAWNBASELINE:
		{
			entity_state_t	*es = void_data;
			int				mask;

			// generate mask to save bandwidth
			mask = 0;
			if (es->modelindex)				mask |= U_MODEL;
			if (es->modelindex2)			mask |= U_MODEL2;
			if (es->modelindex3)			mask |= U_MODEL3;
			if (es->modelindex4)			mask |= U_MODEL4;
			if (es->origin[0])				mask |=	U_ORIGIN1;
			if (es->origin[1])				mask |= U_ORIGIN2;
			if (es->origin[2])				mask |= U_ORIGIN3;
			if (es->angles[0])				mask |= U_ANGLE1;
			if (es->angles[1])				mask |= U_ANGLE2;
			if (es->angles[2])				mask |= U_ANGLE3;
			if (es->frame > 0xFF)			mask |= U_FRAME16;
			else if (es->frame)				mask |= U_FRAME8;
			if (es->skinnum > 0xFFFF)		mask |= U_SKIN16|U_SKIN8;
			else if (es->skinnum > 0xFF)	mask |= U_SKIN16;
			else if (es->skinnum)			mask |= U_SKIN8;
			if (es->effects > 0xFFFF)		mask |= U_EFFECTS16|U_EFFECTS8;
			else if (es->effects > 0xFF)	mask |= U_EFFECTS16;
			else if (es->effects)			mask |= U_EFFECTS8;
			if (es->renderfx > 0xFFFF)		mask |= U_RENDERFX16|U_RENDERFX8;
			else if (es->renderfx > 0xFF)	mask |= U_RENDERFX16;
			else if (es->renderfx)			mask |= U_RENDERFX8;
			if (es->old_origin[0] || es->old_origin[1] || es->old_origin[2])
				mask |= U_OLDORIGIN;
			if (es->sound)					mask |= U_SOUND;
			if (es->event)					mask |= U_EVENT;
			if (es->solid)					mask |= U_SOLID;

			if		(mask & 0xFF000000)		mask |= U_MOREBITS1|U_MOREBITS2|U_MOREBITS3;
			else if	(mask & 0x00FF0000)		mask |= U_MOREBITS1|U_MOREBITS2;
			else if	(mask & 0x0000FF00)		mask |= U_MOREBITS1;

			if (!mask)
				return;
			if (es->number > 0xFF)			mask |= U_NUMBER16;

			WriteByte(dest, SVC_SPAWNBASELINE);
			WriteByte(dest, mask & 0xFF);
			if (mask & U_MOREBITS1)			WriteByte(dest, (mask >>  8) & 0xFF);
			if (mask & U_MOREBITS2)			WriteByte(dest, (mask >> 16) & 0xFF);
			if (mask & U_MOREBITS3)			WriteByte(dest, (mask >> 24) & 0xFF);

			if (mask & U_NUMBER16)			WriteShort(dest, es->number);
			else							WriteByte(dest, es->number);
			if (mask & U_MODEL)				WriteByte(dest, es->modelindex);
			if (mask & U_MODEL2)			WriteByte(dest, es->modelindex2);
			if (mask & U_MODEL3)			WriteByte(dest, es->modelindex3);
			if (mask & U_MODEL4)			WriteByte(dest, es->modelindex4);
			if (mask & U_FRAME8)			WriteByte(dest, es->frame);
			if (mask & U_FRAME16)			WriteShort(dest, es->frame);

			if (mask & U_SKIN8)
			{
				if (mask & U_SKIN16)		WriteLong(dest, es->skinnum);
				else						WriteByte(dest, es->skinnum);
			}
			else if (mask & U_SKIN16)		WriteShort(dest, es->skinnum);

			if (mask & U_EFFECTS8)
			{
				if (mask & U_EFFECTS16)		WriteLong(dest, es->effects);
				else						WriteByte(dest, es->effects);
			}
			else if (mask & U_EFFECTS16)	WriteShort(dest, es->effects);

			if (mask & U_RENDERFX8)
			{
				if (mask & U_RENDERFX16)		WriteLong(dest, es->renderfx);
				else						WriteByte(dest, es->renderfx);
			}
			else if (mask & U_RENDERFX16)	WriteShort(dest, es->renderfx);

			if (mask & U_ORIGIN1)			WriteCoord(dest, es->origin[0]);
			if (mask & U_ORIGIN2)			WriteCoord(dest, es->origin[1]);
			if (mask & U_ORIGIN3)			WriteCoord(dest, es->origin[2]);
			if (mask & U_ANGLE1)			WriteAngle(dest, es->angles[0]);
			if (mask & U_ANGLE2)			WriteAngle(dest, es->angles[1]);
			if (mask & U_ANGLE3)			WriteAngle(dest, es->angles[2]);
			if (mask & U_OLDORIGIN)			WritePosition(dest, es->old_origin);
			if (mask & U_SOUND)				WriteByte(dest, es->sound);
			if (mask & U_EVENT)				WriteByte(dest, es->event);
			if (mask & U_SOLID)				WriteShort(dest, es->solid);
		}
		break;
	case SVC_CENTERPRINT:
		{
			WriteByte(dest, SVC_CENTERPRINT);
			WriteString(dest, (char *)void_data);
		}
		break;
	default:
		assert(0);
		Com_Printf("WriteMessage: unable to create message from id: %d\n", id);
		break;
	}
}

// This is a generic parse function that converts one message in a block
// from binary format to a convenient message structure. It is up to the caller
// to do something with it and free it afterward.
msg_t *ParseMessage(dm2_t *dm2, block_t *src)
{
	int		id, i;
	msg_t	*msg;

	id = ReadByte(src);
	
	switch(id)
	{
	case SVC_MUZZLEFLASH:
	case SVC_MUZZLEFLASH2:
		{
			muzzleflash_t	data;
			
			data.entity = ReadShort(src);
			data.value = ReadByte(src);
			
			msg = CreateMessage(id, &data, sizeof(muzzleflash_t), false);
			VectorCopy(dm2->states[dm2->current_frame & UPDATE_MASK].entities[data.entity].origin, msg->origin);
			return msg;
		}
		break;
	case SVC_TEMP_ENTITY:
		{
			temp_entity_t	data;
			
			data.type = ReadByte(src);
			switch(data.type)
			{
			// case TE_PLASMATRAIL:			// old meaning
			case TE_GREENBLOOD:				// new meaning (3.15+)
				if (dm2->svd.version >= 32)
					goto impact_entity;
				else
					goto line_entity;
			// case TE_GREENBLOOD_old:		// old meaning
			case TE_BLUEHYPERBLASTER:		// new meaning (3.15+)
				if (dm2->svd.version >= 32)
					goto line_entity;
				else
					goto impact_entity;
			// point entity
			case TE_EXPLOSION1:
			case TE_EXPLOSION2:
			case TE_ROCKET_EXPLOSION:
			case TE_GRENADE_EXPLOSION:
			case TE_ROCKET_EXPLOSION_WATER:
			case TE_GRENADE_EXPLOSION_WATER:
			case TE_BFG_EXPLOSION:
			case TE_BFG_BIGEXPLOSION:
			case TE_BOSSTPORT:
			case TE_PLASMA_EXPLOSION:
			case TE_PLAIN_EXPLOSION:
			case TE_CHAINFIST_SMOKE:
			case TE_TRACKER_EXPLOSION:
			case TE_TELEPORT_EFFECT:
			case TE_DBALL_GOAL:
			case TE_NUKEBLAST:
			case TE_WIDOWSPLASH:
			case TE_EXPLOSION1_BIG:
			case TE_EXPLOSION1_NP:
				ReadPosition(src, data.origin);
				break;
			// impact entity
			case TE_GUNSHOT:
			case TE_BLOOD:
			case TE_BLASTER:
			case TE_SHOTGUN:
			case TE_SPARKS:
			case TE_SCREEN_SPARKS:
			case TE_SHIELD_SPARKS:
			case TE_BULLET_SPARKS:
			// case TE_GREENBLOOD_new:
			// case TE_GREENBLOOD_old:
			case TE_BLASTER2:
			case TE_MOREBLOOD:
			case TE_HEATBEAM_SPARKS:
			case TE_HEATBEAM_STEAM:
			case TE_ELECTRIC_SPARKS:
			case TE_FLECHETTE:
impact_entity:
				ReadPosition(src, data.origin);
				ReadDir(src, data.movedir);
				break;
			// line entity
			case TE_RAILTRAIL:
			case TE_BUBBLETRAIL:
			case TE_BFG_LASER:
			// case TE_PLASMATRAIL:
			// case TE_BLUEHYPERBLASTER:
			case TE_DEBUGTRAIL:
			case TE_BUBBLETRAIL2:
line_entity:
				ReadPosition(src, data.origin);
				ReadPosition(src, data.endpos);
				break;
			// special entity
			case TE_SPLASH:
			case TE_LASER_SPARKS:
			case TE_WELDING_SPARKS:
			case TE_TUNNEL_SPARKS:
				data.count = ReadByte(src);
				ReadPosition(src, data.origin);
				ReadDir(src, data.movedir);
				data.style = ReadByte(src);
				break;
			case TE_PARASITE_ATTACK:
			case TE_MEDIC_CABLE_ATTACK:
			case TE_HEATBEAM:
			case TE_MONSTER_HEATBEAM:
				data.entity = ReadShort(src);
				ReadPosition(src, data.origin);
				ReadPosition(src, data.endpos);
				break;
			case TE_GRAPPLE_CABLE:
				data.entity = ReadShort(src);
				ReadPosition(src, data.origin);
				ReadPosition(src, data.endpos);
				ReadPosition(src, data.pos1);
				break;
			case TE_FLAME:
				data.entity = ReadShort(src);
				data.count = ReadShort(src);
				ReadPosition(src, data.endpos);
				ReadPosition(src, data.origin);
				ReadPosition(src, data.pos1);
				ReadPosition(src, data.pos2);
				ReadPosition(src, data.pos3);
				ReadPosition(src, data.pos4);
				break;
			case TE_LIGHTNING:
				data.dest_entity = ReadShort(src);
				data.entity = ReadShort(src);
				ReadPosition(src, data.endpos);
				ReadPosition(src, data.origin);
				break;
			case TE_FLASHLIGHT:
				ReadPosition(src, data.origin);
				data.entity = ReadShort(src);
				break;
			case TE_FORCEWALL:
				ReadPosition(src, data.origin);
				ReadPosition(src, data.endpos);
				data.style = ReadShort(src);
				break;
			case TE_STEAM:
				data.nextid = ReadShort(src);
				data.count = ReadByte(src);
				ReadPosition(src, data.origin);
				ReadDir(src, data.movedir);
				data.style = ReadByte(src);
				data.plat2flags = ReadShort(src);
				if (data.nextid != -1)
					data.wait = ReadLong(src);
				break;
			default:
				Com_Printf("ParseMessage: unknown temp entity type: %d\n", data.type);
				src->p = src->end;
				break;
			}

			msg = CreateMessage(SVC_TEMP_ENTITY, &data, sizeof(temp_entity_t), false);
			memcpy(msg->data, &data, sizeof(temp_entity_t));
			VectorCopy(data.origin, msg->origin);
			return msg;
		}
		break;
	case SVC_LAYOUT:
		{
			char	string[MAX_MSGLEN];
			
			ReadString(src, string, sizeof(string));
			
			return CreateMessage(SVC_LAYOUT, string, strlen(string)+1, true);
		}
		break;
	case SVC_INVENTORY:
		{
			short	inventory[MAX_ITEMS];
			int		i;
			
			for (i = 0; i < MAX_ITEMS; i++)
				inventory[i] = ReadShort(src);
			
			return CreateMessage(SVC_INVENTORY, inventory, sizeof(inventory), true);
		}
		break;
	case SVC_NOP:
		{
			return CreateMessage(SVC_NOP, NULL, 0, false);
		}
		break;
	case SVC_DISCONNECT:
		{
			return CreateMessage(SVC_DISCONNECT, NULL, 0, true);
		}
		break;
	case SVC_RECONNECT:
		{
			return CreateMessage(SVC_RECONNECT, NULL, 0, true);
		}
		break;
	case SVC_SOUND:
		{
			int		mask, mix;
			sound_t	data;
			
			mask = ReadByte(src);
			data.soundnum = ReadByte(src);
			data.volume = (mask & SND_VOLUME) ? ((float)ReadByte(src) / 255) : DEFAULT_SOUND_PACKET_VOLUME;
			data.attenuation = (mask & SND_ATTENUATION) ? ((float)ReadByte(src) / 64) : DEFAULT_SOUND_PACKET_ATTENUATION;
			data.timeofs = (mask & SND_OFFSET) ? ((float)ReadByte(src) * 0.001F) : 0;
			if (mask & SND_ENT)
			{
				mix = ReadShort(src);
				data.entity = (mix >> 3) & 0x3FF;
				data.channel = mix & 0x07;
			}
			else
			{
				data.entity = 0;
				data.channel = 0;
			}
			if (mask & SND_POS)
			{
				data.positioned = true;
				ReadPosition(src, data.origin);
			}
			else
				data.positioned = false;
			
			msg = CreateMessage(SVC_SOUND, &data, sizeof(sound_t), false);

			if (data.positioned)	// multicast from sound's origin
				VectorCopy(data.origin, msg->origin);
			else if (data.entity)	// multicast from entity's origin
				VectorCopy(dm2->states[dm2->current_frame & UPDATE_MASK].entities[data.entity].origin, msg->origin);
			else if (dm2->svd.isdemo == RECORD_CLIENT || dm2->svd.isdemo == RECORD_NETWORK)
			{	// "local sound" (?), multicast from player's origin
				for (i = 0; i < 3; i++)
					msg->origin[i] = dm2->players[0].ps[dm2->current_frame & UPDATE_MASK].pmove.origin[i];
			}

			return msg;
		}
		break;
	case SVC_PRINT:
		{
			print_t	data;
			
			data.level = ReadByte(src);
			ReadString(src, data.string, sizeof(data.string));
			
			return CreateMessage(SVC_PRINT, &data, sizeof(print_t), false);
		}
		break;
	case SVC_STUFFTEXT:
		{
			char	string[MAX_MSGLEN];
			
			ReadString(src, string, sizeof(string));
			
			return CreateMessage(SVC_STUFFTEXT, string, strlen(string)+1, true);
		}
		break;
	case SVC_SERVERDATA:
		{
			serverdata_t	data;
			
			data.version = ReadLong(src);
			data.key = ReadLong(src);
			data.isdemo = ReadByte(src);
			ReadString(src, data.game, sizeof(data.game));
			data.player = ReadShort(src);
			ReadString(src, data.mapname, sizeof(data.mapname));
			
			return CreateMessage(SVC_SERVERDATA, &data, sizeof(serverdata_t), true);
		}
		break;
	case SVC_CONFIGSTRING:
		{
			configstring_t	data;
			
			data.index = ReadShort(src);
			ReadString(src, data.string, sizeof(data.string));
			
			assert(data.index >= 0 && data.index < MAX_CONFIGSTRINGS);
			
			return CreateMessage(SVC_CONFIGSTRING, &data, sizeof(configstring_t), true);
		}
		break;
	case SVC_SPAWNBASELINE:
		{
			entity_state_t	es;
			int				mask;
			
			memset(&es, 0, sizeof(entity_state_t));
			// baselines must be memset to 0 beforehand (but keep es->numbers intact!)
			mask = ReadByte(src);
			if (mask & U_MOREBITS1)	mask |= (ReadByte(src) <<  8);
			if (mask & U_MOREBITS2) mask |= (ReadByte(src) << 16);
			if (mask & U_MOREBITS3) mask |= (ReadByte(src) << 24);
			
			if (mask & U_NUMBER16)
				es.number = ReadShort(src);
			else
				es.number = ReadByte(src);
			
			assert(es.number >= 0 && es.number < MAX_EDICTS);
			
			if (mask & U_MODEL)				es.modelindex = ReadByte(src);
			if (mask & U_MODEL2)			es.modelindex2 = ReadByte(src);
			if (mask & U_MODEL3)			es.modelindex3 = ReadByte(src);
			if (mask & U_MODEL4)			es.modelindex4 = ReadByte(src);
			if (mask & U_FRAME8)			es.frame = ReadByte(src);
			if (mask & U_FRAME16)			es.frame = ReadShort(src);
			if (mask & U_SKIN8)
			{
				if (mask & U_SKIN16)		es.skinnum = ReadLong(src);
				else						es.skinnum = ReadByte(src);
			}
			else if (mask & U_SKIN16)		es.skinnum = ReadShort(src);
			if (mask & U_EFFECTS8)
			{
				if (mask & U_EFFECTS16)		es.effects = ReadLong(src);
				else						es.effects = ReadByte(src);
			}
			else if (mask & U_EFFECTS16)	es.effects = ReadShort(src);
			if (mask & U_RENDERFX8)
			{
				if (mask & U_RENDERFX16)	es.renderfx = ReadLong(src);
				else						es.renderfx = ReadByte(src);
			}
			else if (mask & U_RENDERFX16)	es.renderfx = ReadShort(src);
			if (mask & U_ORIGIN1)			es.origin[0] = ReadCoord(src);
			if (mask & U_ORIGIN2)			es.origin[1] = ReadCoord(src);
			if (mask & U_ORIGIN3)			es.origin[2] = ReadCoord(src);
			if (mask & U_ANGLE1)			es.angles[0] = ReadAngle(src);
			if (mask & U_ANGLE2)			es.angles[1] = ReadAngle(src);
			if (mask & U_ANGLE3)			es.angles[2] = ReadAngle(src);
			if (mask & U_OLDORIGIN)			ReadPosition(src, es.old_origin);
			if (mask & U_SOUND)				es.sound = ReadByte(src);
			if (mask & U_EVENT)				es.event = ReadByte(src);
			if (mask & U_SOLID)				es.solid = ReadShort(src);
			
			return CreateMessage(SVC_SPAWNBASELINE, &es, sizeof(entity_state_t), true);
		}
		break;
	case SVC_CENTERPRINT:
		{
			char	string[MAX_MSGLEN];
			
			ReadString(src, string, sizeof(string));
			
			return CreateMessage(SVC_CENTERPRINT, string, strlen(string)+1, false);
		}
		break;
	case SVC_DOWNLOAD:
		{
			size_t	size, percent, i;
			
			size = ReadShort(src);
			percent = ReadByte(src);
			
			for (i = 0; i < size; i++)
				ReadByte(src);
			
			Com_Printf("Warning: SVC_DOWNLOAD message encountered\n");
			return NULL;
		}
		break;
	case SVC_PLAYERINFO:
		{
			player_state_t	data;
			int				index;

			if (dm2->svd.isdemo == RECORD_RELAY)
				index = ReadByte(src);
			else
				index = 0;

			data = dm2->players[index].ps[dm2->current_frame & UPDATE_MASK];

			ReadPS(src, &data);
			return CreateMessage(SVC_PLAYERINFO, &data, sizeof(player_state_t), false);
		}
		break;
	case SVC_PACKETENTITIES:
		{
			state_t	data;

			data = dm2->states[dm2->current_frame & UPDATE_MASK];

			ReadEntities(src, &data, &dm2->baselines);
			return CreateMessage(SVC_PACKETENTITIES, &data, sizeof(state_t), false);
		}
		break;
	case SVC_FRAME:
		{
			state_t	*from;
			int		count, i;
			frame_t	data;

			if (dm2->svd.isdemo != RECORD_SERVER)
			{
				data.seq1 = ReadLong(src);
				data.seq2 = ReadLong(src);

				if (data.seq2 == -1)
					from = &dm2->baselines;
				else
					from = &dm2->states[data.seq2 & UPDATE_MASK];
				
				assert(data.seq2 == from->frame);
				
				memcpy(data.areas, from->areas, sizeof(data.areas));

				ReadByte(src);		// ???
				count = ReadByte(src);
				for (i = 0; i < count; i++)
					data.areas[i] = ReadByte(src);

				return CreateMessage(SVC_FRAME, &data, sizeof(frame_t), false);
			}
		}
		break;
	default:
		Com_Printf("ParseMessage: Unknown server command id: %d\n", id);
		src->p = src->end;
		return NULL;
	}
	
	return NULL;
}

int	ReadBlock(block_t *block, FILE *fd, int isdemo)
{
	if (fread(&block->size, 1, 4, fd) != 4)
	{
		block->size = -1;
		if (isdemo != RECORD_SERVER)
			return -1;
	}

	block->size = LittleLong(block->size);
	if (block->size == -1)
		return 0;

	if ((signed)fread(block->buffer, 1, block->size, fd) != block->size)
	{
		block->size = -1;
		return -1;
	}
	block->p = block->buffer;
	block->end = block->p + block->size;
	return 0;
}

int ReadPreFrame(dm2_t *dm2, FILE *fd)
{
	block_t		block;
	char		buffer[MAX_SVSLEN];
	msg_t		*msg;
	qboolean	keepgoing;

	InitBlock(&block, buffer, sizeof(buffer));
	rewind(fd);

	keepgoing = true;
	while(keepgoing)
	{
		if (ReadBlock(&block, fd, dm2->svd.isdemo) || block.size == -1)
			return -1;

		while (block.p < block.end)
		{
			msg = ParseMessage(dm2, &block);
			if (msg)
			{
				switch(msg->id)
				{
				case SVC_BAD:
					{
						Sys_Error("ReadPreFrame: SVC_BAD message encountered\n");
						return 1;
					}
				case SVC_STUFFTEXT:
					{
						char	temp[MAX_MSGLEN], *cur, *next;

						// look for "precache" command, which signals end of preframe information
						strcpy(temp, msg->data);
						cur = temp;
						while (cur)
						{
							next = SplitCommandString(cur, ';');
							if (!strcmp(Argv(cur, 0), "precache"))
							{
								keepgoing = false;
								cur = NULL;
							}
							else
								cur = next;
						}
					}
					break;
				case SVC_SERVERDATA:
					{
						dm2->svd = *(serverdata_t *)msg->data;
					}
					break;
				case SVC_CONFIGSTRING:
					{
						configstring_t	*data = msg->data;

						strcpy(dm2->configstrings[data->index], data->string);
					}
					break;
				case SVC_SPAWNBASELINE:
					{
						entity_state_t	*es = msg->data;

						dm2->baselines.entities[es->number] = *es;
					}
					break;
				// rest of message ids are ignored
				default:
					break;
				}
				FreeMessage(msg);
			}
		}
	}

	return 0;
}

void WritePreFrame(serverdata_t *serverdata, relayinfo_t *relayinfo, char configstrings[MAX_CONFIGSTRINGS][64], state_t *baselines, FILE *fd)
{
	block_t			out;
	char			buffer[MAX_SVSLEN];
	int				i, len;
	int				mask;
	entity_state_t	*es;

	InitBlock(&out, buffer, sizeof(buffer));

	WriteByte(&out, SVC_SERVERDATA);
	WriteLong(&out, serverdata->version);
	WriteLong(&out, 0);
	WriteByte(&out, serverdata->isdemo);
	WriteString(&out, serverdata->game);
	WriteShort(&out, serverdata->player);
	WriteString(&out, serverdata->mapname);

/*	if (serverdata->isdemo == RECORD_RELAY)
	{
		WriteByte(&out, SVC_RELAYINFO);
		WriteShort(&out, relayinfo->maxclients);
	}
*/
	for (i = 0; i < MAX_CONFIGSTRINGS; i++)
	{
		if (!configstrings[i][0])
			continue;
		if (i != 0 && strlen(configstrings[i-1]) >= 64)
			continue;

		WriteByte(&out, SVC_CONFIGSTRING);
		WriteShort(&out, i);
		WriteString(&out, configstrings[i]);

		if (out.size > 1024)
		{
			len = LittleLong(out.size);
			fwrite(&len, 4, 1, fd);
			fwrite(out.buffer, out.size, 1, fd);
			ResetBlock(&out);
		}
	}

	for (i = 0; i < MAX_EDICTS; i++)
	{
		es = &baselines->entities[i];

		mask = 0;
		if (es->modelindex)				mask |= U_MODEL;
		if (es->modelindex2)			mask |= U_MODEL2;
		if (es->modelindex3)			mask |= U_MODEL3;
		if (es->modelindex4)			mask |= U_MODEL4;
		if (es->origin[0])				mask |=	U_ORIGIN1;
		if (es->origin[1])				mask |= U_ORIGIN2;
		if (es->origin[2])				mask |= U_ORIGIN3;
		if (es->angles[0])				mask |= U_ANGLE1;
		if (es->angles[1])				mask |= U_ANGLE2;
		if (es->angles[2])				mask |= U_ANGLE3;
		if (es->frame > 0xFF)			mask |= U_FRAME16;
		else if (es->frame)				mask |= U_FRAME8;
		if (es->skinnum > 0xFFFF)		mask |= U_SKIN16|U_SKIN8;
		else if (es->skinnum > 0xFF)	mask |= U_SKIN16;
		else if (es->skinnum)			mask |= U_SKIN8;
		if (es->effects > 0xFFFF)		mask |= U_EFFECTS16|U_EFFECTS8;
		else if (es->effects > 0xFF)	mask |= U_EFFECTS16;
		else if (es->effects)			mask |= U_EFFECTS8;
		if (es->renderfx > 0xFFFF)		mask |= U_RENDERFX16|U_RENDERFX8;
		else if (es->renderfx > 0xFF)	mask |= U_RENDERFX16;
		else if (es->renderfx)			mask |= U_RENDERFX8;
		if (es->old_origin[0] || es->old_origin[1] || es->old_origin[2])
			mask |= U_OLDORIGIN;
		if (es->sound)					mask |= U_SOUND;
		if (es->event)					mask |= U_EVENT;
		if (es->solid)					mask |= U_SOLID;
		
		if		(mask & 0xFF000000)		mask |= U_MOREBITS1|U_MOREBITS2|U_MOREBITS3;
		else if	(mask & 0x00FF0000)		mask |= U_MOREBITS1|U_MOREBITS2;
		else if	(mask & 0x0000FF00)		mask |= U_MOREBITS1;
		
		if (!mask)
			continue;
		if (es->number > 0xFF)			mask |= U_NUMBER16;

		WriteByte(&out, SVC_SPAWNBASELINE);
		WriteByte(&out, mask & 0xFF);
		if (mask & U_MOREBITS1)			WriteByte(&out, (mask >>  8) & 0xFF);
		if (mask & U_MOREBITS2)			WriteByte(&out, (mask >> 16) & 0xFF);
		if (mask & U_MOREBITS3)			WriteByte(&out, (mask >> 24) & 0xFF);
		
		if (mask & U_NUMBER16)			WriteShort(&out, es->number);
		else							WriteByte(&out, es->number);
		if (mask & U_MODEL)				WriteByte(&out, es->modelindex);
		if (mask & U_MODEL2)			WriteByte(&out, es->modelindex2);
		if (mask & U_MODEL3)			WriteByte(&out, es->modelindex3);
		if (mask & U_MODEL4)			WriteByte(&out, es->modelindex4);
		if (mask & U_FRAME8)			WriteByte(&out, es->frame);
		if (mask & U_FRAME16)			WriteShort(&out, es->frame);
		
		if (mask & U_SKIN8)
		{
			if (mask & U_SKIN16)		WriteLong(&out, es->skinnum);
			else						WriteByte(&out, es->skinnum);
		}
		else if (mask & U_SKIN16)		WriteShort(&out, es->skinnum);
		
		if (mask & U_EFFECTS8)
		{
			if (mask & U_EFFECTS16)		WriteLong(&out, es->effects);
			else						WriteByte(&out, es->effects);
		}
		else if (mask & U_EFFECTS16)	WriteShort(&out, es->effects);
		
		if (mask & U_RENDERFX8)
		{
			if (mask & U_RENDERFX16)		WriteLong(&out, es->renderfx);
			else						WriteByte(&out, es->renderfx);
		}
		else if (mask & U_RENDERFX16)	WriteShort(&out, es->renderfx);
		
		if (mask & U_ORIGIN1)			WriteCoord(&out, es->origin[0]);
		if (mask & U_ORIGIN2)			WriteCoord(&out, es->origin[1]);
		if (mask & U_ORIGIN3)			WriteCoord(&out, es->origin[2]);
		if (mask & U_ANGLE1)			WriteAngle(&out, es->angles[0]);
		if (mask & U_ANGLE2)			WriteAngle(&out, es->angles[1]);
		if (mask & U_ANGLE3)			WriteAngle(&out, es->angles[2]);
		if (mask & U_OLDORIGIN)			WritePosition(&out, es->old_origin);
		if (mask & U_SOUND)				WriteByte(&out, es->sound);
		if (mask & U_EVENT)				WriteByte(&out, es->event);
		if (mask & U_SOLID)				WriteShort(&out, es->solid);

		if (out.size > 1024)
		{
			len = LittleLong(out.size);
			fwrite(&len, 4, 1, fd);
			fwrite(out.buffer, out.size, 1, fd);
			ResetBlock(&out);
		}
	}

	WriteByte(&out, SVC_STUFFTEXT);
	WriteString(&out, "precache\n");

	len = LittleLong(out.size);
	fwrite(&len, 4, 1, fd);
	fwrite(out.buffer, out.size, 1, fd);
}
