/*
Copyright (C) 1996-1997 Id Software, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/


#include "quakedef.h"

#define MAX_PARTICLES                      8192 // default max # of particles at one time
#define ABSOLUTE_MIN_PARTICLES	512		// no fewer than this no matter what's on the command line
#define NUMVERTEXNORMALS	      162

extern	float	r_avertexnormals[NUMVERTEXNORMALS][3];

int ramp1[8] = {0x6f, 0x6d, 0x6b, 0x69, 0x67, 0x65, 0x63, 0x61};
int ramp2[8] = {0x6f, 0x6e, 0x6d, 0x6c, 0x6b, 0x6a, 0x68, 0x66};
int ramp3[8] = {0x6d, 0x6b, 6, 5, 4, 3};
int r_numparticles, smoketexture;

vec3_t r_pright, 
       r_pup, 
       r_ppn,
       avelocity = {23, 7, 3},
       avelocities[NUMVERTEXNORMALS];

particle_t *active_particles, 
           *free_particles, 
           *particles;

float	beamlength = 16,
      partstep = 0.01,
      timescale = 0.01;

#define PAR_COLOR_RED      -2


/*
===============
R_InitParticles
===============
*/
void R_InitParticles (void)
{
	int		i;
	i = COM_CheckParm ("-particles");
	if (i)
	   {
		r_numparticles = (int)(Q_atoi(com_argv[i+1]));
		if (r_numparticles < ABSOLUTE_MIN_PARTICLES)
			r_numparticles = ABSOLUTE_MIN_PARTICLES;
	   }
	else
	   {
		r_numparticles = MAX_PARTICLES;
	   }

   particles = (particle_t *)
			      Hunk_AllocName (r_numparticles * sizeof(particle_t), "particles");
}

/*
===============
R_EntityParticles
===============
*/

void R_EntityParticles (entity_t *ent)
{
	int			count;
	int			i;
	particle_t	*p;
	float		angle;
	float		sr, sp, sy, cr, cp, cy;
	vec3_t		forward;
	float		dist;
	
	dist = 64;
	count = 50;

   if (!avelocities[0][0])
      {
      for (i = 0; i < NUMVERTEXNORMALS * 3; i++)
         avelocities[0][i] = (rand()&255) * 0.01;
      }


	for (i = 0; i < NUMVERTEXNORMALS; i++)
	   {
		angle = cl.time * avelocities[i][0];
		sy = sin(angle);
		cy = cos(angle);
		angle = cl.time * avelocities[i][1];
		sp = sin(angle);
		cp = cos(angle);
		angle = cl.time * avelocities[i][2];
		sr = sin(angle);
		cr = cos(angle);
	
		forward[0] = cp*cy;
		forward[1] = cp*sy;
		forward[2] = -sp;

		if (!free_particles)
			return;
		p = free_particles;
		free_particles = p->next;
		p->next = active_particles;
      p->style = 0;
		active_particles = p;

		p->die = cl.time + 0.01;
		p->color = 0x6f;
		p->type = pt_explode;
		
		p->org[0] = ent->origin[0] + r_avertexnormals[i][0]*dist + forward[0]*beamlength;			
		p->org[1] = ent->origin[1] + r_avertexnormals[i][1]*dist + forward[1]*beamlength;			
		p->org[2] = ent->origin[2] + r_avertexnormals[i][2]*dist + forward[2]*beamlength;			
	   }
}


/*
===============
R_ClearParticles
===============
*/
void R_ClearParticles (void)
{
	int		i;
	
	free_particles = &particles[0];
	active_particles = NULL;

	for (i=0 ;i<r_numparticles ; i++)
		particles[i].next = &particles[i+1];
	particles[r_numparticles-1].next = NULL;
}


void R_ReadPointFile_f (void)
{
	FILE	*f;
	vec3_t	org;
	int		r;
	int		c;
	particle_t	*p;
	char	name[MAX_OSPATH];
	
//FIXME        sprintf (name,"maps/%s.pts", sv.name);

	COM_FOpenFile (name, &f);
	if (!f)
	   {
		Con_Printf ("couldn't open %s\n", name);
		return;
	   }
	
	Con_Printf ("Reading %s...\n", name);
	c = 0;
	for ( ;; )
	   {
		r = fscanf (f,"%f %f %f\n", &org[0], &org[1], &org[2]);
		if (r != 3)
			break;
		c++;
		
		if (!free_particles)
		   {
			Con_Printf ("Not enough free particles\n");
			break;
		   }
		p = free_particles;
		free_particles = p->next;
		p->next = active_particles;
      p->style = 0;
		active_particles = p;
		
		p->die = 99999;
		p->color = (-c)&15;
		p->type = pt_static;
		VectorCopy (vec3_origin, p->vel);
		VectorCopy (org, p->org);
	   }

	fclose (f);
	Con_Printf ("%i points read\n", c);
}

/*
===============
R_ParseParticleEffect

Parse an effect out of the server message
===============
*/
void R_ParseParticleEffect (void)
{
	vec3_t		org, dir;
	int			i, count, msgcount, color;
	
	for (i=0 ; i<3 ; i++)
		org[i] = MSG_ReadCoord ();
	for (i=0 ; i<3 ; i++)
		dir[i] = MSG_ReadChar () * (1.0/16);
	msgcount = MSG_ReadByte ();
	color = MSG_ReadByte ();

   if (msgcount == 255)
	   count = 1024;
   else
	   count = msgcount;
	
	R_RunParticleEffect (org, dir, color, count);
}
	
/*
===============
R_ParticleExplosion

===============
*/
void R_ParticleExplosion (vec3_t org)
{
   int			i, j;
	particle_t	*p;
	for (i = 0; i < 3 + (rand() & 3); i++)
	   {
		if (!free_particles)
			return;
		p = free_particles;
		free_particles = p->next;
		p->next = active_particles;
		active_particles = p;

		p->die = cl.time + 1.2;
      p->style = PAR_FIRE;
      p->roll = rand() & 256;
		//p->color = ramp1[0];
		//p->ramp = rand()&3;

   	p->type = pt_explode;
		p->org[0] = org[0] + ((rand()%32)-16);
		p->vel[0] = rand() & 63;
		p->org[1] = org[1] + ((rand()%32)-16);
		p->vel[1] = rand() & 63;
		p->org[2] = org[2] + ((rand()%32)-16);
		p->vel[2] = rand() & 63;	
	   }
}

void R_Smoke (vec3_t org, int color)
{
   int			i, j;
	particle_t	*p;
	for (i = 0; i < 3 + (rand() & 3); i++)
	   {
		if (!free_particles)
			return;
		p = free_particles;
		free_particles = p->next;
		p->next = active_particles;
		active_particles = p;

		p->die = cl.time + 1.2;
      p->style = PAR_SMOKE;
      p->roll = rand() & 256;
      p->type = pt_fire;
      p->color = color;
      p->org[0] = org[0] + ((rand()%32)-16);
      p->vel[0] = rand() & 63;
      p->org[1] = org[1] + ((rand()%32)-16);
      p->vel[1] = rand() & 63;
      p->org[2] = org[2] + ((rand()%32)-16);
      p->vel[2] = rand() & 63; 
     }
}

/*
===============
R_ParticleExplosion2

===============
*/
void R_ParticleExplosion2 (vec3_t org, int colorStart, int colorLength)
{
   R_ParticleExplosion (org);
   return;
}

/*
===============
R_BlobExplosion

===============
*/
void R_BlobExplosion (vec3_t org)
{
	int			i, j;
	particle_t	*p;
	
	for (i = 0; i < 128; i++)
	   {
		if (!free_particles)
			return;
		p = free_particles;
      p->style = 0;
		free_particles = p->next;
		p->next = active_particles;
		active_particles = p;

		p->die = cl.time + 1 + (rand()&8)*0.05;

		if (i & 1)
		   {
			p->type = pt_blob;
			p->color = 66 + rand()%6;
			for (j=0 ; j<3 ; j++)
			   {
				p->org[j] = org[j] + ((rand()%32)-16);
				p->vel[j] = (rand()%512)-256;
			   }
		   }
		else
		   {
			p->type = pt_blob2;
			p->color = 150 + rand()%6;
			for (j=0 ; j<3 ; j++)
			   {
				p->org[j] = org[j] + ((rand()%32)-16);
				p->vel[j] = (rand()%512)-256;
			   }
		   }
	   }
}

/*
===============
R_RunParticleEffect

===============
*/
void R_RunParticleEffect (vec3_t org, vec3_t dir, int color, int count)
{
	int			i, j;
	particle_t	*p;

   if (count > 32)
      count = 30;
	for (i = 0; i < count ; i++)
	   {
		if (!free_particles)
			return;
		p = free_particles;
		free_particles = p->next;
		p->next = active_particles;
		active_particles = p;
		p->die = cl.time + 0.1*(rand()%5);
		p->color = (color & ~7) + (rand() & 7);
      p->style = 0;
		p->type = pt_slowgrav;
   	p->org[0] = org[0] + ((rand()&15)-8);
		p->org[1] = org[1] + ((rand()&15)-8);
		p->org[2] = org[2] + ((rand()&15)-8);
		p->vel[0] = dir[0] * 15;
		p->vel[1] = dir[1] * 15;
		p->vel[2] = dir[2] * 15;
	   }
}


// LordHavoc: added this for spawning sparks/dust (which have strong gravity)
/*
===============
R_SparkShower

===============
*/
void R_SparkShower (vec3_t org, vec3_t dir, int count)
{
	int			i, j;
	particle_t	*p;
	
	for (i=0 ; i<count ; i++)
	{
		if (!free_particles)
			return;
		p = free_particles;
		free_particles = p->next;
		p->next = active_particles;
		active_particles = p;

		p->die = cl.time + 0.015 * (rand()%32);
		p->ramp = (rand()&3);
		p->color = ramp1[(int)p->ramp];
		p->type = pt_dust;
		for (j=0 ; j<3 ; j++)
		{
			p->org[j] = org[j] + ((rand()%4)-2);
			p->vel[j] = (rand()%256)-128;
		}
	}
}


/*
===============
R_LavaSplash

===============
*/
void R_LavaSplash (vec3_t org)
{
	int			i, j, k;
	particle_t	*p;
	float		vel;
	vec3_t		dir;

	for (i=-16 ; i<16 ; i++)
		for (j=-16 ; j<16 ; j++)
			for (k=0 ; k<1 ; k++)
			   {
				if (!free_particles)
					return;
				p = free_particles;
p->style = 0; // Ender

				free_particles = p->next;
				p->next = active_particles;
				active_particles = p;
		
				p->die = cl.time + 2 + (rand()&31) * 0.02;
				p->color = 224 + (rand()&7);
				p->type = pt_slowgrav;
				
				dir[0] = j*8 + (rand()&7);
				dir[1] = i*8 + (rand()&7);
				dir[2] = 256;
	
				p->org[0] = org[0] + dir[0];
				p->org[1] = org[1] + dir[1];
				p->org[2] = org[2] + (rand()&63);
	
				VectorNormalize (dir);						
				vel = 50 + (rand()&63);
				VectorScale (dir, vel, p->vel);
			   }
}

/*
===============
R_TeleportSplash

===============
*/
void R_TeleportSplash (vec3_t org)
{
	int			i, j, k;
	particle_t	*p;
	float		vel;
	vec3_t		dir;

//        for (i=-4 ; i<4 ; i+=4) //-16;16
        for (i=-16 ; i<16 ; i+=4) //-16;16
                for (j=-16 ; j<16 ; j+=4) //-16;16
                        for (k=-16 ; k<32 ; k+=4) //-24; 32
			   {
				if (!free_particles)
					return;
				p = free_particles;
            p->style = 0;
				free_particles = p->next;
				p->next = active_particles;
				active_particles = p;
		
				p->die = cl.time + 0.2 + (rand()&7) * 0.02;
				p->color = 7 + (rand()&7);
				p->type = pt_slowgrav;
				
				dir[0] = j*8;
				dir[1] = i*8;
				dir[2] = k*8;
	
				p->org[0] = org[0] + i + (rand()&3);
				p->org[1] = org[1] + j + (rand()&3);
				p->org[2] = org[2] + k + (rand()&3);
	
				VectorNormalize (dir);						
				vel = 50 + (rand()&63);
				VectorScale (dir, vel, p->vel);
			   }
}

void R_RocketTrail (vec3_t start, vec3_t end, int type)
{
	vec3_t		vec;
	float		   len;
	int			j;
	particle_t	*p;
	int			dec;
	static int	tracercount;
   static int   smoke_roll = 0, smoke_smoke_roll = 0;

	VectorSubtract (end, start, vec);
	len = VectorNormalize (vec);
	if (type < 128)
		dec = 3;
	else
	   {
		dec = 1;
		type -= 128;
	   }

	while (len > 0)
	   {
		len -= dec;

		if (!free_particles)
			return;
		p = free_particles;
		free_particles = p->next;
		p->next = active_particles;
		active_particles = p;
		
		VectorCopy (vec3_origin, p->vel);
		p->die = cl.time + 1;//2;
                p->style = 0;
                switch (type) {        
                 case 0: // PAR_SMOKE:
                  p->style = PAR_SMOKE;
                  p->type = pt_fire;
                  p->die -= .7;
                  p->org[0] = end[0] - 3.5 + (rand() & 7);
                  p->org[1] = end[1] - 3.5 + (rand() & 7);
                  p->org[2] = end[2] - 3.5 + (rand() & 7);
                  p->roll = smoke_roll;
                  smoke_roll += 30;
                  if (smoke_roll > 360)
                   smoke_roll -= 360;   
                  VectorAdd (start, vec, start);
                  return;

                 case 1: // PAR_SMOKE_SMOKE:
                  p->style = PAR_SMOKE_SMOKE;
                  p->type = pt_fire;
                  p->org[0] = end[0] - 3.5 + (rand() & 7);
                  p->org[1] = end[1] - 3.5 + (rand() & 7);
                  p->org[2] = end[2] - 3.5 + (rand() & 7);
                  p->roll = smoke_smoke_roll;
                  smoke_smoke_roll += 30;
                  if (smoke_smoke_roll > 360)
                   smoke_smoke_roll -= 360;   
                  VectorAdd (start, vec, start);
                  return;
         
                 case 2: // PAR_BLOOD:
                  p->style = PAR_BLOOD;
                  p->type = pt_grav;
                  p->org[0] = start[0] + ((rand()%6)-3);
                  p->org[1] = start[1] + ((rand()%6)-3);
                  p->org[2] = start[2] + ((rand()%6)-3);
                  VectorAdd (start, vec, start);
                  return;
            
                 case 3:  // EF_GIB
                 case 5: // EF_TRACER
                  p->type = pt_static;
                  p->die = cl.time + 0.5;
                  if (type == 3)
                   p->color = 52 + ((tracercount&4)<<1);
                  else
                   p->color = 230 + ((tracercount&4)<<1);
                   tracercount++;
                   VectorCopy (start, p->org);
                   if (tracercount & 1) {
                    p->vel[0] = 30 * vec[1];
                    p->vel[1] = 30 * -vec[0];
                   } else {
                    p->vel[0] = 30 * -vec[1];
                    p->vel[1] = 30 * vec[0];
                   }
                   break;

                 case 4: // slight blood
                  p->type = pt_grav;
                  p->color = 67 + (rand()&3);
                  p->color = PAR_COLOR_RED;
                  p->org[0] = start[0] + ((rand()%6)-3);
                  p->org[1] = start[1] + ((rand()%6)-3);
                  p->org[2] = start[2] + ((rand()%6)-3);
                  len -= 3;
                  break;

                case 6: // voor trail
                 p->color = 9*16 + 8 + (rand()&3);
                 p->type = pt_static;
                 p->die = cl.time + 0.3;
                 p->org[0] = start[0] + ((rand()&15)-8);
                 p->org[1] = start[1] + ((rand()&15)-8);
                 p->org[2] = start[2] + ((rand()&15)-8);
                 break;
                }
		
		VectorAdd (start, vec, start);
	   }
}


/*
===============
R_DrawParticles
===============
*/
// extern  cvar_t  sv_gravity;
void tQER_AngleVectors (vec3_t angles, vec3_t forward, vec3_t right, vec3_t up);
void R_DrawParticles (void)
{
	particle_t		*p, *kill;
	int				i;
	float			   grav,
                  grav1,
                  grav2,
	               dvel,
                  time1,
                  scale,
                  scale2,
                  frametime,
                  timediff,
                  last_roll = -9999; 
	vec3_t		   up, 
                  right,
                  porigin,
                  up_right;
   unsigned int   color;
   int            last_texture = -1,
                  new_texture = 0,
                  blend_value = 0xFF;
   
  frametime = host_frametime;

//        frametime = cl.time - cl.oldtime;

   glEnable (GL_BLEND);
	glDepthMask (0);
	glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
   //glPolygonMode (GL_FRONT_AND_BACK, GL_LINE);
   grav1 = frametime * movevars.gravity; // Ender: FIXME
   grav2 = grav1 * 0.025f;
   dvel = .02f * frametime; //4
   time1 = frametime * 5;
   grav = grav2 + grav2;
	for ( ;; ) 
	   {
		kill = active_particles;
		if (kill && kill->die < cl.time)
		   {
			active_particles = kill->next;
         kill->style = 0;
         kill->roll = 0;
			kill->next = free_particles;
			free_particles = kill;
			continue;
		   }
		break;
	   }

	for (p = active_particles; p ;p = p->next)
	   {
		for ( ;; )
		   {
			kill = p->next;
			if (kill && kill->die < cl.time)
			   {
				p->next = kill->next;
            kill->style = 0;
            kill->roll = 0;
				kill->next = free_particles;
				free_particles = kill;
				continue;
			   }
			break;
		   }

      scale = (p->org[0] - r_origin[0]) * vpn[0] 
            + (p->org[1] - r_origin[1]) * vpn[1]
	         + (p->org[2] - r_origin[2]) * vpn[2];
		
      if (scale < 20)
			scale = 1;
		else
			scale = 1 + scale * 0.004;
         
      timediff = p->die - cl.time;
      switch (p->style)
         {
         case PAR_SMOKE:
            new_texture = smoketexture + 1;
            blend_value = (float) timediff * 550 + 31;
            scale *= (8 -  8 * timediff);
            if (p->color) {
             color = d_8to24table[(int)p->color] & 0x7fffffff;
            } else {
             color = 0x00C0C0C0 | (blend_value << 24) | ((rand() & 63) << 16) ;
            }
            break;
         case PAR_SMOKE_SMOKE:
            new_texture = smoketexture;
            blend_value = timediff * 196;
            scale *= (9 - 5 * timediff);
            color = 0x008f8f8f | (blend_value << 24);
            break;
         case PAR_FIRE:
            new_texture = smoketexture;
            blend_value = (float) timediff * 192;
            scale *= (16 * timediff);
            color = d_8to24table[(int)15] & 0x7fffffff;
            blend_value = (float) (1.2 - timediff) * 0xC0;
            color |= (blend_value << 16);
            break;
         case PAR_BLOOD:
            new_texture = particletexture;
            color = (rand() & 63) | 0x8f000000;
            break;
         default:
            new_texture = particletexture;
            color = d_8to24table[(int)p->color] & 0x7fffffff;
         }


      if (last_texture != new_texture)
         {
         GL_Bind(new_texture);
         last_texture = new_texture;
         }
		glColor4ubv ((byte *)&color);
      
      // tQER<1>: 
      // Do not updated the orthogonal vectors if the last roll is the same
      // as the current one. Skip computation.
      // If computation is necessary, use the lookup table version of AngleVectors
      // for faster and coarse results.
      if (p->roll != last_roll)
         {
         up_right[0] = r_refdef.viewangles[0];
         up_right[1] = r_refdef.viewangles[1];
         up_right[2] = r_refdef.viewangles[2] + p->roll;
         tQER_AngleVectors (up_right, vpn, vright, vup);
         VectorScale (vup, 2.5, up);
	      VectorScale (vright, 2.5, right);
         VectorAdd (up, right, up_right);
         last_roll = p->roll;
         }

      scale2 = scale / 2;
      porigin[0] = p->org[0] - up_right[0] * scale2;
      porigin[1] = p->org[1] - up_right[1] * scale2;
      porigin[2] = p->org[2] - up_right[2] * scale2;
      if (last_texture > particletexture)
         {
         glBegin (GL_TRIANGLE_STRIP);
	      glTexCoord2f (0,0);
	      glVertex3fv (porigin);
	      glTexCoord2f (1,0);
	      glVertex3f (porigin[0] + up[0]*scale, porigin[1] + up[1]*scale, porigin[2] + up[2]*scale);
		   glTexCoord2f (0,1);
		   glVertex3f (porigin[0] + right[0]*scale, porigin[1] + right[1]*scale, porigin[2] + right[2]*scale);
		   glTexCoord2f (1,1);
		   glVertex3f (porigin[0] + (up[0] + right[0])*scale, porigin[1] + (up[1] + right[1])*scale, porigin[2] + (up[2] + right[2])*scale);
	      glEnd ();
         }
      else
         {
         glBegin (GL_TRIANGLES);
		   glTexCoord2f (0,0);
		   glVertex3fv (p->org);
		   glTexCoord2f (1,0);
		   glVertex3f (p->org[0] + up[0]*scale, p->org[1] + up[1]*scale, p->org[2] + up[2]*scale);
		   glTexCoord2f (0,1);
		   glVertex3f (p->org[0] + right[0]*scale, p->org[1] + right[1]*scale, p->org[2] + right[2]*scale);
	      glEnd ();
         }

		// hack a scale up to keep particles from disapearing
      p->org[0] += p->vel[0] * frametime;
		p->org[1] += p->vel[1] * frametime;
		p->org[2] += p->vel[2] * frametime;
		
		switch (p->type)
		   {
		   case pt_static:
   			break;
		   case pt_fire:
			   p->vel[2] += grav2;
			   break;

		   case pt_explode:
		   case pt_explode2:
			   p->vel[2] -= grav;
			   break;

	      case pt_blob:
			   p->vel[0] += p->vel[0] * dvel;
			   p->vel[1] += p->vel[1] * dvel;
			   p->vel[2] += p->vel[2] * dvel;
			   p->vel[2] -= grav;
			   break;

		   case pt_blob2:
  				p->vel[0] -= p->vel[0] * dvel;
  				p->vel[1] -= p->vel[1] * dvel;
  				p->vel[2] -= p->vel[2] * dvel;
			   p->vel[2] -= grav;
			   break;

		   case pt_grav:
		   case pt_slowgrav:
			   p->vel[2] -= grav;
			   break;
// LordHavoc: added for gunshots and other things, used for spark showers
		case pt_dust:
                        p->ramp += time1;
			if (p->ramp >= 8)
				p->die = -1;
			else
				p->color = ramp3[(int)p->ramp];
			p->vel[2] -= grav1;
			break;
		   }
	   }
	glDisable (GL_BLEND);
	glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
	glDepthMask (1);
}

