/* This program is released under the GNU public lisence.
 * You can should be able to find a copy of this lisence in this
 * qwf release or at http://www.gnu.org/
 * 
 */

/* Program notes... When generating random numbers I'd presonally
 * recoomend that if you're not using /dev/random to get your random
 * numbers that you consider changing time() to some number off the top
 * of your head. Another semi good idea is processor id.
 */

#include <stdlib.h>
#include <stdio.h>

#define CIPHER_LEN 32

#ifdef LINUX
#include <unistd.h>

/* A file pointer using linux specific read/write function */
static int dev_random;

#else
#include <time.h>
#endif

/* The min and max array sizes */
#define minimum_array_size 1024
#define maximum_array_size 2048

typedef unsigned char byte;


/* The GeneratedData.c file Template */
static char *file1_str1 =
"#include \"GeneratedData.h\"\n"
"\n"
"void getKey%s(byte *key)\n"
"{\n"
"    static const int original_multiplier = %d;\n"
"    int multiplier = original_multiplier;\n"
"\n";

static char *file1_str2 = "    GetKey%d(&multiplier, key);\n";
static char *file1_str3 = "}\n\n";

/* The GetKey file template */
static char *file2_str1 = 
"#include \"GeneratedData.h\"\n"
"#define ARRAY_SIZE %d\n"
"static const int mask = ~ (1 << ((sizeof(mask) * 8) - 1));\n"
"static const int secret_multiplier = %d;\n"
"static const byte data_array[ARRAY_SIZE] = {";
static char *file2_str2 = "};\n\n"
"/* Inline would be much obliged, if possible. */\n"
"void GetKey%d(int *multiplier, byte *key)\n"
"{\n"
"       (*multiplier) *= secret_multiplier;\n"
"       (*multiplier) &= mask;\n"
"       key[%d] = data_array[(*multiplier) %% ARRAY_SIZE];\n"
"}\n\n";

/* In linux we we can use /dev/random to get our random bits and be
 * sure that it is in fact unpredictably random.
 * If you're interesed in random numbers I suggest volume 2 of Knuth's
 * the art of Computer Programming, on Semi Numerical algorithms (the stuff
 * on infinite precision math is also terrific).
 */
#ifdef LINUX
int getRandomInteger(void)
{
	int number;
	int mask;
	read(dev_random, &number, sizeof(number));
	mask = 1 << ((sizeof(mask) * 8) - 1);
	mask = ~mask;
	number &= mask; /* If it's negative make it positive */
	return number;
}

byte getRandomByte(void)
{
	byte b;
	read(dev_random, &b, sizeof(b));
	return b;
}

#else

int getRandomInteger(void)
{
	return rand();
}

byte getRandomByte(void)
{
	int i;
	byte b;
	i = rand();
	b = (byte) i;
	return b;
}

#endif

/* Decide on the of the randomly sized array we're going to hide our data in */
int decideOnArraySize(void)
{
	int size;
	size = getRandomInteger();
	size %= maximum_array_size;
	/* Note we can only add like this and not affect the 
	 * probablity of the size of the array since the difference
	 * between the two arrays equals minum_array_size
	 */
	if(size < minimum_array_size)
		size += minimum_array_size;
	return size;
}

/* randomise the order in which the function are called */
void makeArrayOrder(int *order)
{
	int i;
	int num;
	int place_taken[CIPHER_LEN] = {-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1} ;

	
	for(i = 0; i < CIPHER_LEN; i++) {
		num = (int) getRandomByte();
		num %= CIPHER_LEN;
		while(place_taken[num] != -1) {
			num++;
			num %= CIPHER_LEN;
		}
		place_taken[num] = 1;
		order[i] = num;
	}
}

/* Write Generated Data */
void writeGeneratedData(void)
{
	FILE *file;
	int i;
	int order[CIPHER_LEN];
	
	file = fopen("GeneratedData1.c", "w");
	if(file == NULL) {
		fprintf(stderr, "Couldn't open file GeneratedData1.c to write to\n");
		exit(1);
	}
	/* Write getKey() */
	makeArrayOrder(order);
	fprintf(file, file1_str1, "", getRandomInteger());
	for(i = 0; i < CIPHER_LEN; i++)
		fprintf(file, file1_str2, order[i]);
	fprintf(file, file1_str3);
	fclose(file);

	file = fopen("GeneratedData2.c", "w");
	if(file == NULL) {
		fprintf(stderr, "Couldn't open file GeneratedData2.c to write to\n");
		exit(1);
	}
	/* Write getKey_2() */
	makeArrayOrder(order);
	fprintf(file, file1_str1, "_2", getRandomInteger());
	for(i = 0; i < CIPHER_LEN; i++)
		fprintf(file, file1_str2, order[i]);
	fprintf(file, file1_str3);
	fclose(file);
}

void writeKeyFiles(void)
{
	int i,j;
	int array_size;
	FILE *file;
	char fileName[256];
	
	for(i = 0; i < CIPHER_LEN; i++) {
		/* Initialise the variables that will be used in writing this file */
		array_size = decideOnArraySize();
		sprintf(fileName, "GetKey%d.c", i);
		file = fopen(fileName, "w");

		/* Make sure our file name isn't bad */
		if(file == NULL) {
			fprintf(stderr, "Error could not open file %s to write to.\n", fileName);
			exit(1);
		}

		fprintf(file, file2_str1, array_size, getRandomInteger());

       		for (j = 0; j < array_size; j++) {
			fprintf(file, "0x%X%s", getRandomByte(), (j==array_size-1) ? " ":", ");
			if( j % 10 == 0) {
				fprintf(stdout, ".");
				fflush(stdout);
				fprintf(file, "\n");
			}
		}
		fprintf(file, file2_str2, i, i);
		fclose(file);
		fprintf(stdout, "+");
	}
	fprintf(stdout, "\n");
}

int main(int argc, char *argv[])
{
#ifdef LINUX
	dev_random = open("/dev/random", 4);
	if(dev_random == -1) {
		fprintf(stderr, "Error couldn't open /dev/random to generate our random numbers\n");
		exit(1);
	}
#else
	srand(time(NULL));
#endif
	writeGeneratedData();
	writeKeyFiles();
	return 0;
}
