library GLQuArK;

uses Windows, GL, GLu;

const
  kVersionGL = 101;

type
  Reel  = GLdouble;  { GLdouble = Double }
  PVect = ^TVect;
  TVect = record     { vector or vertex }
           X,Y,Z: Reel;
          end;
  PVertexArray = ^TVertexArray;
  TVertexArray = array[0..99] of PVect;  { array of pointers on vertices }
  PSide = ^TSide;
  TSide = record    { polygonal side definition - READ ONLY }
            NormalVector: TVect;
            DistToOrigin: Reel;
            TexParams: array[1..5] of Reel;  { offset X,Y, angle, scale X,Y }
            TexName: String[23];  { 24 bytes - Pascal-style string - don't use it }
            prvDescS: PVertexArray;   { vertices }
            prvNbS: Integer;    { vertices count }
           end;
  PTexHeader = ^TTexHeader;
  TTexHeader = record    { texture header }
                Name: array[0..15] of Byte;   { not useful }
                W,H: LongInt;                { width, height }
                Indexes: array[0..3] of LongInt;  { offset from header to texture images }
               end;

  PTextureGL = ^TTextureGL;   { internal structure }
  TTextureGL = record
                glW, glH: Integer;
                glScaleW, glScaleH: GLfloat;
                glMeanColor: array[0..2] of GLfloat;
                glBits: record end;
               end;
  PFaceGL  = ^TFaceGL;   { side header build by QuArK with read-write access }
  TFaceGL  = record
              glFace: PSide;      { side definition }
              glTex: PTextureGL;  { texture information }
              glSelection: Byte;  { one of the SEL_xxx constants below }
              glRedraw: ByteBool; { if True, redraw this side - reset it to False }
              { --- DLL-specific data (of any size) --- }
              glDrawStep: Byte;
             end;

  PRenderC = ^TRenderC;  { renderer state information for QuArK <-> DLL communication }
  TRenderC = record
               { --- state flags set by QuArK --- }
              ResetScene: ByteBool;   { the list of sides changed - if handled, the DLL must reset it to zero }
              ResetView: ByteBool;    { the camera position or angle changed - if handled, the DLL must reset it to zero }
              StopDrawing: ByteBool;  { the DLL must check this often and stop rendering if it comes to True - don't change it in the DLL }
              FullRedraw: ByteBool;   { redraw everything - if False, check "glRedraw" for each side }
               { --- scene information set by QuArK }
              SidesGL: Pointer;       { array of TFaceGL structures }
              NbSides: Integer;       { number of such structures }
              Camera: TVect;          { camera position }
              Angle: Reel;            { camera angle, as with QuArK's compass }
              PitchAngle: Reel;       { camera pitch, 0=horizontal, positive=down }
              Reserved: array[0..3] of Integer;
               { --- the DLL can add internal information : put it here instead of as global variables --- }
              Wnd: HWnd;
              DC: HDC;
              RC: HGLRC;
              Optimisation: Pointer;
              NbOpt: Integer;
              Palette: HPalette;
              LevelOfTextures: Integer;
              BlackOutline: Boolean;
              rModelMatrix, rProjMatrix: array[0..15] of GLdouble;
              rViewport: array[0..3] of GLint;
              rMatrixMutex: THandle;
             end;

const
 SEL_NORMAL = 0;
 SEL_UNIQUE = 1;
 SEL_GROUPE = 2;
 SEL_FACE   = 3;

type
  TRenderingModule = record   { information on rendering module (a DLL can hold several rendering modules) }
                      RoutinesBaseIndex: Integer; { DLL routine index of routine "Initialize" }
                      Description: PChar;         { Description as seen by the user }
                      Version: Integer;           { Version tag }
                      Reserved1: Integer;         { 0 }
                      SizeOfFaceGL: Integer;      { size of TFaceGL }
                      MenuCommands: PChar;        { #13-separated list of menu commands, or Nil if none }
                      Reserved2: Integer;         { 0 }
                      Reserved3: Integer;         { 0 }
                     end;
  PDLLInformation = ^TDLLInformation;
  TDLLInformation = record  { DLL information }
                     InterfaceVersion: Integer;   { must be kVersionGL }
                     InternalName: PChar;         { not used in the current version of QuArK }
                     ModulesCount: Integer;       { count of following TRenderingModule structures }
                     Reserved: Integer;           { 0 }
                     Modules: array[0..0] of TRenderingModule;   { array of rendering modules information }
                    end;

const
 DLLInformation : TDLLInformation =
  (InterfaceVersion:   kVersionGL;
   InternalName:       'Quake Army Knife';       { you can put your name here }
   ModulesCount:       1;
   Reserved:           0;
   Modules:
       ((RoutinesBaseIndex:  1;
         Description:        'OpenGL preview';   { this can be seen by the user }
         Version:            312;
         Reserved1:          0;
         SizeOfFaceGL:       SizeOf(TFaceGL);
         MenuCommands:       '&No textures'#13'&Standard textures'#13'&Enhanced textures' +
                               #13'-'#13'&Black outline';    { '-' is a separator }
         Reserved2:          0;
         Reserved3:          0)
        )
  );
  RapportTaille = 1/44;

(***********************************************************)
(**   INITALIZATION ROUTINES                              **)
(***********************************************************)

function Err : Boolean;  { OpenGL error check }
var
 I: Integer;
begin
 Result:=False;
 for I:=1 to 25 do
  begin
   if glGetError = GL_NO_ERROR then Break;
   Result:=True;
  end;
 if Result then
  wglMakeCurrent(0,0);
end;

{-------------------------------------------}
{--  Returns basic DLL information        --}
{-------------------------------------------}

(* This function is required for QuArK rendering modules.
   You must implement it so that it returns a pointer on
   information about your rendering module.
   Match the case : it's "InfoQuArK" and not "infoquark"  *)

function InfoQuArK : PDLLInformation; export;
begin
 InfoQuArK:=@DLLInformation;
end;

{-------------------------------------------}
{--  Frees DLL internal structure         --}
{-------------------------------------------}
procedure Destroy(R: PRenderC; RegistryKey: HKey); export;
var
 I: Integer;
begin
 wglMakeCurrent(0,0);
 with R^ do
  begin
   if NbOpt>0 then FreeMem(Optimisation);
   if RC<>0 then wglDeleteContext(RC);
   if DC<>0 then ReleaseDC(Wnd, DC);
   if rMatrixMutex<>0 then CloseHandle(rMatrixMutex);
  end;
 { save state }
 if RegistryKey<>0 then
  begin
   RegSetValueEx(RegistryKey, 'LevelOfTextures', 0, reg_DWord, @R.LevelOfTextures, SizeOf(R.LevelOfTextures));
   I:=Ord(R.BlackOutline);
   RegSetValueEx(RegistryKey, 'BlackOutline', 0, reg_DWord, @I, SizeOf(I));
  end;
 Dispose(R);
end;

{-------------------------------------------}
{--  Creates DLL internal structure       --}
{-------------------------------------------}
function Initialize(nWnd: HWnd; nPalette: HPalette; RegistryKey: HKey) : PRenderC; export;
var
 pfd: TPixelFormatDescriptor;
 pfi: Integer;
 fa: array[0..3] of GLfloat;
 R: PRenderC;
 J: Integer;
 PixelMap: array[0..255] of GLfloat;
 pe: array[0..255] of TPaletteEntry;

  function Err0 : Boolean;
  begin
   Result:=Err;
   if Result then
    Destroy(R, 0);
  end;

  function GetKeyInt(KeyName: PChar; Default: Integer) : Integer;
  var
   DataSize, DataType: Integer;
  begin
   DataSize:=SizeOf(Result);
   if (RegistryKey=0) or (RegQueryValueEx(RegistryKey, KeyName, Nil, @DataType,
    @Result, @DataSize) <> ERROR_SUCCESS) or (DataType <> REG_DWORD) then
    Result:=Default;
  end;

begin
 Result:=Nil;
 New(R);           { create structure }
 FillChar(R^, SizeOf(TRenderC), 0);
 with R^ do
  begin
    { read initial state }
   LevelOfTextures:=GetKeyInt('LevelOfTextures', 1);
   BlackOutline:=GetKeyInt('BlackOutline', 1) <> 0;

    { initalize display context }
   Wnd:=nWnd;
   FillChar(pfd, SizeOf(pfd), 0);
   pfd.nSize:=SizeOf(pfd);
   pfd.nversion:=1;
   pfd.dwflags:=pfd_Support_OpenGl or pfd_Draw_To_Window;
   pfd.iPixelType:=pfd_Type_RGBA;
   pfd.cColorBits:=24;
   pfd.cDepthBits:=16;
   pfd.iLayerType:=pfd_Main_Plane;
   DC:=GetDC(nWnd);
   pfi:=ChoosePixelFormat(DC, @pfd);
   SetPixelFormat(DC, pfi, @pfd);

   Palette:=nPalette;
   SelectPalette(DC, nPalette, False);
   RealizePalette(DC);

    { create an OpenGL rendering context }
   RC:=wglCreateContext(DC);
   if RC=0 then
    begin
     Destroy(R, 0);
     Exit;
    end;
   wglMakeCurrent(DC,RC);
   if Err0 then Exit;

    { set up OpenGL }
   glClearColor(0.6,0.6,0.6,0);
   glClearDepth(1);
   glEnable(GL_DEPTH_TEST);
   glDepthFunc(GL_LEQUAL);
   glEnable(GL_CULL_FACE);
   glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);
   glEdgeFlag(GL_FALSE);
   glMatrixMode(GL_MODELVIEW);
   glLoadIdentity;
   glScaled(RapportTaille, RapportTaille, RapportTaille);
   glGetDoublev(GL_MODELVIEW_MATRIX, rModelMatrix);
   if Err0 then Exit;

    { set up palette index to color map (for textures) }
   GetPaletteEntries(Palette, 0,256, pe);
   for J:=0 to 255 do
    PixelMap[J]:=pe[J].peRed * (1/255);
   glPixelMapfv(GL_PIXEL_MAP_I_TO_R, 256, PixelMap);
   for J:=0 to 255 do
    PixelMap[J]:=pe[J].peGreen * (1/255);
   glPixelMapfv(GL_PIXEL_MAP_I_TO_G, 256, PixelMap);
   for J:=0 to 255 do
    PixelMap[J]:=pe[J].peBlue * (1/255);
   glPixelMapfv(GL_PIXEL_MAP_I_TO_B, 256, PixelMap);
   if Err0 then Exit;

    { set up texture parameters }
   glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
   glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
   glShadeModel(GL_FLAT);
   if Err0 then Exit;

    { set up lighting }
   fa[0]:=0.7;
   fa[1]:=0.7;
   fa[2]:=0.7;
   fa[3]:=0.7;
   glLightfv(GL_LIGHT0, GL_AMBIENT, fa);
   fa[0]:=0.014;
   fa[1]:=0.014;
   fa[2]:=0.014;
   fa[3]:=1;
   glLightfv(GL_LIGHT0, GL_DIFFUSE, fa);
   glEnable(GL_LIGHT0);
   glEnable(GL_COLOR_MATERIAL);
   if Err0 then Exit;

   wglMakeCurrent(0,0);

   rMatrixMutex:=CreateMutex(Nil, False, Nil);
  end;
 Result:=R;
end;

{-------------------------------------------}
{--  Build internal texture structure     --}
{-------------------------------------------}
function BuildTexture(RStruct: PRenderC; TexHeader: PTexHeader) : PTextureGL; export;
var
 pe: array[0..255] of TPaletteEntry;
 w,h, I, Total: Integer;
 r,g,b, Rapport: Reel;
 P, Dest, Temp: PChar;
 Pal2: array[0..255] of record
        pr,pg,pb: Reel;
       end;
begin
 with RStruct^ do
  begin
    { computes the "mean color" by computing the mean of the square of the color components }
   GetPaletteEntries(Palette, 0,256, pe);
   Total:=TexHeader.W*TexHeader.H;
   for I:=0 to 255 do
    with pe[I], Pal2[I] do
     begin
      pr:=Sqr(Integer(peRed));
      pg:=Sqr(Integer(peGreen));
      pb:=Sqr(Integer(peBlue));
     end;
   r:=0;
   g:=0;
   b:=0;
   P:=PChar(TexHeader) + TexHeader.Indexes[0];   { pointer to main image }
   for I:=1 to Total do
    begin
     with Pal2[Ord(P^)] do
      begin
       r:=r+pr;
       g:=g+pg;
       b:=b+pb;
      end;
     Inc(P);
    end;

   wglMakeCurrent(DC,RC);
   P:=PChar(TexHeader) + TexHeader.Indexes[0];
    { try to select the texture into OpenGL }
   glTexImage2D(GL_TEXTURE_2D, 0, 3, TexHeader.W,
    TexHeader.H, 0, GL_COLOR_INDEX, GL_UNSIGNED_BYTE, P^);
   if glGetError<>0 then
    begin  { error : the width or height is probably not a power of 2 }
     w:=4;
     while w<TexHeader.W do w:=w shl 1;   { nearest larger power of 2 }
     h:=1;
     while h<TexHeader.H do h:=h shl 1;   { nearest larger power of 2 }
      { build an enlarged black-filled texture image }
     GetMem(Result, w*h);
     Dest:=PChar(Result);
     for I:=1 to TexHeader.H do
      begin
       Move(P^, Dest^, TexHeader.W);
       Inc(P, TexHeader.W);
       FillChar(Dest[TexHeader.W], w-TexHeader.W, 0);
       Inc(Dest, w);
      end;
     FillChar(Dest^, (h-TexHeader.H)*w, 0);
      { select it into OpenGL }
     glTexImage2D(GL_TEXTURE_2D, 0, 3, w,
      h, 0, GL_COLOR_INDEX, GL_UNSIGNED_BYTE, Result^);
     FreeMem(Result);
     Result:=Nil;
     if Err then Exit;   { error - give up }
      { read the image back in RGB format }
     GetMem(Temp, w*h*3);
     glGetTexImage(GL_TEXTURE_2D, 0, GL_RGB, GL_UNSIGNED_BYTE, Temp^);
     if Err then
      begin
       FreeMem(Temp);
       Exit;
      end;
      { reduce the image size again by cutting out the black parts added }
     P:=Temp;
     Dest:=P;
     for I:=2 to TexHeader.H do
      begin
       Inc(P, 3*w);
       Inc(Dest, 3*TexHeader.W);
       Move(P^, Dest^, 3*TexHeader.W);
      end;
      { scale the RGB image to the new size }
     GetMem(Result, SizeOf(TTextureGL) + w*h*3);
     gluScaleImage(GL_RGB, TexHeader.W, TexHeader.H, GL_UNSIGNED_BYTE, Temp^,
      w, h, GL_UNSIGNED_BYTE, Result^.glBits);
     FreeMem(Temp);
    end
   else
    begin
      { read the OpenGL texture back in RGB format }
     w:=TexHeader.W;
     h:=TexHeader.H;
     GetMem(Result, SizeOf(TTextureGL) + w*h*3);
     glGetTexImage(GL_TEXTURE_2D, 0, GL_RGB, GL_UNSIGNED_BYTE, Result^.glBits);
    end;
   if Err then
    begin
     FreeMem(Result);
     Result:=Nil;
     Exit;
    end;
   wglMakeCurrent(0,0);
   with Result^ do
    begin   { initalize TTextureGL structure }
     glW:=w;
     glH:=h;
     glScaleW:=1/TexHeader.W;
     glScaleH:=(-1)/TexHeader.H;
     Rapport:=(1/Sqr(255))/Total;
     glMeanColor[0] := Sqrt(r*Rapport);
     glMeanColor[1] := Sqrt(g*Rapport);
     glMeanColor[2] := Sqrt(b*Rapport);
    end;
  end;
end;

{-------------------------------------------}
{--  Free internal texture structure      --}
{-------------------------------------------}
procedure FreeTexture(Tex: PTextureGL); export;
begin
 FreeMem(Tex);
end;

(***********************************************************)
(**   RENDERING ROUTINES                                  **)
(***********************************************************)

procedure NouveauPV(RStruct: PRenderC);
const
 AngleLumZ = -0.2;
var
 R: TRect;
 L: array[0..3] of GLfloat;
 a, Rapport: Reel;
begin
 GetClientRect(RStruct.Wnd, R);
 glViewport(0, 0, R.Right, R.Bottom);
 glMatrixMode(GL_PROJECTION);
 glLoadIdentity;
 Rapport:=R.Right/R.Bottom;
 a:=100/Sqrt(Rapport);
 if a<60 then
  a:=60
 else
  if a>100 then
   a:=100;
 gluPerspective(a, Rapport, {1.0}0.1, {30.0} 77.7);
 with RStruct^ do
  begin
   if PitchAngle<>0 then
    glRotated(PitchAngle, 1,0,0);
   a:=-Angle;
   glRotated(a, 0,1,0);
   glTranslated(RapportTaille*Camera.Y, (-RapportTaille)*Camera.Z, RapportTaille*Camera.X);
   a:=(a+20)*(pi/180);
  end;
 L[0]:=Cos(a) * (-Cos(AngleLumZ));
 L[1]:=Sin(a) * (+Cos(AngleLumZ));
 L[2]:=-Sin(AngleLumZ);
 L[3]:=0;
 glLightfv(GL_LIGHT0, GL_POSITION, L);

 WaitForSingleObject(RStruct.rMatrixMutex, INFINITE);
 glGetDoublev(GL_PROJECTION_MATRIX, RStruct.rProjMatrix);
 glGetIntegerv(GL_VIEWPORT, RStruct.rViewport);
 ReleaseMutex(RStruct.rMatrixMutex);
end;

procedure DessinerPolygone(Painting: PFaceGL);
var
 I: Integer;
begin
 with Painting^.glFace^ do
  begin
   glBegin(GL_POLYGON);
   for I:=prvNbS-1 downto 0 do
    with prvDescS^[I]^ do
     glVertex3d(-Y,Z,-X);
   glEnd;
  end;
end;

procedure DessinerPolygoneTex(Painting: PFaceGL);
var
 I: Integer;
 PX, PY, S, C, A: Reel;
begin
 with Painting^ do
  begin
   case glSelection of
    SEL_NORMAL: glColor3f(0.65,0.65,0.65);
    SEL_GROUPE: glColor3f(1,0,1);
    else        glColor3f(1,1,1);
   end;
   with glFace^ do
    begin
     glNormal3dv(NormalVector);
     glBegin(GL_POLYGON);
     for I:=prvNbS-1 downto 0 do
      with prvDescS^[I]^ do
       begin
        A:=Abs(NormalVector.X);
        S:=Abs(NormalVector.Y);
        C:=Abs(NormalVector.Z);
        if A>S then
         if A>=C then
          begin
           PX:=Y;
           PY:=Z;
          end
         else
          begin
           PX:=X;
           PY:=Y;
          end
        else
         if S>=C then
          begin
           PX:=X;
           PY:=Z;
          end
         else
          begin
           PX:=X;
           PY:=Y;
          end;
        if TexParams[3]<>0 then
         begin
          A:=TexParams[3] * (-pi/180);
          S:=Sin(A);
          C:=Cos(A);
          A:=PX;
          PX:=PX*C - PY*S;
          PY:= A*S + PY*C;
         end;
        if TexParams[4]<>1 then
         PX:=PX/TexParams[4];
        if TexParams[5]<>1 then
         PY:=PY/TexParams[5];
        PX:=PX+TexParams[1];
        PY:=PY-TexParams[2];
        glTexCoord2d(PX,PY);
        glVertex3d(-Y,Z,-X);
       end;
     glEnd;
    end;
  end;
end;

procedure DessinerPolygoneLum(Painting: PFaceGL);
var
 I: Integer;
begin
 with Painting^ do
  begin
   glNormal3dv(glFace^.NormalVector);
   if glTex=Nil then
    case glSelection of
     SEL_NORMAL: glColor3f(0.5,0.5,0.5);
     SEL_GROUPE: glColor3f(0.77,0,0.77);
     else        glColor3f(0.77,0.77,0.77);
    end
   else
    case glSelection of
     SEL_NORMAL: glColor3fv(glTex^.glMeanColor);
     SEL_GROUPE: glColor3f(glTex^.glMeanColor[0] * (1/0.65),
                        0, glTex^.glMeanColor[2] * (1/0.65));
     else        glColor3f(glTex^.glMeanColor[0] * (1/0.65),
                           glTex^.glMeanColor[1] * (1/0.65),
                           glTex^.glMeanColor[2] * (1/0.65));
    end;
   glBegin(GL_POLYGON);
   with glFace^ do
    for I:=prvNbS-1 downto 0 do
     with prvDescS^[I]^ do
      glVertex3d(-Y,Z,-X);
   glEnd;
  end;
end;

procedure PreparerTexture(Tex1: PTextureGL; Extra: Boolean);
begin
 if Extra then
  gluBuild2DMipmaps(GL_TEXTURE_2D, 3, Tex1^.glW,
   Tex1^.glH, GL_RGB, GL_UNSIGNED_BYTE,
   Tex1^.glBits)
 else
  glTexImage2D(GL_TEXTURE_2D, 0, 3, Tex1^.glW,
   Tex1^.glH, 0, GL_RGB, GL_UNSIGNED_BYTE,
   Tex1^.glBits);
 glMatrixMode(GL_TEXTURE);
 glLoadIdentity;
 glScalef(Tex1^.glScaleW, Tex1^.glScaleH, 1);
end;

procedure DessinerLignesNoires(Painting: PFaceGL; BlackOutline, ModeTextures: Boolean);
const
 TaillePoignee = 7;
var
 I: Integer;
 Somme, V: TVect;
 Tampon, Pc: ^TVect;
 Rapport: Reel;
begin
 with Painting^ do
  with glFace^ do
   begin
    if Odd(glSelection) then
     begin
      glColor3f(1,1,0);
      if ModeTextures then
       glDisable(GL_TEXTURE_2D)
      else
       glDepthMask(GL_FALSE);
     end
    else
     if BlackOutline then
      glColor3f(0,0,0)
     else
      Exit;
    glBegin(GL_LINE_LOOP);
    for I:=prvNbS-1 downto 0 do
     with prvDescS^[I]^ do
      glVertex3d(-Y, Z, -X);
    glEnd;
    if Odd(glSelection) then
     begin
      FillChar(Somme, SizeOf(Somme), 0);
      for I:=prvNbS-1 downto 0 do
       with prvDescS^[I]^ do
        begin
         Somme.X:=Somme.X+X;
         Somme.Y:=Somme.Y+Y;
         Somme.Z:=Somme.Z+Z;
        end;
      Rapport:=1/prvNbS;
      Somme.X:=Somme.X*Rapport;
      Somme.Y:=Somme.Y*Rapport;
      Somme.Z:=Somme.Z*Rapport;
      GetMem(Tampon, prvNbS*SizeOf(TVect));
      Pc:=Tampon;
      glBegin(GL_LINE_LOOP);
      for I:=prvNbS-1 downto 0 do
       with prvDescS^[I]^ do
        begin
         V.X:=X-Somme.X;
         V.Y:=Y-Somme.Y;
         V.Z:=Z-Somme.Z;
         Rapport:=TaillePoignee/Sqrt(Sqr(V.X)+Sqr(V.Y)+Sqr(V.Z));
         Pc^.X:=-(Rapport*V.Y + Somme.Y);
         Pc^.Y:=  Rapport*V.Z + Somme.Z;
         Pc^.Z:=-(Rapport*V.X + Somme.X);
         glVertex3dv(Pc^);
         Inc(Pc);
        end;
      glEnd;
      if glSelection = SEL_FACE then
       begin
        Pc:=Tampon;
        glBegin(GL_LINES);
        for I:=prvNbS-1 downto 0 do
         with prvDescS^[I]^ do
          begin
           glVertex3d(-Y, Z, -X);
           glVertex3dv(Pc^);
           Inc(Pc);
          end;
        glEnd;
       end;
      FreeMem(Tampon, prvNbS*SizeOf(TVect));
      if ModeTextures then
       glEnable(GL_TEXTURE_2D)
      else
       glDepthMask(GL_TRUE);
     end;
   end;
end;

procedure DessinerVueRapide(R: PRenderC);
var
 I: Integer;
 Painting: PFaceGL;
 OptPtr: ^PFaceGL;
 Passe: Boolean;
begin
 glRenderMode(GL_RENDER);
 glEnable(GL_LIGHTING);
 glDepthMask(GL_TRUE);
 glDisable(GL_TEXTURE_2D);
 glEnable(GL_CULL_FACE);

 if R.NbOpt>0 then
  for Passe:=False to True do
   begin
    Pointer(OptPtr):=R.Optimisation;
    for I:=1 to R.NbOpt do
     begin
      if (OptPtr^^.glDrawStep=0)
      and (Passe or (OptPtr^^.glSelection<>SEL_NORMAL)) then
       begin
        if R.StopDrawing then Exit;
        DessinerPolygoneLum(OptPtr^);
        DessinerLignesNoires(OptPtr^, R.BlackOutline, False);
       {if OptPtr^^.glSelection = SEL_FACE then
         R.VectNormal:=OptPtr^;}
        OptPtr^^.glDrawStep:=1;
       end;
      Inc(OptPtr);
     end;
   end;

 for Passe:=False to True do
  begin
   Painting:=PFaceGL(R.SidesGL);
   for I:=1 to R.NbSides do
    begin
     if R.StopDrawing then Exit;
     if (Painting^.glDrawStep=0)
     and (Passe or (Painting^.glSelection<>SEL_NORMAL)) then
      begin
       DessinerPolygoneLum(Painting);
       DessinerLignesNoires(Painting, R.BlackOutline, False);
      {if Painting^.glSelection = SEL_FACE then
        R.VectNormal:=Painting;}
       Painting^.glDrawStep:=1;
      end;
     Inc(Painting);
    end;
  end;

 glDepthMask(GL_FALSE);
end;

procedure OptimiserVue(R: PRenderC);
const
 DeltaU = $80000000;
type
 PHitRecord = ^THitRecord;
 THitRecord = record
               Reserved: GLuint;
               case Integer of
                0: (DepthMin, DepthMax: GLuint;
                    Name: GLuint);
                1: (Tri: Integer);
              end;
 PHitRecords = ^THitRecords;
 THitRecords = array[0..999] of THitRecord;
var
 I, N: Integer;
 Painting: PFaceGL;
 TamponInterne: PHitRecords;
 P: PHitRecord;
 OptPtr: ^PFaceGL;

 PROCEDURE TrierTampon(Gauche,Droite : INTEGER);
 VAR
  I,J : INTEGER;
  Pivot: Integer;
  UneCase: GLuint;
 BEGIN
  I:=Gauche; J:=Droite;
  Pivot:=TamponInterne^[(Gauche+Droite) DIV 2].Tri;
  REPEAT
   WHILE TamponInterne^[I].Tri < Pivot DO INC(I);
   WHILE TamponInterne^[J].Tri > Pivot DO DEC(J);
   IF I<=J THEN
    BEGIN
     with TamponInterne^[I] do
      begin
       UneCase:=Tri;
       Tri:=TamponInterne^[J].Tri;
       TamponInterne^[J].Tri:=UneCase;
       UneCase:=Name;
       Name:=TamponInterne^[J].Name;
       TamponInterne^[J].Name:=UneCase;
      end;
     INC(I);
     DEC(J);
    END;
  UNTIL I>J;
  IF J>Gauche THEN TrierTampon(Gauche,J);
  IF I<Droite THEN TrierTampon(I,Droite);
 END;

begin
 I:=R.NbSides * SizeOf(THitRecord);
 GetMem(TamponInterne, I);
 glDisable(GL_CULL_FACE);  { to correct a bug (?) of OpenGL for Win95 }

 glSelectBuffer(I, TamponInterne^);
 glRenderMode(GL_SELECT);
 glPushName(0);
 Painting:=PFaceGL(R.SidesGL);
 for I:=1 to R.NbSides do
  begin
   if R.StopDrawing then
    begin
     glRenderMode(GL_RENDER);
     FreeMem(TamponInterne);
     Exit;
    end;
   glLoadName(GLuint(Painting));
   DessinerPolygone(Painting);
   Inc(Painting);
  end;
 N:=glRenderMode(GL_RENDER);
 P:=PHitRecord(TamponInterne);
 for I:=1 to N do
  begin
   P^.Tri:=P^.DepthMin xor DeltaU;
   Inc(P);
  end;
 if N>1 then
  TrierTampon(0,N-1);
 if R.NbOpt>0 then
  begin
   R.NbOpt:=0;
   FreeMem(R.Optimisation);
  end;
 if N>0 then
  begin
   GetMem(R.Optimisation, N*SizeOf(GLuint));
   Pointer(OptPtr):=R.Optimisation;
   P:=PHitRecord(TamponInterne);
   for I:=1 to N do
    begin
     OptPtr^:=PFaceGL(P^.Name);
     Inc(P);
     Inc(OptPtr);
    end;
   R.NbOpt:=N;
  end;
 FreeMem(TamponInterne);
end;

(*type
 PHitRecord = ^THitRecord;
 THitRecord = record
               Reserved: GLuint;
               case Integer of
                0: (DepthMin, DepthMax: GLuint;
                    Name: GLuint);
                1: (Tri: Integer);
              end;
 PHitRecords = ^THitRecords;
 THitRecords = array[0..999] of THitRecord;

procedure Optimiser(R: PRenderC; TamponInterne: PHitRecords; PasseDeux: Boolean);
const
 DeltaU = $80000000;
var
 I, N: Integer;
 Painting: PFaceGL;
 P: PHitRecord;
 OptPtr: ^PFaceGL;

 PROCEDURE TrierTampon(Gauche,Droite : INTEGER);
 VAR
  I,J : INTEGER;
  Pivot: Integer;
  UneCase: GLuint;
 BEGIN
  I:=Gauche; J:=Droite;
  Pivot:=TamponInterne^[(Gauche+Droite) DIV 2].Tri;
  REPEAT
   WHILE TamponInterne^[I].Tri < Pivot DO INC(I);
   WHILE TamponInterne^[J].Tri > Pivot DO DEC(J);
   IF I<=J THEN
    BEGIN
     with TamponInterne^[I] do
      begin
       UneCase:=Tri;
       Tri:=TamponInterne^[J].Tri;
       TamponInterne^[J].Tri:=UneCase;
       UneCase:=Name;
       Name:=TamponInterne^[J].Name;
       TamponInterne^[J].Name:=UneCase;
      end;
     INC(I);
     DEC(J);
    END;
  UNTIL I>J;
  IF J>Gauche THEN TrierTampon(Gauche,J);
  IF I<Droite THEN TrierTampon(I,Droite);
 END;

begin
 glSelectBuffer(R.NbSides * SizeOf(THitRecord), TamponInterne^);
 glRenderMode(GL_SELECT);
 glPushName(0);
 Painting:=PFaceGL(R.SidesGL);
 for I:=1 to R.NbSides do
  begin
   if (Painting^.glSelection<>SEL_NORMAL) xor PasseDeux then
    begin
     if R.StopDrawing then
      begin
       glRenderMode(GL_RENDER);
       Exit;
      end;
     glLoadName(GLuint(Painting));
     DessinerPolygone(Painting);
    end;
   Inc(Painting);
  end;
 N:=glRenderMode(GL_RENDER);
 P:=PHitRecord(TamponInterne);
 for I:=1 to N do
  begin
   P^.Tri:=P^.DepthMin xor DeltaU;
   Inc(P);
  end;
 if N>1 then
  TrierTampon(0,N-1);
 if N>0 then
  begin
   ReallocMem(R.Optimisation, (N+R.NbOpt)*SizeOf(GLuint));
   Pointer(OptPtr):=PChar(R.Optimisation) + R.NbOpt*SizeOf(GLuint);
   P:=PHitRecord(TamponInterne);
   for I:=1 to N do
    begin
     OptPtr^:=PFaceGL(P^.Name);
     Inc(P);
     Inc(OptPtr);
    end;
   Inc(R.NbOpt, N);
  end;
end;

procedure OptimiserVue(R: PRenderC);
var
 TamponInterne: PHitRecords;
begin
 glDisable(GL_CULL_FACE);  { to correct a bug (?) of OpenGL for Win95 }
 if R.NbOpt>0 then
  begin
   R.NbOpt:=0;
   FreeMem(R.Optimisation);
  end;
 R.Optimisation:=Nil;
 GetMem(TamponInterne, R.NbSides * SizeOf(THitRecord));
 if not R.StopDrawing then Optimiser(R, TamponInterne, False);
 if not R.StopDrawing then Optimiser(R, TamponInterne, True);
 FreeMem(TamponInterne);
end;*)

procedure DessinerVueTextures(R: PRenderC);
var
 I, J: Integer;
 Painting: PFaceGL;
 OptPtr, OptPtr1: ^PFaceGL;
 PasseTex: Integer;
 Passe: Boolean;
 Tex1: PTextureGL;
begin
 if R.NbOpt=0 then Exit;

 glEnable(GL_TEXTURE_2D);
 glEnable(GL_CULL_FACE);

 for PasseTex:=1 to R.LevelOfTextures do
  begin
   case PasseTex of
    1: begin
        glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
        glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
       end;
    2: begin
        glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
        glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST_MIPMAP_LINEAR);
       end;
   end;

   for Passe:=False to True do
    begin
     Pointer(OptPtr1):=R.Optimisation;
     for I:=1 to R.NbOpt do
      begin
       if (OptPtr1^^.glTex<>Nil) and (OptPtr1^^.glDrawStep<=PasseTex)
       and (Passe or (OptPtr1^^.glSelection<>SEL_NORMAL)) then
        begin
         OptPtr:=OptPtr1;
         Tex1:=OptPtr^^.glTex;
         PreparerTexture(Tex1, PasseTex=2);
         for J:=I to R.NbOpt do
          begin
           Painting:=OptPtr^;
           if (Painting^.glTex=Tex1) and (Painting^.glDrawStep<=PasseTex) then
            begin
             if R.StopDrawing then Exit;
             DessinerPolygoneTex(Painting);
             DessinerLignesNoires(Painting, R.BlackOutline, True);
             Painting^.glDrawStep:=PasseTex+1;
            end;
           Inc(OptPtr);
          end;
        end;
       Inc(OptPtr1);
      end;
    end;
  end;
end;

procedure InitRedraw(R: PRenderC);     { prerendering }
var
 Painting: PFaceGL;
 I: Integer;
begin
 Painting:=PFaceGL(R.SidesGL);
 if R.FullRedraw then
  begin
   for I:=1 to R.NbSides do
    begin
     with Painting^ do
      begin
       glDrawStep:=0;     { reset every glDrawStep to zero }
       glRedraw:=False;
      end;
     Inc(Painting);
    end;
   glDepthMask(GL_TRUE);
   glClear(GL_COLOR_BUFFER_BIT or GL_DEPTH_BUFFER_BIT); { clear screen }
   R.FullRedraw:=False;
  end
 else
  for I:=1 to R.NbSides do
   begin
    with Painting^ do
     if glRedraw then     { only reset polygons with glRedraw }
      begin
       glDrawStep:=0;
       glRedraw:=False;
      end;
    Inc(Painting);
   end;
end;

{-------------------------------------------}
{--  Rendering                            --}
{-------------------------------------------}
procedure Rendering(RStruct: PRenderC); export;
begin
 with RStruct^ do
  begin
  {ResetDepthBuffer:=ResetDepthBuffer or ResetScene or ResetView;}
   if ResetScene then    { if the scene changed, clear the Optimisation structure }
    begin
     ResetScene:=False;
     if NbOpt>0 then
      begin
       NbOpt:=0;
       FreeMem(Optimisation);
      end;
    {FullRedraw:=True;  not required : if the program set it to False, it means that only the sides with glRedraw=True were added }
    end;
   wglMakeCurrent(DC,RC);
  {if ResetDepthBuffer then
    begin
     ResetDepthBuffer:=False;
     glDepthMask(GL_TRUE);
     glClear(GL_DEPTH_BUFFER_BIT);
    end;}
   if ResetView then    { if the camera moved, update it in OpenGL }
    begin
     ResetView:=False;
     NouveauPV(RStruct);
     if Err then Exit;
    {FullRedraw:=True;  not required : the program always set it together with ResetView }
    end;
  {VectNormal:=Nil;}
   if not StopDrawing then InitRedraw(RStruct);
   if not StopDrawing then DessinerVueRapide(RStruct);
   if not StopDrawing then OptimiserVue(RStruct);
   if not StopDrawing then DessinerVueTextures(RStruct);
   if not StopDrawing then glFinish;
   if Err then MessageBeep(0);
   wglMakeCurrent(0,0);
  end;
end;

(***********************************************************)
(**   AUXILIARY ROUTINES                                  **)
(***********************************************************)

const
 GLM_CLICK         = $80000000;
 GLM_UPDATE        = 1;

 GLM_NORMAL        = 0;
 GLM_CHECKED       = 1;
 GLM_RADIO         = 2;
 GLM_DISABLED      = 4;

{-------------------------------------------}
{--  Menu management                      --}
{-------------------------------------------}
function MenuCmd(R: PRenderC; Index: Integer) : Integer; export;
{  input : Index = zero-based menu index,
            ORed with GLM_CLICK if clicked
  output :
    if clicked, return the required redraw :
      GLM_NORMAL   if no redraw needed
      GLM_UPDATE   if redraw needed - first set flags like FullRedraw, ResetView, ...
    if not clicked, return the menu command state :
      GLM_NORMAL   not checked
      GLM_CHECKED  checked (with a standard check mark)
      GLM_RADIO    radio-button checked
      GLM_DISABLED disabled (can be ORed with the above)
}
begin
 Result:=GLM_NORMAL;
 if Index and GLM_CLICK = 0 then
  case Index of
   0..2: begin
          Result:=GLM_RADIO;
          if Index=R.LevelOfTextures then
           Result:=Result or GLM_CHECKED;
         end;
   4: if R.BlackOutline then
       Result:=Result or GLM_CHECKED;
  end
 else
  begin
   Index:=Index and not GLM_CLICK;
   case Index of
    0..2: if R.LevelOfTextures<>Index then
           begin
            if R.LevelOfTextures>Index then
             R.FullRedraw:=True;
            R.LevelOfTextures:=Index;
            Result:=GLM_UPDATE;
           end;
    4: begin
        R.BlackOutline:=not R.BlackOutline;
        R.FullRedraw:=True;
       {R.ResetDepthBuffer:=True;}
        Result:=GLM_UPDATE;
       end;
   end;
  end;
end;

{-------------------------------------------}
{--  3D coordinates to window conversion  --}
{-------------------------------------------}
function Coordinates2D(R: PRenderC; var Source: TVect; var Dest: TPoint) : Bool;
{  input : Source.X,Y,Z = source 3D coordinates
  output : Dest.X,Y = computed window coordinates
    note : "window coordinates" means X,Y pixels
}
var
 V: TVect;
begin
 WaitForSingleObject(R.rMatrixMutex, INFINITE);
 Result:=gluProject(-Source.Y, Source.Z, -Source.X,
  R.rModelMatrix, R.rProjMatrix, R.rViewport,
  V.X, V.Y, V.Z) <> GL_FALSE;
 if Result then
  begin
   Dest.X:=Round(V.X);
   Dest.Y:=R.rViewport[3]-Round(V.Y);
  end;
 ReleaseMutex(R.rMatrixMutex);
end;

{-------------------------------------------}
{--  Window to 3D coordinates conversion  --}
{-------------------------------------------}
function Coordinates3D(R: PRenderC; var Source: TPoint; var Dest: TVect) : Bool;
{  input : Source.X,Y = source window coordinates
  output : Dest.X,Y,Z = computed 3D coordinates
    note : the function must compute an arbitrar 3D point that will project to Source.X,Y
}
begin
 WaitForSingleObject(R.rMatrixMutex, INFINITE);
 Result:=gluUnProject(Source.X, R.rViewport[3]-Source.Y, 0.5 { <- arbitrarly choosen },
  R.rModelMatrix, R.rProjMatrix, R.rViewport,
  Dest.Y, Dest.Z, Dest.X) <> GL_FALSE;
 if Result then
  begin
   Dest.Y:=-Dest.Y;
   Dest.X:=-Dest.X;
  end;
 ReleaseMutex(R.rMatrixMutex);
end;

(***********************************************************)

exports
 Initialize     index 1,    { = RoutinesBaseIndex + 0 }
 Destroy        index 2,    { = RoutinesBaseIndex + 1 }
 BuildTexture   index 3,    { = RoutinesBaseIndex + 2 }
 FreeTexture    index 4,    { = RoutinesBaseIndex + 3 }
 Rendering      index 5,    { = RoutinesBaseIndex + 4 }
 MenuCmd        index 6,    { = RoutinesBaseIndex + 5 }
 Coordinates2D  index 7,    { = RoutinesBaseIndex + 6 }
 Coordinates3D  index 8,    { = RoutinesBaseIndex + 7 }
 InfoQuArK;

begin
end.
