/*
Copyright (C) 2000 Jason Wilkins

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include "shaders.h"
#include "sh_internal.h"



static void save_skyparms(shader_t* shader)
{
   if ((shader->flags & SH_FAR_BOX) ||
       (shader->flags & SH_SKYGEN)  ||
       (shader->flags & SH_NEAR_BOX)) {
      sh_output("\tskyparms ");

      if (shader->flags & SH_FAR_BOX) {
         sh_output("%s", shader->far_box_basename);
      }
      else {
         sh_output("-");
      }

      sh_output(" ");

      if (shader->flags & SH_SKYGEN) {
         sh_output("%s", shader->cloud_height.string);
      }
      else {
         sh_output("-");
      }

      sh_output(" ");

      if (shader->flags & SH_NEAR_BOX) {
         sh_output("%s", shader->near_box_basename);
      }
      else {
         sh_output("-");
      }

      sh_output("\n");
   }
}



static void save_cull(shader_t* shader)
{
   if (shader->flags & SH_EXPLICIT_CULL) {
      sh_output("cull %s\n", sh_string_from_cull(shader->cull_face));
   }
}



static void save_vxmod(sh_vxmod_t* vxmod)
{
   sh_output("\tdeformVertexes %s", sh_string_from_vxmod(vxmod->func));

   switch(vxmod->func) {
      case SH_VXMOD_WAVE:
         sh_output(" %s", vxmod->args.wave.div.string);
         sh_output(" %s", sh_string_from_waveform(vxmod->args.wave.form));
         sh_output(" %s", vxmod->args.wave.base.string);
         sh_output(" %s", vxmod->args.wave.amp.string);
         sh_output(" %s", vxmod->args.wave.phase.string);
         sh_output(" %s", vxmod->args.wave.freq.string);
         break;

      case SH_VXMOD_NORMAL:
         sh_output(" %s", vxmod->args.normal.amp.string);
         sh_output(" %s", vxmod->args.normal.freq.string);
         break;

      case SH_VXMOD_BULGE:
         sh_output(" %s", vxmod->args.bulge.width.string);
         sh_output(" %s", vxmod->args.bulge.height.string);
         sh_output(" %s", vxmod->args.bulge.speed.string);
         break;

      case SH_VXMOD_MOVE:
         sh_output(" %s", vxmod->args.move.x.string);
         sh_output(" %s", vxmod->args.move.y.string);
         sh_output(" %s", vxmod->args.move.z.string);
         sh_output(" %s", sh_string_from_waveform(vxmod->args.move.form));
         sh_output(" %s", vxmod->args.move.base.string);
         sh_output(" %s", vxmod->args.move.amp.string);
         sh_output(" %s", vxmod->args.move.phase.string);
         sh_output(" %s", vxmod->args.move.freq.string);
         break;

      case SH_VXMOD_AUTOSPRITE:
      case SH_VXMOD_AUTOSPRITE2:
      case SH_VXMOD_PROJECTION_SHADOW:
      case SH_VXMOD_TEXT0:
      case SH_VXMOD_TEXT1:

#ifdef SH_PHOENIX_QUAKE1_EXT
      case SH_VXMOD_STENCIL_SHADOW_EXT:
#endif

         break;

      default:
         break;
   }

   sh_output("\n");
}



static void save_vxmods(shader_t* shader)
{
   int i;

   for (i = 0; i < shader->vxmod_count; i++) {
      save_vxmod(&shader->vxmods[i]);
   }
}



static void save_fogparms(shader_t* shader)
{
   if (shader->flags & SH_FOGPARMS) {
      sh_output("\tfogparms");
      sh_output(" %s", shader->fog_color[0].string);
      sh_output(" %s", shader->fog_color[1].string);
      sh_output(" %s", shader->fog_color[2].string);
      sh_output(" %s", shader->fog_distance_to_opaque.string);
      sh_output("\n");
   }
}



static void save_nopicmip(shader_t* shader)
{
   if (shader->flags & SH_EXPLICIT_NOPICMIP) {
      sh_output("\tnopicmip\n");
   }
}



static void save_nomipmaps(shader_t* shader)
{
   if (shader->flags & SH_NOMIPMAPS) {
      sh_output("\tnomipmaps\n");
   }
}



static void save_polygon_offset(shader_t* shader)
{
   if (shader->flags & SH_POLYGON_OFFSET) {
      sh_output("\tpolygonOffset\n");
   }
}



static void save_sort(shader_t* shader)
{
   if (shader->flags & SH_EXPLICIT_PORTAL) {
      sh_output("\tportal\n");
   }

   if (shader->flags & SH_EXPLICIT_SORT) {
      sh_output("\tsort %s\n", sh_string_from_sort(shader->sort_priority));
   }
}



static void save_entity_mergable(shader_t* shader)
{
   if (shader->flags & SH_ENTITY_MERGABLE) {
      sh_output("\tentityMergable\n");
   }
}



static void save_tess_size(shader_t* shader)
{
   if (shader->flags & SH_TESS_SIZE) {
      sh_output("\ttessSize %s\n", shader->tess_size.string);
   }
}



static void save_q3map_backshader(shader_t* shader)
{
   if (shader->flags & SH_Q3MAP_BACKSHADER) {
      sh_output("\tq3map_backshader %s\n", shader->q3map_backshader);
   }
}



static void save_q3map_globaltexture(shader_t* shader)
{
   if (shader->flags & SH_Q3MAP_GLOBALTEXTURE) {
      sh_output("\tq3map_globaltexture\n");
   }
}



static void save_q3map_sun(shader_t* shader)
{
   if (shader->flags & SH_Q3MAP_SUN) {
      sh_output("\tq3map_sun");
      sh_output(" %s", shader->q3map_sun.red);
      sh_output(" %s", shader->q3map_sun.green);
      sh_output(" %s", shader->q3map_sun.blue);
      sh_output(" %s", shader->q3map_sun.intensity);
      sh_output(" %s", shader->q3map_sun.degrees);
      sh_output(" %s", shader->q3map_sun.elevation);
      sh_output("\n");
   }
}



static void save_q3map_surfacelight(shader_t* shader)
{
   if (shader->flags & SH_EXPLICIT_Q3MAP_SURFACELIGHT) {
      sh_output("\tq3map_surfacelight %s\n", shader->q3map_surfacelight.string);
   }
}



static void save_q3map_lightsubdivide(shader_t* shader)
{
   if (shader->flags & SH_Q3MAP_LIGHTSUBDIVIDE) {
      sh_output("\tq3map_lightsubdivide %s\n", shader->q3map_lightsubdivide.string);
   }
}



static void save_surfaceparm(shader_t* shader)
{
   _sh_surfaceparm_table_t* table;

   table = _sh_surfaceparm_table;

   while (table->name) {
      if (shader->surfaceparms & table->value) {
         sh_output("\tsurfaceparm %s\n", table->name);
      }

      table++;
   }
}



static void save_qer_editorimage(shader_t* shader)
{
   if (shader->flags & SH_QER_EDITORIMAGE) {
      sh_output("\tqer_editorimage %s\n", shader->qer_editorimage);
   }
}



static void save_qer_nocarve(shader_t* shader)
{
   if (shader->flags & SH_QER_NOCARVE) {
      sh_output("\tqer_nocarve\n");
   }
}



static void save_qer_trans(shader_t* shader)
{
   if (shader->flags & SH_QER_TRANS) {
      sh_output("\tqer_trans %s\n", shader->qer_trans.string);
   }
}



static void save_map(sh_stage_t* stage)
{
   if (stage->anim_map_count == 1) {
      if (stage->maps[0].flags & SH_MAP_CLAMPMAP) {
         sh_output("\t\tclampmap");
      }
      else {
         sh_output("\t\tmap");
      }

      sh_output(" %s\n", stage->maps[0].name);
   }
   else if (stage->anim_map_count > 1) {
      int i;

      sh_output("\t\tanimMap");

      sh_output(" %s", stage->anim_map_fps.string);

      for (i = 0; i < stage->anim_map_count; i++) {
         sh_output(" %s", stage->maps[i].name);
      }

      sh_output("\n");
   }
}



static void save_blendfunc(sh_stage_t* stage)
{
   if (stage->flags & SH_STAGE_BLENDFUNC) {
      sh_output("\t\tblendfunc ");
      
      if (stage->flags & SH_STAGE_EXPLICIT_BLENDFUNC) {
         sh_output("%s ", sh_string_from_blend(stage->blend_src));
         sh_output("%s",  sh_string_from_blend(stage->blend_dst));
      }
      else {
         if (sh_is_blend_filter(stage)) {
            sh_output("filter");
         }
         else if (sh_is_blend_blend(stage)) {
            sh_output("blend");
         }
         else if (sh_is_blend_add(stage)) {
            sh_output("add");
         }/*
         else {
            assert: this should never happen
         }*/
      }

      sh_output("\n");
   }
}



static void save_rgbagen(sh_rgbagen_t* rgbagen, bool is_rgbgen)
{
   sh_output("\t\t");

   if (is_rgbgen) {
      sh_output("rgbGen");
   }
   else {
      sh_output("alphaGen");
   }

   sh_output(" %s", sh_string_from_rgbagen(rgbagen->func));

   switch(rgbagen->func) {
      case SH_RGBAGEN_WAVE:
         sh_output(" %s", sh_string_from_waveform(rgbagen->args.wave.form));
         sh_output(" %s", rgbagen->args.wave.base.string);
         sh_output(" %s", rgbagen->args.wave.amp.string);
         sh_output(" %s", rgbagen->args.wave.phase.string);
         sh_output(" %s", rgbagen->args.wave.freq.string);
         break;

#ifdef SH_PHOENIX_QUAKE1_EXT
      case SH_RGBAGEN_CONST_EXT:
         if (is_rgbgen) {
             sh_output(" %s", rgbagen->args.constant.color[0]);
             sh_output(" %s", rgbagen->args.constant.color[1]);
             sh_output(" %s", rgbagen->args.constant.color[2]);
         }
         else {
             sh_output(" %s", rgbagen->args.constant.color[0]);
         }

         break;

      case SH_RGBAGEN_ONE_MINUS_PORTAL_EXT:
#endif

      case SH_RGBAGEN_IDENTITY_LIGHTING:
      case SH_RGBAGEN_IDENTITY:
      case SH_RGBAGEN_ENTITY:
      case SH_RGBAGEN_ONE_MINUS_ENTITY:
      case SH_RGBAGEN_VERTEX:
      case SH_RGBAGEN_ONE_MINUS_VERTEX:
      case SH_RGBAGEN_PORTAL:
      case SH_RGBAGEN_LIGHTING_DIFFUSE:
      case SH_RGBAGEN_LIGHTING_SPECULAR:
      default:
         break;
   }

   sh_output("\n");
}



static void save_rgbgen(sh_stage_t* stage)
{
   if (stage->flags & SH_STAGE_EXPLICIT_RGBGEN) {
      save_rgbagen(&stage->rgbgen, true);
   }
}



static void save_alphagen(sh_stage_t* stage)
{
   if (stage->flags & SH_STAGE_EXPLICIT_ALPHAGEN) {
      save_rgbagen(&stage->alphagen, false);
   }
}



static void save_tcgen(sh_stage_t* stage)
{
   if (stage->flags & SH_STAGE_EXPLICIT_TCGEN) {
      sh_output("\t\ttcGen %s", sh_string_from_tcgen(stage->tcgen.func));

      switch(stage->tcgen.func) {
         case SH_TCGEN_VECTOR:
            sh_output(" ( %s %s %s )", stage->tcgen.args.vector.sx.string, stage->tcgen.args.vector.sy.string, stage->tcgen.args.vector.sz.string);
            sh_output(" ( %s %s %s )", stage->tcgen.args.vector.tx.string, stage->tcgen.args.vector.ty.string, stage->tcgen.args.vector.tz.string);
            break;

#ifdef SH_PHOENIX_QUAKE1_EXT
         case SH_TCGEN_SKY_EXT:
            sh_output(" %s", stage->tcgen.args.sky.cloud_height.string);
            break;

         case SH_TCGEN_Q1TURB_EXT:
            sh_output(" %s", stage->tcgen.args.q1turb.div.string);
            sh_output(" %s", sh_string_from_waveform(stage->tcgen.args.q1turb.form));
            sh_output(" %s", stage->tcgen.args.q1turb.base.string);
            sh_output(" %s", stage->tcgen.args.q1turb.amp.string);
            sh_output(" %s", stage->tcgen.args.q1turb.phase.string);
            sh_output(" %s", stage->tcgen.args.q1turb.freq.string);
            break;
#endif

         case SH_TCGEN_BASE:
         case SH_TCGEN_LIGHTMAP:
         case SH_TCGEN_ENVIRONMENT:
         default:
            break;
      }

      sh_output("\n");
   }
}



static void save_tcmod(sh_tcmod_t* tcmod)
{
   sh_output("\t\ttcMod %s", sh_string_from_tcmod(tcmod->func));

   switch (tcmod->func) {
      case SH_TCMOD_ROTATE:
         sh_output(" %s", tcmod->args.rotate.speed.string);
         break;

      case SH_TCMOD_SCALE:
         sh_output(" %s", tcmod->args.scale.s.string);
         sh_output(" %s", tcmod->args.scale.t.string);
         break;

      case SH_TCMOD_SCROLL:
         sh_output(" %s", tcmod->args.scroll.s.string);
         sh_output(" %s", tcmod->args.scroll.t.string);
         break;

      case SH_TCMOD_STRETCH:
         sh_output(" %s", sh_string_from_waveform(tcmod->args.stretch.form));
         sh_output(" %s", tcmod->args.stretch.base.string);
         sh_output(" %s", tcmod->args.stretch.amp.string);
         sh_output(" %s", tcmod->args.stretch.phase.string);
         sh_output(" %s", tcmod->args.stretch.freq.string);
         break;

      case SH_TCMOD_TRANSFORM:
         sh_output(" %s", tcmod->args.transform.m00.string);
         sh_output(" %s", tcmod->args.transform.m01.string);
         sh_output(" %s", tcmod->args.transform.m10.string);
         sh_output(" %s", tcmod->args.transform.m11.string);
         sh_output(" %s", tcmod->args.transform.t0.string);
         sh_output(" %s", tcmod->args.transform.t1.string);
         break;

      case SH_TCMOD_TURB:
         sh_output(" %s", tcmod->args.turb.base.string);
         sh_output(" %s", tcmod->args.turb.amp.string);
         sh_output(" %s", tcmod->args.turb.phase.string);
         sh_output(" %s", tcmod->args.turb.freq.string);
         break;
   }

   sh_output("\n");
}



static void save_tcmods(sh_stage_t* stage)
{
   int i;

   for (i = 0; i < stage->tcmod_count; i++) {
      save_tcmod(&stage->tcmods[i]);
   }
}



static void save_depthfunc(sh_stage_t* stage)
{
   if (stage->flags & SH_STAGE_EXPLICIT_DEPTHFUNC) {
      sh_output("\t\tdepthFunc %s\n", sh_string_from_depthfunc(stage->depthfunc));
   }
}



static void save_depthwrite(sh_stage_t* stage)
{
   if (stage->flags & SH_STAGE_EXPLICIT_DEPTHWRITE) {
      sh_output("\t\tdepthWrite\n");
   }
}



static void save_detail(sh_stage_t* stage)
{
   if (stage->flags & SH_STAGE_DETAIL) {
      sh_output("\t\tdetail\n");
   }
}



static void save_alphafunc(sh_stage_t* stage)
{
   if (stage->flags & SH_STAGE_ALPHAFUNC) {
      sh_output("\t\talphaFunc %s\n", sh_string_from_alphafunc(stage->alphafunc));
   }
}



static void save_stage(sh_stage_t* stage)
{
   sh_output("\t{\n");
   save_map(stage);
   save_blendfunc(stage);
   save_rgbgen(stage);
   save_alphagen(stage);
   save_tcgen(stage);
   save_tcmods(stage);
   save_depthfunc(stage);
   save_depthwrite(stage);
   save_detail(stage);
   save_alphafunc(stage);
   sh_output("\t}\n");
}



static void save_stages(shader_t* shader)
{
   int i;

   for (i = 0; i < shader->stage_count; i++) {
      if (i != 0) sh_output("\n");

      save_stage(&shader->stages[i]);
   }
}



void sh_save(shader_t* shader)
{
   sh_output("%s\n", shader->name);
   sh_output("{\n");
   save_skyparms(shader);
   save_cull(shader);
   save_vxmods(shader);
   save_fogparms(shader);
   save_nopicmip(shader);
   save_nomipmaps(shader);
   save_polygon_offset(shader);
   save_sort(shader);
   save_entity_mergable(shader);
   save_tess_size(shader);
   save_q3map_backshader(shader);
   save_q3map_globaltexture(shader);
   save_q3map_sun(shader);
   save_q3map_surfacelight(shader);
   save_q3map_lightsubdivide(shader);
   save_surfaceparm(shader);
   save_qer_editorimage(shader);
   save_qer_nocarve(shader);
   save_qer_trans(shader);

   if (shader->stage_count > 0) sh_output("\n");

   save_stages(shader);
   sh_output("}\n\n");
}


