/*
Copyright (C) 2000 Jason Wilkins

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include "shaders.h"
#include "sh_internal.h"



static void report_skyparms(shader_t* shader)
{
   if ((shader->flags & SH_FAR_BOX) ||
       (shader->flags & SH_SKYGEN)  ||
       (shader->flags & SH_NEAR_BOX)) {
      sh_output("\nskyparms:\n\n");

      sh_output("\tfar box basename: ");

      if (shader->flags & SH_FAR_BOX) {
         sh_output("%s\n", shader->far_box_basename);
      }
      else {
         sh_output("-\n");
      }

      if (shader->flags & SH_SKYGEN) {
         sh_output("\tcloudheight: %s\n", shader->cloud_height.string);
      }
      else {
         sh_output("no cloud layer\n");
      }

      sh_output("\tnear box basename: ");

      if (shader->flags & SH_NEAR_BOX) {
         sh_output("%s\n", shader->near_box_basename);
      }
      else {
         sh_output("-\n");
      }
   }
   else {
      sh_output("\nskyparms: <none>\n");
   }
}



static void report_cull(shader_t* shader)
{
   sh_output("\ncull: %s\n", sh_string_from_cull(shader->cull_face));
}



static void report_vxmod(sh_vxmod_t* vxmod)
{
   sh_output("%s\n", sh_string_from_vxmod(vxmod->func));

   switch(vxmod->func) {
      case SH_VXMOD_WAVE:
         sh_output("\n");
         sh_output("\t\tdiv:       %s\n", vxmod->args.wave.div.string);
         sh_output("\t\twaveform:  %s\n", sh_string_from_waveform(vxmod->args.wave.form));
         sh_output("\t\tbase:      %s\n", vxmod->args.wave.base.string);
         sh_output("\t\tamplitude: %s\n", vxmod->args.wave.amp.string);
         sh_output("\t\tphase:     %s\n", vxmod->args.wave.phase.string);
         sh_output("\t\tfrequency: %s\n", vxmod->args.wave.freq.string);
         break;

      case SH_VXMOD_NORMAL:
         sh_output("\n");
         sh_output("\t\tamplitude: %s\n", vxmod->args.normal.amp.string);
         sh_output("\t\tfrequency: %s\n", vxmod->args.normal.freq.string);
         break;

      case SH_VXMOD_BULGE:
         sh_output("\n");
         sh_output("\t\twidth:     %s\n", vxmod->args.bulge.width.string);
         sh_output("\t\theight:    %s\n", vxmod->args.bulge.height.string);
         sh_output("\t\tspeed:     %s\n", vxmod->args.bulge.speed.string);
         break;

      case SH_VXMOD_MOVE:
         sh_output("\n");
         sh_output("\t\tvector:    <%s, %s, %s>\n", vxmod->args.move.x.string, vxmod->args.move.y.string, vxmod->args.move.z.string);
         sh_output("\t\twaveform:  %s\n", sh_string_from_waveform(vxmod->args.move.form));
         sh_output("\t\tbase:      %s\n", vxmod->args.move.base.string);
         sh_output("\t\tamplitude: %s\n", vxmod->args.move.amp.string);
         sh_output("\t\tphase:     %s\n", vxmod->args.move.phase.string);
         sh_output("\t\tfrequency: %s\n", vxmod->args.move.freq.string);
         break;

      case SH_VXMOD_AUTOSPRITE:
      case SH_VXMOD_AUTOSPRITE2:
      case SH_VXMOD_PROJECTION_SHADOW:
      case SH_VXMOD_TEXT0:
      case SH_VXMOD_TEXT1:

#ifdef SH_PHOENIX_QUAKE1_EXT
      case SH_VXMOD_STENCIL_SHADOW_EXT:
#endif

         break;

      default:
         break;
   }
}



static void report_vxmods(shader_t* shader)
{
   int i;

   if (shader->vxmod_count > 0) {
      sh_output("\ndeformVertexes:\n\n");
      sh_output("\tcount: %d\n", shader->vxmod_count);

      for (i = 0; i < shader->vxmod_count; i++) {
         sh_output("\n");
         sh_output("\tdeformVertexes #%d: ", i+1);
         report_vxmod(&shader->vxmods[i]);
      }
   }
   else {
      sh_output("\ndeformVertexes: <none>\n");
   }
}



static void report_fogparms(shader_t* shader)
{
   if (shader->flags & SH_FOGPARMS) {
      sh_output("\nfogparms:\n");
      sh_output("\tcolor:              <%s, %s, %s>\n", shader->fog_color[0].string, shader->fog_color[1].string, shader->fog_color[2].string);
      sh_output("\tdistance to opaque: %s\n", shader->fog_distance_to_opaque.string);
   }
   else {
      sh_output("\nfogparms: <none>\n");
   }
}



static void report_nopicmip(shader_t* shader)
{
   sh_output("\nnopicmip: ");

   if (shader->flags & SH_NOPICMIP) {
      sh_output("true");
   }
   else {
      sh_output("false");
   }

   sh_output("\n");
}



static void report_nomipmaps(shader_t* shader)
{
   sh_output("\nnomipmaps: ");

   if (shader->flags & SH_NOMIPMAPS) {
      sh_output("true");
   }
   else {
      sh_output("false");
   }

   sh_output("\n");
}



static void report_polygon_offset(shader_t* shader)
{
   sh_output("polygon offset: ");

   if (shader->flags & SH_POLYGON_OFFSET) {
      sh_output("true");
   }
   else {
      sh_output("false");
   }

   sh_output("\n");
}



static void report_sort(shader_t* shader)
{
   sh_output("\nsort priority: %s\n", sh_string_from_sort(shader->sort_priority));
}



static void report_entity_mergable(shader_t* shader)
{
   sh_output("\nentityMergable: ");

   if (shader->flags & SH_ENTITY_MERGABLE) {
      sh_output("true");
   }
   else {
      sh_output("false");
   }

   sh_output("\n");
}


static void report_tess_size(shader_t* shader)
{
   sh_output("\ntessSize: ");

   if (shader->flags & SH_TESS_SIZE) {
      sh_output("%s\n", shader->tess_size.string);
   }
   else {
      sh_output("<none>\n");
   }
}



static void report_q3map_backshader(shader_t* shader)
{
   sh_output("\nq3map_backshader: ");

   if (shader->flags & SH_TESS_SIZE) {
      sh_output("%s\n", shader->q3map_backshader);
   }
   else {
      sh_output("<none>\n");
   }
}



static void report_q3map_globaltexture(shader_t* shader)
{
   sh_output("\n\t\tq3map_globaltexture: ");

   if (shader->flags & SH_Q3MAP_GLOBALTEXTURE) {
      sh_output("true");
   }
   else {
      sh_output("false");
   }

   sh_output("\n");
}



static void report_q3map_sun(shader_t* shader)
{
   sh_output("\n\t\tq3map_sun: ");

   if (shader->flags & SH_Q3MAP_SUN) {
      sh_output("\t\t\tcolor:     <%s, %s, %s>\n", shader->q3map_sun.red, shader->q3map_sun.green, shader->q3map_sun.blue);
      sh_output("\t\t\tintensity: %s\n", shader->q3map_sun.intensity);
      sh_output("\t\t\tdegrees:   %s\n", shader->q3map_sun.degrees);
      sh_output("\t\t\televation: %s\n", shader->q3map_sun.elevation);
   }
   else {
      sh_output("<none>");
   }
   
   sh_output("\n");
}



static void report_q3map_surfacelight(shader_t* shader)
{
   sh_output("\nq3map_surfacelight: %s\n", shader->q3map_surfacelight.string);
}



static void report_q3map_lightsubdivide(shader_t* shader)
{
   sh_output("\nq3map_lightsubdivide: ");

   if (shader->flags & SH_Q3MAP_LIGHTSUBDIVIDE) {
      sh_output("%s\n", shader->q3map_lightsubdivide.string);
   }
   else {
      sh_output("<none>\n");
   }
}



static void report_surfaceparm(shader_t* shader)
{
   _sh_surfaceparm_table_t* table;
   bool found;

   found = false;
   table = _sh_surfaceparm_table;

   sh_output("surfaceparms: ");

   while (table->name) {
      if (shader->surfaceparms & table->value) {
         sh_output("\t%s\n", table->name);
         found = true;
      }

      table++;
   }

   if (!found) {
      sh_output("<none>\n");
   }
}



static void report_qer_editorimage(shader_t* shader)
{
   sh_output("\nqer_editorimage: ");

   if (shader->flags & SH_QER_EDITORIMAGE) {
      sh_output("%s\n", shader->qer_editorimage);
   }
   else {
      sh_output("<none>\n");
   }
}



static void report_qer_nocarve(shader_t* shader)
{
   sh_output("\n\t\tqer_nocarve: ");

   if (shader->flags & SH_QER_NOCARVE) {
      sh_output("true");
   }
   else {
      sh_output("false");
   }

   sh_output("\n");
}



static void report_qer_trans(shader_t* shader)
{
   sh_output("\n\t\tqer_trans: ");

   if (shader->flags & SH_QER_TRANS) {
      sh_output("%s\n", shader->qer_trans.string);
   }
   else {
      sh_output("<none>");
   }

   sh_output("\n");
}



static void report_map(sh_stage_t* stage)
{
   if (stage->anim_map_count == 1) {
      if (stage->maps[0].flags & SH_MAP_CLAMPMAP) {
         sh_output("\t\tclampmap: ");
      }
      else {
         sh_output("\t\tmap: ");
      }

      sh_output("%s\n", stage->maps[9].name);
   }
   else if (stage->anim_map_count > 1) {
      int i;

      sh_output("\t\tanimMap:\n\n");

      sh_output("\t\t\tfps:   %s\n", stage->anim_map_fps.string);
      sh_output("\t\t\tcount: %d\n\n", stage->anim_map_count);

      for (i = 0; i < stage->anim_map_count; i++) {
         sh_output("\t\t\tframe %d: %s\n", i+1, stage->maps[i].name);
      }
   }
   else {
      sh_output("\n\t\tmap: <none>\n");
   }
}



static void report_blendfunc(sh_stage_t* stage)
{
   if (stage->flags & SH_STAGE_BLENDFUNC) {
      sh_output("\n\t\tblendfunc: ");
      
      if (sh_is_blend_filter(stage)) {
         sh_output("filter\n");
      }
      else if (sh_is_blend_blend(stage)) {
         sh_output("blend\n");
      }
      else if (sh_is_blend_add(stage)) {
         sh_output("add\n");
      }
      else {
         sh_output("\n\n");
         sh_output("\t\t\tsource factor:      %s\n", sh_string_from_blend(stage->blend_src));
         sh_output("\t\t\tdestination factor: %s\n", sh_string_from_blend(stage->blend_dst));
      }
   }
   else {
      sh_output("\n\t\tblendFunc: <none>\n");
   }
}



static void report_rgbagen(sh_rgbagen_t* rgbagen, bool is_rgbgen)
{
   sh_output("%s\n", sh_string_from_rgbagen(rgbagen->func));

   switch(rgbagen->func) {
      case SH_RGBAGEN_WAVE:
         sh_output("\n");
         sh_output("\t\t\twaveform:  %s\n", sh_string_from_waveform(rgbagen->args.wave.form));
         sh_output("\t\t\tbase:      %s\n", rgbagen->args.wave.base.string);
         sh_output("\t\t\tamplitude: %s\n", rgbagen->args.wave.amp.string);
         sh_output("\t\t\tphase:     %s\n", rgbagen->args.wave.phase.string);
         sh_output("\t\t\tfrequency: %s\n", rgbagen->args.wave.freq.string);
         break;

#ifdef SH_PHOENIX_QUAKE1_EXT
      case SH_RGBAGEN_CONST_EXT:
         sh_output("\n\t\t\t");

         if (is_rgbgen) {
             sh_output("color: <%s, %s, %s>\n", rgbagen->args.constant.color[0], rgbagen->args.constant.color[1], rgbagen->args.constant.color[2]);
         }
         else {
             sh_output("alpha: %s\n", rgbagen->args.constant.color[0]);
         }

         break;

      case SH_RGBAGEN_ONE_MINUS_PORTAL_EXT:
#else
      (void)is_rgbgen;
#endif

      case SH_RGBAGEN_IDENTITY_LIGHTING:
      case SH_RGBAGEN_IDENTITY:
      case SH_RGBAGEN_ENTITY:
      case SH_RGBAGEN_ONE_MINUS_ENTITY:
      case SH_RGBAGEN_VERTEX:
      case SH_RGBAGEN_ONE_MINUS_VERTEX:
      case SH_RGBAGEN_PORTAL:
      case SH_RGBAGEN_LIGHTING_DIFFUSE:
      case SH_RGBAGEN_LIGHTING_SPECULAR:
      default:
         break;
   }
}



static void report_rgbgen(sh_stage_t* stage)
{
   sh_output("\n\t\trgbGen: ");
   report_rgbagen(&stage->rgbgen, true);
}



static void report_alphagen(sh_stage_t* stage)
{
   sh_output("\n\t\talphaGen: ");
   report_rgbagen(&stage->alphagen, false);
}



static void report_tcgen(sh_stage_t* stage)
{
   sh_output("\n\t\ttcGen: %s\n", sh_string_from_tcgen(stage->tcgen.func));

   switch(stage->tcgen.func) {
      case SH_TCGEN_VECTOR:
         sh_output("\n");
         sh_output("\t\t\ts vector:  <%s, %s, %s>\n", stage->tcgen.args.vector.sx.string, stage->tcgen.args.vector.sy.string, stage->tcgen.args.vector.sz.string);
         sh_output("\t\t\tt vector:  <%s, %s, %s>\n", stage->tcgen.args.vector.tx.string, stage->tcgen.args.vector.ty.string, stage->tcgen.args.vector.tz.string);
         break;

#ifdef SH_PHOENIX_QUAKE1_EXT
      case SH_TCGEN_SKY_EXT:
         sh_output("\n");
         sh_output("\t\t\tcloudheight:  %s\n", stage->tcgen.args.sky.cloud_height.string);
         break;

      case SH_TCGEN_Q1TURB_EXT:
         sh_output("\n");
         sh_output("\t\t\tdiv:       %s\n", stage->tcgen.args.q1turb.div.string);
         sh_output("\t\t\twaveform:  %s\n", sh_waveform(stage->tcgen.args.q1turb.form));
         sh_output("\t\t\tbase:      %s\n", stage->tcgen.args.q1turb.base.string);
         sh_output("\t\t\tamplitude: %s\n", stage->tcgen.args.q1turb.amp.string);
         sh_output("\t\t\tphase:     %s\n", stage->tcgen.args.q1turb.phase.string);
         sh_output("\t\t\tfrequency: %s\n", stage->tcgen.args.q1turb.freq.string);
         break;
#endif

      case SH_TCGEN_BASE:
      case SH_TCGEN_LIGHTMAP:
      case SH_TCGEN_ENVIRONMENT:
      default:
         break;
   }
}



static void report_tcmod(sh_tcmod_t* tcmod)
{
   sh_output("%s\n", sh_string_from_tcmod(tcmod->func));

   switch (tcmod->func) {
      case SH_TCMOD_ROTATE:
         sh_output("\n");
         sh_output("\t\t\t\tspeed: %s\n", tcmod->args.rotate.speed.string);
         break;

      case SH_TCMOD_SCALE:
         sh_output("\n");
         sh_output("\t\t\t\ts: %s\n", tcmod->args.scale.s.string);
         sh_output("\t\t\t\tt: %s\n", tcmod->args.scale.t.string);
         break;

      case SH_TCMOD_SCROLL:
         sh_output("\n");
         sh_output("\t\t\t\ts: %s\n", tcmod->args.scroll.s.string);
         sh_output("\t\t\t\tt: %s\n", tcmod->args.scroll.t.string);
         break;

      case SH_TCMOD_STRETCH:
         sh_output("\n");
         sh_output("\t\t\t\twaveform:  %s\n", sh_string_from_waveform(tcmod->args.stretch.form));
         sh_output("\t\t\t\tbase:      %s\n", tcmod->args.stretch.base.string);
         sh_output("\t\t\t\tamplitude: %s\n", tcmod->args.stretch.amp.string);
         sh_output("\t\t\t\tphase:     %s\n", tcmod->args.stretch.phase.string);
         sh_output("\t\t\t\tfrequency: %s\n", tcmod->args.stretch.freq.string);
         break;

      case SH_TCMOD_TRANSFORM:
         sh_output("\n");
         sh_output("\t\t\t\tm00: %s\n", tcmod->args.transform.m00.string);
         sh_output("\t\t\t\tm01: %s\n", tcmod->args.transform.m01.string);
         sh_output("\t\t\t\tm10: %s\n", tcmod->args.transform.m10.string);
         sh_output("\t\t\t\tm11: %s\n", tcmod->args.transform.m11.string);
         sh_output("\t\t\t\tt0:  %s\n", tcmod->args.transform.t0.string);
         sh_output("\t\t\t\tt1:  %s\n", tcmod->args.transform.t1.string);
         break;

      case SH_TCMOD_TURB:
         sh_output("\n");
         sh_output("\t\t\t\tbase:      %s\n", tcmod->args.turb.base.string);
         sh_output("\t\t\t\tamplitude: %s\n", tcmod->args.turb.amp.string);
         sh_output("\t\t\t\tphase:     %s\n", tcmod->args.turb.phase.string);
         sh_output("\t\t\t\tfrequency: %s\n", tcmod->args.turb.freq.string);
         break;
   }
}



static void report_tcmods(sh_stage_t* stage)
{
   if (stage->tcmod_count > 0) {
      int i;

      sh_output("\n\t\ttcMods:\n\n");
      sh_output("\t\t\tcount: %d\n", stage->tcmod_count);

      for (i = 0; i < stage->tcmod_count; i++) {
         sh_output("\n");
         sh_output("\t\t\ttcMod #%d: ", i+1);
         report_tcmod(&stage->tcmods[i]);
      }
   }
   else {
      sh_output("\n\t\ttcMods: <none>\n");
   }
}



static void report_depthfunc(sh_stage_t* stage)
{
   sh_output("\n\t\tdepthfunc: %s\n", sh_string_from_depthfunc(stage->depthfunc));
}



static void report_depthwrite(sh_stage_t* stage)
{
   sh_output("\n\t\tdepthWrite: ");

   if (stage->flags & SH_STAGE_DEPTHWRITE) {
      sh_output("true");
   }
   else {
      sh_output("false");
   }

   sh_output("\n");
}



static void report_detail(sh_stage_t* stage)
{
   sh_output("\n\t\tdetail: ");

   if (stage->flags & SH_STAGE_DETAIL) {
      sh_output("true");
   }
   else {
      sh_output("false");
   }

   sh_output("\n");
}



static void report_alphafunc(sh_stage_t* stage)
{
   sh_output("\n\t\talphafunc: ");

   if (stage->flags & SH_STAGE_ALPHAFUNC) {
      sh_output("%s\n", sh_string_from_alphafunc(stage->alphafunc));
   }
   else {
      sh_output("<none>\n");
   }
}



static void report_stage(sh_stage_t* stage)
{
   report_map(stage);
   report_blendfunc(stage);
   report_rgbgen(stage);
   report_alphagen(stage);
   report_tcgen(stage);
   report_tcmods(stage);
   report_depthfunc(stage);
   report_depthwrite(stage);
   report_detail(stage);
   report_alphafunc(stage);
}



static void report_stages(shader_t* shader)
{
   if (shader->stage_count > 0) {
      int i;

      sh_output("\nstages:\n\n");
      sh_output("\tcount: %d\n", shader->stage_count);

      for (i = 0; i < shader->stage_count; i++) {
         sh_output("\n");
         sh_output("\tstage #%d: \n\n", i+1);
         report_stage(&shader->stages[i]);
      }
   }
   else {
      sh_output("\nno stages\n");
   }
}



void sh_report(shader_t* shader)
{
   sh_output("------------------------\n");
   sh_output("name: %s\n", shader->name);
   report_skyparms(shader);
   report_cull(shader);
   report_vxmods(shader);
   report_fogparms(shader);
   report_nopicmip(shader);
   report_nomipmaps(shader);
   report_polygon_offset(shader);
   report_sort(shader);
   report_entity_mergable(shader);
   report_tess_size(shader);
   report_q3map_backshader(shader);
   report_q3map_globaltexture(shader);
   report_q3map_sun(shader);
   report_q3map_surfacelight(shader);
   report_q3map_lightsubdivide(shader);
   report_surfaceparm(shader);
   report_qer_editorimage(shader);
   report_qer_nocarve(shader);
   report_qer_trans(shader);
   report_stages(shader);
   sh_output("------------------------\n");
}


