/*
Copyright (C) 2000 Jason Wilkins

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include "shaders.h"
#include "sh_internal.h"



_sh_token_t _sh_token;

int _sh_line = 1;

static int brace_count     = 0;
static int next_line_count = 1;

static bool uninputted  = false;
static int  uninput_cin = '\0';

static sh_input_func_t input_func = sh_input_file_func;



/* sh_input:
 ***************************************************************************/
int sh_input()
{
   int cin;

   if (uninputted) {
      cin = uninput_cin;
      uninputted = false;
   }
   else {
      cin = input_func();
   }

   if (cin == '\n') {
      next_line_count++;
   }

   return cin;
}



/* sh_input_unget:
 ***************************************************************************/
void sh_input_unget(int cin)
{
   uninput_cin = cin;
   uninputted  = true;
}



/* sh_input_set_func:
 ***************************************************************************/
sh_input_func_t sh_input_set_func(sh_input_func_t i)
{
   sh_input_func_t o;
   
   o = input_func;
   input_func = i;

   return o;
}



/***************************************************************************\
 string input
\***************************************************************************/

static const char* curpos;
static const char* endpos;



/* sh_input_string_func:
 ***************************************************************************/
int sh_input_string_func(void)
{
   if (*curpos != '\0') {
      return *curpos++;
   }
   else {
      return EOF;
   }
}



/* sh_input_set_string:
 ***************************************************************************/
void sh_input_set_string(const char* string)
{
   curpos = string;
}



/* sh_input_load_string:
 ***************************************************************************/
bool sh_input_load_string(const char* filename, char* text, size_t* size)
{
   FILE* f;

   f = fopen(filename, "rb");

   if (!f) {
      sh_message(SH_MSG_ERROR, "error opening '%s'\n", filename);
      return false;
   }

   if (text) {
      size_t num_read;
      size_t filesize = *size - 1;


      num_read = fread(text, filesize, 1, f);

      if (num_read != 1) {
         sh_message(SH_MSG_ERROR, "read error loading '%s'\n", filename);
         fclose(f);
         return false;
      }

      text[filesize] = '\0';

      sh_input_set_string(text);
   }
   else {
      fseek(f, 0, SEEK_END);
      *size = ftell(f) + 1;
   }

   fclose(f);

   return true;
}



/***************************************************************************\
 standard file input
\***************************************************************************/

static FILE* _input_file = stdin;



/* sh_input_file_func:
 ***************************************************************************/
int sh_input_file_func(void)
{
   return getc(_input_file);
}



/* sh_input_set_file:
 ***************************************************************************/
FILE* sh_input_set_file(FILE* f)
{
   FILE* o;
   
   o = _input_file;
   _input_file = f;

   return o;
}



/***************************************************************************\

scanner

\***************************************************************************/

/* _sh_begin_scan:
 ***************************************************************************/
void _sh_begin_scan(void)
{
   brace_count = 0;
   _sh_line    = 1;
   next_line_count = 1;
}



/* _sh_end_scan:
 ***************************************************************************/
void _sh_end_scan(void)
{
   brace_count = 0;
   _sh_line    = 0;
   next_line_count  = 0;
}



/* end_of_input
 ***************************************************************************/
static void end_of_input(void)
{
   strncpy(_sh_token.string, "<EOF>", SH_TOKEN_LEN);

   if (brace_count > 0) {
      sh_message(SH_MSG_ERROR, "%d extra opening brace%sdetected in file",
         (brace_count > 1)?("s "):(" "),
         brace_count);
   }
   else if (brace_count < 0) {
      sh_message(SH_MSG_ERROR, "%d extra closing brace%sdetected in file",
         (brace_count < -1)?("s "):(" "),
         -brace_count);
   }
}


/* _sh_scan:
 ***************************************************************************/
_SH_TOKEN _sh_scan()
{
   int cin;

   _sh_token.string[0]  = '\0';
   _sh_token.is_newline = false;

   _sh_line = next_line_count;

   for (;;) {
      do {
         cin = sh_input();

         if (cin == '\n') _sh_token.is_newline = true;

      } while (isspace(cin));

      if (cin == '/') {
         cin = sh_input();

         if (cin == '/') {
            do {
               cin = sh_input();
            } while ((cin != '\n') && (cin != EOF));

            if (cin == '\n') _sh_token.is_newline = true;

            if (cin == EOF) {
               end_of_input();
               return _SH_TOK_EOF;
            }
         }
         else {
            sh_input_unget(cin);
            break;
         }
      }
      else {
         break;
      }
   }

   if (cin != EOF) {
      int pos = 0;

      do {
         _sh_token.string[pos] = (char)cin;
         cin = sh_input();

         // If the token is too big then it will be truncated to a size
         // one larger than is allowable.  The overflow can then be
         // detected and reported by the consumer of the token.
         if (pos < SH_TOKEN_LEN) pos++;
      } while (!isspace(cin) && (cin != EOF));

      sh_input_unget(cin);

      _sh_token.string[pos] = '\0';

      if (strcmp("{", _sh_token.string) == 0) {
         brace_count++;

         if (brace_count > 2) sh_message(SH_MSG_ERROR, "'{' (opening braces) should never be nested more than 2 deep");

         return '{';
      }
      else if (strcmp("}", _sh_token.string) == 0) {
         brace_count--;

         if (brace_count < 0) sh_message(SH_MSG_ERROR, "extra '}' (closing brace) detected");

         return '}';
      }
      else {
         return _SH_TOK_STRING;
      }
   }
   else {
      end_of_input();
      return _SH_TOK_EOF;
   }
}



/* _sh_skip
 ***************************************************************************/
bool _sh_skip()
{
   while (brace_count == 0) {
      if (_sh_scan() == _SH_TOK_EOF) return true;
   }

   while (brace_count != 0) {
      if (_sh_scan() == _SH_TOK_EOF) return true;
   }

   return true;
}


