#ifndef SHADERS_INCLUDED
#define SHADERS_INCLUDED

/*
Copyright (C) 2000 Jason Wilkins

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include <stdlib.h>
#include <math.h>
#include <stdarg.h>
#include <string.h>
#include <stdio.h>
#include <ctype.h>



/***************************************************************************\

basic definitions

\***************************************************************************/

#ifndef __cplusplus
  #define true  1
  #define false 0
  typedef int bool;
#endif



/***************************************************************************\

shader structure

\***************************************************************************/

// limits
#define SH_NAME_LEN           64 // shader and map names can be up to 64 characters
#define SH_TOKEN_LEN          (SH_NAME_LEN+1)
#define SH_NUMBER_LEN         11 
#define SH_MAX_STAGES          8 // there can be up to 8 cloud layers
#define SH_MAX_TCMODS          4 // this is just a guess
#define SH_MAX_VXMODS          4 // this is just a guess

#ifdef SH_PHOENIX_QUAKE1_EXT
   #define SH_MAX_ANIMMAP_FRAMES 10
#else
   #define SH_MAX_ANIMMAP_FRAMES  8
#endif

#define SH_MAX_SHADER_ARGS    12 // tcMod vector has 11 arguments + the command

#define _SH_FLAG(x) ((unsigned)1 << (x))

// global flags
#define SH_SKYGEN            _SH_FLAG( 0)
#define SH_NEAR_BOX          _SH_FLAG( 1)
#define SH_FAR_BOX           _SH_FLAG( 2)
#define SH_CULL              _SH_FLAG( 3)
#define SH_FOGPARMS          _SH_FLAG( 4)
#define SH_NOPICMIP          _SH_FLAG( 5)
#define SH_NOMIPMAPS         _SH_FLAG( 6)
#define SH_POLYGON_OFFSET    _SH_FLAG( 7)
#define SH_ENTITY_MERGABLE   _SH_FLAG( 8)
#define SH_EXPLICIT_SKYGEN   _SH_FLAG( 9)
#define SH_EXPLICIT_CULL     _SH_FLAG(10)
#define SH_EXPLICIT_NOPICMIP _SH_FLAG(11)
#define SH_EXPLICIT_PORTAL   _SH_FLAG(12)
#define SH_EXPLICIT_SORT     _SH_FLAG(13)

#ifdef SH_PHOENIX_QUAKE1_EXT
#define SH_CACHEPIC_EXT      _SH_FLAG(14)
#endif

// stage flags
#define SH_STAGE_MAP_NEEDS_ALPHA     _SH_FLAG( 0)
#define SH_STAGE_BLENDFUNC           _SH_FLAG( 1)
#define SH_STAGE_DEPTHWRITE          _SH_FLAG( 2)
#define SH_STAGE_DETAIL              _SH_FLAG( 3)
#define SH_STAGE_ALPHAFUNC           _SH_FLAG( 4)
#define SH_STAGE_EXPLICIT_BLENDFUNC  _SH_FLAG( 5)
#define SH_STAGE_EXPLICIT_TCGEN      _SH_FLAG( 6)
#define SH_STAGE_EXPLICIT_RGBGEN     _SH_FLAG( 7)
#define SH_STAGE_EXPLICIT_ALPHAGEN   _SH_FLAG( 8)
#define SH_STAGE_EXPLICIT_DEPTHFUNC  _SH_FLAG( 9)
#define SH_STAGE_EXPLICIT_DEPTHWRITE _SH_FLAG(10)

#ifdef SH_PHOENIX_QUAKE1_EXT
#define SH_STAGE_FILTER_EXT          _SH_FLAG(11)
#endif

// map flags
#define SH_MAP_CLAMPMAP         _SH_FLAG(0)
#define SH_MAP_ALPHA            _SH_FLAG(1)
#define SH_MAP_PICMIP           _SH_FLAG(2)
#define SH_MAP_MIPMAPS          _SH_FLAG(3)

#ifdef SH_PHOENIX_QUAKE1_EXT
#define SH_MAP_Q1_SOLID_SKY_EXT _SH_FLAG(4)
#define SH_MAP_Q1_ALPHA_SKY_EXT _SH_FLAG(5)
#endif

// number stored with original string
typedef struct sh_num_s {
   float value;
   char  string[SH_NUMBER_LEN+1];
} sh_num_t;

// wave
enum {
   SH_WAVE_SIN,
   SH_WAVE_TRIANGLE,
   SH_WAVE_SQUARE,
   SH_WAVE_SAWTOOTH,
   SH_WAVE_INVSAWTOOTH,
   SH_WAVE_NOISE
};

// cull
enum {
   SH_CULL_FRONT,
   SH_CULL_BACK,
   SH_CULL_DISABLE,

#ifdef SH_PHOENIX_QUAKE1_EXT
   SH_CULL_BOTH_EXT,
#endif
};

// deformVertexes
enum {
   SH_VXMOD_WAVE,
   SH_VXMOD_NORMAL,
   SH_VXMOD_BULGE,
   SH_VXMOD_MOVE,
   SH_VXMOD_AUTOSPRITE,
   SH_VXMOD_AUTOSPRITE2,
   SH_VXMOD_PROJECTION_SHADOW,
   SH_VXMOD_TEXT0,
   SH_VXMOD_TEXT1,

#ifdef SH_PHOENIX_QUAKE1_EXT
   SH_VXMOD_STENCIL_SHADOW_EXT,
#endif
};

typedef struct sh_vxmod_s {
   int func;

   union {
      // deformVertexes wave
      struct {
         sh_num_t div;
         int      form;
         sh_num_t base;
         sh_num_t amp;
         sh_num_t phase;
         sh_num_t freq;
      } wave;

      // deformVertexes normal
      struct {
         sh_num_t amp;
         sh_num_t freq;
      } normal;

      // deformVertexes bulge
      struct {
         sh_num_t width, height;
         sh_num_t speed;
      } bulge;

      // deformVertexes move
      struct {
         sh_num_t x, y, z;
         int      form;
         sh_num_t base;
         sh_num_t amp;
         sh_num_t phase;
         sh_num_t freq;
      } move;
   } args;
} sh_vxmod_t;

// sort
enum {
   SH_SORT_UNDEFINED      = 0,
   SH_SORT_PORTAL         = 1,
   SH_SORT_SKY            = 2,  // default for skyParms
   SH_SORT_OPAQUE         = 3,  // default with no blendFunc or alphaFunc
   SH_SORT_DECAL          = 4,  // default for polygonOffset (undocumented)
   SH_SORT_DECAL2         = 5,
   SH_SORT_BANNER         = 6,
   SH_SORT_SEE_THROUGH    = 7,  // default for alphaFunc (undocumented)
   SH_SORT_UNDERWATER     = 8,
   SH_SORT_ADDITIVE       = 9,  // default for blendFunc
   SH_SORT_ADDITIVE1      = 10,
   SH_SORT_ADDITIVE2      = 11,
   SH_SORT_FOG            = 12, // (undocumented)
   SH_SORT_FOG1           = 13,
   SH_SORT_FOG2           = 14,
   SH_SORT_STENCIL_SHADOW = 15, // (undocumented)
   SH_SORT_NEAREST        = 16,
   SH_SORT_NUM_SORTS
};

// tcGen
enum {
   SH_TCGEN_BASE,
   SH_TCGEN_LIGHTMAP,
   SH_TCGEN_ENVIRONMENT,
   SH_TCGEN_VECTOR,

#ifdef SH_PHOENIX_QUAKE1_EXT
   SH_TCGEN_SKY_EXT,
   SH_TCGEN_Q1TURB_EXT,
#endif
};

typedef struct sh_tcgen_s {
   int func;

   union {
      // tcGen vector
      struct {
         sh_num_t sx, sy, sz;
         sh_num_t tx, ty, tz;
      } vector;

#ifdef SH_PHOENIX_QUAKE1_EXT
      // tcGen _sky
      struct {
         sh_num_t cloud_height;
      } sky;

      // tcGen _q1turb
      struct {
         sh_num_t div;
         int      form;
         sh_num_t base;
         sh_num_t amp;
         sh_num_t phase;
         sh_num_t freq;
      } q1turb;
#endif
   } args;
} sh_tcgen_t;

// tcMod
enum {
   SH_TCMOD_ROTATE,
   SH_TCMOD_SCALE,
   SH_TCMOD_SCROLL,
   SH_TCMOD_STRETCH,
   SH_TCMOD_TRANSFORM,
   SH_TCMOD_TURB,
};

typedef struct sh_matrix3x2_s {
   sh_num_t m00, m01, t0;
   sh_num_t m10, m11, t1;
} sh_matrix3x2_t;

typedef struct sh_tcmod_s {
   int func;

   union {
      // tcMod rotate
      struct {
         sh_num_t speed;
      } rotate;

      // tcMod scale
      struct {
         sh_num_t s, t;
      } scale;

      // tcMod scroll
      struct {
         sh_num_t s, t;
      } scroll;
    
      // tcMod stretch
      struct {
         int      form;
         sh_num_t base;
         sh_num_t amp;
         sh_num_t phase;
         sh_num_t freq;
      } stretch;

      // tcMod transform
      sh_matrix3x2_t transform;

      // tcMod turb
      struct {
         sh_num_t base;
         sh_num_t amp;
         sh_num_t phase;
         sh_num_t freq;
      } turb;
    } args;
} sh_tcmod_t;

// rgbGen and alphaGen
enum {
   SH_RGBAGEN_IDENTITY_LIGHTING,
   SH_RGBAGEN_IDENTITY,
   SH_RGBAGEN_WAVE,
   SH_RGBAGEN_ENTITY,
   SH_RGBAGEN_ONE_MINUS_ENTITY,
   SH_RGBAGEN_VERTEX,
   SH_RGBAGEN_ONE_MINUS_VERTEX,
   SH_RGBAGEN_PORTAL,
   SH_RGBAGEN_LIGHTING_DIFFUSE,
   SH_RGBAGEN_LIGHTING_SPECULAR,

#ifdef SH_PHOENIX_QUAKE1_EXT
   SH_RGBAGEN_CONST_EXT,
   SH_RGBAGEN_ONE_MINUS_PORTAL_EXT,
#endif
};

typedef struct sh_rgbagen_s {
   int func;

   union {
      // rgbGen wave, alphaGen wave
      struct {
         int      form;
         sh_num_t base;
         sh_num_t amp;
         sh_num_t phase;
         sh_num_t freq;
      } wave;

      // alphaGen portal
      struct {
         sh_num_t dist_to_opaque;
      } portal;

#ifdef SH_PHOENIX_QUAKE1_EXT
      // alphaGen _one_minus_portal
      struct {
         sh_num_t dist_to_clear;
      } one_minus_portal;

      // rgbGen const, alphaGen const
      struct {
         sh_num_t color[3];
      } constant;
#endif
   } args;
} sh_rgbagen_t;

// blendFunc
enum {
   SH_BLEND_UNDEFINED,
   SH_BLEND_ONE,
   SH_BLEND_ZERO,
   SH_BLEND_DST_COLOR,
   SH_BLEND_ONE_MINUS_DST_COLOR,
   SH_BLEND_SRC_ALPHA,
   SH_BLEND_ONE_MINUS_SRC_ALPHA,
   SH_BLEND_DST_ALPHA,
   SH_BLEND_ONE_MINUS_DST_ALPHA,
   SH_BLEND_SRC_COLOR,
   SH_BLEND_ONE_MINUS_SRC_COLOR,
};

// depthFunc
enum {
   SH_DEPTHFUNC_LEQUAL,
   SH_DEPTHFUNC_EQUAL,
};

// alphaFunc
enum {
   SH_ALPHAFUNC_GT0,
   SH_ALPHAFUNC_LT128,
   SH_ALPHAFUNC_GE128,
};

// _filter
enum {
   SH_FILTER_NEAREST,
   SH_FILTER_LINEAR,
   SH_FILTER_NEAREST_MIPMAP_NEAREST,
   SH_FILTER_NEAREST_MIPMAP_LINEAR,
   SH_FILTER_LINEAR_MIPMAP_NEAREST,
   SH_FILTER_LINEAR_MIPMAP_LINEAR
};

// map, clampmap, animMap
typedef struct sh_map_s {
   char     name[SH_NAME_LEN+1];
   unsigned flags;
} sh_map_t;

// stage
typedef struct sh_stage_s {
   // map, clampmap, animMap
   sh_map_t maps[SH_MAX_ANIMMAP_FRAMES];
   int      anim_map_count;
   sh_num_t anim_map_fps;

   // blendFunc
   int blend_src;
   int blend_dst;

   // alphaFunc
   int alphafunc;

   // depthFunc
   int depthfunc;

   // tcMod
   int        tcmod_count;
   sh_tcmod_t tcmods[SH_MAX_TCMODS];

   // tcGen
   sh_tcgen_t tcgen;

   // rgbGen
   sh_rgbagen_t rgbgen;

   // alphaGen
   sh_rgbagen_t alphagen;

   // detail, depthWrite
   unsigned flags;

#ifdef SH_PHOENIX_QUAKE1_EXT
   // _filter
   int min_filter;
   int mag_filter;
#endif
} sh_stage_t;

// shader
typedef struct shader_s {
   char name[SH_NAME_LEN+1];

   // nopicmip, nomipmaps, polygonOffset, entityMergable
   unsigned flags;

   // skyparms
   char     far_box_basename[SH_NAME_LEN+1];
   sh_map_t far_box[6];

   sh_num_t cloud_height;

   char     near_box_basename[SH_NAME_LEN+1];
   sh_map_t near_box[6];

   // cull
   int cull_face;

   // deformVertexes
   int        vxmod_count;
   sh_vxmod_t vxmods[SH_MAX_VXMODS];

   // fogparms
   sh_num_t fog_color[3];
   sh_num_t fog_distance_to_opaque;

   // sort
   int sort_priority;

   // stages
   int        stage_count;
   sh_stage_t stages[SH_MAX_STAGES];
} shader_t;



/***************************************************************************\

numbers

\***************************************************************************/

void sh_number_parse(const char* string, sh_num_t* number);



/***************************************************************************\

parsing and compilation

\***************************************************************************/

typedef bool   (*sh_approve_func_t)(const char* name);
typedef void   (*sh_compile_func_t)(shader_t* shader);

size_t sh_parse(sh_compile_func_t compile, sh_approve_func_t approve);
void   sh_parse_stop(void);
void   sh_report(shader_t* shader);
void   sh_save(shader_t* shader);



/***************************************************************************\

input

\***************************************************************************/

typedef int (*sh_input_func_t)(void);

int             sh_input();
void            sh_input_unget(int cin);
sh_input_func_t sh_input_set_func(sh_input_func_t func);
int             sh_input_string_func(void);
void            sh_input_set_string(const char* string, size_t size);
bool            sh_input_load_string(const char* filename, char* text, size_t* size);
int             sh_input_file_func(void);
FILE*           sh_input_set_file(FILE* file);



/***************************************************************************\

constant to string conversion

\***************************************************************************/

const char* sh_sort_to_string(int s, bool parsable);
const char* sh_cull_to_string(int c);
const char* sh_waveform_to_string(int w);
const char* sh_vxmod_to_string(int v);
const char* sh_blend_to_string(int b);
const char* sh_alphafunc_to_string(int a);
const char* sh_depthfunc_to_string(int d);
const char* sh_rgbagen_to_string(int g);
const char* sh_tcmod_to_string(int t);
const char* sh_tcgen_to_string(int t);



/***************************************************************************\

 messages

\***************************************************************************/

#define SH_MSG_ERROR         _SH_FLAG(0)
#define SH_MSG_WARNING       _SH_FLAG(1)
#define SH_MSG_COMPATIBILITY _SH_FLAG(2)
#define SH_MSG_VERBOSE       _SH_FLAG(3)
#define SH_MSG_DEBUG         _SH_FLAG(4)
#define SH_MSG_ALL           (SH_MSG_ERROR|SH_MSG_WARNING|SH_MSG_COMPATIBILITY|SH_MSG_VERBOSE)

typedef void (*sh_message_func_t)(unsigned type, const char* msg);

void              sh_message(unsigned type, const char* format, ... );
void              sh_message_stderr_func(unsigned type, const char* msg);
sh_message_func_t sh_message_set_func(sh_message_func_t func);
unsigned          sh_message_set_types(unsigned types);
void              sh_message_get_counts(int* ecount, int* wcount, int* ccount);
void              sh_message_reset_counts(unsigned types);
const char*       sh_message_set_file(const char* file);
const char*       sh_message_set_prefix(const char* prefix);



/***************************************************************************\

 memory allocation

\***************************************************************************/

typedef shader_t* (*sh_alloc_func_t)(void);

shader_t*       sh_alloc(void);
sh_alloc_func_t sh_alloc_set_func(sh_alloc_func_t m);
shader_t*       sh_alloc_static_func(void);
shader_t*       sh_alloc_array_func(void);
shader_t*       sh_alloc_set_array(shader_t* array, size_t elements);



/***************************************************************************\

 inline functions and macros

\***************************************************************************/

#include "shader_inline.h"



#endif
