/*
Copyright (C) 2000 Jason Wilkins

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include "shaders.h"
#include "shader_internal.h"



static void save_skyparms(shader_t* shader)
{
   if ((shader->flags & SH_FAR_BOX) ||
       (shader->flags & SH_SKYGEN)  ||
       (shader->flags & SH_NEAR_BOX)) {
      printf("\tskyparms ");

      if (shader->flags & SH_FAR_BOX) {
         printf("%s", shader->far_box_basename);
      }
      else {
         printf("-");
      }

      printf(" ");

      if (shader->flags & SH_SKYGEN) {
         printf("%s", shader->cloud_height.string);
      }
      else {
         printf("-");
      }

      printf(" ");

      if (shader->flags & SH_NEAR_BOX) {
         printf("%s", shader->near_box_basename);
      }
      else {
         printf("-");
      }

      printf("\n");
   }
}



static void save_cull(shader_t* shader)
{
   if (shader->flags & SH_EXPLICIT_CULL) {
      printf("cull %s\n", sh_cull_to_string(shader->cull_face));
   }
}



static void save_vxmod(sh_vxmod_t* vxmod)
{
   printf("\tdeformVertexes %s", sh_vxmod_to_string(vxmod->func));

   switch(vxmod->func) {
      case SH_VXMOD_WAVE:
         printf(" %s", vxmod->args.wave.div.string);
         printf(" %s", sh_waveform_to_string(vxmod->args.wave.form));
         printf(" %s", vxmod->args.wave.base.string);
         printf(" %s", vxmod->args.wave.amp.string);
         printf(" %s", vxmod->args.wave.phase.string);
         printf(" %s", vxmod->args.wave.freq.string);
         break;

      case SH_VXMOD_NORMAL:
         printf(" %s", vxmod->args.normal.amp.string);
         printf(" %s", vxmod->args.normal.freq.string);
         break;

      case SH_VXMOD_BULGE:
         printf(" %s", vxmod->args.bulge.width.string);
         printf(" %s", vxmod->args.bulge.height.string);
         printf(" %s", vxmod->args.bulge.speed.string);
         break;

      case SH_VXMOD_MOVE:
         printf(" %s", vxmod->args.move.x.string);
         printf(" %s", vxmod->args.move.y.string);
         printf(" %s", vxmod->args.move.z.string);
         printf(" %s", sh_waveform_to_string(vxmod->args.move.form));
         printf(" %s", vxmod->args.move.base.string);
         printf(" %s", vxmod->args.move.amp.string);
         printf(" %s", vxmod->args.move.phase.string);
         printf(" %s", vxmod->args.move.freq.string);
         break;

      case SH_VXMOD_AUTOSPRITE:
      case SH_VXMOD_AUTOSPRITE2:
      case SH_VXMOD_PROJECTION_SHADOW:
      case SH_VXMOD_TEXT0:
      case SH_VXMOD_TEXT1:

#ifdef SH_PHOENIX_QUAKE1_EXT
      case SH_VXMOD_STENCIL_SHADOW_EXT:
#endif

         break;

      default:
         break;
   }

   printf("\n");
}



static void save_vxmods(shader_t* shader)
{
   int i;

   for (i = 0; i < shader->vxmod_count; i++) {
      save_vxmod(&shader->vxmods[i]);
   }
}



static void save_fogparms(shader_t* shader)
{
   if (shader->flags & SH_FOGPARMS) {
      printf("\tfogparms");
      printf(" %s", shader->fog_color[0].string);
      printf(" %s", shader->fog_color[1].string);
      printf(" %s", shader->fog_color[2].string);
      printf(" %s", shader->fog_distance_to_opaque.string);
      printf("\n");
   }
}



static void save_nopicmip(shader_t* shader)
{
   if (shader->flags & SH_EXPLICIT_NOPICMIP) {
      printf("\tnopicmip\n");
   }
}



static void save_nomipmaps(shader_t* shader)
{
   if (shader->flags & SH_NOMIPMAPS) {
      printf("\tnomipmaps\n");
   }
}



static void save_polygon_offset(shader_t* shader)
{
   if (shader->flags & SH_POLYGON_OFFSET) {
      printf("\tpolygonOffset\n");
   }
}



static void save_sort(shader_t* shader)
{
   if (shader->flags & SH_EXPLICIT_PORTAL) {
      printf("\tportal\n");
   }

   if (shader->flags & SH_EXPLICIT_SORT) {
      printf("\tsort %s\n", sh_sort_to_string(shader->sort_priority, true));
   }
}



static void save_entity_mergable(shader_t* shader)
{
   if (shader->flags & SH_ENTITY_MERGABLE) {
      printf("\tentityMergable\n");
   }
}



static void save_map(sh_stage_t* stage)
{
   if (stage->anim_map_count == 1) {
      if (stage->maps[0].flags & SH_MAP_CLAMPMAP) {
         printf("\t\tclampmap");
      }
      else {
         printf("\t\tmap");
      }

      printf(" %s\n", stage->maps[0].name);
   }
   else if (stage->anim_map_count > 1) {
      int i;

      printf("\t\tanimMap ");

      printf(" %s", stage->anim_map_fps.string);

      for (i = 0; i < stage->anim_map_count; i++) {
         printf(" %s", stage->maps[i].name);
      }

      printf("\n");
   }
}



static void save_blendfunc(sh_stage_t* stage)
{
   if (stage->flags & SH_STAGE_BLENDFUNC) {
      printf("\t\tblendfunc ");
      
      if (stage->flags & SH_STAGE_EXPLICIT_BLENDFUNC) {
         printf("%s ", sh_blend_to_string(stage->blend_src));
         printf("%s",  sh_blend_to_string(stage->blend_dst));
      }
      else {
         if (sh_is_blend_filter(stage)) {
            printf("filter");
         }
         else if (sh_is_blend_blend(stage)) {
            printf("blend");
         }
         else if (sh_is_blend_add(stage)) {
            printf("add");
         }/*
         else {
            assert: this should never happen
         }*/
      }

      printf("\n");
   }
}



static void save_rgbagen(sh_rgbagen_t* rgbagen, bool is_rgbgen)
{
   printf("\t\t");

   if (is_rgbgen) {
      printf("rgbGen");
   }
   else {
      printf("alphaGen");
   }

   printf(" %s", sh_rgbagen_to_string(rgbagen->func));

   switch(rgbagen->func) {
      case SH_RGBAGEN_WAVE:
         printf(" %s", sh_waveform_to_string(rgbagen->args.wave.form));
         printf(" %s", rgbagen->args.wave.base.string);
         printf(" %s", rgbagen->args.wave.amp.string);
         printf(" %s", rgbagen->args.wave.phase.string);
         printf(" %s", rgbagen->args.wave.freq.string);
         break;

#ifdef SH_PHOENIX_QUAKE1_EXT
      case SH_RGBAGEN_CONST_EXT:
         if (is_rgbgen) {
             printf(" %s", rgbagen->args.constant.color[0]);
             printf(" %s", rgbagen->args.constant.color[1]);
             printf(" %s", rgbagen->args.constant.color[2]);
         }
         else {
             printf(" %s", rgbagen->args.constant.color[0]);
         }

         break;

      case SH_RGBAGEN_ONE_MINUS_PORTAL_EXT:
#endif

      case SH_RGBAGEN_IDENTITY_LIGHTING:
      case SH_RGBAGEN_IDENTITY:
      case SH_RGBAGEN_ENTITY:
      case SH_RGBAGEN_ONE_MINUS_ENTITY:
      case SH_RGBAGEN_VERTEX:
      case SH_RGBAGEN_ONE_MINUS_VERTEX:
      case SH_RGBAGEN_PORTAL:
      case SH_RGBAGEN_LIGHTING_DIFFUSE:
      case SH_RGBAGEN_LIGHTING_SPECULAR:
      default:
         break;
   }

   printf("\n");
}



static void save_rgbgen(sh_stage_t* stage)
{
   if (stage->flags & SH_STAGE_EXPLICIT_RGBGEN) {
      save_rgbagen(&stage->rgbgen, true);
   }
}



static void save_alphagen(sh_stage_t* stage)
{
   if (stage->flags & SH_STAGE_EXPLICIT_ALPHAGEN) {
      save_rgbagen(&stage->alphagen, false);
   }
}



static void save_tcgen(sh_stage_t* stage)
{
   if (stage->flags & SH_STAGE_EXPLICIT_TCGEN) {
      printf("\t\ttcGen %s", sh_tcgen_to_string(stage->tcgen.func));

      switch(stage->tcgen.func) {
         case SH_TCGEN_VECTOR:
            printf(" ( %s %s %s )", stage->tcgen.args.vector.sx.string, stage->tcgen.args.vector.sy.string, stage->tcgen.args.vector.sz.string);
            printf(" ( %s %s %s )", stage->tcgen.args.vector.tx.string, stage->tcgen.args.vector.ty.string, stage->tcgen.args.vector.tz.string);
            break;

#ifdef SH_PHOENIX_QUAKE1_EXT
         case SH_TCGEN_SKY_EXT:
            printf(" %s", stage->tcgen.args.sky.cloud_height.string);
            break;

         case SH_TCGEN_Q1TURB_EXT:
            printf(" %s", stage->tcgen.args.q1turb.div.string);
            printf(" %s", sh_waveform_to_string(stage->tcgen.args.q1turb.form));
            printf(" %s", stage->tcgen.args.q1turb.base.string);
            printf(" %s", stage->tcgen.args.q1turb.amp.string);
            printf(" %s", stage->tcgen.args.q1turb.phase.string);
            printf(" %s", stage->tcgen.args.q1turb.freq.string);
            break;
#endif

         case SH_TCGEN_BASE:
         case SH_TCGEN_LIGHTMAP:
         case SH_TCGEN_ENVIRONMENT:
         default:
            break;
      }

      printf("\n");
   }
}



static void save_tcmod(sh_tcmod_t* tcmod)
{
   printf("\t\ttcMod %s", sh_tcmod_to_string(tcmod->func));

   switch (tcmod->func) {
      case SH_TCMOD_ROTATE:
         printf(" %s", tcmod->args.rotate.speed.string);
         break;

      case SH_TCMOD_SCALE:
         printf(" %s", tcmod->args.scale.s.string);
         printf(" %s", tcmod->args.scale.t.string);
         break;

      case SH_TCMOD_SCROLL:
         printf(" %s", tcmod->args.scroll.s.string);
         printf(" %s", tcmod->args.scroll.t.string);
         break;

      case SH_TCMOD_STRETCH:
         printf(" %s", sh_waveform_to_string(tcmod->args.stretch.form));
         printf(" %s", tcmod->args.stretch.base.string);
         printf(" %s", tcmod->args.stretch.amp.string);
         printf(" %s", tcmod->args.stretch.phase.string);
         printf(" %s", tcmod->args.stretch.freq.string);
         break;

      case SH_TCMOD_TRANSFORM:
         printf(" %s", tcmod->args.transform.m00.string);
         printf(" %s", tcmod->args.transform.m01.string);
         printf(" %s", tcmod->args.transform.m10.string);
         printf(" %s", tcmod->args.transform.m11.string);
         printf(" %s", tcmod->args.transform.t0.string);
         printf(" %s", tcmod->args.transform.t1.string);
         break;

      case SH_TCMOD_TURB:
         printf(" %s", tcmod->args.turb.base.string);
         printf(" %s", tcmod->args.turb.amp.string);
         printf(" %s", tcmod->args.turb.phase.string);
         printf(" %s", tcmod->args.turb.freq.string);
         break;
   }

   printf("\n");
}



static void save_tcmods(sh_stage_t* stage)
{
   int i;

   for (i = 0; i < stage->tcmod_count; i++) {
      save_tcmod(&stage->tcmods[i]);
   }
}



static void save_depthfunc(sh_stage_t* stage)
{
   if (stage->flags & SH_STAGE_EXPLICIT_DEPTHFUNC) {
      printf("\t\tdepthFunc %s\n", sh_depthfunc_to_string(stage->depthfunc));
   }
}



static void save_depthwrite(sh_stage_t* stage)
{
   if (stage->flags & SH_STAGE_EXPLICIT_DEPTHWRITE) {
      printf("\t\tdepthWrite\n");
   }
}



static void save_detail(sh_stage_t* stage)
{
   if (stage->flags & SH_STAGE_DETAIL) {
      printf("\t\tdetail\n");
   }
}



static void save_alphafunc(sh_stage_t* stage)
{
   if (stage->flags & SH_STAGE_ALPHAFUNC) {
      printf("\t\talphaFunc %s\n", sh_alphafunc_to_string(stage->alphafunc));
   }
}



static void save_stage(sh_stage_t* stage)
{
   printf("\t{\n");
   save_map(stage);
   save_blendfunc(stage);
   save_rgbgen(stage);
   save_alphagen(stage);
   save_tcgen(stage);
   save_tcmods(stage);
   save_depthfunc(stage);
   save_depthwrite(stage);
   save_detail(stage);
   save_alphafunc(stage);
   printf("\t}\n");
}



static void save_stages(shader_t* shader)
{
   int i;

   for (i = 0; i < shader->stage_count; i++) {
      if (i != 0) printf("\n");

      save_stage(&shader->stages[i]);
   }
}



void sh_save(shader_t* shader)
{
   printf("%s\n", shader->name);
   printf("{\n");
   save_skyparms(shader);
   save_cull(shader);
   save_vxmods(shader);
   save_fogparms(shader);
   save_nopicmip(shader);
   save_nomipmaps(shader);
   save_polygon_offset(shader);
   save_sort(shader);
   save_entity_mergable(shader);
   save_stages(shader);
   printf("}\n\n");
}


