/*
Copyright (C) 2000 Jason Wilkins

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include "shaders.h"
#include "shader_internal.h"



static sh_stage_t ignored_stage;

static _SH_TOKEN lookahead = _SH_TOK_EOF;

static shader_t*   shader = NULL;
static sh_stage_t* stage  = NULL;

static bool parse_stop;



/* expected:
 ***************************************************************************/
static void expected(const char* expected, const char* got)
{
   sh_message(SH_MSG_ERROR, "expected %s, but found '%s'", expected, got);
}



/* sh_number_parse:
 ***************************************************************************/
void sh_number_parse(const char* string, sh_num_t* number)
{
   char* place;
   int   len;

   len = strlen(string);

   if (len > SH_NUMBER_LEN) {
      sh_message(SH_MSG_WARNING, "number string length exceeds SH_NUMBER_LEN (%d) in shader '%s'", SH_NUMBER_LEN, shader->name);
   }

   number->value = (float)strtod(string, &place);

   if ((place - string) != len) {
      expected("number", string);
   }

   strncpy(number->string, string, SH_NUMBER_LEN);
   number->string[SH_NUMBER_LEN] = '\0';
}



/* parse_clamped_number
 ***************************************************************************/
static void parse_clamped_number(const char* string, sh_num_t* number, const char* name, float low, float high) {
   sh_number_parse(string, number);

   if (number->value < low) {
      sh_message(SH_MSG_WARNING, "%s is not normalized, clamping to %f", name, low);
      number->value = low;
   }
   else if (number->value > high) {
      sh_message(SH_MSG_WARNING, "%s is not normalized, clamping to %f", name, high);
      number->value = high;
   }
}



/***************************************************************************\

General Keyword:

   skyParms <farbox> <cloudheight> <nearbox>

\***************************************************************************/



/* ParseSkyboxBasename:
 ***************************************************************************/
static bool ParseSkyboxBasename(const char* in, char* out, const char* skybox)
{
   if (strcmp("-", in) != 0) {
      if (strlen(in) <= SH_NAME_LEN - sizeof("_xx.tga")) {
         strcpy(out, in);
         return true;
      }
      else {
         sh_message(SH_MSG_WARNING, "%s base name too long", skybox);
      }
   }

   out[0] = '\0';

   return false;
}



/* skyParmsParseFunc:
 ***************************************************************************/
static bool skyParmsParseFunc(int unused, char* argv[])
{
   (void)unused;

   // farbox
   if (ParseSkyboxBasename(argv[1], shader->far_box_basename, "far box")) {
      shader->flags |= SH_FAR_BOX;
   }
   else {
      shader->flags &= ~SH_FAR_BOX;
   }

   // cloudheight
   if (strcmp("-", argv[2]) != 0) {
      sh_number_parse(argv[2], &shader->cloud_height);
      shader->flags |= SH_EXPLICIT_SKYGEN;
   }
   else {
      sh_number_parse("128", &shader->cloud_height);
   }

   shader->flags |= SH_SKYGEN;

   // nearbox
   if (ParseSkyboxBasename(argv[3], shader->near_box_basename, "near box")) {
      sh_message(SH_MSG_COMPATIBILITY, "extension: a near box was specified");
      shader->flags |= SH_NEAR_BOX;
   }
   else {
      shader->flags &= ~SH_NEAR_BOX;
   }

   return true;
}



/* request_sky_box_maps:
 ***************************************************************************/
static void request_sky_box_maps
   (
   shader_t*    shader,
   char*        base,
   bool         is_far_box,
   sh_map_t maps_out[5]
   )
{
   int i;
   char name[SH_NAME_LEN+1];

   static char* suffixes[6] = {
      "_rt.tga",
      "_lf.tga",
      "_ft.tga",
      "_bk.tga",
      "_up.tga",
      "_dn.tga"
   };

   if (((shader->flags & SH_FAR_BOX)  &&  is_far_box) ||
       ((shader->flags & SH_NEAR_BOX) && !is_far_box)) {

      for (i = 0; i < 6; i++) {
         strncpy(name, base, SH_NAME_LEN);
         name[SH_NAME_LEN] = '\0';
         strncat(name, suffixes[i], SH_NAME_LEN+8);
         name[SH_NAME_LEN+8] = '\0';
         strcpy(maps_out[i].name, name);

         if (!is_far_box) maps_out[i].flags |= SH_MAP_ALPHA;
      }

   }
}



/* finalize_skyparms
 ***************************************************************************/
static void finalize_skyparms(shader_t* shader)
{
   if ((shader->stage_count == 0) &&
       (shader->flags & SH_EXPLICIT_SKYGEN)) {
      sh_message(SH_MSG_WARNING, "specified skyparms <cloudheight>, but there are no cloud layers.  specify <cloudheight> as '-' to quiet this warning");
   }

   request_sky_box_maps(shader, shader->far_box_basename,  true,  shader->far_box);
   request_sky_box_maps(shader, shader->near_box_basename, false, shader->near_box);
}



/***************************************************************************\

General Keyword:

   cull front
   cull back
   cull disable
   cull none
   cull _both

\***************************************************************************/



/* cullParseFunc:
 ***************************************************************************/
static bool cullParseFunc(int argc, char* argv[])
{
   if (argc == 1) {
      // if there are no arguments, it defaults to 'cull front'
      shader->cull_face = SH_CULL_FRONT;
   }
   else {
      if (stricmp(argv[1], "front") == 0) {
         shader->cull_face = SH_CULL_FRONT;
      }
      else if (stricmp(argv[1], "back") == 0) {
         shader->cull_face = SH_CULL_BACK;
      }
      else if ((stricmp(argv[1], "disable") == 0) ||
               (stricmp(argv[1], "none") == 0)) {
         shader->cull_face = SH_CULL_DISABLE;
      }

#ifdef SH_PHOENIX_QUAKE1_EXT
      else if (stricmp(argv[1], "_both") == 0) {
         sh_message(SH_MSG_COMPATIBILITY, "_both is an extension");
         shader->cull_face = SH_CULL_BOTH_EXT;
      }
#endif

      else {
         sh_message(SH_MSG_WARNING, "argument '%s' is invalid, it should be one of 'front', 'back', '_both', 'disable', or 'none'", argv[1]);
         return true;
      }
   }

   return true;
}



/* finalize_cull
 ***************************************************************************/
static void finalize_cull()
{
   if (!(shader->flags & SH_EXPLICIT_CULL)) shader->cull_face = SH_CULL_FRONT;
}



/***************************************************************************\

General Keyword:

   deformVertexes wave <div> <func> <base> <amp> <phase> <freq>
   deformVertexes normal <amp> <freq>
   deformVertexes bulge <width> <height> <speed>
   deformVertexes move <x> <y> <z> <func> <base> <amp> <phase> <freq>
   deformVertexes autoSprite
   deformVertexes autoSprite2
   deformVertexes projectionShader
   deformVertexes text0
   deformVertexes text1
   deformVertexes _stencilShadow

\***************************************************************************/



/* ParseWaveForm:
 ***************************************************************************/
static int ParseWaveForm(const char* str)
{
   if (stricmp(str, "sin") == 0) {
      return SH_WAVE_SIN;
   }
   else if (stricmp(str, "triangle") == 0) {
      return SH_WAVE_TRIANGLE;
   }
   else if (stricmp(str, "square") == 0) {
      return SH_WAVE_SQUARE;
   }
   else if (stricmp(str, "sawtooth") == 0) {
      return SH_WAVE_SAWTOOTH;
   }
   else if (stricmp(str, "inversesawtooth") == 0) {
      return SH_WAVE_INVSAWTOOTH;
   }
   else if (stricmp(str, "noise") == 0) {
      sh_message(SH_MSG_COMPATIBILITY, "waveform noise is not documented by Id");
      return SH_WAVE_NOISE;
   }
   else {
      sh_message(SH_MSG_ERROR, "unknown waveform '%s'", str);
      return -1;
   }
}



/* deformVertexesParseFunc:
 ***************************************************************************/
static bool deformVertexesParseFunc(int argc, char* argv[])
{
   sh_vxmod_t* vxmod;
   sh_vxmod_t  ignored_vxmod;

   if (shader->vxmod_count < SH_MAX_VXMODS) {
      vxmod = &shader->vxmods[shader->vxmod_count];
   }
   else {
      sh_message(SH_MSG_ERROR, "number of deformVertexes commands exceeds SH_MAX_VXMODS (%d) in shader %s", SH_MAX_VXMODS, shader->name);
      vxmod = &ignored_vxmod;
   }

   // deformVertexes wave <div> <func> <base> <amp> <phase> <freq>
   if (stricmp(argv[1], "wave") == 0) {
      int form;

      if (argc != 8) {
         sh_message(SH_MSG_WARNING, "usage: deformVertexes wave <div> <func> <base> <amp> <phase> <freq>");
         return true;
      }

      vxmod->func = SH_VXMOD_WAVE;

      sh_number_parse(argv[2], &vxmod->args.wave.div);

      form = ParseWaveForm(argv[3]);

      if (form < 0) return true;

      vxmod->args.wave.form = form;
      
      sh_number_parse(argv[4], &vxmod->args.wave.base);
      sh_number_parse(argv[5], &vxmod->args.wave.amp);
      sh_number_parse(argv[6], &vxmod->args.wave.phase);
      sh_number_parse(argv[7], &vxmod->args.wave.freq);
   }
   // deformVertexes normal <amp> <freq>
   else if (stricmp(argv[1], "normal") == 0) {
      if (argc != 4) {
         sh_message(SH_MSG_WARNING, "usage: deformVertexes normal <amp> <freq>");
         return true;
      }

      vxmod->func = SH_VXMOD_NORMAL;

      sh_number_parse(argv[2], &vxmod->args.normal.amp);
      sh_number_parse(argv[3], &vxmod->args.normal.freq);
   }
   // deformVertexes bulge <width> <height> <speed>
   else if (stricmp(argv[1], "bulge") == 0) {
      if (argc != 5) {
         sh_message(SH_MSG_WARNING, "usage: deformVertexes bulge <width> <height> <speed>");
         return true;
      }

      vxmod->func = SH_VXMOD_BULGE;

      sh_number_parse(argv[2], &vxmod->args.bulge.width);
      sh_number_parse(argv[3], &vxmod->args.bulge.height);
      sh_number_parse(argv[4], &vxmod->args.bulge.speed);
   }
   // deformVertexes move <x> <y> <z> <func> <base> <amp> <phase> <freq>
   else if (stricmp(argv[1], "move") == 0) {
      int form;

      if (argc != 10) {
         sh_message(SH_MSG_WARNING, "usage: deformVertexes move <x> <y> <z> <func> <base> <amplitude> <phase> <freq>");
         return true;
      }

      vxmod->func = SH_VXMOD_MOVE;

      sh_number_parse(argv[2], &vxmod->args.move.x);
      sh_number_parse(argv[3], &vxmod->args.move.y);
      sh_number_parse(argv[4], &vxmod->args.move.z);

      form = ParseWaveForm(argv[5]);

      if (form < 0) return true;

      vxmod->args.move.form = form;

      sh_number_parse(argv[6], &vxmod->args.move.base);
      sh_number_parse(argv[7], &vxmod->args.move.amp);
      sh_number_parse(argv[8], &vxmod->args.move.phase);
      sh_number_parse(argv[9], &vxmod->args.move.freq);
   }
   // deformVertexes autosprite
   else if (stricmp(argv[1], "autosprite") == 0) {
      if (argc != 2) {
         sh_message(SH_MSG_WARNING, "autosprite does not take any arguments");
      }

      vxmod->func = SH_VXMOD_AUTOSPRITE;
   }
   // deformVertexes autosprite2
   else if (stricmp(argv[1], "autosprite2") == 0) {
      if (argc != 2) {
         sh_message(SH_MSG_WARNING, "autosprite2 does not take any arguments");
      }

      vxmod->func = SH_VXMOD_AUTOSPRITE2;
   }
   // deformVertexes projectionShadow
   else if (stricmp(argv[1], "projectionShadow") == 0) {
      sh_message(SH_MSG_COMPATIBILITY, "projectionShadow is not documented by Id");

      if (argc != 2) {
         sh_message(SH_MSG_WARNING, "projectionShadow does not take any arguments");
      }

      vxmod->func = SH_VXMOD_PROJECTION_SHADOW;
   }
   else if (stricmp(argv[1], "text0") == 0) {
      sh_message(SH_MSG_COMPATIBILITY, "text0 is not documented by Id");

      if (argc != 2) {
         sh_message(SH_MSG_WARNING, "text0 does not take any arguments");
      }

      vxmod->func = SH_VXMOD_TEXT0;
   }
   else if (stricmp(argv[1], "text1") == 0) {
      sh_message(SH_MSG_COMPATIBILITY, "text1 is not documented by Id");

      if (argc != 2) {
         sh_message(SH_MSG_WARNING, "text1 does not take any arguments");
      }

      vxmod->func = SH_VXMOD_TEXT1;
   }

#ifdef SH_PHOENIX_QUAKE1_EXT
   // deformVertexes _stencilShadow
   else if (stricmp(argv[1], "_stencilShadow") == 0) {
      sh_message(SH_MSG_COMPATIBILITY, "_stencilShadow is an extension");

      if (argc != 2) {
         sh_message(SH_MSG_WARNING, "_stencilShadow does not take any arguments");
      }

      vxmod->func = SH_VXMOD_STENCIL_SHADOW_EXT;
   }
#endif

   // error
   else {
      sh_message(SH_MSG_WARNING, "unrecognized command subtype '%s'", argv[1]);

      return true;
   }

   if (shader->vxmod_count < SH_MAX_VXMODS) shader->vxmod_count++;

   return true;
}



/***************************************************************************\

General Keyword:

   fogparms <red> <green> <blue> <distance to opaque>
   fogparms ( <red> <green> <blue> ) <distance to opaque>

\***************************************************************************/



/* ParseColor:
 ***************************************************************************/
void ParseColor(sh_num_t color[3], const char* red, const char* green, const char* blue)
{
   parse_clamped_number(red,   &color[0], "red color component",   0, 1);
   parse_clamped_number(green, &color[1], "green color component", 0, 1);
   parse_clamped_number(blue,  &color[2], "blue color component",  0, 1);
}



/* fogparmsParseFunc:
 ***************************************************************************/
static bool fogparmsParseFunc(int argc, char* argv[])
{
   const char *red;
   const char *green;
   const char *blue;
   const char *distance_to_opaque;

   // check arguments
   if (argc == 5) {
      red   = argv[1];
      green = argv[2];
      blue  = argv[3];

      distance_to_opaque = argv[4];
   }
   else if ((argc == 7) &&
            (strcmp(argv[1], "(") == 0) &&
            (strcmp(argv[5], ")") == 0)) {

      red   = argv[2];
      green = argv[3];
      blue  = argv[4];

      distance_to_opaque = argv[6];

   }
   else {
      sh_message(SH_MSG_WARNING, "usage: fogparms <red> <green> <blue> <distance to opaque>");
      return true;
   } 

   // color
   ParseColor(shader->fog_color, red, green, blue);

   // distance to opaque
   sh_number_parse(distance_to_opaque, &shader->fog_distance_to_opaque);

   // finish
   shader->flags |= SH_FOGPARMS;

   return true;
}



/***************************************************************************\

General Keyword:

   nopicmip

\***************************************************************************/



/* nopicmipParseFunc:
 ***************************************************************************/
static bool nopicmipParseFunc(int unused0, char* unused1[])
{
   (void)unused0;
   (void)unused1;

   shader->flags |= SH_NOPICMIP;
   shader->flags |= SH_EXPLICIT_NOPICMIP;

   return true;
}



/***************************************************************************\

General Keyword:

   nomipmaps

\***************************************************************************/



/* nomipmapsParseFunc:
 ***************************************************************************/
static bool nomipmapsParseFunc(int unused0, char* unused1[])
{
   (void)unused0;
   (void)unused1;

   shader->flags |= SH_NOMIPMAPS;

   // nomipmaps implies no picmip
   shader->flags |= SH_NOPICMIP;

   return true;
}



/***************************************************************************\

General Keyword:

   polygonOffset

\***************************************************************************/



/* polygonOffsetParseFunc:
 ***************************************************************************/
static bool polygonOffsetParseFunc(int unused0, char* unused1[])
{
   (void)unused0;
   (void)unused1;

   shader->flags |= SH_POLYGON_OFFSET;

   return true;
}



/***************************************************************************\

General Keyword:

   portal

\***************************************************************************/



/* portalParseFunc:
 ***************************************************************************/
static bool portalParseFunc(int unused0, char* unused1[])
{
   (void)unused0;
   (void)unused1;

   shader->sort_priority = SH_SORT_PORTAL;
   shader->flags |= SH_EXPLICIT_PORTAL;

   return true;
}



/***************************************************************************\

General Keyword:

   sort <simple name>
   sort <#>

\***************************************************************************/



typedef struct sort_parse_table_s {
   char* name;
   int   pri;
} sort_parse_table_t;



static sort_parse_table_t sort_table[] = {
  /* name          priority      */
   { "portal",     SH_SORT_PORTAL     },
   { "sky",        SH_SORT_SKY        },
   { "opaque",     SH_SORT_OPAQUE     },
   { "banner",     SH_SORT_BANNER     },
   { "underwater", SH_SORT_UNDERWATER },
   { "additive",   SH_SORT_ADDITIVE   },
   { "nearest",    SH_SORT_NEAREST    },
   { NULL,         0             }
};



/* sortParseFunc:
 ***************************************************************************/
static bool sortParseFunc(int unused, char* argv[])
{
   int pri;

   (void)unused;

   // sort <priority>
   if (isalpha(argv[1][0])) {
      sort_parse_table_t* table;

      table = sort_table;

      while (table->name) {
         if (stricmp(table->name, argv[1]) == 0) {
            shader->sort_priority = table->pri;
            break;
         }

         table++;
      }

      if (!table->name) {
         sh_message(SH_MSG_WARNING, "unknown priority");
         return true;
      }
   }
   // sort <#>
   else {
      sh_num_t num;

      sh_number_parse(argv[1], &num);
      pri = (int)num.value;

      if (pri < 1) {
         sh_message(SH_MSG_WARNING, "priority must be between 1 and 16, clamping to 1");
         pri = 1;
      }
      else if (pri > 16) {
         sh_message(SH_MSG_WARNING, "priority must be between 1 and 16, clamping to 16");
         pri = 16;
      }

      shader->sort_priority = pri;
   }

   return true;
}



/* finalize_sort:
 ***************************************************************************/
void finalize_sort()
{
   if (!(shader->flags & SH_EXPLICIT_SORT)) {
      if (shader->flags & (SH_SKYGEN|SH_FAR_BOX|SH_NEAR_BOX)) {
         shader->sort_priority = SH_SORT_SKY;
      }
      else if (shader->flags & SH_FOGPARMS) {
         shader->sort_priority = SH_SORT_FOG;
      }
      else if (shader->flags & SH_POLYGON_OFFSET) {
         shader->sort_priority = SH_SORT_DECAL;
      }
      else if (shader->stage_count > 0) {
         if (shader->stages[0].flags & SH_STAGE_BLENDFUNC) {
            shader->sort_priority = SH_SORT_ADDITIVE;
         }
         else if (shader->stages[0].flags & SH_STAGE_ALPHAFUNC) {
            shader->sort_priority = SH_SORT_SEE_THROUGH;
         }
         else {
            shader->sort_priority = SH_SORT_OPAQUE;
         }
      }/*
      else {
         shader remains undefined, this is for shaders with no stages
      }*/
   }
}



/***************************************************************************\

General Keyword:

   entityMergable

\***************************************************************************/



/* entityMergableParseFunc
 ***************************************************************************/
static bool entityMergableParseFunc(int unused0, char* unused1[])
{
   (void)unused0;
   (void)unused1;

   sh_message(SH_MSG_COMPATIBILITY, "entityMergable is not documented by Id");

   shader->flags |= SH_ENTITY_MERGABLE;

   return true;
}


#ifdef SH_PHOENIX_QUAKE1_EXT

/***************************************************************************\

General Keyword:

   _cachepic

\***************************************************************************/



/* cachepicParseFunc
 ***************************************************************************/
static bool cachepicParseFunc(int unused0, char* unused1[])
{
   (void)unused0;
   (void)unused1;

   sh_message(SH_MSG_COMPATIBILITY, "cachepic is an extension");

   shader->flags |= SH_CACHEPIC_EXT;

   return true;
}

#endif



/***************************************************************************\

Stage Keywords:

   map
   clampmap
   animMap

\***************************************************************************/

typedef enum {
   MAP_TYPE_NORMAL,

#ifdef SH_PHOENIX_QUAKE1_EXT
   MAP_TYPE_Q1_SOLID_SKY,
   MAP_TYPE_Q1_ALPHA_SKY,
#endif
};



/* request_map:
 ***************************************************************************/
static bool request_map(sh_stage_t* stage, const char* name, int m, bool clampmap, int type)
{
   // assert (m < SH_MAX_ANIMMAP_FRAMES)

   if (strlen(name) > SH_NAME_LEN) {
      sh_message(SH_MSG_ERROR, "map name '%s' exceeds SH_NAME_LEN (%d) characters", name, SH_NAME_LEN);
      return false;
   }
   else {
      strcpy(stage->maps[m].name, name);
   
      if (clampmap) stage->maps[m].flags |= SH_MAP_CLAMPMAP;

#ifdef SH_PHOENIX_QUAKE1_EXT
      if (type == MAP_TYPE_Q1_SOLID_SKY) {
         stage->maps[m].flags |= SH_MAP_Q1_SOLID_SKY_EXT;
      }
      else if (type == MAP_TYPE_Q1_ALPHA_SKY) {
         stage->maps[m].flags |= SH_MAP_Q1_ALPHA_SKY_EXT;
      }
#else
      (void)type;
#endif

      return true;
   }
}



/* mapParseFunc:
 ***************************************************************************/
static bool mapParseFunc(int unused, char* argv[])
{
   bool clampmap;

   (void)unused;

   clampmap = (stricmp(argv[0], "clampmap") == 0);

   if (clampmap) {
      if (stricmp(argv[1], "$lightmap") == 0) {
         sh_message(SH_MSG_ERROR, "$lightmap cannot be a clampmap");
         return true;
      }

      if (stricmp(argv[1], "$whiteimage") == 0) {
         sh_message(SH_MSG_ERROR, "$whiteimage cannot be a clampmap");
         return true;
      }
   }

   if (request_map( stage, argv[1], 0, clampmap, MAP_TYPE_NORMAL)) {
      stage->anim_map_count = 1;
   }

   return true;
}



/* animMapParseFunc:
 ***************************************************************************/
static bool animMapParseFunc(int argc, char* argv[])
{
   int i;
   int anim_map_count;

   sh_number_parse(argv[1], &stage->anim_map_fps);

   anim_map_count = 0;

   for (i = 2; i < argc; i++) {
      if (stricmp(argv[1], "$lightmap") == 0) {
         sh_message(SH_MSG_ERROR, "$lightmap cannot be an animMap");
         continue;
      }

      if (stricmp(argv[1], "$whiteimage") == 0) {
         sh_message(SH_MSG_ERROR, "$whiteimage cannot be an animMap");
         continue;
      }

      if (request_map(stage, argv[i], anim_map_count, false, MAP_TYPE_NORMAL)) {
         anim_map_count++;
      }
   }

   if (anim_map_count > 0) {
      if (anim_map_count > 8) {
         sh_message(SH_MSG_COMPATIBILITY, "having more than 8 maps is an extension");
      }

      if (anim_map_count > SH_MAX_ANIMMAP_FRAMES) {
         sh_message(SH_MSG_ERROR, "number of animMap frames exceeds SH_MAX_ANIMMAP_FRAMES (%d) in shader '%s'", SH_MAX_ANIMMAP_FRAMES, shader->name);
      }

      stage->anim_map_count = anim_map_count;
   }

   return true;
}



/***************************************************************************\

Stage Keyword:

   blendFunc add
   blendFunc filter
   blendFunc blend
   blendFunc <src> <dst>

\***************************************************************************/



typedef struct {
   char* name;
   int   factor;
   int   is_src;
   int   is_dst;
   int   map_alpha;
} blend_parse_table_t;



static blend_parse_table_t blend_parse_table[] = {
  /* name                      factor                     is_src is_dst map_alpha */
   { "GL_ONE",                 SH_BLEND_ONE,                 true,  true,  false },
   { "GL_ZERO",                SH_BLEND_ZERO,                true,  true,  false },
   { "GL_DST_COLOR",           SH_BLEND_DST_COLOR,           true,  false, false },
   { "GL_ONE_MINUS_DST_COLOR", SH_BLEND_ONE_MINUS_DST_COLOR, true,  false, false },
   { "GL_SRC_ALPHA",           SH_BLEND_SRC_ALPHA,           true,  true,  true  },
   { "GL_ONE_MINUS_SRC_ALPHA", SH_BLEND_ONE_MINUS_SRC_ALPHA, true,  true,  true  },
   { "GL_DST_ALPHA",           SH_BLEND_DST_ALPHA,           true,  true,  false },
   { "GL_ONE_MINUS_DST_ALPHA", SH_BLEND_ONE_MINUS_DST_ALPHA, true,  true,  false },
   { "GL_SRC_COLOR",           SH_BLEND_SRC_COLOR,           false, true,  false },
   { "GL_ONE_MINUS_SRC_COLOR", SH_BLEND_ONE_MINUS_SRC_COLOR, false, true,  false },
   { 0 }
};



/* ParseBlendFunc:
 ***************************************************************************/
int ParseBlendFunc
   (
   const char* arg,
   bool        is_src,
   bool*       map_alpha_out
   )
{
   blend_parse_table_t* table = blend_parse_table;

   while (table->name) {
      if (stricmp(table->name, arg) == 0) {
         if (is_src && !table->is_src) {
            sh_message(SH_MSG_WARNING, "'%s' is not a source blend factor");
            return -1;
         }

         if (!is_src && !table->is_dst) {
            sh_message(SH_MSG_WARNING, "'%s' is not a destination blend factor");
            return -1;
         }

         *map_alpha_out = table->map_alpha;

         return table->factor;
      }

      table++;
   }

   if (is_src) {
      sh_message(SH_MSG_WARNING, "unknown source factor '%s'", arg);
      return -1;
   }
   else {
      sh_message(SH_MSG_WARNING, "unknown destination factor '%s'", arg);
      return -1;
   }
}



/* blendFuncParseFunc:
 ***************************************************************************/
static bool blendFuncParseFunc(int argc, char* argv[])
{
   if (argc == 2) {
      // blendfunc add
      if (stricmp(argv[1], "add") == 0) {
         stage->blend_src = SH_BLEND_ONE;
         stage->blend_dst = SH_BLEND_ONE;

         stage->flags &= ~SH_STAGE_MAP_NEEDS_ALPHA;
      }
      // blendfunc filter
      else if (stricmp(argv[1], "filter") == 0) {
         stage->blend_src = SH_BLEND_DST_COLOR;
         stage->blend_dst = SH_BLEND_ZERO;

         stage->flags &= ~SH_STAGE_MAP_NEEDS_ALPHA;
      }
      // blendfunc blend
      else if (stricmp(argv[1], "blend") == 0) {
         stage->blend_src = SH_BLEND_SRC_ALPHA;
         stage->blend_dst = SH_BLEND_ONE_MINUS_SRC_ALPHA;

         stage->flags |= SH_STAGE_MAP_NEEDS_ALPHA;
      }
      else {
         sh_message(0, "unknown simplified blendFunc '%s'", argv[1]);
         return true;
      }
   }
   // blendfunc <src> <dst>
   else {
      bool src_alpha;
      bool dst_alpha;

      stage->blend_src = ParseBlendFunc(argv[1], true, &src_alpha);

      if (stage->blend_src < 0) return true;

      stage->blend_dst = ParseBlendFunc(argv[2], false, &dst_alpha);

      if (stage->blend_dst < 0) return true;

      if (src_alpha || dst_alpha) {
         stage->flags |= SH_STAGE_MAP_NEEDS_ALPHA;
      }
      else {
         stage->flags &= ~SH_STAGE_MAP_NEEDS_ALPHA;
      }

      stage->flags |= SH_STAGE_EXPLICIT_BLENDFUNC;
   }

   stage->flags |= SH_STAGE_BLENDFUNC;

   return true;
}



/***************************************************************************\

Stage Keyword:

   rgbGen identityLighting
   rgbGen identity
   rgbGen wave
   rgbGen entity
   rgbGen oneMinusEntity
   rgbGen vertex
   rgbGen oneMinusVertex
   rgbGen lightingDiffuse
   rgbGen _const

   alphaGen identity
   alphaGen wave
   alphaGen entity
   alphaGen oneMinusEntity
   alphaGen vertex
   alphaGen oneMinusVertex
   alphaGen lightingSpecular
   alphaGen _const

\***************************************************************************/



/* rgbaGenParseFunc:
 ***************************************************************************/
static bool rgbaGenParseFunc(int argc, char* argv[])
{
   sh_rgbagen_t* gen;
   bool          is_rgbgen;

   if (stricmp(argv[0], "rgbGen") == 0) {
      gen = &stage->rgbgen;
      is_rgbgen = true;
   }
   else {
      gen = &stage->alphagen;
      is_rgbgen = false;
   }

   // rgbGen identityLighting
   if (is_rgbgen && (stricmp(argv[1], "identityLighting") == 0)) {
      if (argc != 2) {
         sh_message(SH_MSG_ERROR, "identityLighting does not take any arguments");
         return true;
      }

      gen->func = SH_RGBAGEN_IDENTITY_LIGHTING;
   }
   // rgbGen|alphaGen identity
   else if (stricmp(argv[1], "identity") == 0) {
      if (argc != 2) {
         sh_message(SH_MSG_ERROR, "identity does not take any arguments");
         return true;
      }

      gen->func = SH_RGBAGEN_IDENTITY;
   }
   // rgbGen|alphaGen wave
   else if (stricmp(argv[1], "wave") == 0) {
      int form;

      if (argc != 7) {
         sh_message(SH_MSG_ERROR, "usage: %s wave <func> <base> <amp> <phase> <freq>", argv[0]);
         return true;
      }

      gen->func = SH_RGBAGEN_WAVE;

      form = ParseWaveForm(argv[2]);

      if (form < 0) return true;

      gen->args.wave.form = form;
     
      sh_number_parse(argv[3], &gen->args.wave.base);
      sh_number_parse(argv[4], &gen->args.wave.amp);
      sh_number_parse(argv[5], &gen->args.wave.phase);
      sh_number_parse(argv[6], &gen->args.wave.freq);
   }
   // rgbGen|alphaGen entity
   else if (stricmp(argv[1], "entity") == 0) {
      if (argc != 2) {
         sh_message(SH_MSG_ERROR, "entity does not take arguments");
         return true;
      }

      gen->func = SH_RGBAGEN_ENTITY;
   }
   // rgbGen|alphaGen oneMinusEntity
   else if (stricmp(argv[1], "oneMinusEntity") == 0) {
      if (argc != 2) {
         sh_message(SH_MSG_ERROR, "oneMinusEntity does not take arguments");
         return true;
      }

      gen->func = SH_RGBAGEN_ONE_MINUS_ENTITY;
   }
   // rgbGen|alphaGen vertex
   else if (stricmp(argv[1], "vertex") == 0) {
      if (argc != 2) {
         sh_message(SH_MSG_ERROR, "vertex does not take arguments");
         return true;
      }

      gen->func = SH_RGBAGEN_VERTEX;
   }
   // rgbGen|alphaGen oneMinusVertex
   else if (stricmp(argv[1], "oneMinusVertex") == 0) {
      if (argc != 2) {
         sh_message(SH_MSG_ERROR, "oneMinusVertex does not take arguments");
         return true;
      }

      gen->func = SH_RGBAGEN_ONE_MINUS_VERTEX;
   }
   // alphaGen portal
   else if (!is_rgbgen && (stricmp(argv[1], "portal") == 0)) {
      if (argc != 3) {
         sh_message(SH_MSG_ERROR, "usage: alphaGen portal <distance to opaque>");
         return true;
      }

      gen->func = SH_RGBAGEN_PORTAL;

      sh_number_parse(argv[2], &gen->args.portal.dist_to_opaque);
   }
   // rgbGen lightingDiffuse
   else if (is_rgbgen && stricmp(argv[1], "lightingDiffuse") == 0) {
      if (argc != 2) {
         sh_message(SH_MSG_ERROR, "lightingDiffuse does not take arguments");
         return true;
      }

      gen->func = SH_RGBAGEN_LIGHTING_DIFFUSE;
   }
   // alphaGen lightingSpecular
   else if (!is_rgbgen && stricmp(argv[1], "lightingSpecular") == 0) {
      sh_message(SH_MSG_COMPATIBILITY, "lightingSpecular is an extension");

      if (argc != 2) {
         sh_message(SH_MSG_ERROR, "lightingSpecular does not take arguments");
         return true;
      }

      gen->func = SH_RGBAGEN_LIGHTING_SPECULAR;
   }

#ifdef SH_PHOENIX_QUAKE1_EXT
   // alphaGen _oneMinusPortal
   else if (!is_rgbgen && (stricmp(argv[1], "_oneMinusPortal") == 0)) {
      sh_message(SH_MSG_COMPATIBILITY, "oneMinusPortal is an extension");

      if (argc != 3) {
         sh_message(SH_MSG_ERROR, "usage: alphaGen portal <distance to clear>");
         return true;
      }

      gen->func = SH_RGBAGEN_ONE_MINUS_PORTAL_EXT;

      sh_number_parse(argv[2], &gen->args.one_minus_portal.dist_to_clear);
   }
   else if (stricmp(argv[1], "_const") == 0) {
      // rgbGen _const
      sh_message(SH_MSG_COMPATIBILITY, "const is an extension");

      if (is_rgbgen) {
         if (argc != 5) {
            sh_message(SH_MSG_ERROR, "usage: rgbGen _const <red> <green> <blue>");
            return true;
         }

         ParseColor(gen->args.constant.color, argv[2], argv[3], argv[4]);
      }
      else {
         if (argc != 3) {
            sh_message(SH_MSG_ERROR, "usage: alphaGen _const <alpha>");
            return true;
         }

         ParseColor(gen->args.constant.color, argv[2], argv[2], argv[2]);
      }

      gen->func = SH_RGBAGEN_CONST_EXT;
   }
#endif

   // error
   else {
      sh_message(SH_MSG_ERROR, "unknown function '%s'", argv[1]);
      return true;
   }

   if (gen == &stage->rgbgen) {
      stage->flags |= SH_STAGE_EXPLICIT_RGBGEN;
   }
   else {
      stage->flags |= SH_STAGE_EXPLICIT_ALPHAGEN;
   }

   return true;
}



/* finalize_rgbgen:
 ***************************************************************************/
static void finalize_rgbgen()
{
   if (!(stage->flags & SH_STAGE_EXPLICIT_RGBGEN)) {
      if ((stage->flags & SH_STAGE_BLENDFUNC) && sh_is_blend_filter(stage)) {
         stage->rgbgen.func = SH_RGBAGEN_IDENTITY;
      }
      else {
         stage->rgbgen.func = SH_RGBAGEN_IDENTITY_LIGHTING;
      }
   }
}



/* finalize_alphagen:
 ***************************************************************************/
void finalize_alphagen()
{
   if (!(stage->flags & SH_STAGE_EXPLICIT_ALPHAGEN)) {
      stage->alphagen.func  = SH_RGBAGEN_IDENTITY;
   }
}



/***************************************************************************\

Stage Keyword:

    tcGen base
    tcGen lightmap
    tcGen environment
    tcGen vector
    tcGen _sky
    tcGen _Q1Turb

\***************************************************************************/



/* tcGenParseFunc:
 ***************************************************************************/
static bool tcGenParseFunc(int argc, char* argv[])
{
   // tcGen base
   if (stricmp(argv[1], "base") == 0) {
      if (argc != 2) {
         sh_message(SH_MSG_WARNING, "base does not take arguments");
         return true;
      }

      stage->tcgen.func = SH_TCGEN_BASE;
   }
   // tcGen lightmap
   else if (stricmp(argv[1], "lightmap") == 0) {
      if (argc != 2) {
         sh_message(SH_MSG_WARNING, "lightmap does not take arguments");
         return true;
      }

      stage->tcgen.func = SH_TCGEN_LIGHTMAP;
   }
   // tcGen environment
   else if (stricmp(argv[1], "environment") == 0) {
      if (argc != 2) {
         sh_message(SH_MSG_WARNING, "environment does not take arguments");
         return true;
      }

      stage->tcgen.func = SH_TCGEN_ENVIRONMENT;
   }
   // tcGen vector
   else if (stricmp(argv[1], "vector") == 0) {
      if (argc != 12) {
         sh_message(SH_MSG_WARNING, "usage: tcGen vector ( <sx> <sy> <sz> ) ( <tx> <ty> <tz> )");
         return true;
      }

      if (strcmp(argv[2],  "(") ||
          strcmp(argv[6],  ")") ||
          strcmp(argv[7],  "(") ||
          strcmp(argv[11], ")")) {
         sh_message(SH_MSG_WARNING, "make sure to put parentheses around the vectors and to space them out properly");
      }

      stage->tcgen.func = SH_TCGEN_VECTOR;
      sh_number_parse(argv[3],  &stage->tcgen.args.vector.sx);
      sh_number_parse(argv[4],  &stage->tcgen.args.vector.sy);
      sh_number_parse(argv[5],  &stage->tcgen.args.vector.sz);
      sh_number_parse(argv[8],  &stage->tcgen.args.vector.tx);
      sh_number_parse(argv[9],  &stage->tcgen.args.vector.ty);
      sh_number_parse(argv[10], &stage->tcgen.args.vector.tz);
   }

#ifdef SH_PHOENIX_QUAKE1_EXT
   // tcGen _sky
   else if (stricmp(argv[1], "_sky") == 0) {
      sh_message(SH_MSG_COMPATIBILITY, "sky is an extension");

      if (argc != 3) {
         sh_message(SH_MSG_WARNING, "usage: tcGen _sky <cloudheight>");
         return true;
      }

      stage->tcgen.func = SH_TCGEN_SKY_EXT;
      sh_number_parse(argv[2], &stage->tcgen.args.sky.cloud_height);
   }
   // tcGen _q1turb
   else if (stricmp(argv[1], "_q1turb") == 0) {
      int form;

      sh_message(SH_MSG_COMPATIBILITY, "q1turb is an extension");

      if (argc != 8) {
         sh_message(SH_MSG_WARNING, "usage: tcGen _q1turb <div> <func> <base> <amp> <phase> <freq>");
         return true;
      }

      stage->tcgen.func = SH_TCGEN_Q1TURB_EXT;

      sh_number_parse(argv[2], &stage->tcgen.args.q1turb.div);

      form = ParseWaveForm(argv[3]);

      if (form < 0) return true;

      sh_number_parse(argv[4], &stage->tcgen.args.q1turb.base);
      sh_number_parse(argv[5], &stage->tcgen.args.q1turb.amp);
      sh_number_parse(argv[6], &stage->tcgen.args.q1turb.phase);
      sh_number_parse(argv[7], &stage->tcgen.args.q1turb.freq);
   }
#endif

   // error
   else {
      sh_message(SH_MSG_WARNING, "unknown tcGen '%s'", argv[1]);
      return true;
   }

   stage->flags |= SH_STAGE_EXPLICIT_TCGEN;

   return true;
}



/* finalize_tcgen:
 ***************************************************************************/
static void finalize_tcgen()
{
   if (!(stage->flags & SH_STAGE_EXPLICIT_TCGEN)) {
      if (stage->anim_map_count == 1) {
         if (stricmp(stage->maps[0].name, "$lightmap") == 0) {
            stage->tcgen.func = SH_TCGEN_LIGHTMAP;
         }
         else {
            stage->tcgen.func = SH_TCGEN_BASE;
         }
      }
      else {
         stage->tcgen.func = SH_TCGEN_BASE;
      }
   }
}



/***************************************************************************\

Stage Keyword:

    tcMod rotate <speed>
    tcMod scale <s> <t>
    tcMod scroll <s> <t>
    tcMod stretch <func> <base> <amp> <phase> <freq>
    tcMod transform <m00> <m01> <m10> <m11> <t0> <t1>
    tcMod turb <base> <amp> <phase> <freq>

\***************************************************************************/



/* tcModParseFunc:
 ***************************************************************************/
static bool tcModParseFunc(int argc, char* argv[])
{
   sh_tcmod_t* tcmod;
   sh_tcmod_t  ignored_tcmod;

   if (stage->tcmod_count < SH_MAX_TCMODS) {
      tcmod = &stage->tcmods[stage->tcmod_count];
   }
   else {
      sh_message(SH_MSG_ERROR, "number of tcMod commands exceeds SH_MAX_TCMODS (%d) in shader %s", SH_MAX_TCMODS, shader->name);
      tcmod = &ignored_tcmod;
   }

   // tcMod rotate <speed>
   if (stricmp(argv[1], "rotate") == 0) {
      if (argc != 3) {
         sh_message(SH_MSG_WARNING, "usage: tcMod rotate <speed>");
         return true;
      }

      tcmod->func = SH_TCMOD_ROTATE;
      sh_number_parse(argv[2], &tcmod->args.rotate.speed);
   }
   // tcMod scale <s> <t>
   else if (stricmp(argv[1], "scale") == 0) {
      if (argc != 4) {
         sh_message(SH_MSG_WARNING, "usage: tcMod scale <s> <t>");
         return true;
      }

      tcmod->func = SH_TCMOD_SCALE;
      sh_number_parse(argv[2], &tcmod->args.scale.s);
      sh_number_parse(argv[3], &tcmod->args.scale.t);
   }
   // tcMod scroll <s> <t>
   else if (stricmp(argv[1], "scroll") == 0) {
      if (argc != 4) {
         sh_message(SH_MSG_WARNING, "usage: tcMod scroll <s> <t>");
         return true;
      }

      tcmod->func = SH_TCMOD_SCROLL;
      sh_number_parse(argv[2], &tcmod->args.scroll.s);
      sh_number_parse(argv[3], &tcmod->args.scroll.t);
   }
   // tcMod stretch <func> <base> <amp> <phase> <freq>
   else if (stricmp(argv[1], "stretch") == 0) {
      int form;

      if (argc != 7) {
         sh_message(SH_MSG_WARNING, "usage: tcMod stretch <func> <base> <amp> <phase> <freq>");
         return true;
      }

      tcmod->func = SH_TCMOD_STRETCH;

      form = ParseWaveForm(argv[2]);

      if (form < 0) return true;

      tcmod->args.stretch.form  = form;
      sh_number_parse(argv[3], &tcmod->args.stretch.base);
      sh_number_parse(argv[4], &tcmod->args.stretch.amp);
      sh_number_parse(argv[5], &tcmod->args.stretch.phase);
      sh_number_parse(argv[6], &tcmod->args.stretch.freq);
   }
   // tcMod transform <m00> <m01> <m10> <m11> <t0> <t1>
   else if (stricmp(argv[1], "transform") == 0) {
      if (argc != 8) {
         sh_message(SH_MSG_WARNING, "usage: tcMod transform <m00> <m01> <m10> <m11> <t0> <t1>");
         return true;
      }

      tcmod->func = SH_TCMOD_TRANSFORM;
      sh_number_parse(argv[3], &tcmod->args.transform.m00);
      sh_number_parse(argv[3], &tcmod->args.transform.m01);
      sh_number_parse(argv[4], &tcmod->args.transform.m10);
      sh_number_parse(argv[5], &tcmod->args.transform.m11);
      sh_number_parse(argv[6], &tcmod->args.transform.t0);
      sh_number_parse(argv[7], &tcmod->args.transform.t1);
   }
   // tcMod turb <base> <amp> <phase> <freq>
   else if (stricmp(argv[1], "turb") == 0) {
      if (argc != 6) {
         sh_message(SH_MSG_WARNING, "usage: tcMod turb <base> <amp> <phase> <freq>");
         return true;
      }

      tcmod->func = SH_TCMOD_TURB;
      sh_number_parse(argv[2], &tcmod->args.turb.base);
      sh_number_parse(argv[3], &tcmod->args.turb.amp);
      sh_number_parse(argv[4], &tcmod->args.turb.phase);
      sh_number_parse(argv[5], &tcmod->args.turb.freq);
   }
   // error
   else {
      sh_message(SH_MSG_WARNING, "unknown tcMod '%s'", argv[2]);
      return true;
   }

   if (stage->tcmod_count < SH_MAX_TCMODS) stage->tcmod_count++;

   return true;
}


/***************************************************************************\

Stage Keyword:

    depthFunc lequal
    depthFunc equal

\***************************************************************************/



/* depthFuncParseFunc
 ***************************************************************************/
static bool depthFuncParseFunc(int unused, char* argv[])
{
   (void)unused;

   if (stricmp(argv[1], "lequal") == 0) {
      stage->depthfunc = SH_DEPTHFUNC_LEQUAL;
   }
   else if (stricmp(argv[1], "equal") == 0) {
      stage->depthfunc = SH_DEPTHFUNC_EQUAL;
   }
   else {
      sh_message(SH_MSG_WARNING, "unknown depthFunc '%s'", argv[1]);
      return true;
   }

   stage->flags |= SH_STAGE_EXPLICIT_DEPTHFUNC;

   return true;
}



/* finalize_depthfunc
 ***************************************************************************/
void finalize_depthfunc()
{
   if (!(stage->flags & SH_STAGE_EXPLICIT_DEPTHFUNC)) {
      stage->depthfunc = SH_DEPTHFUNC_LEQUAL;
   }
}



/***************************************************************************\

Stage Keyword:

    depthWrite

\***************************************************************************/



/* depthWriteParseFunc
 ***************************************************************************/
static bool depthWriteParseFunc(int unused0, char* unused1[])
{
   (void)unused0;
   (void)unused1;

   stage->flags |= SH_STAGE_DEPTHWRITE;
   stage->flags |= SH_STAGE_EXPLICIT_DEPTHWRITE;
   return true;
}



/* finalize_depthwrite
 ***************************************************************************/
static void finalize_depthwrite()
{
   if (!(stage->flags & SH_STAGE_EXPLICIT_DEPTHWRITE) && sh_is_opaque_stage(stage)) {
      stage->flags |= SH_STAGE_DEPTHWRITE;
   }
}



/***************************************************************************\

Stage Keyword:

    detail

\***************************************************************************/



/* detailParseFunc
 ***************************************************************************/
static bool detailParseFunc(int unused0, char* unused1[])
{
   (void)unused0;
   (void)unused1;

   stage->flags |= SH_STAGE_DETAIL;

   return true;
}



/***************************************************************************\

Stage Keyword:

    alphaFunc

\***************************************************************************/



/* alphaFuncParseFunc
 ***************************************************************************/
static bool alphaFuncParseFunc(int unused, char* argv[])
{
   (void)unused;

   if (stricmp(argv[1], "GT0") == 0) {
      stage->alphafunc = SH_ALPHAFUNC_GT0;
   }
   else if (stricmp(argv[1], "LT128") == 0) {
      stage->alphafunc = SH_ALPHAFUNC_LT128;
   }
   else if (stricmp(argv[1], "GE128") == 0) {
      stage->alphafunc = SH_ALPHAFUNC_GE128;
   }
   else {
      sh_message(SH_MSG_WARNING, "unknown alphaFunc '%s'", argv[1]);
      return true;
   }

   stage->flags |= SH_STAGE_MAP_NEEDS_ALPHA;
   stage->flags |= SH_STAGE_ALPHAFUNC;

   return true;
}



#ifdef SH_PHOENIX_QUAKE1_EXT

/***************************************************************************\

Stage Keyword:

    _Q1SkyMap

\***************************************************************************/



/* q1SkyMapParseFunc
 ***************************************************************************/
static bool q1SkyMapParseFunc(int unused, char* argv[])
{
   bool is_alpha_map;

   (void)unused;

   sh_message(SH_MSG_COMPATIBILITY, "Q1SkyMap is an extension");

   if (strnicmp(argv[2], "$bsp#sky", 8) != 0) {
      sh_message(SH_MSG_WARNING, "Q1SkyMap only works with sky maps from BSP files");
      return true;
   }

   if (stricmp(argv[1], "alpha") == 0) {
      is_alpha_map = true;
   }
   else if (stricmp(argv[1], "solid") == 0) {
      is_alpha_map = false;
   }
   else {
      sh_message(SH_MSG_WARNING, "type must by solid or alpha");
      return true;
   }

   if (request_map(stage, argv[2], 0, false, is_alpha_map?MAP_TYPE_Q1_ALPHA_SKY:MAP_TYPE_Q1_SOLID_SKY)) {
      stage->anim_map_count = 1;
   }

   return true;
}



/***************************************************************************\

Stage Keyword:

   _filter

\***************************************************************************/



typedef struct filter_parse_table_s {
   char* name;
   int   value;
   bool  is_mag_filter;
} filter_parse_table_t;



filter_parse_table_t filter_parse_table[] =
{
   { "GL_NEAREST",                SH_FILTER_NEAREST, true                 },
   { "GL_LINEAR",                 SH_FILTER_LINEAR,  true                 },
   { "GL_NEAREST_MIPMAP_NEAREST", SH_FILTER_NEAREST_MIPMAP_NEAREST, false },
   { "GL_NEAREST_MIPMAP_LINEAR",  SH_FILTER_NEAREST_MIPMAP_LINEAR,  false },
   { "GL_LINEAR_MIPMAP_NEAREST",  SH_FILTER_LINEAR_MIPMAP_NEAREST,  false },
   { "GL_LINEAR_MIPMAP_LINEAR",   SH_FILTER_LINEAR_MIPMAP_LINEAR,   false },
   { 0 }
};



/* ParseFilter
 ***************************************************************************/
int ParseFilter(const char* name, bool is_mag_filter)
{
   filter_parse_table_t* table;

   table = filter_parse_table;

   while (table->name) {
      if (stricmp(table->name, name) == 0) {
         if (is_mag_filter && !table->is_mag_filter) {
            sh_message(SH_MSG_WARNING, "'%s' can not be a magnification filter", name);
            return -1;
         }

         return table->value;
      }

      table++;
   }

   sh_message(SH_MSG_WARNING, "no such filter '%s'", name);

   return -1;
}



/* filterParseFunc
 ***************************************************************************/
static bool filterParseFunc(int unused, char* argv[])
{
   (void)unused;

   sh_message(SH_MSG_COMPATIBILITY, "filter is an extension");

   stage->min_filter = ParseFilter(argv[0], false);

   if (stage->min_filter < 0) {
      return true;
   }

   stage->mag_filter = ParseFilter(argv[1], true);

   if (stage->mag_filter < 0) {
      return true;
   }

   stage->flags |= SH_STAGE_FILTER_EXT;

   return true;
}

#endif



/***************************************************************************\

End of Command Parse Functions

\***************************************************************************/



/* match
 ***************************************************************************/
bool match(_SH_TOKEN id)
{
   if (id == lookahead) {
      lookahead = _sh_scan();
      return true;
   }
   else {
      switch(id) {
         case _SH_TOK_STRING:
            expected("string", token.string);
            break;
         case _SH_TOK_EOF:
            expected("<EOF>", token.string);
            break;
         case '{':
            expected("{", token.string);
            break;
         case '}':
            expected("}", token.string);
            break;
         default:
            // assert: this should never happen
            expected("<opps!>", token.string);
            break;
      }

      return false;
   }
}



typedef int (*command_parse_func_t)(int argc, char* argv[]);

typedef struct {
   char*                name;
   int                  min_argc;
   int                  max_argc;
   bool             is_stage_command;
   bool             is_ignored;
   command_parse_func_t Parse;
} command_parse_table_t;



static command_parse_table_t command_table[] =
{
    // global commands
    { "skyParms",             4,  4, false, false, skyParmsParseFunc       },
    { "cull",                 1,  2, false, false, cullParseFunc           },
    { "deformVertexes",       2, 10, false, false, deformVertexesParseFunc },
    { "fogparms",             5,  7, false, false, fogparmsParseFunc       },
    { "nopicmip",             1,  1, false, false, nopicmipParseFunc       },
    { "nomipmaps",            1,  1, false, false, nomipmapsParseFunc      },
    { "polygonOffset",        1,  1, false, false, polygonOffsetParseFunc  },
    { "portal",               1,  1, false, false, portalParseFunc         },
    { "sort",                 2,  2, false, false, sortParseFunc           },

    // undocumented global commands
    { "entityMergable",       1,  1, false, false, entityMergableParseFunc },
    { "fogonly",              1,  1, false, true,  NULL },
    { "light",                2,  2, false, true,  NULL },
    { "foggen",               6,  6, false, true,  NULL },
    { "lightning",            1,  1, false, true,  NULL },
    { "cloudparms",           3,  3, false, true,  NULL },
    { "sky",                  2,  2, false, true,  NULL },

#ifdef SH_PHOENIX_QUAKE1_EXT
    // extensions
    { "_cachepic",            1,  1, false, false, cachepicParseFunc },
    { "_skyRotate",           4,  4, false, true,  NULL              },
    { "_skyBoxFaces",         1,  7, false, true,  NULL              },
#endif

    // q3bsp keywords (ignored by the renderer)
    { "tessSize",             1,  1, false, true,  NULL },
    { "q3map_backshader",     1,  1, false, true,  NULL },
    { "q3map_globaltexture",  1,  1, false, true,  NULL },
    { "q3map_sun",            1,  1, false, true,  NULL },
    { "q3map_surfaceLight",   1,  1, false, true,  NULL },
    { "q3map_lightimage",     1,  1, false, true,  NULL },
    { "q3map_lightsubdivide", 1,  1, false, true,  NULL },
    { "surfaceparm",          1,  1, false, true,  NULL },

    // undocumented q3bsp keywords
    { "q3map_backsplash",     1,  1, false, true,  NULL },
    { "q3map_flare",          2,  2, false, true,  NULL },

    // editor commands (ignored by the renderer)
    { "qer_editorimage",      1,  1, false, true,  NULL },
    { "qer_nocarve",          1,  1, false, true,  NULL },
    { "qer_trans",            1,  1, false, true,  NULL },

    // stage commands
    { "map",                  2,  2, true,  false, mapParseFunc        },
    { "clampmap",             2,  2, true,  false, mapParseFunc        },
    { "animMap",              3, 12, true,  false, animMapParseFunc    },
    { "blendFunc",            2,  3, true,  false, blendFuncParseFunc  },
    { "rgbGen",               2,  7, true,  false, rgbaGenParseFunc    },
    { "alphaGen",             2,  7, true,  false, rgbaGenParseFunc    },
    { "tcGen",                2, 12, true,  false, tcGenParseFunc      },
    { "tcMod",                3,  8, true,  false, tcModParseFunc      },
    { "depthFunc",            2,  2, true,  false, depthFuncParseFunc  },
    { "depthWrite",           1,  1, true,  false, depthWriteParseFunc },
    { "detail",               1,  1, true,  false, detailParseFunc     },
    { "alphaFunc",            2,  2, true,  false, alphaFuncParseFunc  },

#ifdef SH_PHOENIX_QUAKE1_EXT
    // extensions
    { "_q1SkyMap",            2,  2, true,  false, q1SkyMapParseFunc },
    { "_filter",              3,  3, true,  false, filterParseFunc   },
    { "alphaMap",             2,  2, true,  true, NULL },
#endif

    { 0 }
};



/* DispatchCommandParser
 ***************************************************************************/
static void DispatchCommandParser(int argc, char* argv[SH_MAX_SHADER_ARGS], bool is_stage_command)
{
   command_parse_table_t* command;

   for (command = command_table; ; command++) {
      if (!command->name) {
         sh_message(SH_MSG_WARNING, "unrecognized command '%s'", argv[0]);
         return;
      }

      if (is_stage_command && !command->is_stage_command) {
         if (stricmp(command->name, argv[0]) == 0) {
            sh_message(SH_MSG_WARNING, "'%s' is not a stage command", argv[0]);
            return;
         }
         else {
            continue;
         }
      }

      if (!is_stage_command &&  command->is_stage_command) {
         if (stricmp(command->name, argv[0]) == 0) {
            sh_message(SH_MSG_WARNING, "'%s' is not a global command", command);
            return;
         }
         else {
            continue;
         }
      }

      if (stricmp(command->name, argv[0]) == 0) {
         if (command->is_ignored) {
         }
         else if (argc > command->max_argc) {
            sh_message(SH_MSG_WARNING, "too many arguments for '%s'", argv[0]);
         }
         else if (argc < command->min_argc) {
            sh_message(SH_MSG_WARNING, "too few arguments for '%s'", argv[0]);
         }
         else {
            sh_message_set_prefix(command->name);
            command->Parse(argc, argv);
            sh_message_set_prefix(NULL);
         }

         return;
      }
   }
}





/* ParseArguments
 *   command := STRING (STRING)* NEWLINE
 ***************************************************************************/
static bool ParseArguments(int* pargc, char* pargv[SH_MAX_SHADER_ARGS])
{
   *pargc = 1;
   strncpy(pargv[0], token.string, SH_TOKEN_LEN);

   match(_SH_TOK_STRING);

   while (!token.is_newline) {
      if (*pargc < SH_MAX_SHADER_ARGS) {
         strncpy(pargv[*pargc], token.string, SH_TOKEN_LEN);
         (*pargc)++;
      }

      if (!match(_SH_TOK_STRING)) return false;
   }

   return true;
}



typedef struct arg_s {
   char string[SH_TOKEN_LEN+1];
} arg_t;


/* parse_shader_command
 ***************************************************************************/
static bool parse_shader_command(bool is_stage_command)
{
   int argc;

   static arg_t argv_str[SH_MAX_SHADER_ARGS];
   static char* argv[SH_MAX_SHADER_ARGS] = {
      argv_str[0].string,
      argv_str[1].string,
      argv_str[2].string,
      argv_str[3].string,
      argv_str[4].string,
      argv_str[5].string,
      argv_str[6].string,
      argv_str[7].string,
      argv_str[8].string,
      argv_str[9].string,
      argv_str[10].string,
      argv_str[11].string,
   };

   if (ParseArguments(&argc, argv)) {
      DispatchCommandParser(argc, argv, is_stage_command);
      return true;
   }
   else {
      return false;
   }
}



/* initialize_stage
 ***************************************************************************/
static void initialize_stage()
{
   if (shader->stage_count < SH_MAX_STAGES) {
      stage = &shader->stages[shader->stage_count];
   }
   else {
      if (shader->stage_count == SH_MAX_STAGES) {
         sh_message(SH_MSG_ERROR, "number of stages exceeds SH_MAX_STAGES (%d) in shader '%s'", SH_MAX_STAGES, shader->name);
      }

      stage = &ignored_stage;
   }

   memset(stage, 0, sizeof(sh_stage_t));
}



/* finalize_stage
 ***************************************************************************/
static void finalize_stage()
{
   int i, j;

   finalize_rgbgen();
   finalize_alphagen();
   finalize_tcgen();
   finalize_depthfunc();
   finalize_depthwrite();

   shader->stage_count++;

   if (sh_is_opaque_stage(stage)) {
      if (shader->stage_count > 1) {
         sh_message(SH_MSG_WARNING, "stage %d will overwrite previous stages",
            shader->stage_count);
      }
      else if (shader->flags & SH_FAR_BOX) {
         sh_message(SH_MSG_WARNING, "stage 1 will overwrite the sky far box");
      }
   }

   for (i = 0; i < shader->stage_count; i++) {
      if (shader->stages[i].flags & SH_STAGE_MAP_NEEDS_ALPHA) {
         for (j = 0; j < shader->stages[i].anim_map_count; j++) {
            shader->stages[i].flags |= SH_MAP_ALPHA;
         }
      }
   }

   if (sh_is_blend_src_only(stage)) {
      sh_message(SH_MSG_VERBOSE, "blendFunc GL_ONE GL_ZERO will look equivalent to no blendfunc, but the Z buffer will not be written");
   }
   else if (sh_is_blend_dst_only(stage)) {
      sh_message(SH_MSG_VERBOSE, "blendFunc GL_ZERO GL_ONE means that no color will be written for this stage, only the Z buffer");
   }

   stage = NULL;
}



/* parse_shader_stage
 *   shader_stage := '{' stage_command* '}'
 ***************************************************************************/
static bool parse_shader_stage(void)
{
   match('{');
   initialize_stage();

   while (lookahead != '}') {
      if (lookahead == _SH_TOK_STRING) {
         parse_shader_command(true);
      }
      else {
         expected("stage command", token.string);
         return false;
      }
   }

   finalize_stage();
   match('}');

   return true;
}



/* initialize_shader
 ***************************************************************************/
static bool initialize_shader(const char* name)
{
   shader = sh_alloc();

   if (shader == NULL) return false;

   memset(shader, 0, sizeof(shader_t));
   strcpy(shader->name, name);

   return true;
}



/* finalize_shader:
 ***************************************************************************/
static void finalize_shader()
{
   finalize_skyparms(shader);
   finalize_sort(shader);
   finalize_cull(shader);
}



/* parse_shader
 *   shader := STRING '{' (global_command|stage)* '}'
 ***************************************************************************/
bool parse_shader(sh_approve_func_t approve)
{
   char name[SH_TOKEN_LEN+1];
   
   strncpy(name, token.string, SH_TOKEN_LEN);
   name[SH_TOKEN_LEN] = '\0';

   match(_SH_TOK_STRING);

   if (strlen(name) <= SH_NAME_LEN) {
      if (approve && !approve(name)) return false;

      if (!initialize_shader(name)) return false;

      sh_message(SH_MSG_VERBOSE, "parsing shader '%s'", name);
   }
   else {
      sh_message(SH_MSG_ERROR, "shader name '%s' too long", name);
      return false;
   }

   if (lookahead == '{') {
      match('{');

      while (lookahead != '}') {
         if (lookahead == '{') {
            if (!parse_shader_stage()) return false;
         }
         else if (lookahead == _SH_TOK_STRING) {
            if (!parse_shader_command(false)) return false;
         }
         else {
            expected("global command or stage", token.string);
            return false;
         }
      }

      match('}');
      finalize_shader();
      return true;
   }
   else {
      expected("beginning of shader body", token.string);
      return false;
   }
}



/* sh_parse:
 *   shaders := (shader)* EOF
 ***************************************************************************/
size_t sh_parse(sh_compile_func_t compile, sh_approve_func_t approve)
{
   int count = 0;

   _sh_begin_scan();
   parse_stop = false;
   lookahead  = _sh_scan();

   while (lookahead != _SH_TOK_EOF) {
      if (lookahead == _SH_TOK_STRING) {
         if (parse_shader(approve)) {
            if (compile) compile(shader);

            _sh_alloc_keep();
            count++;

            if (parse_stop) {
               return count;
            }
            else {
               continue;
            }
         }
      }
      else {
         expected("shader name", token.string);
      }

      _sh_skip();
   }

   match(_SH_TOK_EOF);

   _sh_end_scan();
   lookahead = 0;

   return count;
}



/* sh_parse_stop:
 ***************************************************************************/
void sh_parse_stop(void)
{
   parse_stop = true;
}


