// fenix@io.com: vertex arrays

/*
Copyright (C) 1996-1997 Id Software, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

#include "quakedef.h"

#include "varrays.h"
#include "shaders.h"

#define R_MIN_VERTEX_COUNT 128



cvar_t  r_vertex_arrays = { "r_vertex_arrays", "0" };

qboolean is_vertex_arrays_supported = false;

vavertex_t* global_vertex_array;
unsigned*   global_indice_array;

int global_vertex_array_size;
int global_indice_array_size;

texcoord_t* global_lightmap_texcoords;

int mode_table[] = {
    /* R_POINTS         */ GL_POINTS,
    /* R_LINES          */ GL_LINES,
    /* R_LINE_STRIP     */ GL_LINE_STRIP,
    /* R_LINE_LOOP      */ GL_LINE_LOOP,
    /* R_TRIANGLES      */ GL_TRIANGLES,
    /* R_TRIANGLE_STRIP */ GL_TRIANGLE_STRIP,
    /* R_TRIANGLE_FAN   */ GL_TRIANGLE_FAN,
    /* R_QUADS          */ GL_QUADS,
    /* R_QUAD_STRIP     */ GL_QUAD_STRIP,
    /* R_POLYGON        */ GL_POLYGON
};


int blendfunc_table[] = {
   /* R_ONE                 */ GL_ONE,
   /* R_ZERO                */ GL_ZERO,
   /* R_DST_COLOR           */ GL_DST_COLOR,
   /* R_ONE_MINUS_DST_COLOR */ GL_ONE_MINUS_DST_COLOR,
   /* R_SRC_ALPHA           */ GL_SRC_ALPHA,
   /* R_ONE_MINUS_SRC_ALPHA */ GL_ONE_MINUS_SRC_ALPHA,
   /* R_ONE_SRC_COLOR       */ GL_SRC_COLOR,
   /* R_ONE_MINUS_SRC_COLOR */ GL_ONE_MINUS_SRC_COLOR,
};

int func_table[] = {
   /* R_NEVER    */ GL_NEVER,
   /* R_ALWAYS   */ GL_ALWAYS,
   /* R_LESS     */ GL_LESS,
   /* R_LEQUAL   */ GL_LEQUAL,
   /* R_EQUAL    */ GL_EQUAL,
   /* R_GEQUAL   */ GL_GEQUAL,
   /* R_GREATER  */ GL_GREATER,
   /* R_NOTEQUAL */ GL_NOTEQUAL
};

int polygon_face_table[] = {
   /* R_FRONT */          GL_FRONT,
   /* R_BACK  */          GL_BACK,
   /* R_FRONT_AND_BACK */ GL_FRONT_AND_BACK
};

vavertex_t* current_vertex_array;
unsigned current_vertex_fields;

texcoord_t* current_lightmap_texcoords;



/* R_InitVertexArrays
 ****************************************************************************/
void R_InitVertexArrays(void)
{
    int pnum = COM_CheckParm("-varraysize");

    if (pnum != 0)
    {
        global_vertex_array_size = Q_atoi(com_argv[pnum+1]);

        if (global_vertex_array_size < R_MIN_VERTEX_COUNT)
        {
            global_vertex_array_size = R_MIN_VERTEX_COUNT;
        }
    }
    else
    {
        global_vertex_array_size = R_MIN_VERTEX_COUNT;
    }

    global_indice_array_size = global_vertex_array_size;

    global_vertex_array = Hunk_AllocName(sizeof(vavertex_t) * global_vertex_array_size, "vertex_array");
    global_indice_array = Hunk_AllocName(sizeof(vavertex_t) * global_indice_array_size, "indice_array");

    global_lightmap_texcoords = Hunk_AllocName(sizeof(texcoord_t) * global_vertex_array_size, "lightmap_texcoords");
}



/* R_VertexArray
 ****************************************************************************/
void R_VertexArray(vavertex_t* varray, unsigned fields)
{
   current_vertex_array  = varray;
   current_vertex_fields = 0;

   if (fields & R_VERTEX3) {
      glEnableClientState(GL_VERTEX_ARRAY);
      glVertexPointer(3, GL_FLOAT, sizeof(vavertex_t), &varray->x);
      current_vertex_fields |= R_VERTEX3;
   }
   else if (fields & R_VERTEX2) {
      glEnableClientState(GL_VERTEX_ARRAY);
      glVertexPointer(2, GL_FLOAT, sizeof(vavertex_t), &varray->x);
      current_vertex_fields |= R_VERTEX2;
   }
   else {
      glDisableClientState(GL_VERTEX_ARRAY);
   }

   if (fields & R_COLOR4) {
      glEnableClientState(GL_COLOR_ARRAY);
      glColorPointer(4, GL_UNSIGNED_BYTE, sizeof(vavertex_t), &varray->r);
      current_vertex_fields |= R_COLOR4;
   }
   else if (fields & R_COLOR3) {
      glEnableClientState(GL_COLOR_ARRAY);
      glColorPointer(3, GL_UNSIGNED_BYTE, sizeof(vavertex_t), &varray->r);
      current_vertex_fields |= R_COLOR3;
   }
   else {
      glDisableClientState(GL_COLOR_ARRAY);
   }

   if (fields & R_TEXCOORD2) {
      glEnableClientState(GL_TEXTURE_COORD_ARRAY);
      glTexCoordPointer(2, GL_FLOAT, sizeof(vavertex_t), &varray->s);
      current_vertex_fields |= R_TEXCOORD2;
   }
   else {
      glDisableClientState(GL_TEXTURE_COORD_ARRAY);
   }
}



/* R_Lightmap:
 ****************************************************************************/
void R_Lightmap(texcoord_t* texcoords, int lightmap)
{
   current_lightmap_texcoords = texcoords;
//   texture_maps[R_MAP_LIGHTMAP].index = texture_maps[lightmap].index;
}



/* SHADER_R_DrawArray
 ****************************************************************************/
void SHADER_R_DrawArray(R_MODE mode, int first, size_t count)
{
   int i, j;

   shader_stage_t* stage = current_shader->stages;

   qboolean is_blend_enabled       = false;
   qboolean is_blend_dirty         = false;

   qboolean is_alpha_test_enabled  = true;
   qboolean is_alpha_test_dirty    = false;

   qboolean is_depth_test_dirty    = false;

   qboolean is_depth_write_enabled = true;

   qboolean is_cull_face_enabled   = true;
   qboolean is_cull_face_dirty     = false;

   unsigned saved_vertex_fields;

   qboolean is_last_stage          = false;

   // cull
   if (current_shader->flags & R_SHADER_HAS_CULL) {
      if (!is_cull_face_enabled) {
         glEnable(GL_CULL_FACE);
         is_cull_face_enabled = true;
      }

      if (current_shader->cull_face != GL_FRONT) {
         glCullFace(polygon_face_table[current_shader->cull_face]);
         is_cull_face_dirty = true;
      }
   }
   else if (is_cull_face_enabled) {
      glDisable(GL_CULL_FACE);
      is_cull_face_enabled = false;
   }

   // save matrix
   if (current_shader->flags & R_SHADER_SAVE_MATRIX) {
      glPushMatrix();
   }

   // deformVertexes
   if (current_shader->num_vxmods > 0) {
      for (j = 0; j < current_shader->num_vxmods; j++) {
         current_shader->vxmods[j].func(
            &current_shader->vxmods[j],
            mode,
            current_vertex_array,
            first,
            count);
      }
   }

   // polygonOffset
   if (current_shader->flags & R_SHADER_POLYGON_OFFSET) glEnable(GL_POLYGON_OFFSET_FILL);

   // skyParms
   if (current_shader->flags & R_SHADER_HAS_SKYPARMS) {
      // TODO: sub-divide the polygons for the sky

      if (current_shader->flags & R_SHADER_HAS_FARBOX) {
         // TODO: draw the farbox as the first layer
      }
   }

   // save fields so they can be restored
   saved_vertex_fields = current_vertex_fields;

   for (i = 0; i < current_shader->num_stages; i++) {
      unsigned fields;

      // skip if this is detail stage and the player does not want detail
      if ((stage->flags & R_DETAIL) && !r_detail.value) continue;

      if (i == current_shader->num_stages-1) is_last_stage = true;

      // setup texture map
      R_Bind(stage);

      // adjust enabled fields
      fields = ((stage->on_fields|saved_vertex_fields) & ~(stage->off_fields));

      if (fields != current_vertex_fields) {
         R_VertexArray(current_vertex_array, fields);
      }

      // save colors
      if (!is_last_stage &&
          (stage->flags & R_STAGE_SAVE_COLORS) &&
          (current_vertex_fields & (R_COLOR3|R_COLOR4))) {
         R_SaveRGB(current_vertex_array, first, count);
      }

      // save alpha
      if (!is_last_stage &&
          (stage->flags & R_STAGE_SAVE_ALPHA) &&
          (current_vertex_fields & R_COLOR4)) {
         R_SaveAlpha(current_vertex_array, first, count);
      }

      // perform color and alpha generation
      stage->rgbagen_func(
         &stage->rgbgen,
         &stage->alphagen,
         current_vertex_array,
         first,
         count);

      // perform color modification (not implemented)

      // perform alpha modification (not implemented)

      // save texcoords
      if (!is_last_stage &&
          (stage->flags & R_STAGE_SAVE_TEXCOORDS) &&
          (current_vertex_fields & R_TEXCOORD2)) {
         R_SaveST(current_vertex_array, first, count);
      }

      // perform texture coordinate generation
      stage->tcgen.func(&stage->tcgen, current_vertex_array, first, count);

      // perform texture coordinate modifications
      if (stage->num_tcmods > 0) {
         for (j = 0; j < stage->num_tcmods; j++) {
            stage->tcmods[j].func(&stage->tcmods[j], current_vertex_array, first, count);
         }

         R_ApplyShaderTexMatrix(current_vertex_array, first, count);
      }

      // setup blend state
      if (stage->flags & R_HAS_BLENDFUNC) {
         if (!is_blend_enabled) {
            glEnable(GL_BLEND);
            is_blend_enabled = true;
         }
       
         glBlendFunc(
            blendfunc_table[stage->blend_src],
            blendfunc_table[stage->blend_dst]);

         is_blend_dirty = true;
      }
      else if (is_blend_enabled) {
         glDisable(GL_BLEND);
         is_blend_enabled = false;
      }

      // setup depth test
      if (stage->depth_test_func != R_LEQUAL) {
         glDepthFunc(func_table[stage->depth_test_func]);
         is_depth_test_dirty = true;
      }
      else if (is_depth_test_dirty) {
         glDepthFunc(GL_LEQUAL);
      }

      // setup depth writing
      if (stage->flags & R_DEPTHWRITE) {
         if (!is_depth_write_enabled) glDepthMask(true);

         is_depth_write_enabled = true;
      }
      else {
         if (is_depth_write_enabled) glDepthMask(false);

         is_depth_write_enabled = false;
      }

      // setup alpha test
      if (stage->flags & R_HAS_ALPHAFUNC) {
         if (!is_alpha_test_enabled) glEnable(GL_ALPHA_TEST);

         is_alpha_test_enabled = true;

         glAlphaFunc(func_table[stage->alpha_test_func], stage->alpha_test_ref);
         is_alpha_test_dirty = true;
      }
      else if (is_alpha_test_enabled) {
         glDisable(GL_ALPHA_TEST);
         is_alpha_test_enabled = false;
      }

      // draw
      R_DrawArray(mode, first, count);

      // restore colors
      if (!is_last_stage &&
          (current_vertex_fields & (R_COLOR3|R_COLOR4)) &&
          (stage->flags & R_STAGE_SAVE_COLORS)) {
         R_RestoreRGB(current_vertex_array, first, count);
      }

      // restore alpha
      if (!is_last_stage &&
          (current_vertex_fields & R_COLOR4) &&
          (stage->flags & R_STAGE_SAVE_ALPHA)) {
         R_RestoreAlpha(current_vertex_array, first, count);
      }

      // restore texture coordinates 
      if (!is_last_stage &&
          (current_vertex_fields & R_TEXCOORD2) &&
          (stage->flags & R_STAGE_SAVE_TEXCOORDS)) {
         R_RestoreST(current_vertex_array, first, count);
      }

      stage++;
   }

   // skyParms
   if (current_shader->flags & R_SHADER_HAS_NEARBOX) {
      // TODO: draw the nearbox as the last layer
   }

   // fogparms
   if (current_shader->flags & R_SHADER_HAS_FOGPARMS) {
      // TODO: do an additional pass to layer in fog
   }

   // deformVertexes

   // cull
   if (is_cull_face_dirty) glCullFace(GL_FRONT);

   if (!is_cull_face_enabled) glEnable(GL_CULL_FACE);

   // restore blend state to default
   if (is_blend_dirty) glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

   if (is_blend_enabled) glDisable(GL_BLEND);

   // restore alpha test state to default
   if (is_alpha_test_dirty) glAlphaFunc(GL_GREATER, 0.666);

   if (is_alpha_test_enabled) glDisable(GL_ALPHA_TEST);

   // restore depth test state to default
   if (is_depth_test_dirty) glDepthFunc(GL_LEQUAL);

   // restore buffer masks to default
   if (!is_depth_write_enabled) glDepthMask(true);

   // restore polygon offset state
   if (current_shader->flags & R_SHADER_POLYGON_OFFSET) glDisable(GL_POLYGON_OFFSET_FILL);

   // restore matrix
   if (current_shader->flags & R_SHADER_SAVE_MATRIX) glPopMatrix();

   // restore vertex fields
   if (saved_vertex_fields != current_vertex_fields) {
      R_VertexArray(current_vertex_array, saved_vertex_fields);
   }

   // restore color
   glColor4f(1, 1, 1, 1);

   // restore texture environment
   glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
}



/* _R_DrawVertex
 ****************************************************************************/
__inline void _R_DrawVertex(vavertex_t* varray, texcoord_t* larray)
{
   if (current_vertex_fields & R_COLOR4) {
      glColor4ubv(&varray->r);
   }
   else if (current_vertex_fields & R_COLOR3) {
      glColor3ubv(&varray->r);
   }

//   if (is_mtex_active) {
//      qglMTexCoord2fSGIS(TEXTURE0_SGIS, varray->s, varray->t);
//      qglMTexCoord2fSGIS(TEXTURE1_SGIS, larray->s, larray->t);
//   } else

   if (current_vertex_fields & R_TEXCOORD2) {
      glTexCoord2fv(&varray->s);
   }
  
   if (current_vertex_fields & R_VERTEX3) {
      glVertex3fv(&varray->x);
   }
   else if (current_vertex_fields & R_VERTEX2) {
      glVertex2fv(&varray->x);
   }
}



/* R_DrawArray
 ****************************************************************************/
void R_DrawArray(R_MODE mode, int first, size_t count)
{
   if (is_vertex_arrays_supported && r_vertex_arrays.value) {
      glDrawArrays(mode_table[mode], first, count);
   }
   else {
      vavertex_t *varray;
      texcoord_t *larray;
      int i;

      varray = current_vertex_array       + first;
      larray = current_lightmap_texcoords + first;

      glBegin(mode_table[mode]);

      for (i = 0; i < count; i++) {
         _R_DrawVertex(varray, larray);
         varray++;
         larray++;
      }

      glEnd();
   }
}



/* R_DrawElements
 ****************************************************************************/
void R_DrawElements(R_MODE mode, size_t count, unsigned* indices)
{
   if (is_vertex_arrays_supported && r_vertex_arrays.value) {
      glDrawElements(mode_table[mode], count, GL_UNSIGNED_INT, indices);
   }
   else {
      vavertex_t *varray;
      texcoord_t *larray;
      int i;

      glBegin(mode_table[mode]);

      for (i = 0; i < count; i++) {
         varray = current_vertex_array + indices[i];
         larray = current_lightmap_texcoords + indices[i];
         _R_DrawVertex(varray, larray);
      }

      glEnd();
   }
}



/* R_DrawRangeElements
 ****************************************************************************/
void R_DrawRangeElements(R_MODE mode, unsigned start, unsigned end, size_t count, unsigned* indices)
{
   if (is_vertex_arrays_supported && r_vertex_arrays.value) {
      //glDrawRangeElements(mode_table[mode], start, end, count, GL_UNSIGNED_INT, indices);
   }
   else {
      R_DrawElements(mode, count, indices);
   }
}


