// fenix@io.com: shaders

/*
Copyright (C) 1996-1997 Id Software, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

#include "quakedef.h"

#include "shaders.h"

#define R_MIN_SHADER_COUNT    1024
#define R_MIN_MAP_COUNT       1024



typedef struct shader_command_s {
    char*        name;
    int          min_argc;
    int          max_argc;
    qboolean     is_stage_command;
    qboolean     is_ignored;
    parse_func_t parse;
} shader_command_t;



int       shader_array_size;
shader_t* shaders;
int       num_shaders;

int            texture_map_array_size;
texture_map_t* texture_maps;
int            num_texture_maps;

shader_t* current_shader;

static qboolean nomipmaps;
static qboolean nopicmip;
static qboolean force_depth_write;
static qboolean clampmap;

static char map_name[R_MAX_ANIMMAPS][R_SHADER_NAME_LEN+1];
static int  num_maps;
static qboolean map_alpha;

static qboolean used_blendfunc;

static qboolean used_skyparms;

static char farbox_base_name[R_SHADER_NAME_LEN+1];
static char nearbox_base_name[R_SHADER_NAME_LEN+1];

static vavertex_t* saved_vertexes;

static unsigned char entity_color[4];

static qboolean rgbgen_func_global;
static qboolean alphagen_func_global;

static qboolean has_rgbgen;

static qboolean explicit_tcgen;

qboolean overbright = false;

const texmat_t identity_texmat = {
   1, 0,
   0, 1,
   
   0, 0
};

texmat_t shader_texmat = {
   1, 0,
   0, 1,
   
   0, 0
};

qboolean is_shader_texmat_dirty;

cvar_t r_detail = { "r_detail", "1" };

static unsigned int question_texture[64] =
{
	0,0,0,0,0,0,0,0,
	0,0,0,1,1,1,0,0,
	0,0,1,0,0,0,1,0,
	0,0,0,0,0,1,0,0,
	0,0,0,0,1,0,0,0,
	0,0,0,0,0,0,0,0,
	0,0,0,0,1,0,0,0,
	0,0,0,0,0,0,0,0
};

static unsigned int exclaim_texture[64] =
{
	0,0,0,0,0,0,0,0,
	0,0,0,0,1,0,0,0,
	0,0,0,0,1,0,0,0,
	0,0,0,0,1,0,0,0,
	0,0,0,0,1,0,0,0,
	0,0,0,0,0,0,0,0,
	0,0,0,0,1,0,0,0,
	0,0,0,0,0,0,0,0,
};

static unsigned int checker_texture[4] =
{
	0x00000000,0xFFFFFFFF,
	0xFFFFFFFF,0x00000000
};

static unsigned int whiteimage_texture[1] = { 0xFFFFFFFF };



void InitTextureMaps(void)
{
   int i;

   for (i = 0; i < 64; i++)
   {
      if (question_texture[i] == 0) {
         question_texture[i] = LittleLong(0xEE888888);
      }
      else {
         question_texture[i] = LittleLong(0x1100FF00);
      }

      if (exclaim_texture[i] == 1) {
         exclaim_texture[i] = LittleLong(0xEE888888);
      }
      else {
         exclaim_texture[i] = LittleLong(0x110000FF);
      }
   }

   texture_maps[R_MAP_WHITEIMAGE].index = 
      GL_LoadTexture32(
         "whiteimage",
         1,
         1,
         whiteimage_texture,
         false,
         false,
         false,
         false);

   texture_maps[R_MAP_QUESTION].index =
      GL_LoadTexture32(
         "question",
         8,
         8,
         question_texture,
         false,
         false,
         true,
         false);

   texture_maps[R_MAP_LIGHTMAP].index = texture_maps[R_MAP_WHITEIMAGE].index;

   texture_maps[R_MAP_EXCLAIM].index =
      GL_LoadTexture32(
         "exclaim",
         8,
         8,
         exclaim_texture,
         false,
         false,
         true,
         false);

   texture_maps[R_MAP_CHECKER].index =
      GL_LoadTexture32(
         "checker",
         2,
         2,
         checker_texture,
         false,
         false,
         true,
         false);

   num_texture_maps = R_MAP_FIRST_USER_MAP;
}



void R_ReloadTexture32
   (
   int       texnum,
   unsigned* data,
   int       width,
   int       height,
   qboolean  mipmap,
   qboolean  picmip,
   qboolean  alpha,
   qboolean  clampmap
   )
{
   GL_Bind(texnum);
   GL_Upload32(data, width, height, mipmap, picmip, alpha, clampmap);
}



void R_ConcatenateTexMatrix(texmat_t* m1, texmat_t* m2, texmat_t* out)
{
   texmat_t tmp;

   tmp.m00 = (m2->m00 * m1->m00) + (m2->m01 * m1->m10);
   tmp.m01 = (m2->m00 * m1->m01) + (m2->m01 * m1->m11);
   tmp.m10 = (m2->m10 * m1->m00) + (m2->m11 * m1->m10);
   tmp.m11 = (m2->m10 * m1->m01) + (m2->m11 * m1->m11);

   tmp.t0 = (m2->t0 * m1->m00) + (m2->t1 * m1->m10) + m1->t0;
   tmp.t1 = (m2->t0 * m1->m01) + (m2->t1 * m1->m11) + m1->t1;

   *out = tmp;
}



void R_ApplyTexMatrixOnce(texmat_t* m, float os, float ot, float* s, float* t)
{
   *s = (m->m00 * os) + (m->m10 * ot) + m->t0;
   *t = (m->m01 * os) + (m->m11 * ot) + m->t1;
}



void R_ApplyTexMatrix(texmat_t* m, vavertex_t* varray, int first, int count)
{
   int i;
   float s, t;

   varray += first;

   for (i = 0; i < count; i++) {
        s = (varray->s * m->m00) + (varray->t * m->m10) + m->t0;
        t = (varray->s * m->m01) + (varray->t * m->m11) + m->t1;
        varray->s = s;
        varray->t = t;
        varray++;
   }
}



void R_MakeScaleTexMatrix(texmat_t* m, float s, float t)
{
   m->m00 = s;
   m->m01 = 0;

   m->m10 = 0;
   m->m11 = t;

   m->t0  = 0;
   m->t1  = 0;
}



void R_MakeRotateTexMatrix(texmat_t* m, float a)
{
   float c = cos(a * -(M_PI / 180.0));
   float s = sin(a * -(M_PI / 180.0));

   m->m00 =  c;
   m->m01 =  s;

   m->m10 = -s;
   m->m11 =  c;

   m->t0 = 0;
   m->t1 = 0;
}



void R_MakeTranslateTexMatrix(texmat_t* m, float s, float t)
{
   m->m00 = 1;
   m->m01 = 0;

   m->m10 = 0;
   m->m11 = 1;

   m->t0 = s;
   m->t1 = t;
}



void R_InitShaders(void)
{
    int i;
    int pnum = COM_CheckParm("-numshaders");

    if (pnum != 0)
    {
        shader_array_size = atoi(com_argv[pnum+1]);

        if (shader_array_size < R_MIN_SHADER_COUNT)
        {
            shader_array_size = R_MIN_SHADER_COUNT;
        }
    }
    else
    {
        shader_array_size = R_MIN_SHADER_COUNT;
    }

    shaders = Hunk_AllocName(sizeof(shader_t) * shader_array_size, "shaders");

    for (i = 0; i < shader_array_size; i++) {
       shaders[i].flags = 0;
    }

    pnum = COM_CheckParm("-nummaps");

    if (pnum != 0)
    {
        texture_map_array_size = atoi(com_argv[pnum+1]);

        if (texture_map_array_size < R_MIN_MAP_COUNT)
        {
            texture_map_array_size = R_MIN_MAP_COUNT;
        }
    }
    else
    {
        texture_map_array_size = R_MIN_MAP_COUNT;
    }

    texture_maps = Hunk_AllocName(sizeof(texture_map_t) * texture_map_array_size, "texture_maps");

    saved_vertexes = Hunk_AllocName(sizeof(vavertex_t) * global_vertex_array_size, "saved_vertexes");

    InitTextureMaps();

    Cvar_RegisterVariable(&r_detail);
}



void R_Shader(int shader_num)
{
   current_shader = shaders + shader_num;
}



void R_ShaderName(const char* name)
{
   R_Shader(R_GetShader(name));
}



int R_RequestShader(const char* name)
{
    shader_t* shader;
    int id;

    id = R_GetShader(name);

    if (id >= 0) return id;

    if (num_shaders == shader_array_size) Sys_Error("please increase -numshaders");

    shader = &shaders[num_shaders];

    strncpy(shader->name, name, R_SHADER_NAME_LEN);
    shader->name[R_SHADER_NAME_LEN] = '\0';

    shader->num_stages = 0;

    shader->flags = 0;

    // assume compilable until marked otherwise
    shader->flags |= R_IS_COMPILABLE;

    // default cull face
    shader->cull_face  = R_FRONT;
    shader->flags     |= R_SHADER_HAS_CULL;

    // default cloud height
    shader->cloudheight = 128;

    num_shaders++;

    return num_shaders - 1;
}



void R_LoadShaders(const char* name)
{
   int   mark;
   char* text;

   mark = Hunk_LowMark();

   text = (char *)COM_LoadHunkFile((char*)name);

   if (text) {
      Con_Printf("...loading '%s'\n", name);
      R_ParseShaders(text, com_filesize);
   }
   else {
      Con_Printf("error: shader file %s not found.\n", name);
   }

   Hunk_FreeToLowMark(mark);
}


static char* curpos;
static char* endpos;

// Aftershock 0.1
static char*
nexttok(qboolean* is_eol_out)
{
    char *tok;
    
    while (curpos < endpos)
    {
	/* Skip leading whitespace */
	while (*curpos == ' ' || *curpos == '\t' || *curpos == '\n' ||
	      *curpos == '\r')
	    if (++curpos == endpos) return NULL;

	/* Check for comment */
	if (curpos[0] == '/' && curpos[1] == '/')
	{
	    /* Skip to end of comment line */
	    while (*curpos++ != '\n')
		if (curpos == endpos) return NULL;
	    /* Restart with leading whitespace */
	    continue;
	}

	/* Seek to end of token */
	tok = curpos;
	while (*curpos != ' ' && *curpos != '\t' && *curpos != '\n' &&
	      *curpos != '\r')
	    if (++curpos == endpos) break;

	/* Zero whitespace character and advance */
    if ((curpos < endpos) && (*curpos == '\r')) *curpos++ = '\0';

    if (is_eol_out) {
       if (((curpos < endpos) && (*curpos == '\n')) || (curpos == endpos)) {
          *is_eol_out = true;
       }
       else {
          *is_eol_out = false;
       }
    }

    if (curpos < endpos) *curpos++ = '\0';

	return tok;
    }

    return NULL;
}



// Aftershock 0.1
// fenix@io.com: modified
static qboolean
shader_skip(void)
{
    char *tok;
    int brace_count;

    /* Opening brace */
    tok = nexttok(NULL);

    if (!tok) {
       Con_Printf("error: unexpected end of file\n.");
       return false;
    }

    if (tok[0] != '{') {
       Con_Printf("error: expected '{', found '%s'\n", tok);
       return false;
    }

    for (brace_count = 1; brace_count > 0 && curpos < endpos; curpos++)
    {
	if (*curpos == '{')
	    brace_count++;
	else if (*curpos == '}')
	    brace_count--;
    }

    if (curpos == endpos) {
       Con_Printf("error: unexpected end of input\n");
       return false;
    }

    return true;
}


void tcGenLightmap(struct tcgen_s* tcgen, vavertex_t* varray, int first, int count);


static void mapFinalize(shader_stage_t* stage)
{
   int m;

   for (m = 0; m < num_maps; m++) {
      if (strnicmp(map_name[m], "$bsp#", 5) == 0) {
         strncpy(stage->map[m].bspmap_name, map_name[m] + 5, 16);
         stage->map[m].bspmap_name[16] = '\0';

         stage->map[m].bspmap_unloaded = true;

         stage->map[m].map_num = R_MAP_QUESTION;

         stage->map[m].bspmap_mipmaps  = !nomipmaps;
         stage->map[m].bspmap_picmip   = !nopicmip;
         stage->map[m].bspmap_alpha    = map_alpha;
         stage->map[m].bspmap_clampmap = clampmap;
      }
      else {
         stage->map[m].map_num =
            R_SafeLoadMap(map_name[m], !nomipmaps, !nopicmip, map_alpha, clampmap);

         stage->map[m].bspmap_name[0] = '\0';
      }
   }

   if (m == 1 && !explicit_tcgen && (stricmp(map_name[0], "$lightmap") == 0)) {
      stage->tcgen.func = tcGenLightmap;
      stage->flags |= R_STAGE_SAVE_TEXCOORDS;
      stage->on_fields |= R_TEXCOORD2;
   }

   stage->num_maps = num_maps;
}



void R_SaveST(vavertex_t* varray, int first, int count)
{
   vavertex_t* saved = saved_vertexes;
   int i;

   varray += first;

   for (i = 0; i < count; i++) {
      saved->s = varray->s;
      saved->t = varray->t;
      varray++;
      saved++;
   }
}



void R_RestoreST(vavertex_t* varray, int first, int count)
{
   vavertex_t* saved = saved_vertexes;
   int i;

   varray += first;

   for (i = 0; i < count; i++) {
      varray->s = saved->s;
      varray->t = saved->t;
      varray++;
      saved++;
   }
}



static void tcModRotate(struct tcmod_s* tcmod, vavertex_t* varray, int first, int count)
{
   texmat_t translate1;
   texmat_t translate2;
   texmat_t rotate;
   float s_half, t_half;

   (void)varray;
   (void)first;
   (void)count;

   s_half = 0.5 / sqrt(
       (shader_texmat.m00 * shader_texmat.m00) +
       (shader_texmat.m01 * shader_texmat.m01));

   t_half = 0.5 / sqrt(
       (shader_texmat.m10 * shader_texmat.m10) +
       (shader_texmat.m11 * shader_texmat.m11));

   R_MakeTranslateTexMatrix(&translate1,  s_half,  t_half);
   R_MakeRotateTexMatrix(&rotate, tcmod->args.rotate.speed * realtime);
   R_MakeTranslateTexMatrix(&translate2, -s_half, -t_half);

   R_ConcatenateTexMatrix(&shader_texmat, &translate1,  &shader_texmat);
   R_ConcatenateTexMatrix(&shader_texmat, &rotate,      &shader_texmat);
   R_ConcatenateTexMatrix(&shader_texmat, &translate2,  &shader_texmat);

   is_shader_texmat_dirty = true;
}



static void tcModScale(struct tcmod_s* tcmod, vavertex_t* varray, int first, int count)
{
   texmat_t scale;

   (void)varray;
   (void)first;
   (void)count;

   #define ARG (tcmod->args.scale)

   R_MakeScaleTexMatrix(&scale, 1.0 / ARG.s, 1.0 / ARG.t);
   R_ConcatenateTexMatrix(&shader_texmat, &scale, &shader_texmat);

   is_shader_texmat_dirty = true;

   #undef ARG
}



static void tcModScroll(struct tcmod_s* tcmod, vavertex_t* varray, int first, int count)
{
   texmat_t translate;
   texmat_t recip;
   float  s,  t;
   float ls, lt;

   (void)varray;
   (void)first;
   (void)count;

   // TODO: The shader docs mention that a 'mod' is used to save
   //       precision.  I do not know if it is actually necessary,
   //       but I will leave it out because some shaders work
   //       better without it and I do not know how to implement
   //       it correctly.

   #define ARG (tcmod->args.scroll)

   ls = 1.0 / sqrt((shader_texmat.m00 * shader_texmat.m00) +
                   (shader_texmat.m01 * shader_texmat.m01));

   lt = 1.0 / sqrt((shader_texmat.m10 * shader_texmat.m10) +
                   (shader_texmat.m11 * shader_texmat.m11));

   ls *= ls;
   lt *= lt;

   recip.m00 = shader_texmat.m00 * ls;
   recip.m01 = shader_texmat.m01 * ls;

   recip.m10 = shader_texmat.m10 * lt;
   recip.m11 = shader_texmat.m11 * lt;

   s = realtime * ((ARG.s * recip.m00) + (ARG.t * recip.m10));
   t = realtime * ((ARG.s * recip.m01) + (ARG.t * recip.m11));

   R_MakeTranslateTexMatrix(&translate, s, t);
   R_ConcatenateTexMatrix(&shader_texmat, &translate, &shader_texmat);

   is_shader_texmat_dirty = true;

   #undef ARG
}



static float WaveForm(R_WAVEFORM func, float ox)
{
   float x, y;

   x = ox - floor(ox);

   switch (func) {
      case R_SIN:
         // a simple sin wave
         // TODO: this could be a table
         // look up, but speed is not
         // as important here

         y = sin(2 * M_PI * x);

         break;

      case R_TRIANGLE:
         // mirrors a sin wave,
         // but is linear.

         if (x < 0.25) {
            y = (2 * x) + 0.5;
         }
         else if (x < 0.75) {
            y = 1 - (2 * (x - 0.25));
         }
         else {
            y = 2 * (x - 0.75);
         }

         break;

      case R_SQUARE:
         // mirrors the sin wave,
         // but is either completely
         // positive or negative

         if (x < 0.5) {
            y =  1;
         }
         else {
            y = -1;
         }
         
         break;

      case R_SAWTOOTH:
         // raises linearly and falls
         // back suddenly at the beginning
         // of the next period

         y = x;

         break;

      case R_INVSAWTOOTH:
         // falls linearly and raises
         // back suddenly at the beginning
         // of the next period

         y = 1 - x;

         break;

      case R_NOISE:
         // random noise
         y = ((float)rand() * (1.0f / (float)(RAND_MAX >> 1))) - 1;

         break;

      default:
         // to catch any errors, this is
         // simply a flat-line.

         Con_DPrintf("error: unknown shader waveform\n");

         y = 0;

         break;
   }

   return y;
}



static void tcModStretch(struct tcmod_s* tcmod, vavertex_t* varray, int first, int count)
{
   texmat_t translate1;
   texmat_t translate2;
   texmat_t scale;
   float s_half, t_half;
   float y;

   (void)varray;
   (void)first;
   (void)count;

   #define ARG (tcmod->args.stretch)

   y = (ARG.amp * WaveForm(ARG.form, ARG.freq * (realtime + ARG.phase))) + ARG.base;

   s_half = 0.5 / sqrt(
       (shader_texmat.m00 * shader_texmat.m00) +
       (shader_texmat.m01 * shader_texmat.m01));

   t_half = 0.5 / sqrt(
       (shader_texmat.m10 * shader_texmat.m10) +
       (shader_texmat.m11 * shader_texmat.m11));

   R_MakeTranslateTexMatrix(&translate1,  s_half,  t_half);
   R_MakeScaleTexMatrix(&scale, y, y);
   R_MakeTranslateTexMatrix(&translate2, -s_half, -t_half);

   R_ConcatenateTexMatrix(&shader_texmat, &translate1,  &shader_texmat);
   R_ConcatenateTexMatrix(&shader_texmat, &scale,       &shader_texmat);
   R_ConcatenateTexMatrix(&shader_texmat, &translate2,  &shader_texmat);

   is_shader_texmat_dirty = true;

   #undef ARG
}



static void tcModTransform(struct tcmod_s* tcmod, vavertex_t* varray, int first, int count)
{
   (void)varray;
   (void)first;
   (void)count;

   R_ConcatenateTexMatrix(&shader_texmat, &(tcmod->args.transform), &shader_texmat);

   is_shader_texmat_dirty = true;
}



static void tcModTurb(struct tcmod_s* tcmod, vavertex_t* varray, int first, int count)
{
   texmat_t scale;
   float sy, ty;

   #define ARG (tcmod->args.turb)

   sy = (ARG.amp * WaveForm(R_SIN, ARG.freq * (realtime + ARG.phase))) + ARG.base;
   ty = (ARG.amp * WaveForm(R_SIN, ARG.freq * (realtime + ARG.phase + 0.5))) + ARG.base;

   R_MakeScaleTexMatrix(&scale, 1 + sy, 1 + ty);
   R_ConcatenateTexMatrix(&shader_texmat, &scale, &shader_texmat);

   #undef ARG
}



static void tcModWave(struct tcmod_s* tcmod, vavertex_t* varray, int first, int count)
{
   int i;

   #define ARG (tcmod->args.wave)

   // Wave is a non-affine transformation, so it
   // cannot be concatenated into the matrix. For
   // that reason, the vertices need to be updated
   // and the matrix reset.
   R_ApplyShaderTexMatrix(varray, first, count);

   for (i = 0; i < count; i++) {
      float os, ot;
      float t;
      float div;

      t = ARG.freq * (realtime + ARG.phase);

      div = 1.0 / ARG.div;

      os = varray->s * div;
      ot = varray->t * div;

      varray->s = os + (ARG.amp * WaveForm(ARG.form, ot + t)) + ARG.base;
      varray->t = ot + (ARG.amp * WaveForm(ARG.form, os + t)) + ARG.base;

      varray++;
    }

   #undef ARG
}



static R_WAVEFORM ParseWaveForm(const char* str)
{
   if (stricmp(str, "sin") == 0) {
      return R_SIN;
   }
   else if (stricmp(str, "triangle") == 0) {
      return R_TRIANGLE;
   }
   else if (stricmp(str, "square") == 0) {
      return R_SQUARE;
   }
   else if (stricmp(str, "sawtooth") == 0) {
      return R_SAWTOOTH;
   }
   else if (stricmp(str, "inversesawtooth") == 0) {
      return R_INVSAWTOOTH;
   }
   else if (stricmp(str, "noise") == 0) {
      Con_Printf("warning waveform: noise is not documented.\n");
      return R_NOISE;
   }
   else {
      return -1;
   }
}



static qboolean tcModParseFunc(shader_t* shader, shader_stage_t* stage, int argc, char* argv[])
{
   tcmod_t* tcmod = &stage->tcmods[stage->num_tcmods];

   if (stricmp(argv[0], "rotate") == 0) {
      if (argc != 2) {
         Con_Printf("warning: tcMod: wrong number of arguments for rotate.\n");
         return true;
      }

      tcmod->func = tcModRotate;
      tcmod->args.rotate.speed = atof(argv[1]);

      stage->flags |= R_STAGE_SAVE_TEXCOORDS;
   }
   else if (stricmp(argv[0], "scale") == 0) {
      if (argc != 3) {
         Con_Printf("warning: tcMod: wrong number of arguments for scale.\n");
         return true;
      }

      tcmod->func = tcModScale;
      tcmod->args.scale.s = atof(argv[1]);
      tcmod->args.scale.t = atof(argv[2]);

      stage->flags |= R_STAGE_SAVE_TEXCOORDS;
   }
   else if (stricmp(argv[0], "scroll") == 0) {
      if (argc != 3) {
         Con_Printf("warning: tcMod: wrong number of arguments for scroll.\n");
         return true;
      }

      tcmod->func = tcModScroll;
      tcmod->args.scroll.s = atof(argv[1]);
      tcmod->args.scroll.t = atof(argv[2]);

      stage->flags |= R_STAGE_SAVE_TEXCOORDS;
   }
   else if (stricmp(argv[0], "stretch") == 0) {
      int form;

      if (argc != 6) {
         Con_Printf("warning: tcMod: wrong number of arguments for stretch.\n");
         return true;
      }

      tcmod->func = tcModStretch;

      form = ParseWaveForm(argv[1]);

      if (form < 0) {
         Con_Printf("warning: tcMod: stretch: unknown waveform.\n");
         return true;
      }

      tcmod->args.stretch.form  = form;
      tcmod->args.stretch.base  = atof(argv[2]);
      tcmod->args.stretch.amp   = atof(argv[3]);
      tcmod->args.stretch.phase = atof(argv[4]);
      tcmod->args.stretch.freq  = atof(argv[5]);

      stage->flags |= R_STAGE_SAVE_TEXCOORDS;
   }
   else if (stricmp(argv[0], "transform") == 0) {
      if (argc != 7) {
         Con_Printf("warning: tcMod: wrong number of arguments for transform.\n");
         return true;
      }

      tcmod->func = tcModTransform;

      tcmod->args.transform.m00 = atof(argv[1]);
      tcmod->args.transform.m01 = atof(argv[2]);
      tcmod->args.transform.m10 = atof(argv[3]);
      tcmod->args.transform.m11 = atof(argv[4]);
      tcmod->args.transform.t0  = atof(argv[5]);
      tcmod->args.transform.t1  = atof(argv[6]);

      stage->flags |= R_STAGE_SAVE_TEXCOORDS;
   }
   else if (stricmp(argv[0], "turb") == 0) {
      if (argc != 5) {
         Con_Printf("warning: tcMod: wrong number of arguments for turb.\n");
         return true;
      }

      tcmod->func = tcModTurb;

      tcmod->args.turb.base  = atof(argv[1]);
      tcmod->args.turb.amp   = atof(argv[2]);
      tcmod->args.turb.phase = atof(argv[3]);
      tcmod->args.turb.freq  = atof(argv[4]);

      stage->flags |= R_STAGE_SAVE_TEXCOORDS;
   }
   else if (stricmp(argv[0], "_wave") == 0) {
      int form;

      Con_DPrintf("warning: tcMod: _wave is an extension.\n");

      if (argc != 7) {
         Con_Printf("warning: tcMod: wrong number of arguments for _wave.\n");
         return true;
      }

      tcmod->func = tcModWave;

      tcmod->args.wave.div = atof(argv[1]);

      form = ParseWaveForm(argv[2]);

      if (form < 0) {
         Con_Printf("warning: tcMod: _wave: unknown waveform.\n");
         return true;
      }

      tcmod->args.wave.base  = atof(argv[3]);
      tcmod->args.wave.amp   = atof(argv[4]);
      tcmod->args.wave.phase = atof(argv[5]);
      tcmod->args.wave.freq  = atof(argv[6]);

      // wave is non-affine, so a software path must be used
      shader->flags &= ~R_IS_COMPILABLE;

      stage->flags |= R_STAGE_SAVE_TEXCOORDS;
   }
   else {
      tcmod->func = NULL;

      Con_Printf("warning: tcMod: unknown tcMod type %s.\n", argv[0]);
      return true;
   }

   stage->num_tcmods++;

   return true;
}



typedef struct {
   char*          name;
   R_BLEND_FACTOR factor;
   int            is_src;
   int            is_dst;
   int            map_alpha;
} blend_parse_table_t;



static blend_parse_table_t blend_parse_table[] = {
   { "GL_ONE",                 R_ONE,                 true,  true,  false },
   { "GL_ZERO",                R_ZERO,                true,  true,  false },
   { "GL_DST_COLOR",           R_DST_COLOR,           true,  false, false },
   { "GL_ONE_MINUS_DST_COLOR", R_ONE_MINUS_DST_COLOR, true,  false, false },
   { "GL_SRC_ALPHA",           R_SRC_ALPHA,           true,  true,  true  },
   { "GL_ONE_MINUS_SRC_ALPHA", R_ONE_MINUS_SRC_ALPHA, true,  true,  true  },
   { "GL_SRC_COLOR",           R_SRC_COLOR,           false, true,  false },
   { "GL_ONE_MINUS_SRC_COLOR", R_ONE_MINUS_SRC_COLOR, false, true,  false },
   { 0 }
};



R_BLEND_FACTOR ParseBlendFunc(const char* arg, qboolean is_parsing_for_src, qboolean* map_alpha_out)
{
   blend_parse_table_t* table = blend_parse_table;

   while (table->name) {
      if (stricmp(table->name, arg) == 0) {
         if (is_parsing_for_src && !table->is_src) {
            Con_Printf("warning: blendFunc: '%s' is not a source blend factor.\n");
            return -1;
         }

         if (!is_parsing_for_src && !table->is_dst) {
            Con_Printf("warning: blendFunc: '%s' is not a source blend factor.\n");
            return -1;
         }

         if (map_alpha_out) *map_alpha_out = table->map_alpha;

         return table->factor;
      }

      table++;
   }

   if (is_parsing_for_src) {
      Con_Printf("warning: blendFunc: unknown source factor.\n");
      return -1;
   }
   else {
      Con_Printf("warning: blendFunc: unknown destination factor.\n");
      return -1;
   }

} 



static qboolean blendFuncParseFunc
   (
   shader_t*       shader,
   shader_stage_t* stage,
   int             argc,
   char*           argv[]
   )
{
   if (argc == 1) {
      if (stricmp(argv[0], "add") == 0) {
         stage->blend_src = R_ONE;
         stage->blend_dst = R_ONE;
      }
      else if (stricmp(argv[0], "filter") == 0) {
         stage->blend_src = R_DST_COLOR;
         stage->blend_dst = R_ZERO;
      }
      else if (stricmp(argv[0], "blend") == 0) {
         stage->blend_src = R_SRC_ALPHA;
         stage->blend_dst = R_ONE_MINUS_SRC_ALPHA;

         map_alpha = true;
      }
      else {
         Con_Printf("warning: blendFunc: unknown blendFunc '%s'.\n", argv[0]);
         return true;
      }
   }
   else {
      qboolean src_alpha;
      qboolean dst_alpha;

      stage->blend_src = ParseBlendFunc(argv[0], true, &src_alpha);

      if (stage->blend_src < 0) return true;

      stage->blend_dst = ParseBlendFunc(argv[1], false, &dst_alpha);

      if (stage->blend_dst < 0) return true;

      map_alpha = src_alpha || dst_alpha;
   }

   stage->flags |= R_HAS_BLENDFUNC;
   stage->flags &= ~R_DEPTHWRITE;

   used_blendfunc = true;

   return true;
}



static qboolean alphaFuncParseFunc
   (
   shader_t*       shader,
   shader_stage_t* stage,
   int             argc,
   char*           argv[]
   )
{
   if (stricmp(argv[0], "GT0") == 0) {
      stage->alpha_test_func = R_GREATER;
      stage->alpha_test_ref  = 0;
   }
   else if (stricmp(argv[0], "LT128") == 0) {
      stage->alpha_test_func = R_LESS;
      stage->alpha_test_ref  = 128;
   }
   else if (stricmp(argv[0], "GE128") == 0) {
      stage->alpha_test_func = R_GEQUAL;
      stage->alpha_test_ref  = 128;
   }
   else {
      Con_Printf("warning: alphaFunc: unknown alpha test '%s'.\n", argv[0]);
      return true;
   }

   stage->flags |= R_HAS_ALPHAFUNC;

   map_alpha = true;

   return true;
}



static qboolean depthWriteParseFunc
   (
   shader_t*       shader,
   shader_stage_t* stage,
   int             argc,
   char*           argv[]
   )
{
   force_depth_write = true;

   return true;
}



static qboolean depthFuncParseFunc
   (
   shader_t*       shader,
   shader_stage_t* stage,
   int             argc,
   char*           argv[]
   )
{
   if (stricmp(argv[0], "lequal") == 0) {
      stage->depth_test_func = R_LEQUAL;
   }
   else if (stricmp(argv[0], "equal") == 0) {
      stage->depth_test_func = R_EQUAL;
   }
   else {
      Con_Printf("warning: depthFunc: unknown depth test '%s'.\n", argv[0]);
      return true;
   }

   return true;
}






/* skyParmsParseFunc:
 ***************************************************************************/
static qboolean skyParmsParseFunc
   (
   shader_t*       shader,
   shader_stage_t* unused,
   int             argc,
   char*           argv[]
   )
{
   (void)unused;

   // farbox
   strncpy(farbox_base_name, argv[0], R_SHADER_NAME_LEN);
   farbox_base_name[R_SHADER_NAME_LEN] = '\0';

   if (strlen(argv[0]) > R_SHADER_NAME_LEN) {
      Con_Printf("warning: skyParms: far-box base name too long");
   }

   // cloudheight
   shader->cloudheight = atoi(argv[1]);

   // nearbox
   strncpy(nearbox_base_name, argv[2], R_SHADER_NAME_LEN);
   nearbox_base_name[R_SHADER_NAME_LEN] = '\0';

   if (strlen(argv[2]) > R_SHADER_NAME_LEN) {
      Con_Printf("warning: skyParms: near-box base name too long");
   }

   used_skyparms = true;

   return true;
}



/* LoadSkyBoxMaps:
 ***************************************************************************/
static void LoadSkyBoxMaps
   (
   shader_t* shader,
   char*     base,
   qboolean  alpha,
   int       indexes_out[5]
   )
{
   int i;
   char name[R_SHADER_NAME_LEN + 9];

   static char* suffixes[6] = {
      "_rt.tga",
      "_lt.tga",
      "_ft.tga",
      "_bk.tga",
      "_up.tga",
      "_dn.tga" // TODO: should I extend shaders to draw the bottom map?
   };

   for (i = 0; i < 5; i++) {
      strncpy(name, base, R_SHADER_NAME_LEN);
      name[R_SHADER_NAME_LEN] = '\0';
      strncat(name, suffixes[i], R_SHADER_NAME_LEN+8);
      name[R_SHADER_NAME_LEN+8] = '\0';

      indexes_out[i] = R_SafeLoadMap(name, !nomipmaps, !nopicmip, alpha, false);
   }
}



/* skyParmsFinalize:
 ***************************************************************************/
static void skyParmsFinalize(shader_t* shader)
{
   if (stricmp(farbox_base_name, "-") != 0) {
      LoadSkyBoxMaps(shader, farbox_base_name, false, shader->farbox);
      shader->flags |= R_SHADER_HAS_FARBOX;
   }

   if (stricmp(nearbox_base_name, "-") != 0) {
      LoadSkyBoxMaps(shader, nearbox_base_name, true, shader->nearbox);
      shader->flags |= R_SHADER_HAS_NEARBOX;
   }

   shader->flags |= R_SHADER_HAS_SKYPARMS;
}



/* cullParseFunc:
 ***************************************************************************/
static qboolean cullParseFunc
   (
   shader_t*       shader,
   shader_stage_t* unused,
   int             argc,
   char*           argv[]
   )
{
   (void)unused;

   if (argc == 0) {
      // is there are no arguments, it defaults to 'cull front'
      shader->cull_face  = R_FRONT;
      shader->flags     |= R_SHADER_HAS_CULL;
      return true;
   }
   else {
      if (stricmp(argv[0], "front") == 0) {
         shader->cull_face  = R_FRONT;
         shader->flags     |= R_SHADER_HAS_CULL;
         return true;
      }
      else if (stricmp(argv[0], "back") == 0) {
         shader->cull_face  = R_BACK;
         shader->flags     |= R_SHADER_HAS_CULL;
         return true;
      }
      else if ((stricmp(argv[0], "disable") == 0) ||
               (stricmp(argv[0], "none") == 0)) {
         shader->flags &= ~R_SHADER_HAS_CULL;
         return true;
      }
      else {
         Con_Printf("warning: cull: argument should be 'front', 'back', 'disable', or 'none'.\n");
         return true;
      }
   }
}



/* deformVertexesWave:
 ***************************************************************************/
void deformVertexesWave
   (
   struct vxmod_s* vxmod,
   int             mode,
   vavertex_t*     varray,
   int             first,
   int             count
   )
{
}



/* deformVertexesNormal:
 ***************************************************************************/
void deformVertexesNormal
   (
   struct vxmod_s* vxmod,
   int             mode,
   vavertex_t*     varray,
   int             first,
   int             count
   )
{
}



/* deformVertexesBulge:
 ***************************************************************************/
void deformVertexesBulge
   (
   struct vxmod_s* vxmod,
   int             mode,
   vavertex_t*     varray,
   int             first,
   int             count
   )
{
}



/* deformVertexesMove:
 ***************************************************************************/
void deformVertexesMove
   (
   struct vxmod_s* vxmod,
   int             mode,
   vavertex_t*     varray,
   int             first,
   int             count
   )
{
   #define ARG (vxmod->args.move)

   float s =
     (ARG.amp * WaveForm(ARG.form, ARG.freq * (realtime + ARG.phase))) + ARG.base;

   glTranslatef(ARG.x * s, ARG.y * s, ARG.z * s);

   #undef ARG
}



/* deformVertexesAutoSprite:
 ***************************************************************************/
void deformVertexesAutoSprite
   (
   struct vxmod_s* unused,
   int             mode,
   vavertex_t*     varray,
   int             first,
   int             count
   )
{
   vec3_t w_origin;
   vec3_t o_origin;
   vec3_t one;
   vec3_t two;
   vec3_t side;
   float matrix[16];
   float len;

   (void)unused;

   switch (mode) {
      case R_TRIANGLE_STRIP:
      case R_QUAD_STRIP:
      case R_POLYGON:
      case R_QUADS:
      case R_TRIANGLE_FAN:

         if (count != 4) return;

         o_origin[0] = (varray[first].x + varray[first + 1].x + varray[first + 2].x + varray[first + 3].x) * (1.0/4.0);
         o_origin[1] = (varray[first].y + varray[first + 1].y + varray[first + 2].y + varray[first + 3].y) * (1.0/4.0);
         o_origin[2] = (varray[first].z + varray[first + 1].z + varray[first + 2].z + varray[first + 3].z) * (1.0/4.0);
   
         break;

      case R_TRIANGLES:

         if (count != 6) return;

         o_origin[0] = (varray[first].x + varray[first + 1].x + varray[first + 2].x + varray[first + 3].x + varray[first + 4].x + varray[first + 5].x) * (1.0/6.0);
         o_origin[1] = (varray[first].y + varray[first + 1].y + varray[first + 2].y + varray[first + 3].y + varray[first + 4].y + varray[first + 5].y) * (1.0/6.0);
         o_origin[2] = (varray[first].z + varray[first + 1].z + varray[first + 2].z + varray[first + 3].z + varray[first + 4].z + varray[first + 5].z) * (1.0/6.0);

         break;

      default:
         return;
   }

   glGetFloatv (GL_MODELVIEW_MATRIX, matrix);
   glLoadIdentity();

   w_origin[0] = (o_origin[0] * matrix[0]) + (o_origin[1] * matrix[4]) + (o_origin[2] * matrix[8] ) + matrix[12];
   w_origin[1] = (o_origin[0] * matrix[1]) + (o_origin[1] * matrix[5]) + (o_origin[2] * matrix[9] ) + matrix[13];
   w_origin[2] = (o_origin[0] * matrix[2]) + (o_origin[1] * matrix[6]) + (o_origin[2] * matrix[10]) + matrix[14];

   one[0] = (varray[first].x * matrix[0]) + (varray[first].y * matrix[4]) + (varray[first].z * matrix[8] ) + matrix[12];
   one[1] = (varray[first].x * matrix[1]) + (varray[first].y * matrix[5]) + (varray[first].z * matrix[9] ) + matrix[13];
   one[2] = (varray[first].x * matrix[2]) + (varray[first].y * matrix[6]) + (varray[first].z * matrix[10]) + matrix[14];

   two[0] = (varray[first + 1].x * matrix[0]) + (varray[first + 1].y * matrix[4]) + (varray[first + 1].z * matrix[8] ) + matrix[12];
   two[1] = (varray[first + 1].x * matrix[1]) + (varray[first + 1].y * matrix[5]) + (varray[first + 1].z * matrix[9] ) + matrix[13];
   two[2] = (varray[first + 1].x * matrix[2]) + (varray[first + 1].y * matrix[6]) + (varray[first + 1].z * matrix[10]) + matrix[14];

   VectorSubtract(one, two, side);
   len = Length(side) * 0.5;

   varray += first;

   switch (mode) {
      case R_TRIANGLE_STRIP:
      case R_QUAD_STRIP:
         // upper left
         varray->x = w_origin[0] - len;
         varray->y = w_origin[1] - len;
         varray->z = w_origin[2];
         varray++;

         // lower left
         varray->x = w_origin[0] - len;
         varray->y = w_origin[1] + len;
         varray->z = w_origin[2];
         varray++;

         // upper right
         varray->x = w_origin[0] + len;
         varray->y = w_origin[1] - len;
         varray->z = w_origin[2];
         varray++;

         // lower right
         varray->x = w_origin[0] + len;
         varray->y = w_origin[1] + len;
         varray->z = w_origin[2];

         break;

      case R_POLYGON:
      case R_QUADS:
      case R_TRIANGLE_FAN:
      default:
         // upper left
         varray->x = w_origin[0] - len;
         varray->y = w_origin[1] - len;
         varray->z = w_origin[2];
         varray++;

         // lower left
         varray->x = w_origin[0] - len;
         varray->y = w_origin[1] + len;
         varray->z = w_origin[2];
         varray++;

         // lower right
         varray->x = w_origin[0] + len;
         varray->y = w_origin[1] + len;
         varray->z = w_origin[2];
         varray++;

         // upper right
         varray->x = w_origin[0] + len;
         varray->y = w_origin[1] - len;
         varray->z = w_origin[2];

         break;

      case R_TRIANGLES:
         // lower left
         varray->x = w_origin[0] - len;
         varray->y = w_origin[1] + len;
         varray->z = w_origin[2];
         varray++;

         // lower right
         varray->x = w_origin[0] + len;
         varray->y = w_origin[1] + len;
         varray->z = w_origin[2];
         varray++;

         // upper left
         varray->x = w_origin[0] - len;
         varray->y = w_origin[1] - len;
         varray->z = w_origin[2];
         varray++;

         // upper left
         varray->x = w_origin[0] - len;
         varray->y = w_origin[1] - len;
         varray->z = w_origin[2];
         varray++;

         // lower right
         varray->x = w_origin[0] + len;
         varray->y = w_origin[1] + len;
         varray->z = w_origin[2];
         varray++;

         // upper right
         varray->x = w_origin[0] + len;
         varray->y = w_origin[1] - len;
         varray->z = w_origin[2];

         break;
   }
}



/* deformVertexesAutoSprite2:
 ***************************************************************************/
void deformVertexesAutoSprite2
   (
   struct vxmod_s* unused,
   int             mode,
   vavertex_t*     varray,
   int             first,
   int             count
   )
{
   vec3_t side;
   vec3_t w_endpoint1;
   vec3_t w_endpoint2;
   vec3_t o_endpoint1;
   vec3_t o_endpoint2;
   vec3_t ul, ll, ur, lr;
   vec3_t one, two;
   float matrix[16];
   float len;
   float len1;
   float len2;

   (void)unused;

   switch (mode) {
      case R_TRIANGLE_STRIP:
      case R_QUAD_STRIP:

         if (count != 4) return;

         VectorSubtract((&varray[first].x), (&varray[first+1].x), side);
         len1 = Length(side);

         VectorSubtract((&varray[first].x), (&varray[first+2].x), side);
         len2 = Length(side);

         if (len1 < len2) {
           o_endpoint1[0] = (varray[first].x   + varray[first+1].x) * 0.5;
           o_endpoint1[1] = (varray[first].y   + varray[first+1].y) * 0.5;
           o_endpoint1[2] = (varray[first].z   + varray[first+1].z) * 0.5;

           o_endpoint2[0] = (varray[first+2].x + varray[first+3].x) * 0.5;
           o_endpoint2[1] = (varray[first+2].y + varray[first+3].y) * 0.5;
           o_endpoint2[2] = (varray[first+2].z + varray[first+3].z) * 0.5;
         }
         else {
           o_endpoint1[0] = (varray[first].x   + varray[first+2].x) * 0.5;
           o_endpoint1[1] = (varray[first].y   + varray[first+2].y) * 0.5;
           o_endpoint1[2] = (varray[first].z   + varray[first+2].z) * 0.5;

           o_endpoint2[0] = (varray[first+1].x + varray[first+3].x) * 0.5;
           o_endpoint2[1] = (varray[first+1].y + varray[first+3].y) * 0.5;
           o_endpoint2[2] = (varray[first+1].z + varray[first+3].z) * 0.5;
         }

         break;

      case R_POLYGON:
      case R_QUADS:
      case R_TRIANGLE_FAN:

         if (count != 4) return;

         VectorSubtract((&varray[first].x), (&varray[first+1].x), side);
         len1 = Length(side);

         VectorSubtract((&varray[first+1].x), (&varray[first+2].x), side);
         len2 = Length(side);

         if (len1 < len2) {
           o_endpoint1[0] = (varray[first].x   + varray[first+1].x) * 0.5;
           o_endpoint1[1] = (varray[first].y   + varray[first+1].y) * 0.5;
           o_endpoint1[2] = (varray[first].z   + varray[first+1].z) * 0.5;

           o_endpoint2[0] = (varray[first+2].x + varray[first+3].x) * 0.5;
           o_endpoint2[1] = (varray[first+2].y + varray[first+3].y) * 0.5;
           o_endpoint2[2] = (varray[first+2].z + varray[first+3].z) * 0.5;
         }
         else {
           o_endpoint1[0] = (varray[first].x   + varray[first+3].x) * 0.5;
           o_endpoint1[1] = (varray[first].y   + varray[first+3].y) * 0.5;
           o_endpoint1[2] = (varray[first].z   + varray[first+3].z) * 0.5;

           o_endpoint2[0] = (varray[first+1].x + varray[first+2].x) * 0.5;
           o_endpoint2[1] = (varray[first+1].y + varray[first+2].y) * 0.5;
           o_endpoint2[2] = (varray[first+1].z + varray[first+2].z) * 0.5;
         }

         break;

      case R_TRIANGLES:

      default:
         return;
   }

   glGetFloatv (GL_MODELVIEW_MATRIX, matrix);
   glLoadIdentity();

   w_endpoint1[0] = (o_endpoint1[0] * matrix[0]) + (o_endpoint1[1] * matrix[4]) + (o_endpoint1[2] * matrix[8] ) + matrix[12];
   w_endpoint1[1] = (o_endpoint1[0] * matrix[1]) + (o_endpoint1[1] * matrix[5]) + (o_endpoint1[2] * matrix[9] ) + matrix[13];
   w_endpoint1[2] = (o_endpoint1[0] * matrix[2]) + (o_endpoint1[1] * matrix[6]) + (o_endpoint1[2] * matrix[10]) + matrix[14];

   w_endpoint2[0] = (o_endpoint2[0] * matrix[0]) + (o_endpoint2[1] * matrix[4]) + (o_endpoint2[2] * matrix[8] ) + matrix[12];
   w_endpoint2[1] = (o_endpoint2[0] * matrix[1]) + (o_endpoint2[1] * matrix[5]) + (o_endpoint2[2] * matrix[9] ) + matrix[13];
   w_endpoint2[2] = (o_endpoint2[0] * matrix[2]) + (o_endpoint2[1] * matrix[6]) + (o_endpoint2[2] * matrix[10]) + matrix[14];

   one[0] = (varray[first].x * matrix[0]) + (varray[first].y * matrix[4]) + (varray[first].z * matrix[8] ) + matrix[12];
   one[1] = (varray[first].x * matrix[1]) + (varray[first].y * matrix[5]) + (varray[first].z * matrix[9] ) + matrix[13];
   one[2] = (varray[first].x * matrix[2]) + (varray[first].y * matrix[6]) + (varray[first].z * matrix[10]) + matrix[14];

   switch (mode) {
      case R_TRIANGLE_STRIP:
      case R_QUAD_STRIP:

         if (len1 < len2) {
            two[0] = (varray[first + 1].x * matrix[0]) + (varray[first + 1].y * matrix[4]) + (varray[first + 1].z * matrix[8] ) + matrix[12];
            two[1] = (varray[first + 1].x * matrix[1]) + (varray[first + 1].y * matrix[5]) + (varray[first + 1].z * matrix[9] ) + matrix[13];
            two[2] = (varray[first + 1].x * matrix[2]) + (varray[first + 1].y * matrix[6]) + (varray[first + 1].z * matrix[10]) + matrix[14];
         }
         else {
            two[0] = (varray[first + 2].x * matrix[0]) + (varray[first + 2].y * matrix[4]) + (varray[first + 2].z * matrix[8] ) + matrix[12];
            two[1] = (varray[first + 2].x * matrix[1]) + (varray[first + 2].y * matrix[5]) + (varray[first + 2].z * matrix[9] ) + matrix[13];
            two[2] = (varray[first + 2].x * matrix[2]) + (varray[first + 2].y * matrix[6]) + (varray[first + 2].z * matrix[10]) + matrix[14];
         }

         break;

      case R_POLYGON:
      case R_QUADS:
      case R_TRIANGLE_FAN:

         if (len1 < len2) {
            two[0] = (varray[first + 1].x * matrix[0]) + (varray[first + 1].y * matrix[4]) + (varray[first + 1].z * matrix[8] ) + matrix[12];
            two[1] = (varray[first + 1].x * matrix[1]) + (varray[first + 1].y * matrix[5]) + (varray[first + 1].z * matrix[9] ) + matrix[13];
            two[2] = (varray[first + 1].x * matrix[2]) + (varray[first + 1].y * matrix[6]) + (varray[first + 1].z * matrix[10]) + matrix[14];
         }
         else {
            two[0] = (varray[first + 3].x * matrix[0]) + (varray[first + 3].y * matrix[4]) + (varray[first + 3].z * matrix[8] ) + matrix[12];
            two[1] = (varray[first + 3].x * matrix[1]) + (varray[first + 3].y * matrix[5]) + (varray[first + 3].z * matrix[9] ) + matrix[13];
            two[2] = (varray[first + 3].x * matrix[2]) + (varray[first + 3].y * matrix[6]) + (varray[first + 3].z * matrix[10]) + matrix[14];
         }
         
         break;

      case R_TRIANGLES:

      default:
         break;
   }

   VectorSubtract(one, two, side);
   len = Length(side) * 0.5;

   if (fabs(w_endpoint1[0] - w_endpoint2[0]) > fabs(w_endpoint1[1] - w_endpoint2[1])) {
      ul[0] = w_endpoint1[0];
      ul[1] = w_endpoint1[1] - len;
      ul[2] = w_endpoint1[2];

      ll[0] = w_endpoint1[0];
      ll[1] = w_endpoint1[1] + len;
      ll[2] = w_endpoint1[2];

      ur[0] = w_endpoint2[0];
      ur[1] = w_endpoint2[1] - len;
      ur[2] = w_endpoint2[2];

      lr[0] = w_endpoint2[0];
      lr[1] = w_endpoint2[1] + len;
      lr[2] = w_endpoint2[2];
   }
   else {
      ul[0] = w_endpoint1[0] - len;
      ul[1] = w_endpoint1[1];
      ul[2] = w_endpoint1[2];

      ll[0] = w_endpoint2[0] - len;
      ll[1] = w_endpoint2[1];
      ll[2] = w_endpoint2[2];

      ur[0] = w_endpoint1[0] + len;
      ur[1] = w_endpoint1[1];
      ur[2] = w_endpoint1[2];

      lr[0] = w_endpoint2[0] + len;
      lr[1] = w_endpoint2[1];
      lr[2] = w_endpoint2[2];
   }

   varray += first;

   switch (mode) {
      case R_TRIANGLE_STRIP:
      case R_QUAD_STRIP:
         // upper left
         varray->x = ul[0];
         varray->y = ul[1];
         varray->z = ul[2];
         varray++;

         // lower left
         varray->x = ll[0];
         varray->y = ll[1];
         varray->z = ll[2];
         varray++;

         // upper right
         varray->x = ur[0];
         varray->y = ur[1];
         varray->z = ur[2];
         varray++;

         // lower right
         varray->x = lr[0];
         varray->y = lr[1];
         varray->z = lr[2];

         break;

      case R_POLYGON:
      case R_QUADS:
      case R_TRIANGLE_FAN:
      default:

         // upper left
         varray->x = ul[0];
         varray->y = ul[1];
         varray->z = ul[2];
         varray++;

         // lower left
         varray->x = ll[0];
         varray->y = ll[1];
         varray->z = ll[2];
         varray++;

         // lower right
         varray->x = lr[0];
         varray->y = lr[1];
         varray->z = lr[2];
         varray++;

         // upper right
         varray->x = ur[0];
         varray->y = ur[1];
         varray->z = ur[2];

         break;

      case R_TRIANGLES:
         break;
   }
}



/* deformVertexesParseFunc:
 ***************************************************************************/
static qboolean deformVertexesParseFunc
   (
   shader_t*       shader,
   shader_stage_t* unused,
   int             argc,
   char*           argv[]
   )
{
   vxmod_t* vxmod = &shader->vxmods[shader->num_vxmods];

   (void)unused;

   if (stricmp(argv[0], "wave") == 0) {
      int form;

      if (argc != 7) {
         Con_Printf("warning: deformVertexes: wave: wrong number of arguments.\n");
         return true;
      }

      vxmod->func = deformVertexesWave;

      vxmod->args.wave.div = atof(argv[1]);

      form = ParseWaveForm(argv[2]);

      if (form < 0) {
         Con_Printf("warning: deformVertexes: wave: unknown waveform.\n");
         return true;
      }

      vxmod->args.wave.form = form;
      
      vxmod->args.wave.base  = atof(argv[3]);
      vxmod->args.wave.amp   = atof(argv[4]);
      vxmod->args.wave.phase = atof(argv[5]);
      vxmod->args.wave.freq  = atof(argv[6]);
   }
   else if (stricmp(argv[0], "normal") == 0) {
      int form;

      if (argc != 7) {
         Con_Printf("warning: deformVertexes: normal: wrong number of arguments.\n");
         return true;
      }

      vxmod->func = deformVertexesNormal;

      vxmod->args.normal.div = atof(argv[1]);

      form = ParseWaveForm(argv[2]);

      if (form < 0) {
         Con_Printf("warning: deformVertexes: normal: unknown waveform.\n");
         return true;
      }

      vxmod->args.normal.form = form;
      
      vxmod->args.normal.base  = atof(argv[3]);
      vxmod->args.normal.amp   = atof(argv[4]);
      vxmod->args.normal.phase = atof(argv[5]);
      vxmod->args.normal.freq  = atof(argv[6]);
   }
   else if (stricmp(argv[0], "bulge") == 0) {
      if (argc != 4) {
         Con_Printf("warning: deformVertexes: bulge: wrong number of arguments.\n");
      }

      vxmod->func = deformVertexesBulge;

      vxmod->args.bulge.width  = atof(argv[1]);
      vxmod->args.bulge.height = atof(argv[2]);
      vxmod->args.bulge.speed  = atof(argv[3]);
   }
   else if (stricmp(argv[0], "move") == 0) {
      int form;

      if (argc != 9) {
         Con_Printf("warning: deformVertexes: move: wrong number of arguments.\n");
      }

      vxmod->func = deformVertexesMove;

      vxmod->args.move.x = atof(argv[1]);
      vxmod->args.move.y = atof(argv[2]);
      vxmod->args.move.z = atof(argv[3]);

      form = ParseWaveForm(argv[4]);

      if (form < 0) {
         Con_Printf("warning: deformVertexes: move: unknown waveform.\n");
         return true;
      }

      vxmod->args.move.form  = form;

      vxmod->args.move.base  = atof(argv[5]);
      vxmod->args.move.amp   = atof(argv[6]);
      vxmod->args.move.phase = atof(argv[7]);
      vxmod->args.move.freq  = atof(argv[8]);

      shader->flags |= R_SHADER_SAVE_MATRIX;
   }
   else if (stricmp(argv[0], "autosprite") == 0) {
      if (argc != 1) {
         Con_Printf("warning: deformVertexes: autosprite: does not take any arguments.\n");
      }

      vxmod->func = deformVertexesAutoSprite;

      shader->flags |= R_SHADER_SAVE_MATRIX;
   }
   else if (stricmp(argv[0], "autosprite2") == 0) {
      if (argc != 1) {
         Con_Printf("warning: deformVertexes: autosprite2: does not take any arguments.\n");
      }

      vxmod->func = deformVertexesAutoSprite2;

      shader->flags |= R_SHADER_SAVE_MATRIX;
   }
   else {
      vxmod->func = NULL;

      Con_Printf("warning: deformVertexes: unrecognized deform type.\n");

      return true;
   }

   shader->num_vxmods++;

   return true;
}



/* fogparmsParseFunc:
 ***************************************************************************/
static qboolean fogparmsParseFunc
   (
   shader_t*       shader,
   shader_stage_t* unused,
   int             argc,
   char*           argv[]
   )
{
   (void)unused;

   shader->fog_color[0] = atof(argv[0]);

   if ((shader->fog_color[0] < 0) || (shader->fog_color[0] > 1)) {
      shader->fog_color[0] = CLAMP(0, shader->fog_color[0], 1);
      Con_Printf("warning: fogparms: red color value need to be normalized between 0 and 1.\n");
   }

   shader->fog_color[1] = atof(argv[1]);

   if ((shader->fog_color[1] < 0) || (shader->fog_color[1] > 1)) {
      shader->fog_color[1] = CLAMP(0, shader->fog_color[1], 1);
      Con_Printf("warning: fogparms: green color value need to be normalized between 0 and 1.\n");
   }

   shader->fog_color[2] = atof(argv[2]);

   if ((shader->fog_color[2] < 0) || (shader->fog_color[2] > 1)) {
      shader->fog_color[2] = CLAMP(0, shader->fog_color[2], 1);
      Con_Printf("warning: fogparms: blue color value need to be normalized between 0 and 1.\n");
   }

   shader->fog_distance_to_opaque = atof(argv[4]);

   shader->flags |= R_SHADER_HAS_FOGPARMS;

   return true;
}



/* nomipmapsParseFunc:
 ***************************************************************************/
static qboolean nomipmapsParseFunc
   (
   shader_t*       shader,
   shader_stage_t* stage,
   int             argc,
   char*           argv[]
   )
{
   nomipmaps = true;

   // nomipmaps implies no picmip
   nopicmip = true;

   return true;
}



/* nopicmipParseFunc:
 ***************************************************************************/
static qboolean nopicmipParseFunc
   (
   shader_t*       shader,
   shader_stage_t* stage,
   int             argc,
   char*           argv[]
   )
{
   nopicmip = true;

   return true;
}



/* polygonOffsetParseFunc:
 ***************************************************************************/
static qboolean polygonOffsetParseFunc
   (
   shader_t*       shader,
   shader_stage_t* stage,
   int             argc,
   char*           argv[]
   )
{
   shader->flags |= R_SHADER_POLYGON_OFFSET;

   return true;
}



/* portalParseFunc:
 ***************************************************************************/
static qboolean portalParseFunc
   (
   shader_t*       shader,
   shader_stage_t* stage,
   int             argc,
   char*           argv[]
   )
{
   shader->sort_priority = R_SORT_PORTAL;
   shader->flags |= R_SHADER_HAS_SORT;

   return true;
}



typedef struct sort_parse_table_s {
   char*                  name;
   R_SHADER_SORT_PRIORITY pri;
} sort_parse_table_t;



static sort_parse_table_t sort_table[] = {
   { "portal",     R_SORT_PORTAL     },
   { "sky",        R_SORT_SKY        },
   { "opaque",     R_SORT_OPAQUE     },
   { "banner",     R_SORT_BANNER     },
   { "underwater", R_SORT_UNDERWATER },
   { "additive",   R_SORT_ADDITIVE   },
   { "nearest",    R_SORT_NEAREST    },
   { NULL,         0                 }
};



/* sortParseFunc:
 ***************************************************************************/
static qboolean sortParseFunc
   (
   shader_t*       shader,
   shader_stage_t* unused,
   int             argc,
   char*           argv[]
   )
{
   int pri = atoi(argv[0]);

   (void)unused;

   Con_Printf("warning: sort: not fully implemented.\n");

   if (pri == 0) {
      sort_parse_table_t* table;

      if (stricmp(argv[0], "0") == 0) {
         Con_Printf("warning: sort: zero is no a valid sort priority.\n");
         return true;
      }

      table = sort_table;

      while (table->name) {
         if (stricmp(table->name, argv[0]) == 0) {
            shader->sort_priority = table->pri;
            break;
         }

         table++;
      }

      if (!table->name) {
         Con_Printf("warning: sort: unknown sort priority.\n");
         return true;
      }
   }
   else {
      if ((pri < 1) || (pri > 16)) {
         pri = CLAMP(1, pri, 16);
         Con_Printf("warning: sort: priority must be between 1 and 16.\n");
      }

      shader->sort_priority = pri;
   }

   shader->flags |= R_SHADER_HAS_SORT;

   return true;
}



/* RequestMap:
 ***************************************************************************/
static void RequestMap(const char* name, int m)
{
   strncpy(map_name[m], name, R_SHADER_NAME_LEN);
   map_name[m][R_SHADER_NAME_LEN] = '\0';

   if (strlen(name) > R_SHADER_NAME_LEN) {
      Con_Printf("warning: map: name too long");
   }
}



/* mapParseFunc:
 ***************************************************************************/
static qboolean mapParseFunc
   (
   shader_t*       shader,
   shader_stage_t* stage,
   int             argc,
   char*           argv[]
   )
{
   RequestMap(argv[0], 0);
   clampmap = false;
   num_maps = 1;

   return true;
}



/* clampmapParseFunc:
 ***************************************************************************/
static qboolean clampmapParseFunc
   (
   shader_t*       shader,
   shader_stage_t* stage,
   int             argc,
   char*           argv[]
   )
{
   RequestMap(argv[0], 0);
   clampmap = true;
   num_maps = 1;

   return true;
}



/* AnimMapParseFunc:
 ***************************************************************************/
static qboolean AnimMapParseFunc
   (
   shader_t*       shader,
   shader_stage_t* stage,
   int             argc,
   char*           argv[]
   )
{
   int m;

   stage->anim_map_interval = atof(argv[0]);

   num_maps = argc - 1;
 
   for (m = 0; m < num_maps; m++) {
      RequestMap(argv[m+1], m);
   }

   clampmap = false;

   return true;
}



/* rgbGenIdentityLighting:
 ***************************************************************************/
void rgbGenIdentityLighting
   (
   struct rgbgen_s* rgbgen,
   unsigned char    color_out[4]
   )
{
   if (overbright) {
      color_out[0] = 128;
      color_out[1] = 128;
      color_out[2] = 128;
   }
   else {
      color_out[0] = 255;
      color_out[1] = 255;
      color_out[2] = 255;
   }
}



/* rgbGenIdentity:
 ***************************************************************************/
void rgbGenIdentity
   (
   struct rgbgen_s* rgbgen,
   unsigned char    color_out[4]
   )
{
   color_out[0] = 255;
   color_out[1] = 255;
   color_out[2] = 255;
}



/* rgbGenWave:
 ***************************************************************************/
void rgbGenWave
   (
   struct rgbgen_s* rgbgen,
   unsigned char    color_out[4]
   )
{
   int i;
   float y;

   #define ARG (rgbgen->args.wave)

   y = 255 * (ARG.amp * WaveForm(ARG.form, ARG.freq * (realtime + ARG.phase))) + ARG.base;
   y = CLAMP(0, y, 255);

   color_out[0] = y;
   color_out[1] = y;
   color_out[2] = y;

   #undef ARG
}



/* R_EntityColor:
 ***************************************************************************/
void R_EntityColor
   (
   float r,
   float g,
   float b,
   float a
   )
{
   r *= 255;
   g *= 255;
   b *= 255;
   a *= 255;

   entity_color[0] = CLAMP(0, r, 255);
   entity_color[1] = CLAMP(0, g, 255);
   entity_color[2] = CLAMP(0, b, 255);
   entity_color[3] = CLAMP(0, a, 255);
}



/* rgbGenEntity:
 ***************************************************************************/
void rgbGenEntity
   (
   struct rgbgen_s* rgbgen,
   unsigned char    color_out[4]
   )
{
   color_out[0] = entity_color[0];
   color_out[1] = entity_color[1];
   color_out[2] = entity_color[2];
}



/* rgbGenOneMinusEntity:
 ***************************************************************************/
void rgbGenOneMinusEntity
   (
   struct rgbgen_s* rgbgen,
   unsigned char    color_out[4]
   )
{
   color_out[0] = 255 - entity_color[0];
   color_out[1] = 255 - entity_color[1];
   color_out[2] = 255 - entity_color[2];
}



/* rgbGenVertex:
 ***************************************************************************/
void rgbGenVertex
   (
   struct rgbgen_s* rgbgen,
   vavertex_t*      varray,
   int              first,
   int              count
   )
{
   // do nothing
   (void)rgbgen;
   (void)varray;
   (void)first;
   (void)count;
}



/* rgbGenExactVertex:
 ***************************************************************************/
void rgbGenExactVertex
   (
   struct rgbgen_s* rgbgen,
   vavertex_t*      varray,
   int              first,
   int              count
   )
{
   // do nothing
   (void)rgbgen;
   (void)varray;
   (void)first;
   (void)count;
}



/* rgbGenOneMinusVertex:
 ***************************************************************************/
void rgbGenOneMinusVertex
   (
   struct rgbgen_s* rgbgen,
   vavertex_t*      varray,
   int              first,
   int              count
   )
{
   int i;

   varray += first;

   for (i = 0; i < count; i++) {
      varray->r = 255 - varray->r;
      varray->g = 255 - varray->g;
      varray->b = 255 - varray->b;
      varray++;
   }
}



/* rgbGenLightingDiffuse:
 ***************************************************************************/
void rgbGenLightingDiffuse
   (
   struct rgbgen_s* rgbgen,
   vavertex_t*      varray,
   int              first,
   int              count
   )
{
   // TODO: implement rgbGenLightingDiffuse()
}



/* rgbGenParseFunc:
 ***************************************************************************/
static qboolean rgbGenParseFunc
   (
   shader_t*       unused,
   shader_stage_t* stage,
   int             argc,
   char*           argv[]
   )
{
   (void)unused;

   if (stricmp(argv[0], "identityLighting") == 0) {
      if (argc != 1) {
         Con_Printf("warning: rgbGen: identityLighting takes no arguments.\n");
         return true;
      }

      stage->rgbgen.func.global = rgbGenIdentityLighting;
      rgbgen_func_global = true;
      stage->flags &= ~R_STAGE_SAVE_COLORS;
   }
   else if (stricmp(argv[0], "identity") == 0) {
      if (argc != 1) {
         Con_Printf("warning: rgbGen: identity takes no arguments.\n");
         return true;
      }

      stage->rgbgen.func.global = rgbGenIdentity;
      rgbgen_func_global = true;
      stage->flags &= ~R_STAGE_SAVE_COLORS;
   }
   else if (stricmp(argv[0], "wave") == 0) {
      int form;

      if (argc != 6) {
         Con_Printf("warning: rgbGen: wave: wrong number of arguments.\n");
         return true;
      }

      stage->rgbgen.func.global = rgbGenWave;
      rgbgen_func_global = true;

      form = ParseWaveForm(argv[1]);

      if (form < 0) {
         Con_Printf("warning: rgbGen: wave: unknown waveform.\n");
         return true;
      }

      stage->rgbgen.args.wave.form = form;
     
      stage->rgbgen.args.wave.base  = atof(argv[2]);
      stage->rgbgen.args.wave.amp   = atof(argv[3]);
      stage->rgbgen.args.wave.phase = atof(argv[4]);
      stage->rgbgen.args.wave.freq  = atof(argv[5]);

      stage->flags &= ~R_STAGE_SAVE_COLORS;
   }
   else if (stricmp(argv[0], "entity") == 0) {
      if (argc != 1) {
         Con_Printf("warning: rgbGen: entity takes no arguments.\n");
         return true;
      }

      stage->rgbgen.func.global = rgbGenEntity;
      rgbgen_func_global = true;
      stage->flags &= ~R_STAGE_SAVE_COLORS;
   }
   else if (stricmp(argv[0], "oneMinusEntity") == 0) {
      if (argc != 1) {
         Con_Printf("warning rgbGen: oneMinusEntity takes no arguments.\n");
         return true;
      }

      stage->rgbgen.func.global = rgbGenOneMinusEntity;
      rgbgen_func_global = true;
      stage->flags &= ~R_STAGE_SAVE_COLORS;
   }
   else if (stricmp(argv[0], "vertex") == 0) {
      if (argc != 1) {
         Con_Printf("warning rgbGen: vertex takes no arguments.\n");
         return true;
      }

      stage->rgbgen.func.local = rgbGenVertex;
      rgbgen_func_global = false;
      stage->flags &= ~R_STAGE_SAVE_COLORS;
   }
   else if (stricmp(argv[0], "exactVertex") == 0) {
      Con_Printf("warning rgbGen: exactVertex is not documented.\n");

      if (argc != 1) {
         Con_Printf("warning rgbGen: exactVertex takes no arguments.\n");
         return true;
      }

      stage->rgbgen.func.local = rgbGenExactVertex;
      rgbgen_func_global = false;
      stage->flags &= ~R_STAGE_SAVE_COLORS;
   }
   else if (stricmp(argv[0], "oneMinusVertex") == 0) {
      if (argc != 1) {
         Con_Printf("warning rgbGen: oneMinusVertex takes no arguments.\n");
         return true;
      }

      stage->rgbgen.func.local = rgbGenOneMinusVertex;
      rgbgen_func_global = false;
      stage->flags |= R_STAGE_SAVE_COLORS;
   }
   else if (stricmp(argv[0], "lightingDiffuse") == 0) {
      if (argc != 1) {
         Con_Printf("warning rgbGen: lightingDiffuse takes no arguments.\n");
         return true;
      }

      stage->rgbgen.func.local = rgbGenLightingDiffuse;
      rgbgen_func_global = false;
      stage->flags |= R_STAGE_SAVE_COLORS;
   }
   else {
      Con_Printf("warning: rgbGen: unknown rgbGen '%s'.\n", argv[0]);
      return true;
   }

   has_rgbgen = true;

   return true;
}



/* R_SaveRGB:
 ***************************************************************************/
void R_SaveRGB(vavertex_t* varray, int first, int count)
{
   vavertex_t* saved = saved_vertexes;
   int i;

   varray += first;

   for (i = 0; i < count; i++) {
      saved->r = varray->r;
      saved->g = varray->g;
      saved->b = varray->b;
      varray++;
      saved++;
   }
}



/* R_RestoreRGB:
 ***************************************************************************/
void R_RestoreRGB(vavertex_t* varray, int first, int count)
{
   vavertex_t* saved = saved_vertexes;
   int i;

   varray += first;

   for (i = 0; i < count; i++) {
      varray->r = saved->r;
      varray->g = saved->g;
      varray->b = saved->b;
      varray++;
      saved++;
   }
}



/* alphaGenIdentity:
 ***************************************************************************/
void alphaGenIdentity
   (
   struct alphagen_s* alphagen,
   unsigned char    color_out[4]
   )
{
   color_out[3] = 255;
}



/* alphaGenWave:
 ***************************************************************************/
void alphaGenWave
   (
   struct alphagen_s* alphagen,
   unsigned char    color_out[4]
   )
{
   int i;
   float y;

   #define ARG (alphagen->args.wave)

   y = 255 * (ARG.amp * WaveForm(ARG.form, ARG.freq * (realtime + ARG.phase))) + ARG.base;
   color_out[3] = CLAMP(0, y, 255);

   #undef ARG
}



/* alphaGenEntity:
 ***************************************************************************/
void alphaGenEntity
   (
   struct alphagen_s* alphagen,
   unsigned char    color_out[4]
   )
{
   color_out[3] = entity_color[3];
}



/* alphaGenOneMinusEntity:
 ***************************************************************************/
void alphaGenOneMinusEntity
   (
   struct alphagen_s* alphagen,
   unsigned char    color_out[4]
   )
{
   color_out[3] = 255 - entity_color[3];
}



/* alphaGenVertex:
 ***************************************************************************/
void alphaGenVertex
   (
   struct alphagen_s* alphagen,
   vavertex_t*        varray,
   int                first,
   int                count
   )
{
   // do nothing
   (void)alphagen;
   (void)varray;
   (void)first;
   (void)count;
}



/* alphaGenOneMinusVertex:
 ***************************************************************************/
void alphaGenOneMinusVertex
   (
   struct alphagen_s* alphagen,
   vavertex_t*        varray,
   int                first,
   int                count
   )
{
   int i;

   varray += first;

   for (i = 0; i < count; i++) {
      varray->a = 255 - varray->a;
      varray++;
   }
}



/* alphaGenLightingSpecular:
 ***************************************************************************/
void alphaGenLightingSpecular
   (
   struct alphagen_s* alphagen,
   vavertex_t*      varray,
   int              first,
   int              count
   )
{
   // TODO: implement alphaGenLightingDiffuse()
}



/* alphaGenPortal:
 ***************************************************************************/
void alphaGenPortal
   (
   struct alphagen_s* alphagen,
   vavertex_t*      varray,
   int              first,
   int              count
   )
{
   int i;
   float  matrix[16];
   vec3_t v;
   float a;

   glGetFloatv (GL_MODELVIEW_MATRIX, matrix);

   varray += first;

   for (i = 0; i < count; i++) {
      v[0] = (varray->x * matrix[0]) + (varray->y * matrix[4]) + (varray->z * matrix[8] ) + matrix[12];
      v[1] = (varray->x * matrix[1]) + (varray->y * matrix[5]) + (varray->z * matrix[9] ) + matrix[13];
      v[2] = (varray->x * matrix[2]) + (varray->y * matrix[6]) + (varray->z * matrix[10]) + matrix[14];

      a = 255 * (Length(v) / alphagen->args.portal.dist_to_opaque);

      varray->a = CLAMP(0, a, 255);
      varray++;
   }
}



/* alphaGenParseFunc:
 ***************************************************************************/
static qboolean alphaGenParseFunc
   (
   shader_t*       unused,
   shader_stage_t* stage,
   int             argc,
   char*           argv[]
   )
{
   (void)unused;

   if (stricmp(argv[0], "identity") == 0) {
      if (argc != 1) {
         Con_Printf("warning: alphaGen: identity takes no arguments.\n");
         return true;
      }

      stage->alphagen.func.global = alphaGenIdentity;
      alphagen_func_global = true;
      stage->flags &= ~R_STAGE_SAVE_ALPHA;
   }
   else if (stricmp(argv[0], "wave") == 0) {
      int form;

      if (argc != 6) {
         Con_Printf("warning: alphaGen: wave: wrong number of arguments.\n");
         return true;
      }

      stage->alphagen.func.global = alphaGenWave;
      alphagen_func_global = true;

      form = ParseWaveForm(argv[1]);

      if (form < 0) {
         Con_Printf("warning: alphaGen: wave: unknown waveform.\n");
         return true;
      }

      stage->alphagen.args.wave.form = form;
     
      stage->alphagen.args.wave.base  = atof(argv[2]);
      stage->alphagen.args.wave.amp   = atof(argv[3]);
      stage->alphagen.args.wave.phase = atof(argv[4]);
      stage->alphagen.args.wave.freq  = atof(argv[5]);

      stage->flags &= ~R_STAGE_SAVE_ALPHA;
   }
   else if (stricmp(argv[0], "entity") == 0) {
      if (argc != 1) {
         Con_Printf("warning: alphaGen: entity takes no arguments.\n");
         return true;
      }

      stage->alphagen.func.global = alphaGenEntity;
      alphagen_func_global = true;
      stage->flags &= ~R_STAGE_SAVE_ALPHA;
   }
   else if (stricmp(argv[0], "oneMinusEntity") == 0) {
      if (argc != 1) {
         Con_Printf("warning alphaGen: oneMinusEntity takes no arguments.\n");
         return true;
      }

      stage->alphagen.func.global = alphaGenOneMinusEntity;
      alphagen_func_global = true;
      stage->flags &= ~R_STAGE_SAVE_ALPHA;
   }
   else if (stricmp(argv[0], "vertex") == 0) {
      if (argc != 1) {
         Con_Printf("warning alphaGen: vertex takes no arguments.\n");
         return true;
      }

      stage->alphagen.func.local = alphaGenVertex;
      alphagen_func_global = false;
      stage->flags &= ~R_STAGE_SAVE_ALPHA;
   }
   else if (stricmp(argv[0], "oneMinusVertex") == 0) {
      if (argc != 1) {
         Con_Printf("warning alphaGen: oneMinusVertex takes no arguments.\n");
         return true;
      }

      stage->alphagen.func.local = alphaGenOneMinusVertex;
      alphagen_func_global = false;
      stage->flags |= R_STAGE_SAVE_ALPHA;
   }
   else if (stricmp(argv[0], "lightingSpecular") == 0) {
      if (argc != 1) {
         Con_Printf("warning alphaGen: lightingSpecular takes no arguments.\n");
         return true;
      }

      stage->alphagen.func.local = alphaGenLightingSpecular;
      alphagen_func_global = false;
      stage->flags |= R_STAGE_SAVE_ALPHA;
   }
   else if (stricmp(argv[0], "portal") == 0) {
      if (argc != 2) {
         Con_Printf("warning alphaGen: portal: wrong number of arguments.\n");
         return true;
      }

      stage->alphagen.func.local = alphaGenPortal;
      alphagen_func_global = false;
      stage->alphagen.args.portal.dist_to_opaque = atof(argv[1]);
      stage->flags |= R_STAGE_SAVE_ALPHA;
   }
   else {
      Con_Printf("warning: alphaGen: unknown alphaGen '%s'.\n", argv[0]);
      return true;
   }

   return true;
}



/* R_SaveAlpha:
 ***************************************************************************/
void R_SaveAlpha(vavertex_t* varray, int first, int count)
{
   vavertex_t* saved = saved_vertexes;
   int i;

   varray += first;

   for (i = 0; i < count; i++) {
      saved->a = varray->a;
      varray++;
      saved++;
   }
}



/* R_RestoreAlpha:
 ***************************************************************************/
void R_RestoreAlpha(vavertex_t* varray, int first, int count)
{
   vavertex_t* saved = saved_vertexes;
   int i;

   varray += first;

   for (i = 0; i < count; i++) {
      varray->a = saved->a;
      varray++;
      saved++;
   }
}



/* rgbGenSetArray:
 ***************************************************************************/
void rgbGenSetArray
   (
   unsigned char color[4],
   vavertex_t*   varray,
   int           first,
   int           count
   )
{
   int i;

   varray += first;

   for (i = 0; i < count; i++) {
      varray->r = color[0];
      varray->b = color[1];
      varray->g = color[2];
      varray++;
   }
}



/* alphaGenSetArray:
 ***************************************************************************/
void alphaGenSetArray
   (
   unsigned char color[4],
   vavertex_t*   varray,
   int           first,
   int           count
   )
{
   int i;

   varray += first;

   for (i = 0; i < count; i++) {
      varray->a = color[3];
      varray++;
   }
}



/* rgbaGenBothGlobal:
 ***************************************************************************/
void rgbaGenBothGlobal
   (
   struct rgbgen_s*   rgbgen,
   struct alphagen_s* alphagen,
   vavertex_t*        varray,
   int                first,
   int                count
   )
{
   unsigned char color[4];

   rgbgen->func.global(rgbgen, color);
   alphagen->func.global(alphagen, color);

   glColor4ubv(color);
}



/* rgbaGenBothLocal:
 ***************************************************************************/
void rgbaGenBothLocal
   (
   struct rgbgen_s*   rgbgen,
   struct alphagen_s* alphagen,
   vavertex_t*        varray,
   int                first,
   int                count
   )
{
   rgbgen->func.local(rgbgen, varray, first, count);
   alphagen->func.local(alphagen, varray, first, count);
}



/* rgbaGenColorLocalAlphaGlobal:
 ***************************************************************************/
void rgbaGenColorLocalAlphaGlobal
   (
   struct rgbgen_s*   rgbgen,
   struct alphagen_s* alphagen,
   vavertex_t*        varray,
   int                first,
   int                count
   )
{
   unsigned char color[4];

   rgbgen->func.local(rgbgen, varray, first, count);

   if (alphagen->func.global != alphaGenIdentity) {
      alphagen->func.global(alphagen, color);
      alphaGenSetArray(color, varray, first, count);
   }
}



/* rgbaGenColorGlobalAlphaLocal:
 ***************************************************************************/
void rgbaGenColorGlobalAlphaLocal
   (
   struct rgbgen_s*   rgbgen,
   struct alphagen_s* alphagen,
   vavertex_t*        varray,
   int                first,
   int                count
   )
{
   unsigned char color[4];

   rgbgen->func.global(rgbgen, color);
   rgbGenSetArray(color, varray, first, count);

   alphagen->func.local(alphagen, varray, first, count);
}



/* rgbaGenFinalize:
 ***************************************************************************/
void rgbaGenFinalize(shader_stage_t* stage)
{
   if (rgbgen_func_global) {
      if (alphagen_func_global) {
         stage->rgbagen_func = rgbaGenBothGlobal;
         stage->off_fields &= ~(R_COLOR3|R_COLOR4);
      }
      else {
         stage->rgbagen_func = rgbaGenColorGlobalAlphaLocal;
         stage->on_fields  |=  R_COLOR4;
         stage->on_fields  &= ~R_COLOR3;
         stage->off_fields |=  R_COLOR3;
      }
   }
   else {
      if (alphagen_func_global) {
         stage->rgbagen_func = rgbaGenColorLocalAlphaGlobal;

         if (stage->alphagen.func.global == alphaGenIdentity) {
            stage->on_fields  |=  R_COLOR3;
            stage->on_fields  &= ~R_COLOR4;
            stage->off_fields |=  R_COLOR4;
         }
         else {
            stage->on_fields  |=  R_COLOR4;
            stage->on_fields  &= ~R_COLOR3;
            stage->off_fields |=  R_COLOR3;
         }
      }
      else {
         stage->rgbagen_func = rgbaGenBothLocal;
         stage->on_fields  |=  R_COLOR4;
         stage->on_fields  &= ~R_COLOR3;
         stage->off_fields |=  R_COLOR3;
      }
   }

   if ((alphagen_func_global && stage->alphagen.func.global == alphaGenIdentity) &&
       (rgbgen_func_global && stage->rgbgen.func.global == rgbGenIdentity)) {
      stage->flags &= ~R_STAGE_MODULATE;
   }
   else {
      stage->flags |= R_STAGE_MODULATE;
   }
}



/* tcGenBase:
 ***************************************************************************/
void tcGenBase(struct tcgen_s* tcgen, vavertex_t* varray, int first, int count)
{
   // do nothing
   (void)tcgen;
   (void)varray;
   (void)first;
   (void)count;
}



/* tcGenLightmap:
 ***************************************************************************/
void tcGenLightmap(struct tcgen_s* tcgen, vavertex_t* varray, int first, int count)
{
   int i;
      
   if (current_lightmap_texcoords) {
      texcoord_t* larray;

      larray  = current_lightmap_texcoords + first;
      varray += first;

      for (i = 0; i < count; i++) {
         varray->s = larray->s;
         varray->t = larray->t;
         varray++;
         larray++;
      }
   }
   else {
      for (i = 0; i < count; i++) {
         varray->s = 0;
         varray->t = 0;
         varray++;
      }
   }

}



/* tcGenEnvironment:
 ***************************************************************************/
void tcGenEnvironment(struct tcgen_s* tcgen, vavertex_t* varray, int first, int count)
{
}



/* tcGenVector:
 ***************************************************************************/
void tcGenVector
   (
   struct tcgen_s* tcgen,
   vavertex_t*     varray,
   int             first,
   int             count
   )
{
   int i;

   #define ARG (tcgen->args.vector)

   varray += first;

   for (i = 0; i < count; i++) {
      varray->s = (varray->x * ARG.sx) + (varray->y * ARG.sy) + (varray->z * ARG.sz);
      varray->t = (varray->x * ARG.tz) + (varray->y * ARG.ty) + (varray->z * ARG.tz);
      varray++;
   }
}



/* tcGenParseFunc:
 ***************************************************************************/
static qboolean tcGenParseFunc
   (
   shader_t*       shader,
   shader_stage_t* stage,
   int             argc,
   char*           argv[]
   )
{
   if (stricmp(argv[0], "base") == 0) {
      if (argc != 1) {
         Con_Printf("warning tcGen: base takes no arguments.\n");
         return true;
      }

      stage->tcgen.func = tcGenBase;
   }
   else if (stricmp(argv[0], "lightmap") == 0) {
      if (argc != 1) {
         Con_Printf("warning tcGen: lightmap takes no arguments.\n");
         return true;
      }

      stage->tcgen.func = tcGenLightmap;
      stage->flags |= R_STAGE_SAVE_TEXCOORDS;
      stage->on_fields |= R_TEXCOORD2;
   }
   else if (stricmp(argv[0], "environment") == 0) {
      if (argc != 1) {
         Con_Printf("warning tcGen: environment takes no arguments.\n");
         return true;
      }

      stage->tcgen.func = tcGenEnvironment;
      stage->flags |= R_STAGE_SAVE_TEXCOORDS;
      stage->on_fields |= R_TEXCOORD2;
   }
   else if (stricmp(argv[0], "vector") == 0) {
      if (argc != 11) {
         Con_Printf("warning tcGen: vector takes 6 arguments, make sure that you space out the parentheses.\n");
         return true;
      }

      if (strcmp(argv[1],  "(") ||
          strcmp(argv[5],  ")") ||
          strcmp(argv[6],  "(") ||
          strcmp(argv[10], ")")) {
         Con_Printf("warning tcGen: make sure to put parentheses around the vectors and space them out.");
      }

      stage->tcgen.func = tcGenVector;

      stage->tcgen.args.vector.sx = atoi(argv[2]);
      stage->tcgen.args.vector.sy = atoi(argv[3]);
      stage->tcgen.args.vector.sz = atoi(argv[4]);
      stage->tcgen.args.vector.tx = atoi(argv[7]);
      stage->tcgen.args.vector.ty = atoi(argv[8]);
      stage->tcgen.args.vector.tz = atoi(argv[9]);

      stage->flags |= R_STAGE_SAVE_TEXCOORDS;
      stage->on_fields |= R_TEXCOORD2;
   }
   else {
      Con_Printf("warning: blendFunc: unknown blendFunc '%s'.\n", argv[0]);
      return true;
   }

   explicit_tcgen = true;

   return true;
}



static qboolean detailParseFunc
   (
   shader_t*       shader,
   shader_stage_t* stage,
   int             argc,
   char*           argv[]
   )
{
   stage->flags |= R_DETAIL;

   return true;
}



/* entityMergableParseFunc
 ***************************************************************************/
static qboolean entityMergableParseFunc
   (
   shader_t*       shader,
   shader_stage_t* stage,
   int             argc,
   char*           argv[]
   )
{
   Con_Printf("warning: entityMergable is not documented.\n");

   shader->flags |= R_SHADER_ENTITY_MERGABLE;

   return true;
}


typedef struct filter_parse_table_s {
   char*    name;
   R_FILTER value;
   qboolean is_mag_filter;
} filter_parse_table_t;

filter_parse_table_t filter_parse_table[] =
{
   { "GL_NEAREST",                R_NEAREST, true                 },
   { "GL_LINEAR",                 R_LINEAR,  true                 },
   { "GL_NEAREST_MIPMAP_NEAREST", R_NEAREST_MIPMAP_NEAREST, false },
   { "GL_NEAREST_MIPMAP_LINEAR",  R_NEAREST_MIPMAP_LINEAR,  false },
   { "GL_LINEAR_MIPMAP_NEAREST",  R_LINEAR_MIPMAP_NEAREST,  false },
   { "GL_LINEAR_MIPMAP_LINEAR",   R_LINEAR_MIPMAP_LINEAR,   false },
   { 0 }
};



R_FILTER Filter(const char* name, qboolean is_mag_filter)
{
   filter_parse_table_t* table;

   table = filter_parse_table;

   while (table->name) {
      if (stricmp(table->name, name) == 0) {
         if (is_mag_filter && !table->is_mag_filter) {
            Con_Printf("warning: _filter: %s is can not be a magnification name.\n", name);
            return -1;
         }

         return table->value;
      }

      table++;
   }

   Con_Printf("warning: _filter: no such name '%s'.\n", name);

   return -1;
}



/* filterParseFunc
 ***************************************************************************/
static qboolean filterParseFunc
   (
   shader_t*       shader,
   shader_stage_t* stage,
   int             argc,
   char*           argv[]
   )
{
   Con_DPrintf("warning: _filter is an extension.\n");
   stage->min_filter = Filter(argv[0], false);

   if (stage->min_filter < 0) {
      return true;
   }

   stage->mag_filter = Filter(argv[1], true);

   if (stage->mag_filter < 0) {
      return true;
   }

   stage->flags |= R_STAGE_HAS_FILTER;

   return true;
}



/* cachepicParseFunc
 ***************************************************************************/
static qboolean cachepicParseFunc
   (
   shader_t*       shader,
   shader_stage_t* stage,
   int             argc,
   char*           argv[]
   )
{
   Con_DPrintf("warning: _cachepic is an extension.\n");

   return true;
}



static shader_command_t commands[] =
{
    // global commands
    { "skyParms",             3,  3, false, false, skyParmsParseFunc       },
    { "cull",                 0,  1, false, false, cullParseFunc           },
    { "deformVertexes",       1,  9, false, false, deformVertexesParseFunc },
    { "fogparms",             4,  4, false, false, fogparmsParseFunc       },
    { "nopicmip",             0,  0, false, false, nopicmipParseFunc       },
    { "nomipmaps",            0,  0, false, false, nomipmapsParseFunc      },
    { "polygonOffset",        0,  0, false, false, polygonOffsetParseFunc  },
    { "portal",               0,  0, false, false, portalParseFunc         },
    { "sort",                 1,  1, false, false, sortParseFunc           },

    // undocumented global commands
    { "entityMergable",       0,  0, false, false, entityMergableParseFunc },

    // extensions
    { "_cachepic",            0,  0, false, false, cachepicParseFunc },

    // q3bsp keywords (ignored by the renderer)
    { "tessSize",             0,  0, false, true,  NULL },
    { "q3map_backshader",     0,  0, false, true,  NULL },
    { "q3map_globaltexture",  0,  0, false, true,  NULL },
    { "q3map_sun",            0,  0, false, true,  NULL },
    { "q3map_surfaceLight",   0,  0, false, true,  NULL },
    { "q3map_lightimage",     0,  0, false, true,  NULL },
    { "q3map_lightsubdivide", 0,  0, false, true,  NULL },
    { "surfaceparm",          0,  0, false, true,  NULL },

    // editor commands (ignored by the renderer)
    { "qer_editorimage",      0,  0, false, true,  NULL },
    { "qer_nocarve",          0,  0, false, true,  NULL },
    { "qer_trans",            0,  0, false, true,  NULL },

    // stage commands
    { "map",                  1,  1, true,  false, mapParseFunc        },
    { "clampmap",             1,  1, true,  false, clampmapParseFunc   },
    { "AnimMap",              2,  9, true,  false, AnimMapParseFunc    },
    { "blendFunc",            1,  2, true,  false, blendFuncParseFunc  },
    { "rgbGen",               1,  6, true,  false, rgbGenParseFunc     },
    { "alphaGen",             1,  1, true,  false, alphaGenParseFunc   },
    { "tcGen",                1, 11, true,  false, tcGenParseFunc      },
    { "tcMod",                2,  7, true,  false, tcModParseFunc      },
    { "depthFunc",            1,  1, true,  false, depthFuncParseFunc  },
    { "depthWrite",           0,  0, true,  false, depthWriteParseFunc },
    { "detail",               0,  0, true,  false, detailParseFunc     },
    { "alphaFunc",            1,  1, true,  false, alphaFuncParseFunc  },

    // extensions
    { "_filter",              2,  2, true,  false, filterParseFunc  },

    { 0 }
};



static qboolean R_ParseShaderCommand(shader_t* shader, shader_stage_t* stage, char* command, qboolean is_eol)
{
    shader_command_t* cmd;

    

    for (cmd = commands; true; cmd++) {
       if (!cmd->name) {
          Con_Printf("warning: unrecognized command '%s'.\n", command);
          break;
       }

       if (stage && !cmd->is_stage_command) {
          if (stricmp(cmd->name, command) == 0) {
             Con_Printf("warning: '%s' is not a stage command.\n", command);
             break;
          }
          else {
             continue;
          }
       }

       if (!stage &&  cmd->is_stage_command) {
          if (stricmp(cmd->name, command) == 0) {
             Con_Printf("warning: '%s' is not a global command.\n", command);
             break;
          }
          else {
             continue;
          }
       }

       if (stricmp(cmd->name, command) == 0) {
          char* argv[R_MAX_SHADER_ARGS];
          int   argc = 0;

          if (cmd->is_ignored) break;

          while (!is_eol) {
             argv[argc] = nexttok(&is_eol);

             if (argv[argc] == NULL) {
                Con_Printf("error: unexpected end of input parsing '%s'.\n", command);
                return false;
             }

             argc++;

             if (argc == R_MAX_SHADER_ARGS) {
                 while (!is_eol) {
                    nexttok(&is_eol);
                 }

                 Con_Printf("warning: number of arguments for '%s' exceeded maximum for any command.\n", command);
                 return true;
             }
          }

          if (argc > cmd->max_argc) {
             Con_Printf("warning: too many arguments for '%s'.\n", command);
             return true;
          }

          if (argc < cmd->min_argc) {
             Con_Printf("warning: too few arguments for '%s'.\n", command);
             return true;
          }

          return cmd->parse(shader, stage, argc, argv);
       }
    }

    while (!is_eol) {
      nexttok(&is_eol);
    }

    return true;
}



static qboolean R_ParseShaderStage(shader_t* shader)
{
   char*           tok;
   shader_stage_t* stage;

   if (shader->num_stages == R_MAX_SHADER_STAGES) {
      Con_Printf("warning: too many shader stages for %s\n", shader->name);
      return true;
   }

   force_depth_write = false;

   map_alpha = false;

   num_maps = 1;

   used_blendfunc = false;

   explicit_tcgen = false;

   map_name[0][0] = '\0';

   farbox_base_name[0]  = '\0';
   nearbox_base_name[0] = '\0';

   stage = &shader->stages[shader->num_stages];

   stage->flags = 0;

   stage->on_fields  = 0;
   stage->off_fields = 0;

   stage->flags |= R_DEPTHWRITE;

   stage->depth_test_func = R_LEQUAL;

   stage->num_tcmods = 0;

   has_rgbgen = false;

   stage->alphagen.func.global = alphaGenIdentity;
   alphagen_func_global = true;

   stage->tcgen.func = tcGenBase;

   for (;;) {
      qboolean is_eol;

      tok = nexttok(&is_eol);

      if (tok == NULL) {
         Con_Printf("error: unexpected end of input\n");
         return false;
      }

      if (tok[0] == '}') {
         if (force_depth_write) stage->flags |= R_DEPTHWRITE;

         mapFinalize(stage);

         // assign default rgbGen if none was specified
         if (!has_rgbgen) {
            if (stage->blend_dst == R_ZERO || stage->blend_src == R_ZERO) {
               // filter stages get 'rgbGen identity' by default
               stage->rgbgen.func.global = rgbGenIdentity;
            }
            else {
               // additive and blend stages get 'rgbGen identityLighting' by default
               stage->rgbgen.func.global = rgbGenIdentityLighting;
            }

            rgbgen_func_global = true;
         }

         rgbaGenFinalize(stage);

         shader->num_stages++;

         if (!used_blendfunc && ((shader->num_stages > 1) || used_skyparms)) {
            Con_Printf(
               "warning: no blendfunc, stage #%d will overwrite previous stages.\n",
               shader->num_stages);
         }

         return true;
      }
      else {
         if (!R_ParseShaderCommand(shader, stage, tok, is_eol)) return false;
      }
   }
}



void R_ParseShaders(char* text, size_t size)
{
   char*     tok;
   shader_t* shader;
   int       id;

   curpos = text;
   endpos = text + size - 1;

   for (;;) {
      tok = nexttok(NULL);
      
      if (tok == NULL) break;

      id = R_GetShader(tok);

      if (id < 0) {
         if (!shader_skip()) return;
         continue;
      }

      shader = &shaders[id];

      if (shader->flags & R_IS_LOADED) {
         if (!shader_skip()) return;
         continue;
      }

      if (strlen(tok) > R_SHADER_NAME_LEN) Con_Printf("warning: shader name too long");

      // shader body
      tok = nexttok(NULL);

      if (tok[0] != '{') {
         Con_Printf("R_ParseShaders: expected '{', found '%s'\n", tok);

         return;
      }

      nomipmaps = false;
      nopicmip  = false;

      used_skyparms = false;

      Con_DPrintf("   parsing '%s'\n", shader->name);

      // stages and commands
      for (;;) {
         qboolean is_eol;

         tok = nexttok(&is_eol);
         
         if (tok == NULL) {
            // error
            Con_Printf("error: unexpected end of input\n");
            return;
         }

         if (tok[0] == '{') {
            // stage
            if (!R_ParseShaderStage(shader)) return;
         }
         else if (tok[0] == '}') {
            // end of shader body
            if (used_skyparms) {
               skyParmsFinalize(shader);
            }

            shader->flags |= R_IS_LOADED;
            break;
         }
         else {
            // shader command
            if (!R_ParseShaderCommand(shader, NULL, tok, is_eol)) return;
         }
      }
   }

   // make default shaders
}



void R_ClearShaders(void)
{
}



int R_GetShader(const char* name)
{
    int i;

    for (i = 0; i < num_shaders; i++) {
       if (strncmp(name, shaders[i].name, R_SHADER_NAME_LEN) == 0) return i;
    }

    return R_NOT_FOUND;
}



int R_LoadMapFromWad
   (
   const char* name,
   qboolean    mipmap,
   qboolean    picmip,
   qboolean    alpha,
   qboolean    clampmap
   )
{
    qpic_t* pic;

    pic = W_GetLumpName((char*)name);

    texture_maps[num_texture_maps].index =
       GL_LoadTexture (
          (char*)name,
          pic->width,
          pic->height,
          pic->data,
          mipmap,
          picmip,
          alpha,
          clampmap);

    num_texture_maps++;

    return num_texture_maps - 1;
}



int R_LoadMap(const char* name, qboolean mipmap, qboolean picmip, qboolean alpha, qboolean clampmap)
{
   const char *ext;

   if (stricmp(map_name[0], "$lightmap") == 0) {
      return R_MAP_LIGHTMAP;
   }
   else if (stricmp(map_name[0], "$whiteimage") == 0) {
      return R_MAP_WHITEIMAGE;
   }
   else if (strnicmp(name, "$gfx.wad#", 9) == 0) {
      return R_LoadMapFromWad(name + 9, mipmap, picmip, alpha, clampmap);
   }

   ext = COM_FileExtension((char*)name);

   if (stricmp(ext, "lmp") == 0) {
      return R_LoadMapFromLump(name, mipmap, picmip, alpha, clampmap);
   }
   else if (stricmp(ext, "tga") == 0) {
      return R_LoadMapFromTarga(name, mipmap, picmip, alpha, clampmap);
   }
   else if (stricmp(ext, "jpg") == 0) {
      return R_LoadMapFromJPEG(name, mipmap, picmip, alpha, clampmap);
   }
   else {
      return R_UNKNOWN_TYPE;
   }
}



/* R_SafeLoadMap:
 ***************************************************************************/
int R_SafeLoadMap
   (
   const char* name,
   qboolean    mipmap,
   qboolean    picmip,
   qboolean    alpha,
   qboolean    clampmap
   )
{
   int map;

   map = R_LoadMap(name, mipmap, picmip, alpha, clampmap);

   if (map < 0) {
      if (map == R_NOT_FOUND) {
         map = R_MAP_QUESTION;
         Con_Printf("warning: load map: image file '%s' not found\n", name);
      }
      else if (map == R_OUT_OF_MEMORY) {
         map = R_MAP_EXCLAIM;
         Con_Printf("warning: load map: out of space, please increase -nummaps\n");
      }
      else if (map == R_UNKNOWN_TYPE) {
         map = R_MAP_QUESTION;
         Con_Printf("warning: load map: unknown map type.\n");
      }
      else {
         map = R_MAP_EXCLAIM;
         Con_Printf("warning: load map: error loading '%s'.\n", name);
      }
   }

   return map;
}



int R_LoadMapFromTarga
   (
   const char* name,
   qboolean    mipmap,
   qboolean    picmip,
   qboolean    alpha,
   qboolean    clampmap
   )
{
   FILE* f;

   if (num_texture_maps > texture_map_array_size) return R_OUT_OF_MEMORY;

   COM_FOpenFile ((char*)name, &f);

   if (!f) return R_NOT_FOUND;

   LoadTGA(f);

   strncpy(texture_maps[num_texture_maps].name, name, R_SHADER_NAME_LEN);
   texture_maps[num_texture_maps].name[R_SHADER_NAME_LEN] = '\0';

   texture_maps[num_texture_maps].index =
      GL_LoadTexture32(
         (char*)name,
         targa_header.width,
         targa_header.height,
         (int*)targa_rgba,
         mipmap,
         picmip,
         alpha,
         clampmap);

   free (targa_rgba);

   num_texture_maps++;

   return num_texture_maps - 1;
}



int R_LoadMapFromLump
   (
   const char* name,
   qboolean mipmap,
   qboolean picmip,
   qboolean alpha,
   qboolean clampmap
   )
{
   qpic_t* qpic;
   int     mark;

   if (num_texture_maps > texture_map_array_size) return R_OUT_OF_MEMORY;

   mark = Hunk_LowMark();

   qpic = (qpic_t *)COM_LoadHunkFile((char*)name);

   if (!qpic) return R_NOT_FOUND;

   SwapPic(qpic);

   strncpy(texture_maps[num_texture_maps].name, name, R_SHADER_NAME_LEN);
   texture_maps[num_texture_maps].name[R_SHADER_NAME_LEN] = '\0';

   texture_maps[num_texture_maps].index =
      GL_LoadTexture(
         (char*)name,
         qpic->width,
         qpic->height,
         qpic->data,
         mipmap,
         picmip,
         alpha,
         clampmap);

   num_texture_maps++;

   Hunk_FreeToLowMark(mark);

   return num_texture_maps - 1;
}



int R_LoadMapFromJPEG
   (
   const char* name,
   qboolean    mipmap,
   qboolean    picmip,
   qboolean    alpha,
   qboolean    clampmap
   )
{
   return R_UNKNOWN_TYPE;
}



int filter_table[] = {
   GL_NEAREST,
   GL_LINEAR,
   GL_NEAREST_MIPMAP_NEAREST,
   GL_NEAREST_MIPMAP_LINEAR,
   GL_LINEAR_MIPMAP_NEAREST,
   GL_LINEAR_MIPMAP_LINEAR
};



/* R_Bind:
 ***************************************************************************/
void R_Bind(shader_stage_t* stage)
{
   if (stage->num_maps == 1) {
      GL_Bind(texture_maps[stage->map[0].map_num].index);
   }
   else {
      float t = realtime * stage->anim_map_interval;

      GL_Bind(texture_maps[stage->map[(int)((t - floor(t)) * stage->num_maps)].map_num].index);
   }

   if (stage->flags & R_STAGE_MODULATE) {
      glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
   }
   else {
      glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
   }

   if (stage->flags & R_STAGE_HAS_FILTER) {
      glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, filter_table[stage->mag_filter]);
      glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, filter_table[stage->min_filter]);
   }
}



/* R_RequestShaderPic:
 ***************************************************************************/
void R_RequestShaderPic(shader_pic_t* pic, const char* name, float width, float height)
{
    pic->shader_num = R_RequestShader(name);
	pic->sl         = 0;
	pic->sh         = 1;
	pic->tl         = 0;
	pic->th         = 1;
	pic->width      = width;
	pic->height     = height;
    pic->r          = 255;
    pic->g          = 255;
    pic->b          = 255;
    pic->a          = 255;

    return;
}



