/*
Copyright (C) 1996-1997 Id Software, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

// draw.c -- this is the only file outside the refresh that touches the
// vid buffer

#include "quakedef.h"

// fenix@io.com: vertex arrays
#include "varrays.h"
// --

// fenix@io.com: shaders
#include "shaders.h"
// --

// fenix@io.com: variable console alpha
cvar_t  r_console_alpha = { "r_console_alpha", "0.6" };
// --

// fenix@io.com: colored characters
unsigned char col_white[4] = { 255, 255, 255, 255 };
// --

#define GL_COLOR_INDEX8_EXT     0x80E5

extern unsigned char d_15to8table[65536];

cvar_t		gl_nobind = {"gl_nobind", "0"};
cvar_t		gl_max_size = {"gl_max_size", "1024"};
cvar_t		gl_picmip = {"gl_picmip", "0"};

// fenix@io.com: shaders
shader_pic_t draw_disc;
qboolean     disc_initialized;

shader_pic_t draw_backtile;
// --

int			translate_texture;
int         char_texture;

// fenix@io.com: shaders
int conchars;
// --

typedef struct
{
	int		texnum;
	float	sl, tl, sh, th;
} glpic_t;

// fenix@io.com: shaders
shader_pic_t console;
int console_line;
// --

int		gl_lightmap_format = 4;
int		gl_solid_format = 3;
int		gl_alpha_format = 4;

int		gl_filter_min = GL_LINEAR_MIPMAP_NEAREST;
int		gl_filter_max = GL_LINEAR;


int		texels;

typedef struct
{
	int		texnum;
	char	identifier[64];
	int		width, height;
	qboolean	mipmap;
} gltexture_t;

#define	MAX_GLTEXTURES	1024
gltexture_t	gltextures[MAX_GLTEXTURES];
int			numgltextures;


void GL_Bind (int texnum)
{
    // fenix@io.com: shaders
    // TODO: change this to ? texture
	// if (gl_nobind.value)
	//	texnum = char_texture;
    // --
	if (currenttexture == texnum)
		return;
	currenttexture = texnum;
#ifdef _WIN32
	bindTexFunc (GL_TEXTURE_2D, texnum);
#else
	glBindTexture(GL_TEXTURE_2D, texnum);
#endif
}


/*
=============================================================================

  scrap allocation

  Allocate all the little status bar obejcts into a single texture
  to crutch up stupid hardware / drivers

=============================================================================
*/

#define	MAX_SCRAPS		2
#define	BLOCK_WIDTH		256
#define	BLOCK_HEIGHT	256

int			scrap_allocated[MAX_SCRAPS][BLOCK_WIDTH];
byte		scrap_texels[MAX_SCRAPS][BLOCK_WIDTH*BLOCK_HEIGHT*4];
qboolean	scrap_dirty;
int			scrap_texnum;

// returns a texture number and the position inside it
int Scrap_AllocBlock (int w, int h, int *x, int *y)
{
	int		i, j;
	int		best, best2;
	int		bestx;
	int		texnum;

	for (texnum=0 ; texnum<MAX_SCRAPS ; texnum++)
	{
		best = BLOCK_HEIGHT;

		for (i=0 ; i<BLOCK_WIDTH-w ; i++)
		{
			best2 = 0;

			for (j=0 ; j<w ; j++)
			{
				if (scrap_allocated[texnum][i+j] >= best)
					break;
				if (scrap_allocated[texnum][i+j] > best2)
					best2 = scrap_allocated[texnum][i+j];
			}
			if (j == w)
			{	// this is a valid spot
				*x = i;
				*y = best = best2;
			}
		}

		if (best + h > BLOCK_HEIGHT)
			continue;

		for (i=0 ; i<w ; i++)
			scrap_allocated[texnum][*x + i] = best + h;

		return texnum;
	}

	Sys_Error ("Scrap_AllocBlock: full");
}

int	scrap_uploads;

void Scrap_Upload (void)
{
	int		texnum;

	scrap_uploads++;

	for (texnum=0 ; texnum<MAX_SCRAPS ; texnum++) {
		GL_Bind(scrap_texnum + texnum);
        // fenix@io.com: texture maps
		GL_Upload8 (scrap_texels[texnum], BLOCK_WIDTH, BLOCK_HEIGHT, false, true, true, false);
        // --
	}
	scrap_dirty = false;
}

//=============================================================================
/* Support Routines */

typedef struct cachepic_s
{
	char		name[MAX_QPATH];
	qpic_t		pic;
	byte		padding[32];	// for appended glpic
} cachepic_t;

#define	MAX_CACHED_PICS		128
cachepic_t	menu_cachepics[MAX_CACHED_PICS];
int			menu_numcachepics;

byte		menuplyr_pixels[4096];

int		pic_texels;
int		pic_count;

qpic_t *Draw_PicFromWad (char *name)
{
	qpic_t	*p;
	glpic_t	*gl;

	p = W_GetLumpName (name);
	gl = (glpic_t *)p->data;

    Con_Printf("%s: %d, %d\n", name, p->width, p->height);

	// load little ones into the scrap
	if (p->width < 64 && p->height < 64)
	{
		int		x, y;
		int		i, j, k;
		int		texnum;

		texnum = Scrap_AllocBlock (p->width, p->height, &x, &y);
		scrap_dirty = true;
		k = 0;
		for (i=0 ; i<p->height ; i++)
			for (j=0 ; j<p->width ; j++, k++)
				scrap_texels[texnum][(y+i)*BLOCK_WIDTH + x + j] = p->data[k];
		texnum += scrap_texnum;
		gl->texnum = texnum;
		gl->sl = (x+0.01)/(float)BLOCK_WIDTH;
		gl->sh = (x+p->width-0.01)/(float)BLOCK_WIDTH;
		gl->tl = (y+0.01)/(float)BLOCK_WIDTH;
		gl->th = (y+p->height-0.01)/(float)BLOCK_WIDTH;

		pic_count++;
		pic_texels += p->width*p->height;
	}
	else
	{
		gl->texnum = GL_LoadPicTexture (p);
		gl->sl = 0;
		gl->sh = 1;
		gl->tl = 0;
		gl->th = 1;
	}
	return p;
}


/*
================
Draw_CachePic
================
*/
qpic_t	*Draw_CachePic (char *path)
{
	cachepic_t	*pic;
	int			i;
	qpic_t		*dat;
	glpic_t		*gl;

	for (pic=menu_cachepics, i=0 ; i<menu_numcachepics ; pic++, i++)
		if (!strcmp (path, pic->name))
			return &pic->pic;

	if (menu_numcachepics == MAX_CACHED_PICS)
		Sys_Error ("menu_numcachepics == MAX_CACHED_PICS");
	menu_numcachepics++;
	strcpy (pic->name, path);

//
// load the pic from disk
//
	dat = (qpic_t *)COM_LoadTempFile (path);	
	if (!dat)
		Sys_Error ("Draw_CachePic: failed to load %s", path);
	SwapPic (dat);

	// HACK HACK HACK --- we need to keep the bytes for
	// the translatable player picture just for the menu
	// configuration dialog
	if (!strcmp (path, "gfx/menuplyr.lmp"))
		memcpy (menuplyr_pixels, dat->data, dat->width*dat->height);

	pic->pic.width = dat->width;
	pic->pic.height = dat->height;

	gl = (glpic_t *)pic->pic.data;
	gl->texnum = GL_LoadPicTexture (dat);
	gl->sl = 0;
	gl->sh = 1;
	gl->tl = 0;
	gl->th = 1;

	return &pic->pic;
}


typedef struct
{
	char *name;
	int	minimize, maximize;
} glmode_t;

glmode_t modes[] = {
	{"GL_NEAREST", GL_NEAREST, GL_NEAREST},
	{"GL_LINEAR", GL_LINEAR, GL_LINEAR},
	{"GL_NEAREST_MIPMAP_NEAREST", GL_NEAREST_MIPMAP_NEAREST, GL_NEAREST},
	{"GL_LINEAR_MIPMAP_NEAREST", GL_LINEAR_MIPMAP_NEAREST, GL_LINEAR},
	{"GL_NEAREST_MIPMAP_LINEAR", GL_NEAREST_MIPMAP_LINEAR, GL_NEAREST},
	{"GL_LINEAR_MIPMAP_LINEAR", GL_LINEAR_MIPMAP_LINEAR, GL_LINEAR}
};

/*
===============
Draw_TextureMode_f
===============
*/
void Draw_TextureMode_f (void)
{
	int		i;
	gltexture_t	*glt;

	if (Cmd_Argc() == 1)
	{
		for (i=0 ; i< 6 ; i++)
			if (gl_filter_min == modes[i].minimize)
			{
				Con_Printf ("%s\n", modes[i].name);
				return;
			}
		Con_Printf ("current filter is unknown???\n");
		return;
	}

	for (i=0 ; i< 6 ; i++)
	{
		if (!Q_strcasecmp (modes[i].name, Cmd_Argv(1) ) )
			break;
	}
	if (i == 6)
	{
		Con_Printf ("bad filter name\n");
		return;
	}

	gl_filter_min = modes[i].minimize;
	gl_filter_max = modes[i].maximize;

	// change all the existing mipmap texture objects
	for (i=0, glt=gltextures ; i<numgltextures ; i++, glt++)
	{
		if (glt->mipmap)
		{
			GL_Bind (glt->texnum);
			glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, gl_filter_min);
			glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, gl_filter_max);
		}
	}
}



/*
===============
Draw_Init
fenix@io.com: shaders
===============
*/
void Draw_Init (void)
{
	int		i;
    int     map;

	Cvar_RegisterVariable (&gl_nobind);
	Cvar_RegisterVariable (&gl_max_size);
	Cvar_RegisterVariable (&gl_picmip);

	// 3dfx can only handle 256 wide textures
	if (!Q_strncasecmp ((char *)gl_renderer, "3dfx",4) ||
		strstr((char *)gl_renderer, "Glide"))
		Cvar_Set ("gl_max_size", "256");

	Cmd_AddCommand ("gl_texturemode", &Draw_TextureMode_f);

	// save a texture slot for translated picture
	translate_texture = texture_extension_number++;

	// save slots for scraps
	scrap_texnum = texture_extension_number;
	texture_extension_number += MAX_SCRAPS;

    R_InitShaders();

    conchars = R_RequestShader("conchars");

    console_line = R_RequestShader("console_line");

    R_RequestShaderPic(&console, "console",  vid.width, vid.height);

    R_RequestShaderPic(&draw_disc, "disc", 24, 24);
    disc_initialized = true;

    R_RequestShaderPic(&draw_backtile, "backtile", 64, 64);
}



/*
================
Draw_Character

Draws one 8*8 graphics character with 0 being transparent.
It can be clipped to the top of the screen to allow the console to be
smoothly scrolled off.

fenix@io.com: vertex arrays, colored characters, shaders
================
*/
void Draw_Character (int x, int y, int num, unsigned char color[4])
{
	int				row, col;
	float			frow, fcol, size;
    vavertex_t      *varray;

	num &= 255;
	
	if (num == 32) return; // space

	if (y <= -8) return; // totally off screen

	row = num>>4;
	col = num&15;

	frow = row*0.0625;
	fcol = col*0.0625;
	size = 0.0625;

	GL_Bind (char_texture);
    R_Shader(conchars);

    R_VertexArray(global_vertex_array, R_VERTEX2|R_TEXCOORD2|R_COLOR4);

    varray = global_vertex_array;

    varray->s = fcol;
    varray->t = frow;
    varray->r = color[0];
    varray->g = color[1];
    varray->b = color[2];
    varray->a = color[3];
    varray->x = x;
    varray->y = y;
    varray++;

    varray->s = fcol + size;
    varray->t = frow;
    varray->r = color[0];
    varray->g = color[1];
    varray->b = color[2];
    varray->a = color[3];
    varray->x = x + 8;
    varray->y = y;
    varray++;

    varray->s = fcol + size;
    varray->t = frow + size;
    varray->r = color[0];
    varray->g = color[1];
    varray->b = color[2];
    varray->a = color[3];
    varray->x = x + 8;
    varray->y = y + 8;
    varray++;

    varray->s = fcol;
    varray->t = frow + size;
    varray->r = color[0];
    varray->g = color[1];
    varray->b = color[2];
    varray->a = color[3];
    varray->x = x;
    varray->y = y + 8;

    SHADER_R_DrawArray(R_QUADS, 0, 4);
}



/*
================
Draw_String
================
*/
void Draw_String (int x, int y, char *str)
{
	while (*str)
	{
        // fenix@io.com: colored characters
		Draw_Character (x, y, *str, col_white);
        // --
		str++;
		x += 8;
	}
}

/*
================
Draw_DebugChar

Draws a single character directly to the upper right corner of the screen.
This is for debugging lockups by drawing different chars in different parts
of the code.
================
*/
void Draw_DebugChar (char num)
{
}


/*
=============
_Draw_Pic

fenix@io.com: vertex arrays
=============
*/
void _Draw_Pic (int x, int y, qpic_t *pic)
{
    glpic_t    *gl;
    vavertex_t *varray;

    if (scrap_dirty) Scrap_Upload ();

    gl = (glpic_t *)pic->data;

    GL_Bind (gl->texnum);

    R_VertexArray (global_vertex_array, R_VERTEX2|R_TEXCOORD2);

    varray = global_vertex_array;

    varray->s = gl->sl;
    varray->t = gl->tl;
    varray->x = x;
    varray->y = y;
    varray++;

    varray->s = gl->sh;
    varray->t = gl->tl;
    varray->x = x + pic->width;
    varray->y = y;
    varray++;

    varray->s = gl->sh;
    varray->t = gl->th;
    varray->x = x + pic->width;
    varray->y = y + pic->height;
    varray++;

    varray->s = gl->sl;
    varray->t = gl->th;
    varray->x = x;
    varray->y = y + pic->height;

    R_DrawArray (R_QUADS, 0, 4);
}



/*
=============
SHADER_Draw_Pic

fenix@io.com: vertex arrays, shaders
=============
*/
void SHADER_Draw_Pic (int x, int y, shader_pic_t *pic)
{
    vavertex_t *varray;

    R_Shader(pic->shader_num);

    R_VertexArray (global_vertex_array, R_VERTEX2|R_TEXCOORD2|R_COLOR4);

    varray = global_vertex_array;

    varray->s = pic->sl;
    varray->t = pic->tl;
    varray->r = pic->r;
    varray->g = pic->g;
    varray->b = pic->b;
    varray->a = pic->a;
    varray->x = x;
    varray->y = y;
    varray++;

    varray->s = pic->sh;
    varray->t = pic->tl;
    varray->r = pic->r;
    varray->g = pic->g;
    varray->b = pic->b;
    varray->a = pic->a;
    varray->x = x + pic->width;
    varray->y = y;
    varray++;

    varray->s = pic->sh;
    varray->t = pic->th;
    varray->r = pic->r;
    varray->g = pic->g;
    varray->b = pic->b;
    varray->a = pic->a;
    varray->x = x + pic->width;
    varray->y = y + pic->height;
    varray++;

    varray->s = pic->sl;
    varray->t = pic->th;
    varray->r = pic->r;
    varray->g = pic->g;
    varray->b = pic->b;
    varray->a = pic->a;
    varray->x = x;
    varray->y = y + pic->height;

    SHADER_R_DrawArray (R_QUADS, 0, 4);
}



/*
=============
Draw_AlphaPic

fenix@io.com: vertex arrays
=============
*/
void Draw_AlphaPic (int x, int y, qpic_t *pic, float alpha)
{
	glDisable (GL_ALPHA_TEST);
	glEnable (GL_BLEND);
    glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
    glColor4f (1, 1, 1, alpha);
    _Draw_Pic (x, y, pic);
    glColor4f (1, 1, 1, 1);
	glEnable (GL_ALPHA_TEST);
	glDisable (GL_BLEND);
    glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
}



/*
=============
Draw_Pic

fenix@io.com: vertex arrays
=============
*/
void Draw_Pic (int x, int y, qpic_t *pic)
{
    glColor4f (1, 1, 1, 1);
    _Draw_Pic(x, y, pic);
}



/*
=============
SHADER_Draw_SubDividedPic

fenix@io.com: sub-divided pics, shaders
=============
*/
void SHADER_Draw_SubDividedPic (int ox, int oy, int x_subs, int y_subs, shader_pic_t *pic)
{
    vavertex_t *varray;
    int i, j;
    int count;
    float ds, dt, dx, dy;
    float tl, th, sl, sh, xl, yl, yh;

    count = 2 * (x_subs + 1);

    if (count > global_vertex_array_size) {
       Con_Printf("Draw_SubDividedPic: too many pic subdivisions\n");
       x_subs = 1;
       y_subs = 1;
    }

    if (x_subs < 0) x_subs = 1;

    if (y_subs < 0) y_subs = 1;

    ds = (float)(pic->sh - pic->sl) / (float)x_subs;
    dt = (float)(pic->th - pic->tl) / (float)y_subs;

    dx = (float)pic->width  / (float)x_subs;
    dy = (float)pic->height / (float)y_subs;

    R_Shader(pic->shader_num);

    R_VertexArray (global_vertex_array, R_VERTEX2|R_TEXCOORD2|R_COLOR4);

    for (i = 0; i < y_subs; i++)
    {
       tl = pic->tl + (i * dt);
       th = pic->tl + ((i + 1) * dt);

       yl = oy + (i * dy);
       yh = oy + ((i + 1) * dy);

       varray = global_vertex_array;
       
       for (j = 0; j < x_subs; j++) {
           sl = pic->sl + (j * ds);
           xl = ox + (j * dx);

           varray->s = sl;
           varray->t = th;
           varray->r = pic->r;
           varray->g = pic->g;
           varray->b = pic->b;
           varray->a = pic->a;
           varray->x = xl;
           varray->y = yh;
           varray++;

           varray->s = sl;
           varray->t = tl;
           varray->r = pic->r;
           varray->g = pic->g;
           varray->b = pic->b;
           varray->a = pic->a;
           varray->x = xl;
           varray->y = yl;
           varray++;
       }

       varray->s = pic->sh;
       varray->t = th;
       varray->r = pic->r;
       varray->g = pic->g;
       varray->b = pic->b;
       varray->a = pic->a;
       varray->x = ox + pic->width;
       varray->y = yh;
       varray++;

       varray->s = pic->sh;
       varray->t = tl;
       varray->r = pic->r;
       varray->g = pic->g;
       varray->b = pic->b;
       varray->a = pic->a;
       varray->x = ox + pic->width;
       varray->y = yl;

       SHADER_R_DrawArray (R_QUAD_STRIP, 0, count);
    }
}



/*
=============
Draw_TransPic
=============
*/
void Draw_TransPic (int x, int y, qpic_t *pic)
{
	byte	*dest, *source, tbyte;
	unsigned short	*pusdest;
	int				v, u;

	if (x < 0 || (unsigned)(x + pic->width) > vid.width || y < 0 ||
		 (unsigned)(y + pic->height) > vid.height)
	{
		Sys_Error ("Draw_TransPic: bad coordinates");
	}
		
	Draw_Pic (x, y, pic);
}


/*
=============
Draw_TransPicTranslate

Only used for the player color selection menu
fenix@io.com: vertex arrays
=============
*/
void Draw_TransPicTranslate (int x, int y, qpic_t *pic, byte *translation)
{
	int				v, u, c;
	unsigned		trans[64*64], *dest;
	byte			*src;
	int				p;
    vavertex_t      *varray;

	GL_Bind (translate_texture);

	c = pic->width * pic->height;

	dest = trans;
	for (v=0 ; v<64 ; v++, dest += 64)
	{
		src = &menuplyr_pixels[ ((v*pic->height)>>6) *pic->width];
		for (u=0 ; u<64 ; u++)
		{
			p = src[(u*pic->width)>>6];
			if (p == 255)
				dest[u] = p;
			else
				dest[u] =  d_8to24table[translation[p]];
		}
	}

	glTexImage2D (GL_TEXTURE_2D, 0, gl_alpha_format, 64, 64, 0, GL_RGBA, GL_UNSIGNED_BYTE, trans);

	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

	glColor3f (1,1,1);

    R_VertexArray(global_vertex_array, R_VERTEX2|R_TEXCOORD2);

    varray = global_vertex_array;

    varray->s = 0;
    varray->t = 0;
    varray->x = x;
    varray->y = y;
    varray++;

    varray->s = 1;
    varray->t = 0;
    varray->x = x + pic->width;
    varray->y = y;
    varray++;

    varray->s = 1;
    varray->t = 1;
    varray->x = x + pic->width;
    varray->y = y + pic->height;
    varray++;

    varray->s = 0;
    varray->t = 1;
    varray->x = x;
    varray->y = y + pic->height;

    R_DrawArray(R_QUADS, 0, 4);
}



// fenix@io.com: console line, shaders
// draw line and version string on bottom of console
void Draw_ConsoleLine(int lines)
{
    int  len;
    int  i;
    int  x, y;
    unsigned char little_color[4];
    unsigned char color[4];
    vavertex_t* varray;

    *((unsigned long*)little_color) =
       LittleLong(strtoul(r_console_line_color.string, NULL, 16));

    color[0] = little_color[3];
    color[1] = little_color[2];
    color[2] = little_color[1];
    color[3] = little_color[0];

    R_Shader(console_line);

    R_VertexArray(global_vertex_array, R_VERTEX2|R_COLOR4);

    varray = global_vertex_array;

    varray->r = color[0];
    varray->g = color[1];
    varray->b = color[2];
    varray->a = color[3];
    varray->x = 0;
    varray->y = lines;
    varray++;

    varray->r = color[0];
    varray->g = color[1];
    varray->b = color[2];
    varray->a = color[3];
    varray->x = vid.width;
    varray->y = lines;
    varray++;

    varray->r = color[0];
    varray->g = color[1];
    varray->b = color[2];
    varray->a = color[3];
    varray->x = vid.width;
    varray->y = lines+2;
    varray++;

    varray->r = color[0];
    varray->g = color[1];
    varray->b = color[2];
    varray->a = color[3];
    varray->x = 0;
    varray->y = lines+2;

    SHADER_R_DrawArray(R_QUADS, 0, 4);

    len = strlen(r_console_line_string.string);
    x = vid.width - (8 * len) - 1;
    y = lines - 9;

    for (i = 0; i < len; i++) {
        Draw_Character(x, y, r_console_line_string.string[i], color);
        x += 8;
    }
}



/*
================
Draw_ConsoleBackground

fenix@io.com: sub-divided console, variable console alpha, shaders, variable console height, console line, stretchy console
================
*/
void Draw_ConsoleBackground (int lines)
{
    int y;

    if (r_console_subdivide_x.value < 1) 
    {
        Cvar_SetValue("r_console_subdivide_x", 1);
    }

    if (r_console_subdivide_y.value < 1)
    {
        Cvar_SetValue("r_console_subdivide_y", 1);
    }

    if (scr_console_stretch.value) {
       y = 0;
       console.height = lines;
    }
    else {
       y = lines - vid.height;
       console.height = vid.height;
    }

	if (con_forcedup)
    {
       console.a = 255;
    }
	else
    {
       float alpha;
       
       alpha = (float)(r_console_alpha.value *
          (lines / (vid.height * scr_console_height.value)));

       alpha *= 255;

       console.a = CLAMP(0, alpha, 255);
    }

    SHADER_Draw_SubDividedPic(
       0,
       y,
       r_console_subdivide_x.value,
       r_console_subdivide_y.value,
       &console);

    Draw_ConsoleLine(lines);
}


/*
=============
Draw_TileClear

This repeats a 64*64 tile graphic to fill the screen around a sized down
refresh window.

fenix@io.com: vertex arrays, shaders
=============
*/
void Draw_TileClear (int x, int y, int w, int h)
{
    vavertex_t* varray;

	R_Shader(draw_backtile.shader_num);

    R_VertexArray(global_vertex_array, R_TEXCOORD2|R_VERTEX2);

    varray = global_vertex_array;

    varray->s = x / 64.0;
    varray->t = y / 64.0;
    varray->x = x;
    varray->y = y;
    varray++;

    varray->s = (x + w) / 64.0;
    varray->t = y / 64.0;
    varray->x = x + w;
    varray->y = y;
    varray++;

    varray->s = (x + w) / 64.0;
    varray->t = (y + h) / 64.0;
    varray->x = x + w;
    varray->y = y + h;
    varray++;

    varray->s = x / 64.0;
    varray->t = (y + h) / 64.0;
    varray->x = x;
    varray->y = y + h;

    SHADER_R_DrawArray(R_QUADS, 0, 4);
}



/*
=============
Draw_Fill

Fills a box of pixels with a single color

// fenix@io.com: vertex arrays
// fenix@io.com: TODO - Switch this to use glClear, glClearColor, and glScissor
=============
*/
void Draw_Fill (int x, int y, int w, int h, int c)
{
    vavertex_t* varray;

	glDisable (GL_TEXTURE_2D);
	glColor3f (host_basepal[c*3]/255.0,
		host_basepal[c*3+1]/255.0,
		host_basepal[c*3+2]/255.0);

    R_VertexArray (global_vertex_array, R_VERTEX2);

    varray = global_vertex_array;

    varray->x = x;
    varray->y = y;
    varray++;

    varray->x = x + w;
    varray->y = y;
    varray++;

    varray->x = x + w;
    varray->y = y + h;
    varray++;

    varray->x = x;
    varray->y = y + h;

    R_DrawArray(R_QUADS, 0, 4);

	glColor3f (1,1,1);
	glEnable (GL_TEXTURE_2D);
}


//=============================================================================


/*
================
Draw_FadeScreen

fenix@io.com: vertex arrays
================
*/
void Draw_FadeScreen (void)
{
    vavertex_t* varray;

	glEnable (GL_BLEND);
	glDisable (GL_TEXTURE_2D);
	glColor4f (0, 0, 0, 0.8f);

    R_VertexArray (global_vertex_array, R_VERTEX2);

    varray = global_vertex_array;

    varray->x = 0;
    varray->y = 0;
    varray++;

    varray->x = vid.width;
    varray->y = 0;
    varray++;

    varray->x = vid.width;
    varray->y = vid.height;
    varray++;

    varray->x = 0;
    varray->y = vid.height;

    R_DrawArray(R_QUADS, 0, 4);

	glColor4f (1,1,1,1);
	glEnable (GL_TEXTURE_2D);
	glDisable (GL_BLEND);

	Sbar_Changed();
}


//=============================================================================

/*
================
Draw_BeginDisc

Draws the little blue disc in the corner of the screen.
Call before beginning any disc IO.

fenix@io.com: shaders
================
*/
void Draw_BeginDisc (void)
{
    if (disc_initialized) {
        glDrawBuffer(GL_FRONT);
        SHADER_Draw_Pic(vid.width - 24, 0, &draw_disc);
        glDrawBuffer(GL_BACK);
    }
}


/*
================
Draw_EndDisc

Erases the disc icon.
Call after completing any disc IO
================
*/
void Draw_EndDisc (void)
{
}

/*
================
GL_Set2D

Setup as if the screen was 320*200
================
*/
void GL_Set2D (void)
{
	glViewport (glx, gly, glwidth, glheight);

	glMatrixMode(GL_PROJECTION);
    glLoadIdentity ();
	glOrtho  (0, vid.width, vid.height, 0, -99999, 99999);

	glMatrixMode(GL_MODELVIEW);
    glLoadIdentity ();

	glDisable (GL_DEPTH_TEST);
	glDisable (GL_CULL_FACE);
	glDisable (GL_BLEND);
	glEnable (GL_ALPHA_TEST);
//	glDisable (GL_ALPHA_TEST);

	glColor4f (1,1,1,1);
}

//====================================================================

/*
================
GL_FindTexture
================
*/
int GL_FindTexture (char *identifier)
{
	int		i;
	gltexture_t	*glt;

	for (i=0, glt=gltextures ; i<numgltextures ; i++, glt++)
	{
		if (!strcmp (identifier, glt->identifier))
			return gltextures[i].texnum;
	}

	return -1;
}

/*
================
GL_ResampleTexture
================
*/
void GL_ResampleTexture (unsigned *in, int inwidth, int inheight, unsigned *out,  int outwidth, int outheight)
{
	int		i, j;
	unsigned	*inrow;
	unsigned	frac, fracstep;

	fracstep = inwidth*0x10000/outwidth;
	for (i=0 ; i<outheight ; i++, out += outwidth)
	{
		inrow = in + inwidth*(i*inheight/outheight);
		frac = fracstep >> 1;
		for (j=0 ; j<outwidth ; j+=4)
		{
			out[j] = inrow[frac>>16];
			frac += fracstep;
			out[j+1] = inrow[frac>>16];
			frac += fracstep;
			out[j+2] = inrow[frac>>16];
			frac += fracstep;
			out[j+3] = inrow[frac>>16];
			frac += fracstep;
		}
	}
}

/*
================
GL_Resample8BitTexture -- JACK
================
*/
void GL_Resample8BitTexture (unsigned char *in, int inwidth, int inheight, unsigned char *out,  int outwidth, int outheight)
{
	int		i, j;
	unsigned	char *inrow;
	unsigned	frac, fracstep;

	fracstep = inwidth*0x10000/outwidth;
	for (i=0 ; i<outheight ; i++, out += outwidth)
	{
		inrow = in + inwidth*(i*inheight/outheight);
		frac = fracstep >> 1;
		for (j=0 ; j<outwidth ; j+=4)
		{
			out[j] = inrow[frac>>16];
			frac += fracstep;
			out[j+1] = inrow[frac>>16];
			frac += fracstep;
			out[j+2] = inrow[frac>>16];
			frac += fracstep;
			out[j+3] = inrow[frac>>16];
			frac += fracstep;
		}
	}
}


/*
================
GL_MipMap

Operates in place, quartering the size of the texture
================
*/
void GL_MipMap (byte *in, int width, int height)
{
	int		i, j;
	byte	*out;

	width <<=2;
	height >>= 1;
	out = in;
	for (i=0 ; i<height ; i++, in+=width)
	{
		for (j=0 ; j<width ; j+=8, out+=4, in+=8)
		{
			out[0] = (in[0] + in[4] + in[width+0] + in[width+4])>>2;
			out[1] = (in[1] + in[5] + in[width+1] + in[width+5])>>2;
			out[2] = (in[2] + in[6] + in[width+2] + in[width+6])>>2;
			out[3] = (in[3] + in[7] + in[width+3] + in[width+7])>>2;
		}
	}
}

/*
================
GL_MipMap8Bit

Mipping for 8 bit textures
================
*/
void GL_MipMap8Bit (byte *in, int width, int height)
{
	int		i, j;
	unsigned short     r,g,b;
	byte	*out, *at1, *at2, *at3, *at4;

//	width <<=2;
	height >>= 1;
	out = in;
	for (i=0 ; i<height ; i++, in+=width)
	{
		for (j=0 ; j<width ; j+=2, out+=1, in+=2)
		{
			at1 = (byte *) (d_8to24table + in[0]);
			at2 = (byte *) (d_8to24table + in[1]);
			at3 = (byte *) (d_8to24table + in[width+0]);
			at4 = (byte *) (d_8to24table + in[width+1]);

 			r = (at1[0]+at2[0]+at3[0]+at4[0]); r>>=5;
 			g = (at1[1]+at2[1]+at3[1]+at4[1]); g>>=5;
 			b = (at1[2]+at2[2]+at3[2]+at4[2]); b>>=5;

			out[0] = d_15to8table[(r<<0) + (g<<5) + (b<<10)];
		}
	}
}

/*
===============
GL_Upload32
fenix@io.com: shaders
===============
*/
void GL_Upload32
   (
   unsigned *data,
   int width,
   int height,
   qboolean mipmap,
   qboolean picmip,
   qboolean alpha,
   qboolean clampmap
   )
{
	int			samples;
    static	unsigned	scaled[1024*512];	// [512*256];
	int			scaled_width, scaled_height;

	for (scaled_width = 1 ; scaled_width < width ; scaled_width<<=1)
		;
	for (scaled_height = 1 ; scaled_height < height ; scaled_height<<=1)
		;

    if (picmip) 
    {
    	scaled_width >>= (int)gl_picmip.value;
	    scaled_height >>= (int)gl_picmip.value;
    }

	if (scaled_width > gl_max_size.value)
		scaled_width = gl_max_size.value;
	if (scaled_height > gl_max_size.value)
		scaled_height = gl_max_size.value;

	if (scaled_width * scaled_height > sizeof(scaled)/4)
		Sys_Error ("GL_LoadTexture: too big");

	samples = alpha ? gl_alpha_format : gl_solid_format;

#if 0
	if (mipmap)
		gluBuild2DMipmaps (GL_TEXTURE_2D, samples, width, height, GL_RGBA, GL_UNSIGNED_BYTE, trans);
	else if (scaled_width == width && scaled_height == height)
		glTexImage2D (GL_TEXTURE_2D, 0, samples, width, height, 0, GL_RGBA, GL_UNSIGNED_BYTE, trans);
	else
	{
		gluScaleImage (GL_RGBA, width, height, GL_UNSIGNED_BYTE, trans,
			scaled_width, scaled_height, GL_UNSIGNED_BYTE, scaled);
		glTexImage2D (GL_TEXTURE_2D, 0, samples, scaled_width, scaled_height, 0, GL_RGBA, GL_UNSIGNED_BYTE, scaled);
	}
#else
texels += scaled_width * scaled_height;

	if (scaled_width == width && scaled_height == height)
	{
		if (!mipmap)
		{
			glTexImage2D (GL_TEXTURE_2D, 0, samples, scaled_width, scaled_height, 0, GL_RGBA, GL_UNSIGNED_BYTE, data);
			goto done;
		}
		memcpy (scaled, data, width*height*4);
	}
	else
		GL_ResampleTexture (data, width, height, scaled, scaled_width, scaled_height);

	glTexImage2D (GL_TEXTURE_2D, 0, samples, scaled_width, scaled_height, 0, GL_RGBA, GL_UNSIGNED_BYTE, scaled);
	if (mipmap)
	{
		int		miplevel;

		miplevel = 0;
		while (scaled_width > 1 || scaled_height > 1)
		{
			GL_MipMap ((byte *)scaled, scaled_width, scaled_height);
			scaled_width >>= 1;
			scaled_height >>= 1;
			if (scaled_width < 1)
				scaled_width = 1;
			if (scaled_height < 1)
				scaled_height = 1;
			miplevel++;
			glTexImage2D (GL_TEXTURE_2D, miplevel, samples, scaled_width, scaled_height, 0, GL_RGBA, GL_UNSIGNED_BYTE, scaled);
		}
	}
done: ;
#endif


	if (mipmap)
	{
		glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, gl_filter_min);
		glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, gl_filter_max);
	}
	else
	{
		glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, gl_filter_max);
		glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, gl_filter_max);
	}

    if (clampmap)
    {
        glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
        glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
    }
}

// fenix@io.com: texture maps
void GL_Upload8_EXT (byte *data, int width, int height, qboolean mipmap, qboolean picmip, qboolean alpha, qboolean clampmap) 
{
	int			i, s;
	qboolean	noalpha;
	int			p;
	static unsigned j;
	int			samples;
    static	unsigned char scaled[1024*512];	// [512*256];
	int			scaled_width, scaled_height;

	s = width*height;
	// if there are no transparent pixels, make it a 3 component
	// texture even if it was specified as otherwise
	if (alpha)
	{
		noalpha = true;
		for (i=0 ; i<s ; i++)
		{
			if (data[i] == 255)
				noalpha = false;
		}

		if (alpha && noalpha)
			alpha = false;
	}
	for (scaled_width = 1 ; scaled_width < width ; scaled_width<<=1)
		;
	for (scaled_height = 1 ; scaled_height < height ; scaled_height<<=1)
		;

    if (picmip) {
      	scaled_width >>= (int)gl_picmip.value;
	    scaled_height >>= (int)gl_picmip.value;
    }

	if (scaled_width > gl_max_size.value)
		scaled_width = gl_max_size.value;
	if (scaled_height > gl_max_size.value)
		scaled_height = gl_max_size.value;

	if (scaled_width * scaled_height > sizeof(scaled))
		Sys_Error ("GL_LoadTexture: too big");

	samples = 1; // alpha ? gl_alpha_format : gl_solid_format;

	texels += scaled_width * scaled_height;

	if (scaled_width == width && scaled_height == height)
	{
		if (!mipmap)
		{
			glTexImage2D (GL_TEXTURE_2D, 0, GL_COLOR_INDEX8_EXT, scaled_width, scaled_height, 0, GL_COLOR_INDEX , GL_UNSIGNED_BYTE, data);
			goto done;
		}
		memcpy (scaled, data, width*height);
	}
	else
		GL_Resample8BitTexture (data, width, height, scaled, scaled_width, scaled_height);

	glTexImage2D (GL_TEXTURE_2D, 0, GL_COLOR_INDEX8_EXT, scaled_width, scaled_height, 0, GL_COLOR_INDEX, GL_UNSIGNED_BYTE, scaled);
	if (mipmap)
	{
		int		miplevel;

		miplevel = 0;
		while (scaled_width > 1 || scaled_height > 1)
		{
			GL_MipMap8Bit ((byte *)scaled, scaled_width, scaled_height);
			scaled_width >>= 1;
			scaled_height >>= 1;
			if (scaled_width < 1)
				scaled_width = 1;
			if (scaled_height < 1)
				scaled_height = 1;
			miplevel++;
			glTexImage2D (GL_TEXTURE_2D, miplevel, GL_COLOR_INDEX8_EXT, scaled_width, scaled_height, 0, GL_COLOR_INDEX, GL_UNSIGNED_BYTE, scaled);
		}
	}
done: ;


	if (mipmap)
	{
		glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, gl_filter_min);
		glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, gl_filter_max);
	}
	else
	{
		glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, gl_filter_max);
		glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, gl_filter_max);
	}

    if (clampmap)
    {
        glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
        glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
    }
}

/*
===============
GL_Upload8
fenix@io.com: texture maps
===============
*/
void GL_Upload8
   (
   byte*    data,
   int      width,
   int      height,
   qboolean mipmap,
   qboolean picmip,
   qboolean alpha,
   qboolean clampmap
   )
{
static	unsigned	trans[640*480];		// FIXME, temporary
	int			i, s;
	qboolean	noalpha;
	int			p;

	s = width*height;
	// if there are no transparent pixels, make it a 3 component
	// texture even if it was specified as otherwise
	if (alpha)
	{
		noalpha = true;
		for (i=0 ; i<s ; i++)
		{
			p = data[i];
			if (p == 255)
				noalpha = false;
			trans[i] = d_8to24table[p];
		}

		if (alpha && noalpha)
			alpha = false;
	}
	else
	{
		if (s&3)
			Sys_Error ("GL_Upload8: s&3");
		for (i=0 ; i<s ; i+=4)
		{
			trans[i] = d_8to24table[data[i]];
			trans[i+1] = d_8to24table[data[i+1]];
			trans[i+2] = d_8to24table[data[i+2]];
			trans[i+3] = d_8to24table[data[i+3]];
		}
	}

 	if (VID_Is8bit() && !alpha && (data!=scrap_texels[0])) {
 		GL_Upload8_EXT (data, width, height, mipmap, picmip, alpha, clampmap);
 		return;
	}

	GL_Upload32 (trans, width, height, mipmap, picmip, alpha, clampmap);
}

/*
================
GL_LoadTexture
fenix@io.com: texture maps
================
*/
int GL_LoadTexture
   (
   char*    identifier,
   int      width,
   int      height,
   byte*    data,
   qboolean mipmap,
   qboolean picmip,
   qboolean alpha,
   qboolean clampmap
   )
{
	qboolean	noalpha;
	int			i, p, s;
	gltexture_t	*glt;

	// see if the texture is allready present
	if (identifier[0])
	{
		for (i=0, glt=gltextures ; i<numgltextures ; i++, glt++)
		{
			if (!strcmp (identifier, glt->identifier))
			{
				if (width != glt->width || height != glt->height)
					Sys_Error ("GL_LoadTexture: cache mismatch");
				return gltextures[i].texnum;
			}
		}
	}
	else {
		glt = &gltextures[numgltextures];
		numgltextures++;
	}

	strcpy (glt->identifier, identifier);
	glt->texnum = texture_extension_number;
	glt->width = width;
	glt->height = height;
	glt->mipmap = mipmap;

	GL_Bind(texture_extension_number );

	GL_Upload8 (data, width, height, mipmap, picmip, alpha, clampmap);

	texture_extension_number++;

	return texture_extension_number-1;
}



/*
================
GL_LoadTexture32
fenix@io.com: texture maps
================
*/
int GL_LoadTexture32
   (
   char* identifier,
   int   width,
   int   height,
   int*  data,
   qboolean mipmap,
   qboolean picmip,
   qboolean alpha,
   qboolean clampmap
   )
{
	qboolean	noalpha;
	int			i, p, s;
	gltexture_t	*glt;

	// see if the texture is allready present
	if (identifier[0])
	{
		for (i=0, glt=gltextures ; i<numgltextures ; i++, glt++)
		{
			if (!strcmp (identifier, glt->identifier))
			{
				if (width != glt->width || height != glt->height)
					Sys_Error ("GL_LoadTexture: cache mismatch");
				return gltextures[i].texnum;
			}
		}
	}
	else {
		glt = &gltextures[numgltextures];
		numgltextures++;
	}

	strcpy (glt->identifier, identifier);
	glt->texnum = texture_extension_number;
	glt->width = width;
	glt->height = height;
	glt->mipmap = mipmap;

	GL_Bind(texture_extension_number );

	GL_Upload32 (data, width, height, mipmap, picmip, alpha, clampmap);

	texture_extension_number++;

	return texture_extension_number-1;
}



/*
================
GL_LoadPicTexture
================
*/
int GL_LoadPicTexture (qpic_t *pic)
{
    // fenix@io.com: texture maps
	return GL_LoadTexture ("", pic->width, pic->height, pic->data, false, true, true, false);
    // --
}

/****************************************/

static GLenum oldtarget = TEXTURE0_SGIS;

void GL_SelectTexture (GLenum target) 
{
    // fenix@io.com: disable multitexture
    #if 0
	if (!gl_mtexable)
		return;
	qglSelectTextureSGIS(target);
	if (target == oldtarget) 
		return;
	cnttextures[oldtarget-TEXTURE0_SGIS] = currenttexture;
	currenttexture = cnttextures[target-TEXTURE0_SGIS];
	oldtarget = target;
    #endif
}
