// fenix@io.com: vertex arrays

/*
Copyright (C) 1996-1997 Id Software, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

#include "quakedef.h"

#include "varrays.h"
#include "shaders.h"

#define R_MIN_VERTEX_COUNT 128



cvar_t  r_vertex_arrays = { "r_vertex_arrays", "0" };

qboolean is_vertex_arrays_supported = false;

vavertex_t* global_vertex_array;
unsigned*   global_indice_array;

int global_vertex_array_size;
int global_indice_array_size;

int mode_table[] = {
    /* R_POINTS         */ GL_POINTS,
    /* R_LINES          */ GL_LINES,
    /* R_LINE_STRIP     */ GL_LINE_STRIP,
    /* R_LINE_LOOP      */ GL_LINE_LOOP,
    /* R_TRIANGLES      */ GL_TRIANGLES,
    /* R_TRIANGLE_STRIP */ GL_TRIANGLE_STRIP,
    /* R_TRIANGLE_FAN   */ GL_TRIANGLE_FAN,
    /* R_QUADS          */ GL_QUADS,
    /* R_QUAD_STRIP     */ GL_QUAD_STRIP,
    /* R_POLYGON        */ GL_POLYGON
};


int blendfunc_table[] = {
   /* R_ONE                 */ GL_ONE,
   /* R_ZERO                */ GL_ZERO,
   /* R_DST_COLOR           */ GL_DST_COLOR,
   /* R_ONE_MINUS_DST_COLOR */ GL_ONE_MINUS_DST_COLOR,
   /* R_SRC_ALPHA           */ GL_SRC_ALPHA,
   /* R_ONE_MINUS_SRC_ALPHA */ GL_ONE_MINUS_SRC_ALPHA,
   /* R_ONE_SRC_COLOR       */ GL_SRC_COLOR,
   /* R_ONE_MINUS_SRC_COLOR */ GL_ONE_MINUS_SRC_COLOR,
};


qboolean color3_enabled;
qboolean color4_enabled;
qboolean texcoord2_enabled;
qboolean texcoord2x2_enabled;
qboolean vertex2_enabled;
qboolean vertex3_enabled;
vavertex_t* current_vertex_array;



/* R_InitVertexArrays
 ****************************************************************************/
void R_InitVertexArrays(void)
{
    int pnum = COM_CheckParm("-varraysize");

    if (pnum != 0)
    {
        global_vertex_array_size = Q_atoi(com_argv[pnum+1]);

        if (global_vertex_array_size < R_MIN_VERTEX_COUNT)
        {
            global_vertex_array_size = R_MIN_VERTEX_COUNT;
        }
    }
    else
    {
        global_vertex_array_size = R_MIN_VERTEX_COUNT;
    }

    global_indice_array_size = global_vertex_array_size;

    global_vertex_array = Hunk_AllocName(sizeof(vavertex_t) * global_vertex_array_size, "vertex_array");
    global_indice_array = Hunk_AllocName(sizeof(vavertex_t) * global_indice_array_size, "indice_array");
}



/* R_VertexArray
 ****************************************************************************/
void R_VertexArray(vavertex_t* va, unsigned fields)
{
   current_vertex_array = va;

   if (fields & R_VERTEX3) {
      glEnableClientState(GL_VERTEX_ARRAY);
      glVertexPointer(3, GL_FLOAT, sizeof(vavertex_t), &va->x);
      vertex2_enabled = false;
      vertex3_enabled = true;
   }
   else if (fields & R_VERTEX2) {
      glEnableClientState(GL_VERTEX_ARRAY);
      glVertexPointer(2, GL_FLOAT, sizeof(vavertex_t), &va->x);
      vertex2_enabled = true;
      vertex3_enabled = false;
   }
   else {
      glDisableClientState(GL_VERTEX_ARRAY);
      vertex2_enabled = false;
      vertex3_enabled = false;
   }

   if (fields & R_COLOR4) {
      glEnableClientState(GL_COLOR_ARRAY);
      glColorPointer(4, GL_UNSIGNED_BYTE, sizeof(vavertex_t), &va->r);
      color3_enabled = false;
      color4_enabled = true;
   }
   else if (fields & R_COLOR3) {
      glEnableClientState(GL_COLOR_ARRAY);
      glColorPointer(3, GL_UNSIGNED_BYTE, sizeof(vavertex_t), &va->r);
      color3_enabled = true;
      color4_enabled = false;
   }
   else {
      glDisableClientState(GL_COLOR_ARRAY);
      color3_enabled = false;
      color4_enabled = false;
   }

   if (fields & R_TEXCOORD2x2) {
      glDisableClientState(GL_TEXTURE_COORD_ARRAY);
      texcoord2_enabled   = false;
      texcoord2x2_enabled = true;
   }
   else if (fields & R_TEXCOORD2) {
      glEnableClientState(GL_TEXTURE_COORD_ARRAY);
      glTexCoordPointer(2, GL_FLOAT, sizeof(vavertex_t), &va->s);
      texcoord2_enabled   = true;
      texcoord2x2_enabled = false;
   }
   else {
      glDisableClientState(GL_TEXTURE_COORD_ARRAY);
      texcoord2_enabled   = false;
      texcoord2x2_enabled = false;
   }
}



/* SHADER_R_DrawArray
 ****************************************************************************/
void SHADER_R_DrawArray(R_MODE mode, int first, size_t count)
{
   int       s, t;
   shader_t* shader;

   shader = &shaders[current_shader];

   for (s = 0; s < shader->num_stages; s++) {
      shader_stage_t* stage;

      stage = &shader->stages[s];

      GL_Bind(texture_maps[stage->map].index);

      if (stage->num_tcmods > 0) {
         glMatrixMode(GL_TEXTURE);
         glPushMatrix();
         glLoadIdentity();

         // workaround: something bizzare is happening,
         // unless stage has a scroll, subsequent stages
         // do not scroll
         glTranslatef(0, 0, 0);

         for (t = 0; t < stage->num_tcmods; t++) {
            stage->tcmods[t].func(&stage->tcmods[t], current_vertex_array, first, count);
         }
      }

      if (stage->flags & R_HAS_BLENDFUNC) {
         glEnable(GL_BLEND);
         glBlendFunc(
            blendfunc_table[stage->blend_src],
            blendfunc_table[stage->blend_dst]);
      }

      R_DrawArray(mode, first, count);

      if (stage->num_tcmods > 0) {
         glPopMatrix();
         glMatrixMode(GL_MODELVIEW);

         R_RestoreST(current_vertex_array, first, count);
      }

      if (stage->flags & R_HAS_BLENDFUNC) {
         glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
         glDisable(GL_BLEND);
      }
   }

}



/* _R_DrawVertex
 ****************************************************************************/
__inline void _R_DrawVertex(vavertex_t* varray)
{
   if (color4_enabled) {
      glColor4ubv(&varray->r);
   }
   else if (color3_enabled) {
      glColor3ubv(&varray->r);
   }

   if (texcoord2x2_enabled) {
      qglMTexCoord2fSGIS (TEXTURE0_SGIS, varray->s,  varray->t);
      qglMTexCoord2fSGIS (TEXTURE1_SGIS, varray->s1, varray->t1);
   } else if (texcoord2_enabled) {
      glTexCoord2fv(&varray->s);
   }
  
   if (vertex3_enabled) {
      glVertex3fv(&varray->x);
   }
   else if (vertex2_enabled) {
      glVertex2fv(&varray->x);
   }
}



/* R_DrawArray
 ****************************************************************************/
void R_DrawArray(R_MODE mode, int first, size_t count)
{
   if (is_vertex_arrays_supported && r_vertex_arrays.value && !texcoord2x2_enabled) {
      glDrawArrays(mode_table[mode], first, count);
   }
   else {
      vavertex_t *varray;
      int i;

      varray = current_vertex_array;

      glBegin(mode_table[mode]);

      for (i = 0; i < count; i++) {
         _R_DrawVertex(varray);
         varray++;
      }

      glEnd();
   }
}



/* R_DrawElements
 ****************************************************************************/
void R_DrawElements(R_MODE mode, size_t count, unsigned* indices)
{
   if (is_vertex_arrays_supported && r_vertex_arrays.value && !texcoord2x2_enabled) {
      glDrawElements(mode_table[mode], count, GL_UNSIGNED_INT, indices);
   }
   else {
      vavertex_t *varray;
      int i;

      glBegin(mode_table[mode]);

      for (i = 0; i < count; i++) {
         varray = current_vertex_array + indices[i];
         _R_DrawVertex(varray);
      }

      glEnd();
   }
}



/* R_DrawRangeElements
 ****************************************************************************/
void R_DrawRangeElements(R_MODE mode, unsigned start, unsigned end, size_t count, unsigned* indices)
{
   if (is_vertex_arrays_supported && r_vertex_arrays.value && !texcoord2x2_enabled) {
      //glDrawRangeElements(mode_table[mode], start, end, count, GL_UNSIGNED_INT, indices);
   }
   else {
      R_DrawElements(mode, count, indices);
   }
}





