// fenix@io.com: shaders

/*
Copyright (C) 1996-1997 Id Software, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

#include "quakedef.h"

#include "shaders.h"

#define R_MIN_SHADER_COUNT    64
#define R_MIN_MAP_COUNT       64



int       shader_array_size;
shader_t* shaders;
int       num_shaders;

int            texture_map_array_size;
texture_map_t* texture_maps;
int            num_texture_maps;

int current_shader;

static qboolean nomipmaps;
static qboolean nopicmip;

float* saved_st;
qboolean restore_st;



void R_InitShaders(void)
{
    int pnum = COM_CheckParm("-numshaders");

    if (pnum != 0)
    {
        shader_array_size = Q_atoi(com_argv[pnum+1]);

        if (shader_array_size < R_MIN_SHADER_COUNT)
        {
            shader_array_size = R_MIN_SHADER_COUNT;
        }
    }
    else
    {
        shader_array_size = R_MIN_SHADER_COUNT;
    }

    shaders = Hunk_AllocName(sizeof(shader_t) * shader_array_size, "shaders");

    pnum = COM_CheckParm("-nummaps");

    if (pnum != 0)
    {
        texture_map_array_size = Q_atoi(com_argv[pnum+1]);

        if (texture_map_array_size < R_MIN_MAP_COUNT)
        {
            texture_map_array_size = R_MIN_MAP_COUNT;
        }
    }
    else
    {
        texture_map_array_size = R_MIN_MAP_COUNT;
    }

    texture_maps = Hunk_AllocName(sizeof(texture_map_t) * texture_map_array_size, "texture_maps");

    saved_st = Hunk_AllocName(2 * sizeof(float) * global_vertex_array_size, "saved_st");
}



void R_Shader(int shader_num)
{
   current_shader = shader_num;
}



void R_ShaderName(const char* name)
{
   R_Shader(R_GetShader(name));
}



int R_RequestShader(const char* name)
{
    shader_t* shader;

    if (num_shaders == shader_array_size) return R_OUT_OF_MEMORY;

    shader = &shaders[num_shaders];

    strncpy(shader->name, name, R_SHADER_NAME_SIZE);
    shader->name[R_SHADER_NAME_SIZE] = '\0';

    shader->num_stages = 0;

    shader->is_loaded = false;

    num_shaders++;

    return num_shaders - 1;
}



void R_LoadShaders(const char* name)
{
   int   mark;
   char* text;

   mark = Hunk_LowMark();

   text = (char *)COM_LoadTempFile((char*)name);

   if (text) {
      Con_Printf("%s\n", name);
      R_ParseShaders(text, com_filesize);
   }
   else {
      Con_Printf("error: shader file %s not found.\n", name);
   }

   Hunk_FreeToLowMark(mark);
}


static char* curpos;
static char* endpos;

// Aftershock 0.1
static char*
nexttok(void)
{
    char *tok;
    
    while (curpos < endpos)
    {
	/* Skip leading whitespace */
	while (*curpos == ' ' || *curpos == '\t' || *curpos == '\n' ||
	      *curpos == '\r')
	    if (++curpos == endpos) return NULL;

	/* Check for comment */
	if (curpos[0] == '/' && curpos[1] == '/')
	{
	    /* Skip to end of comment line */
	    while (*curpos++ != '\n')
		if (curpos == endpos) return NULL;
	    /* Restart with leading whitespace */
	    continue;
	}

	/* Seek to end of token */
	tok = curpos;
	while (*curpos != ' ' && *curpos != '\t' && *curpos != '\n' &&
	      *curpos != '\r')
	    if (++curpos == endpos) break;

	/* Zero whitespace character and advance by one */
	*curpos++ = '\0';
	return tok;
    }
    return NULL;
}


// Aftershock 0.1
static char *
nextarg(void)
{
    char *arg;

    while (curpos < endpos)
    {
	/* Skip leading whitespace */
	while (*curpos == ' ' || *curpos == '\t')
	    if (++curpos == endpos) return NULL;

	/* Check for newline or comment */
	if (*curpos == '\n' || *curpos == '\r' ||
	    (curpos[0] == '/' && curpos[1] == '/'))
	    return NULL;
	
	/* Seek to end of token */
	arg = curpos;
	while (*curpos != ' ' && *curpos != '\t' && *curpos != '\n' &&
	      *curpos != '\r')
	    if (++curpos == endpos) break;

	/* Zero whitespace character and advance by one */
	*curpos++ = '\0';
	return arg;
    }
    return NULL;
}


// Aftershock 0.1
// fenix@io.com: modified
static qboolean
shader_skip(void)
{
    char *tok;
    int brace_count;

    /* Opening brace */
    tok = nexttok();

    if (!tok) {
       Con_Printf("error: unexpected end of file\n.");
       return false;
    }

    if (tok[0] != '{') {
       Con_Printf("error: expected '{', found '%s'\n", tok);
       return false;
    }

    for (brace_count = 1; brace_count > 0 && curpos < endpos; curpos++)
    {
	if (*curpos == '{')
	    brace_count++;
	else if (*curpos == '}')
	    brace_count--;
    }

    if (curpos == endpos) {
       Con_Printf("error: unexpected end of input\n");
       return false;
    }

    return true;
}



static qboolean mapParseFunc(shader_t* shader, shader_stage_t* stage, int argc, char* argv[])
{
   stage->map = R_LoadMap(argv[0], !nomipmaps, false);

   if (stage->map < 0) {
      if (stage->map == R_NOT_FOUND) {
         Con_Printf("warning: map: image file \"%s\" not found\n", argv[0]);
         return true;
      }
      else if (stage->map == R_OUT_OF_MEMORY) {
         Con_Printf("warning: map: out of space, please increase -nummaps\n");
         return true;
      }
      else {
         Con_Printf("warning: map: error loading \"%s\"\n", argv[0]);
         return true;
      }
   }
   else {
      return true;
   }
}



static R_WAVEFORM WaveForm(const char* str)
{
   if (stricmp(str, "sin") == 0) {
      return R_SIN;
   }
   else if (stricmp(str, "triangle") == 0) {
      return R_TRIANGLE;
   }
   else if (stricmp(str, "square") == 0) {
      return R_SQUARE;
   }
   else if (stricmp(str, "sawtooth") == 0) {
      return R_SAWTOOTH;
   }
   else if (stricmp(str, "inversesawtooth") == 0) {
      return R_INVSAWTOOTH;
   }
   else {
      return -1;
   }
}



static void tcModRotate(struct tcmod_s* tcmod, vavertex_t* va, int first, int count)
{
   (void)va;
   (void)first;
   (void)count;

   glTranslatef( 0.5,  0.5, 0);
   glRotatef(tcmod->args.rotate * realtime, 0, 0, 1);
   glTranslatef(-0.5, -0.5, 0);
}



static void tcModScale(struct tcmod_s* tcmod, vavertex_t* va, int first, int count)
{
   (void)va;
   (void)first;
   (void)count;

   glScalef(1 / tcmod->args.scale.s, 1 / tcmod->args.scale.t, 1);
}



static void tcModScroll(struct tcmod_s* tcmod, vavertex_t* va, int first, int count)
{
   (void)va;
   (void)first;
   (void)count;

   glTranslatef(
      realtime * tcmod->args.scroll.s,
      realtime * tcmod->args.scroll.t,
      0);
}



static float Function(R_WAVEFORM func, float ox)
{
   float x, y;

   x = ox - floor(ox);

   if (func == R_SIN) {
      y = sin(2 * M_PI * x);
   }
   else if (func == R_TRIANGLE) {
      if (x < 0.25)
      {
         y = (4.0 * x);
      }
      else if (x < 0.75)
      {
         y = 1 - ((x - 0.25) * 4);
      }
      else
      {
         y = (4.0 * (x - 0.75)) - 1;
      }
   }
   else if (func == R_SQUARE) {
      if (x < 0.5)
      {
         y =  1;
      }
      else
      {
         y = -1;
      }
   }
   else if (func == R_SAWTOOTH) {
      y = (2 * x) - 1;
   }
   else if (func == R_INVSAWTOOTH) {
      y = (2 * (1 - x)) - 1;
   }
   else {
      y = 0;
   }

   return y;
}



static void tcModStretch(struct tcmod_s* tcmod, vavertex_t* va, int first, int count)
{
   float y;

   (void)va;
   (void)first;
   (void)count;

   #define ARG (tcmod->args.stretch)

   y = (ARG.amp * Function(ARG.func, ARG.freq * (realtime + ARG.phase))) + ARG.base;

   if (y == 0) {
      y = 1000000;
   }
   else {
      y = 1 / y;
   }

   glTranslatef( 0.5,  0.5, 0);
   glScalef(y, y, 1);
   glTranslatef(-0.5, -0.5, 0);

   #undef ARG
}



static void tcModTransform(struct tcmod_s* tcmod, vavertex_t* va, int first, int count)
{
   float matrix[4][4];

   (void)va;
   (void)first;
   (void)count;

   #define ARG (tcmod->args.transform)

   matrix[0][0] = ARG.m00;
   matrix[0][1] = ARG.m01;
   matrix[0][2] = 0;
   matrix[0][3] = 0;

   matrix[1][0] = ARG.m10;
   matrix[1][1] = ARG.m11;
   matrix[1][2] = 0;
   matrix[1][3] = 0;

   matrix[2][0] = 0;
   matrix[2][1] = 0;
   matrix[2][2] = 1;
   matrix[2][3] = 0;

   matrix[3][0] = ARG.t0;
   matrix[3][1] = ARG.t1;
   matrix[3][2] = 0;
   matrix[3][3] = 1;

   glLoadMatrixf((float*)matrix);

   #undef ARG
}



static void tcModTurb(struct tcmod_s* tcmod, vavertex_t* varray, int first, int count)
{
   int    i;
   float* save = saved_st;

   #define ARG (tcmod->args.turb)

   for (i = 0; i < count; i++) {
      float os, ot;
      float t;

      t = ARG.freq * (realtime + ARG.phase);

      os = varray->s;
      ot = varray->t;

      varray->s = os + (ARG.amp * Function(ARG.func, ot + t)) + ARG.base;
      varray->t = ot + (ARG.amp * Function(ARG.func, os + t)) + ARG.base;

      varray++;

      *save++ = os;
      *save++ = ot;
    }

    restore_st = true;

   #undef ARG
}



void R_RestoreST(vavertex_t* varray, int first, int count)
{
   float* saved;
   int    i;

   if (!restore_st) return;

   saved = saved_st;

   for (i = 0; i < count; i++) {
      varray->s = *saved++;
      varray->t = *saved++;
      varray++;
   }

   restore_st = false;
}


static qboolean tcModParseFunc(shader_t* shader, shader_stage_t* stage, int argc, char* argv[])
{
   tcmod_t* tcmod = &stage->tcmods[stage->num_tcmods];

   if (stricmp(argv[0], "rotate") == 0) {
      if (argc != 2) {
         Con_Printf("warning: tcMod: wrong number of arguments for rotate.\n");
         return true;
      }

      tcmod->func = tcModRotate;
      tcmod->args.rotate = Q_atof(argv[1]);
   }
   else if (stricmp(argv[0], "scale") == 0) {
      if (argc != 3) {
         Con_Printf("warning: tcMod: wrong number of arguments for scale.\n");
         return true;
      }

      tcmod->func = tcModScale;
      tcmod->args.scale.s = Q_atof(argv[1]);
      tcmod->args.scale.t = Q_atof(argv[2]);
   }
   else if (stricmp(argv[0], "scroll") == 0) {
      if (argc != 3) {
         Con_Printf("warning: tcMod: wrong number of arguments for scroll.\n");
         return true;
      }

      tcmod->func = tcModScroll;
      tcmod->args.scroll.s = Q_atof(argv[1]);
      tcmod->args.scroll.t = Q_atof(argv[2]);
   }
   else if (stricmp(argv[0], "stretch") == 0) {
      int func;

      if (argc != 6) {
         Con_Printf("warning: tcMod: wrong number of arguments for stretch.\n");
         return true;
      }

      tcmod->func = tcModStretch;

      func = WaveForm(argv[1]);

      if (func < 0) {
         Con_Printf("warning: tcMod: stretch: unknown waveform.\n");
         return true;
      }

      tcmod->args.stretch.func  = func;
      tcmod->args.stretch.base  = Q_atof(argv[2]);
      tcmod->args.stretch.amp   = Q_atof(argv[3]);
      tcmod->args.stretch.phase = Q_atof(argv[4]);
      tcmod->args.stretch.freq  = Q_atof(argv[5]);
   }
   else if (stricmp(argv[0], "transform") == 0) {
      if (argc != 7) {
         Con_Printf("warning: tcMod: wrong number of arguments for transform.\n");
         return true;
      }

      tcmod->func = tcModTransform;

      tcmod->args.transform.m00 = Q_atof(argv[1]);
      tcmod->args.transform.m01 = Q_atof(argv[2]);
      tcmod->args.transform.m10 = Q_atof(argv[3]);
      tcmod->args.transform.m11 = Q_atof(argv[4]);
      tcmod->args.transform.t0  = Q_atof(argv[5]);
      tcmod->args.transform.t1  = Q_atof(argv[6]);
   }
   else if (stricmp(argv[0], "turb") == 0) {
      int func;

      if (argc != 6) {
         Con_Printf("warning: tcMod: wrong number of arguments for turb.\n");
         return true;
      }

      tcmod->func = tcModTurb;

      func = WaveForm(argv[1]);

      if (func < 0) {
         Con_Printf("warning: tcMod: turb: unknown waveform.\n");
         return true;
      }

      tcmod->args.turb.func  = func;
      tcmod->args.turb.base  = Q_atof(argv[2]);
      tcmod->args.turb.amp   = Q_atof(argv[3]);
      tcmod->args.turb.phase = Q_atof(argv[4]);
      tcmod->args.turb.freq  = Q_atof(argv[5]);
   }
   else {
      tcmod->func = NULL;

      Con_Printf("warning: tcMod: unknown tcMod type %s.\n", argv[0]);
      return true;
   }

   stage->num_tcmods++;

   return true;
}



typedef struct {
   char*          name;
   R_BLEND_FACTOR factor;
   int            is_src;
   int            is_dst;
} blend_parse_table_t;



static blend_parse_table_t blend_parse_table[] = {
   { "GL_ONE",                 R_ONE,                 true,  true  },
   { "GL_ZERO",                R_ZERO,                true,  true  },
   { "GL_DST_COLOR",           R_DST_COLOR,           true,  false },
   { "GL_ONE_MINUS_DST_COLOR", R_ONE_MINUS_DST_COLOR, true,  false },
   { "GL_SRC_ALPHA",           R_SRC_ALPHA,           true,  true  },
   { "GL_ONE_MINUS_SRC_ALPHA", R_ONE_MINUS_SRC_ALPHA, true,  true  },
   { "GL_SRC_COLOR",           R_SRC_COLOR,           false, true  },
   { "GL_ONE_MINUS_SRC_COLOR", R_ONE_MINUS_SRC_COLOR, false, true  },
   { 0 }
};



R_BLEND_FACTOR ParseBlendFunc(const char* arg, qboolean is_parsing_for_src)
{
   blend_parse_table_t* table = blend_parse_table;

   while (table->name) {
      if (stricmp(table->name, arg) == 0) {
         if (is_parsing_for_src && !table->is_src) {
            Con_Printf("warning: blendfunc: '%s' is not a source blend factor.\n");
            return -1;
         }

         if (!is_parsing_for_src && !table->is_dst) {
            Con_Printf("warning: blendfunc: '%s' is not a source blend factor.\n");
            return -1;
         }

         return table->factor;
      }

      table++;
   }

   if (is_parsing_for_src) {
      Con_Printf("warning: blendfunc: unknown source factor.\n");
      return -1;
   }
   else {
      Con_Printf("warning: blendfunc: unknown destination factor.\n");
      return -1;
   }

} 



static qboolean blendfuncParseFunc
   (
   shader_t*       shader,
   shader_stage_t* stage,
   int             argc,
   char*           argv[]
   )
{
   if (argc == 1) {
      if (stricmp(argv[0], "add") == 0) {
         stage->blend_src = R_ONE;
         stage->blend_dst = R_ONE;
      }
      else if (stricmp(argv[0], "filter") == 0) {
         stage->blend_src = R_DST_COLOR;
         stage->blend_dst = R_ZERO;
      }
      else if (stricmp(argv[0], "blend") == 0) {
         stage->blend_src = R_SRC_ALPHA;
         stage->blend_dst = R_ONE_MINUS_SRC_ALPHA;
      }
      else {
         Con_Printf("warning: blendfunc: unknown blendfunc '%s'.\n", argv[0]);
         return true;
      }
   }
   else {
      stage->blend_src = ParseBlendFunc(argv[0], true);

      if (stage->blend_src < 0) return true;

      stage->blend_dst = ParseBlendFunc(argv[1], false);

      if (stage->blend_dst < 0) return true;
   }

   stage->flags |= R_HAS_BLENDFUNC;

   return true;
}



static qboolean nomipmapsParseFunc
   (
   shader_t*       shader,
   shader_stage_t* stage,
   int             argc,
   char*           argv[]
   )
{
   nomipmaps = true;

   return true;
}



static qboolean nopicmipParseFunc
   (
   shader_t*       shader,
   shader_stage_t* stage,
   int             argc,
   char*           argv[]
   )
{
   nopicmip = true;

   return true;
}



static shader_command_t commands[] =
{
    { "map",       1, 1, true,  mapParseFunc       },
    { "tcMod",     2, 7, true,  tcModParseFunc     },
    { "blendfunc", 1, 2, true,  blendfuncParseFunc },
    { "nomipmaps", 0, 0, false, nomipmapsParseFunc },
    { "nopicmip",  0, 0, false, nopicmipParseFunc  },
    { 0 }
};



static qboolean R_ParseShaderCommand(shader_t* shader, shader_stage_t* stage, char* command)
{
    shader_command_t* cmd;

    for (cmd = commands; cmd->name != 0; cmd++) {

       if (( stage && !cmd->is_stage_command) ||
           (!stage &&  cmd->is_stage_command)) {
          continue;
       }

       if (stricmp(cmd->name, command) == 0) {
          char* argv[R_MAX_SHADER_ARGS];
          int   argc = 0;

          while ((argv[argc] = nextarg()) != NULL) {
             argc++;
          }

          if (argc > cmd->max_argc) {
             Con_Printf("error: too many arguments");
             return false;
          }

          if (argc < cmd->min_argc) {
             Con_Printf("error: too few arguments");
             return false;
          }

          return cmd->parse(shader, stage, argc, argv);
       }
    }

    Con_Printf("warning: unrecognized command '%s'.\n", command);

    while (*curpos != '\n') {
       if (++curpos == endpos) break;
    }

    return true;
}



static qboolean R_ParseShaderStage(shader_t* shader)
{
   char*           tok;
   shader_stage_t* stage;

   if (shader->num_stages == R_MAX_SHADER_STAGES) {
      Con_Printf("warning: too many shader stages for %s\n", shader->name);
      return true;
   }

   stage = &shader->stages[shader->num_stages];

   stage->map        = 0;
   stage->num_tcmods = 0;

   stage->flags = 0;

   for (;;) {
      tok = nexttok();

      if (tok == NULL) {
         Con_Printf("error: unexpected end of input\n");
         return false;
      }

      if (tok[0] == '}') {
         shader->num_stages++;
         return true;
      }
      else {
         if (!R_ParseShaderCommand(shader, stage, tok)) return false;
      }
   }
}



void R_ParseShaders(char* text, size_t size)
{
   char*     tok;
   shader_t* shader;
   int       id;

   curpos = text;
   endpos = text + size;

   for (;;) {
      tok = nexttok();
      
      if (tok == NULL) break;

      id = R_GetShader(tok);

      if (id < 0) {
         if (!shader_skip()) return;
         continue;
      }

      shader = &shaders[id];

      // shader body
      tok = nexttok();

      if (tok[0] != '{') {
         Con_Printf("R_ParseShaders: expected '{', found '%s'\n", tok);

         return;
      }

      // stages and commands
      for (;;) {
         tok = nexttok();
         
         if (tok == NULL) {
            // error
            Con_Printf("error: unexpected end of input\n");
            return;
         }

         nomipmaps = false;
         nopicmip  = false;

         if (tok[0] == '{') {
            // stage
            if (!R_ParseShaderStage(shader)) return;
         }
         else if (tok[0] == '}') {
            // end of shader body
            shader->is_loaded = true;
            Con_Printf("   %s\n", shader->name);
            break;
         }
         else {
            // shader command
            if (!R_ParseShaderCommand(shader, NULL, tok)) return;
         }
      }
   }

   // make default shaders
}



void R_ClearShaders(void)
{
}



int R_GetShader(const char* name)
{
    int i;

    for (i = 0; i < num_shaders; i++) {
       if (strncmp(name, shaders[i].name, R_SHADER_NAME_SIZE) == 0) return i;
    }

    return R_NOT_FOUND;
}



int R_LoadMap(const char* name, qboolean mipmap, qboolean alpha)
{
    const char *ext;

    ext = COM_FileExtension((char*)name);

    if (stricmp(ext, "lmp") == 0) {
        return R_LoadMapFromLump(name, mipmap, alpha);
    }
    else if (stricmp(ext, "tga") == 0) {
        return R_LoadMapFromTarga(name, mipmap, alpha);
    }
    else {
        return R_UNKNOWN_TYPE;
    }
}



int R_LoadMapFromTarga(const char* name, qboolean mipmap, qboolean alpha)
{
   FILE* f;

   if (num_texture_maps > texture_map_array_size) return R_OUT_OF_MEMORY;

   COM_FOpenFile ((char*)name, &f);

   if (!f) return R_NOT_FOUND;

   LoadTGA(f);

   strncpy(texture_maps[num_texture_maps].name, name, R_SHADER_NAME_SIZE);
   texture_maps[num_texture_maps].name[R_SHADER_NAME_SIZE-1] = '\0';

   texture_maps[num_texture_maps].index =
      GL_LoadTexture32(
         (char*)name,
         targa_header.width,
         targa_header.height,
         (int*)targa_rgba,
         mipmap,
         alpha);

   free (targa_rgba);

   num_texture_maps++;

   return num_texture_maps - 1;
}



int R_LoadMapFromLump(const char* name, qboolean mipmap, qboolean alpha)
{
   qpic_t* qpic;
   int     mark;

   if (num_texture_maps > texture_map_array_size) return R_OUT_OF_MEMORY;

   mark = Hunk_LowMark();

   qpic = (qpic_t *)COM_LoadHunkFile((char*)name);

   if (!qpic) return R_NOT_FOUND;

   SwapPic(qpic);

   strncpy(texture_maps[num_texture_maps].name, name, R_SHADER_NAME_SIZE);
   texture_maps[num_texture_maps].name[R_SHADER_NAME_SIZE-1] = '\0';

   texture_maps[num_texture_maps].index =
      GL_LoadTexture(
         (char*)name,
         qpic->width,
         qpic->height,
         qpic->data,
         mipmap,
         alpha);

   num_texture_maps++;

   Hunk_FreeToLowMark(mark);

   return num_texture_maps - 1;
}
