/* Basic demo parsing code for Quake2
   1998/1999 by Stefan Schwoon
   (schwoon@informatik.uni-hildesheim.de)
*/

/* 24.11.1998: Added support for up to 3.17. Changes:
	- support for new tempentity types
	- support for vwep (does not necessitate change in this code)
	- support for download
	- support for server side recordings (not sure if handled correctly)
*/

/* 11.01.1999: Added support for 3.20. Changes:
	- support for new tempentity types
*/

#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <stdlib.h>

#define PROGRAM_NAME "skel"

#define MAX_MSGLEN "1400"

typedef signed char s_char;
#define char unsigned char

enum {
	SW_SOME_SWITCH_WITH_ARG,
	SW_SOME_OTHER_SWITCH,
	NUM_SWITCHES
};

FILE *pfile;
char rewriting = 0;
char printing = 0;

char flag[NUM_SWITCHES];
char *p_arg = NULL;
char *p_cblocksize = MAX_MSGLEN;
int p_blocksize;

enum { FL_PRINT, FL_WRITE, FL_NONE };
enum { ARG_OPT, ARG_NEEDED };

struct { char *name; int flag, type; char **parm; char need_arg; }
optname[] =
	{ { "-example", SW_SOME_SWITCH_WITH_ARG, FL_PRINT, &p_arg, ARG_OPT },
	  { "-nop", SW_SOME_OTHER_SWITCH, FL_WRITE, NULL, 0 },
	  { "-blocksize", -1, FL_NONE, &p_cblocksize, ARG_NEEDED },
	  { NULL, 0, 0, NULL }
	};

#define MAX_EDICTS 1024

typedef struct {
	char active, removed;
	char modelindex, modelindex2, modelindex3, modelindex4;
	short frame;
	long skinnum, vwep;
	long effects, renderfx;
	short origin[3];
	char angle[3];
	short oldorigin[3];
	char sound, event;
	short solid;
} t_entity;

typedef struct {
	char pm_type;
	short origin[3];
	short velocity[3];
	char teleport_time;
	char pm_flags;
	short gravity;
	short delta_angles[3];
	s_char viewoffset[3];
	short viewangles[3];
	s_char kickangles[3];
	char gunindex;
	char gunframe;
	char gunoffset[3];
	s_char gunangles[3];
	char blend[4];
	char fov;
	char rdflags;
	short stats[32];
} t_playerinfo;

char fnindex;
s_char previndex;
t_entity *baseline;
t_entity *frame[16];
t_playerinfo plinf[16];
long framenum[16];
char isdemo;

enum { RECORD_NETWORK, RECORD_CLIENT, RECORD_SERVER };

#define BASELINESIZE (MAX_EDICTS * sizeof(t_entity))
#define BASELINESIZE (MAX_EDICTS * sizeof(t_entity))

char *inblk, *tmpblk, *outblk;

char *inptr;
long outlen;

void error (const char* msg, ...)
{
	va_list	the_args;
	va_start(the_args,msg);
	fprintf(stderr,"%s: ",PROGRAM_NAME);
	vfprintf(stderr,msg,the_args);
	fprintf(stderr,"\n");
	va_end(the_args);
	exit(1);
}

void copy_msg(int i)
{
	memcpy(outblk + outlen, inptr, i);
	outlen += i;
	inptr += i;
}

void insert_msg(char *msg, int i)
{
	memcpy(outblk + outlen, msg, i);
	outlen += i;
}

void discard_msg(int i)
{
	inptr += i;
}

enum {
	SVC_BAD, SVC_MUZZLEFLASH, SVC_MUZZLFLASH2, SVC_TEMP_ENTITY,
	SVC_LAYOUT, SVC_INVENTORY, SVC_NOP, SVC_DISCONNECT,
	SVC_RECONNECT, SVC_SOUND, SVC_PRINT, SVC_STUFFTEXT,
	SVC_SERVERDATA, SVC_CONFIGSTRING, SVC_SPAWNBASELINE,
	SVC_CENTERPRINT, SVC_DOWNLOAD, SVC_PLAYERINFO,
	SVC_PACKETENTITIES, SVC_DELTAPACKETENTITIES, SVC_FRAME
};

enum {
	TE_GUNSHOT, TE_BLOOD, TE_BLASTER, TE_RAILTRAIL,
	TE_SHOTGUN, TE_EXPLOSION1, TE_EXPLOSION2, TE_ROCKET_EXPLOSION,
	TE_GRENADE_EXPLOSION, TE_SPARKS, TE_SPLASH, TE_BUBBLETRAIL,
	TE_SCREEN_SPARKS, TE_SHIELD_SPARKS, TE_BULLET_SPARKS, TE_LASER_SPARKS,
	TE_PARASITE_ATTACK, TE_ROCKET_EXPLOSION_WATER,
		TE_GRENADE_EXPLOSION_WATER, TE_MEDIC_CABLE_ATTACK,
	TE_BFG_EXPLOSION, TE_BFG_BIGEXPLOSION, TE_BOSSTPORT, TE_BFG_LASER,
	TE_GRAPPLE_CABLE, TE_WELDING_SPARKS, TE_GREENBLOOD, TE_BLUEHYPERBLASTER,
	TE_PLASMA_EXPLOSION, TE_TUNNEL_SPARKS, TE_BLASTER2, TE_RAILTRAIL2,
	TE_FLAME, TE_LIGHTNING, TE_DEBUGTRAIL, TE_PLAIN_EXPLOSION,
	TE_FLASHLIGHT, TE_FORCEWALL, TE_HEATBEAM, TE_MONSTER_HEATBEAM,
	TE_STEAM, TE_BUBBLETRAIL2, TE_MOREBLOOD, TE_HEATSTEAM_SPARKS,
	TE_HEATBEAM_STEAM, TE_CHAINFIST_SMOKE,
		TE_ELECTRIC_SPARKS, TE_TRACKER_EXPLOSION,
	TE_TELEPORT_EFFECT, TE_DBALL_GOAL, TE_WIDOWBEAMOUT, TE_NUKEBLAST,
	TE_WIDOWSPLASH, TE_EXPLOSION1_BIG, TE_EXPLOSION1_NP, TE_FLECHETTE
};

void msg_bad()
{
	error("bad message encountered");
}

void msg_muzzleflash()
{
	copy_msg(4);
}

void msg_muzzlflash2()
{
	copy_msg(4);
}

#define PE_SIZE 8
#define IE_SIZE 9
#define LE_SIZE 14

char te_size[]
	= { IE_SIZE, IE_SIZE, IE_SIZE, LE_SIZE,
	    IE_SIZE, PE_SIZE, PE_SIZE, PE_SIZE,
	    PE_SIZE, IE_SIZE,    11  , LE_SIZE,
	    IE_SIZE, IE_SIZE, IE_SIZE,    11  ,
	       16  , PE_SIZE, PE_SIZE,    16  ,
	    PE_SIZE, PE_SIZE, PE_SIZE, LE_SIZE,
	       22  ,    11  , IE_SIZE, LE_SIZE, 
	    PE_SIZE,    11  , IE_SIZE,	   2  ,
	       42  ,	18  , LE_SIZE, PE_SIZE,
	       10  ,	16  ,	16   ,	  16  ,
	       15  , LE_SIZE, IE_SIZE, IE_SIZE,
	    IE_SIZE, PE_SIZE, LE_SIZE, PE_SIZE,
	    PE_SIZE, PE_SIZE,	10   , PE_SIZE,
	    PE_SIZE, PE_SIZE, PE_SIZE, IE_SIZE
	  };

void msg_temp_entity()
{
	char enttype = inptr[1];
	long nextid = *(long*)(inptr+2);
	if (enttype > TE_FLECHETTE || enttype == TE_RAILTRAIL2)
		error("bad temp_entity type");
	copy_msg(te_size[(int)(enttype)]);
	if (enttype == TE_STEAM && nextid != -1) copy_msg(4);
}

void msg_layout()
{
	char *ptr = inptr + 1;
	while (*ptr++);
	copy_msg(ptr-inptr);
}

void msg_inventory()
{
	#define MAX_ITEMS 256
	copy_msg(MAX_ITEMS*2+1);
}

void msg_nop()
{
	copy_msg(1);
}

void msg_disconnect()
{
	copy_msg(1);
}

void msg_reconnect()
{
	copy_msg(1);
}

void msg_sound()
{
	char mask = inptr[1];
	int len = 3;
	if (mask & 0x01) len++;
	if (mask & 0x02) len++;
	if (mask & 0x10) len++;
	if (mask & 0x08) len += 2;
	if (mask & 0x04) len += 6;
	copy_msg(len);
}

void msg_print()
{
	char *ptr = inptr + 2;
	while (*ptr++);
	copy_msg(ptr-inptr);
}

void msg_stufftext()
{
	char *ptr = inptr + 1;
	while (*ptr++);
	copy_msg(ptr-inptr);
}

void msg_serverdata()
{
	char *ptr = inptr + 10;

	isdemo = inptr[9];
	while (*ptr++);
	ptr += 2;
	while (*ptr++);
	copy_msg(ptr-inptr);
}

void msg_configstring()
{
	char *ptr = inptr + 3;
	while (*ptr++);
	copy_msg(ptr-inptr);
}

int bl_bytes(long bit1, long bit2)
{
	return bit1? (bit2? 4 : 1) : (bit2? 2 : 0);
}

short readshort(char **ptr)
{	*ptr += 2; return *(short*)(*ptr-2);	}

char readchar(char **ptr)
{	return *(*ptr)++;	}

long readlong(char **ptr)
{	*ptr += 4; return *(long*)(*ptr-4);	}

void readshorts(char **ptr, short *origin)
{
	origin[0] = readshort(ptr);
	origin[1] = readshort(ptr);
	origin[2] = readshort(ptr);
}

void readchars(char **ptr, char *angles)
{
	angles[0] = readchar(ptr);
	angles[1] = readchar(ptr);
	angles[2] = readchar(ptr);
}

void bl_read(long *data, long mask, long bit1, long bit2, char **ptr)
{
	if (mask & bit1) 
		*data = (mask & bit2)? readlong(ptr) : readchar(ptr);
	else if (mask & bit2)
		*data = (unsigned short) readshort(ptr);
}

int parse_baseline(t_entity *line)
{
	char *ptr = inptr + 1;
	long mask = inptr[0];
	short entity;
	t_entity *ent;

	if (mask & 0x80) mask |= inptr[1] << 8;
	if (mask & 0x8000) mask |= inptr[2] << 16;
	if (mask & 0x800000) mask |= inptr[3] << 24;
	ptr += !!(mask & 0x80) + !!(mask & 0x8000) + !!(mask & 0x800000);
	entity = (mask & 0x00000100)? readshort(&ptr) : readchar(&ptr);
	if (!entity)
	{
		discard_msg(ptr-inptr);
		return 0;
	}
	ent = &line[entity];
	if (line != baseline)
	{
		ent->active = !(mask & 0x00000040);
		if (!ent->active)
			memcpy(ent,&baseline[entity],sizeof(t_entity));
	}
	if (mask & 0x00000800) ent->modelindex = readchar(&ptr);
	if (mask & 0x00100000) ent->modelindex2 = readchar(&ptr);
	if (mask & 0x00200000) ent->modelindex3 = readchar(&ptr);
	if (mask & 0x00400000) ent->modelindex4 = readchar(&ptr);
	if (mask & 0x00000010) ent->frame = readchar(&ptr);
	if (mask & 0x00020000) ent->frame = readshort(&ptr);
	bl_read(&ent->skinnum, mask, 0x00010000, 0x02000000, &ptr);
	ent->vwep = ent->skinnum >> 8;
	ent->skinnum &= 0xff;
	bl_read(&ent->effects, mask, 0x00004000, 0x00080000, &ptr);
	bl_read(&ent->renderfx, mask, 0x00001000, 0x00040000, &ptr);
	if (mask & 0x00000001) ent->origin[0] = readshort(&ptr);
	if (mask & 0x00000002) ent->origin[1] = readshort(&ptr);
	if (mask & 0x00000200) ent->origin[2] = readshort(&ptr);
	if (mask & 0x00000400) ent->angle[0] = readchar(&ptr);
	if (mask & 0x00000004) ent->angle[1] = readchar(&ptr);
	if (mask & 0x00000008) ent->angle[2] = readchar(&ptr);
	if (mask & 0x01000000) readshorts(&ptr,ent->oldorigin);
	if (mask & 0x04000000) ent->sound = readchar(&ptr);
	if (mask & 0x00000020) ent->event = readchar(&ptr);
	if (mask & 0x08000000) ent->solid = readshort(&ptr);
	((line != baseline)? discard_msg : copy_msg)(ptr-inptr);
	return entity;
}

void msg_spawnbaseline()
{
	copy_msg(1);
	parse_baseline(baseline);
}

void msg_centerprint()
{
	char *ptr = inptr + 1;
	while (*ptr++);
	copy_msg(ptr-inptr);
}

void msg_download()
{
	short size = *(short*)(inptr+1);
	size *= !!~size;	/* heh */
	copy_msg(4+size);
}

void writeshort(char **ptr, short value, unsigned long *mask, long bit)
{
	*(short*)(*ptr) = value;
	*ptr += 2;
	*mask |= bit;
}

void writechar(char **ptr, char value, unsigned long *mask, long bit)
{
	*(*ptr)++ = value;
	*mask |= bit;
}

void writelong(char **ptr, long value, unsigned long *mask, long bit)
{
	*(long*)(*ptr) = value;
	*ptr += 4;
	*mask |= bit;
}

void writeshorts(char **ptr, short *values, unsigned long *mask, long bit)
{
	writeshort(ptr,values[0],mask,bit);
	writeshort(ptr,values[1],mask,bit);
	writeshort(ptr,values[2],mask,bit);
}

void writechars(char **ptr, char *values, unsigned long *mask, long bit)
{
	writechar(ptr,values[0],mask,bit);
	writechar(ptr,values[1],mask,bit);
	writechar(ptr,values[2],mask,bit);
}

/* pi is being delta'ed against delta and written to dst */
int write_playerinfo(t_playerinfo *pi, t_playerinfo *delta, char *dst)
{
	#define CHANGED(x) (pi->x != delta->x)
	#define CHANGED_VEC(x) (pi->x[0] != delta->x[0] \
				|| pi->x[1] != delta->x[1] \
				|| pi->x[2] != delta->x[2])
	char *ptr = dst;
	char *maskptr;
	int i;
	unsigned long mask = 0;

	writechar(&ptr,SVC_PLAYERINFO,&mask,0);
	writeshort(&ptr,0,&mask,0);
	if (CHANGED(pm_type))
		writechar(&ptr,pi->pm_type,&mask,0x0001);
	if (CHANGED_VEC(origin))
		writeshorts(&ptr,pi->origin,&mask,0x0002);
	if (CHANGED_VEC(velocity))
		writeshorts(&ptr,pi->velocity,&mask,0x0004);
	if (CHANGED(teleport_time))
		writechar(&ptr,pi->teleport_time,&mask,0x0008);
	if (CHANGED(pm_flags))
		writechar(&ptr,pi->pm_flags,&mask,0x0010);
	if (CHANGED(gravity))
		writeshort(&ptr,pi->gravity,&mask,0x0020);
	if (CHANGED_VEC(delta_angles))
		writeshorts(&ptr,pi->delta_angles,&mask,0x0040);
	if (CHANGED_VEC(viewoffset))
		writechars(&ptr,pi->viewoffset,&mask,0x0080);
	if (CHANGED_VEC(viewangles))
		writeshorts(&ptr,pi->viewangles,&mask,0x0100);
	if (CHANGED_VEC(kickangles))
		writechars(&ptr,pi->kickangles,&mask,0x0200);
	if (CHANGED(gunindex))
		writechar(&ptr,pi->gunindex,&mask,0x1000);
	if (CHANGED(gunframe)||CHANGED_VEC(gunoffset)||CHANGED_VEC(gunangles))
	{
		writechar(&ptr,pi->gunframe,&mask,0x2000);
		writechars(&ptr,pi->gunoffset,&mask,0x2000);
		writechars(&ptr,pi->gunangles,&mask,0x2000);
	}
	if (CHANGED(blend[0]) || CHANGED(blend[1])
	 || CHANGED(blend[2]) || CHANGED(blend[3]))
	{
		writechar(&ptr,pi->blend[0],&mask,0x0400);
		writechar(&ptr,pi->blend[1],&mask,0x0400);
		writechar(&ptr,pi->blend[2],&mask,0x0400);
		writechar(&ptr,pi->blend[3],&mask,0x0400);
	}
	if (CHANGED(fov))
		writechar(&ptr,pi->fov,&mask,0x0800);
	if (CHANGED(rdflags))
		writechar(&ptr,pi->rdflags,&mask,0x4000);
	*(short*)(dst+1) = mask;
	maskptr = ptr;
	writelong(&ptr,mask = 0,&mask,0);
	for (i = 0; i < 32; i++)
		if (CHANGED(stats[i]))
			writeshort(&ptr,pi->stats[i],&mask,1 << i);
	*(long*)maskptr = mask;
	return ptr-dst;
	#undef CHANGED
	#undef CHANGED_VEC
}

void msg_playerinfo()
{
	char *ptr = inptr + 1;
	int i = 0;
	t_playerinfo *pi = &plinf[fnindex];
	unsigned long mask = readshort(&ptr);

	if (mask & 0x0001) pi->pm_type = readchar(&ptr);
	if (mask & 0x0002) readshorts(&ptr,pi->origin);
	if (mask & 0x0004) readshorts(&ptr,pi->velocity);
	if (mask & 0x0008) pi->teleport_time = readchar(&ptr);
	if (mask & 0x0010) pi->pm_flags = readchar(&ptr);
	if (mask & 0x0020) pi->gravity = readshort(&ptr);
	if (mask & 0x0040) readshorts(&ptr,pi->delta_angles);
	if (mask & 0x0080) readchars(&ptr,pi->viewoffset);
	if (mask & 0x0100) readshorts(&ptr,pi->viewangles);
	if (mask & 0x0200) readchars(&ptr,pi->kickangles);
	if (mask & 0x1000) pi->gunindex = readchar(&ptr);
	if (mask & 0x2000)
	{
		pi->gunframe = readchar(&ptr);
		readchars(&ptr,pi->gunoffset);
		readchars(&ptr,pi->gunangles);
	}
	if (mask & 0x0400)
	{
		pi->blend[0] = readchar(&ptr);
		pi->blend[1] = readchar(&ptr);
		pi->blend[2] = readchar(&ptr);
		pi->blend[3] = readchar(&ptr);
	}
	if (mask & 0x0800) pi->fov = readchar(&ptr);
	if (mask & 0x4000) pi->rdflags = readchar(&ptr);
	mask = readlong(&ptr);
	while (mask)
	{
		if (mask & 0x0001) pi->stats[i] = readshort(&ptr);
		mask >>= 1;
		i++;
	}
	discard_msg(ptr-inptr);
}

int write_packetentities(t_entity *ent, t_entity *delta, char *dst)
{
	#define CHANGED(x) (e->x != d->x)
	int i;
	char *ptr = dst;
	unsigned long mask;
	t_entity tmpent;

	writechar(&ptr,SVC_PACKETENTITIES,&mask,0);
	for (i = 1; i < MAX_EDICTS; i++)
	{
		char *maskptr = ptr;
		t_entity *e = ent+i, *d = delta+i;
		int len;

		if (!e->active && !d->active) continue;
		writelong(&ptr,mask = 0,&mask,0);
		if (i & 0xff00) writeshort(&ptr,i,&mask,0x00000100);
		else writechar(&ptr,i,&mask,0);
		if (!e->active)
		{
			mask |= 0x00000040;
			goto movemask;
		}
		if (!d->active)
		{
			memcpy(&tmpent,d,sizeof(t_entity));
			memcpy(d,&baseline[i],sizeof(t_entity));
		}
		if (CHANGED(modelindex))
			writechar(&ptr,e->modelindex,&mask,0x00000800);
		if (CHANGED(modelindex2))
			writechar(&ptr,e->modelindex2,&mask,0x00100000);
		if (CHANGED(modelindex3))
			writechar(&ptr,e->modelindex3,&mask,0x00200000);
		if (CHANGED(modelindex4))
			writechar(&ptr,e->modelindex4,&mask,0x00400000);
		if (CHANGED(frame))
			if (e->frame & 0xff00)
				writeshort(&ptr,e->frame,&mask,0x00020000);
			else
				writechar(&ptr,e->frame,&mask,0x00000010);
		if (CHANGED(skinnum) || CHANGED(vwep))
		{
			e->skinnum |= e->vwep << 8;
			if (e->skinnum & 0xffff0000)
				writelong(&ptr,e->skinnum,&mask,0x02010000);
			else if (e->skinnum & 0xff00)
				writeshort(&ptr,e->skinnum,&mask,0x02000000);
			else
				writechar(&ptr,e->skinnum,&mask,0x00010000);
			e->skinnum &= 0xff;
		}
		if (CHANGED(effects))
			if (e->effects & 0xffff0000)
				writelong(&ptr,e->effects,&mask,0x00084000);
			else if (e->effects & 0xff00)
				writeshort(&ptr,e->effects,&mask,0x00080000);
			else
				writechar(&ptr,e->effects,&mask,0x00004000);
		if (CHANGED(renderfx))
			if (e->renderfx & 0xffff0000)
				writelong(&ptr,e->renderfx,&mask,0x00041000);
			else if (e->renderfx & 0xff00)
				writeshort(&ptr,e->renderfx,&mask,0x00040000);
			else
				writechar(&ptr,e->renderfx,&mask,0x00001000);
		if (CHANGED(origin[0]))
			writeshort(&ptr,e->origin[0],&mask,0x00000001);
		if (CHANGED(origin[1]))
			writeshort(&ptr,e->origin[1],&mask,0x00000002);
		if (CHANGED(origin[2]))
			writeshort(&ptr,e->origin[2],&mask,0x00000200);
		if (CHANGED(angle[0]))
			writechar(&ptr,e->angle[0],&mask,0x00000400);
		if (CHANGED(angle[1]))
			writechar(&ptr,e->angle[1],&mask,0x00000004);
		if (CHANGED(angle[2]))
			writechar(&ptr,e->angle[2],&mask,0x00000008);
		if (CHANGED(oldorigin[0]) || CHANGED(oldorigin[1])
				|| CHANGED(oldorigin[2]))
			writeshorts(&ptr,e->oldorigin,&mask,0x01000000);
		if (CHANGED(sound))
			writechar(&ptr,e->sound,&mask,0x04000000);
		if (e->event)
			writechar(&ptr,e->event,&mask,0x00000020);
		if (CHANGED(solid))
			writeshort(&ptr,e->solid,&mask,0x08000000);
		movemask:
		if (mask & 0xff000000) mask |= 0x00800000;
		if (mask & 0x00ff0000) mask |= 0x00008000;
		if (mask & 0x0000ff00) mask |= 0x00000080;
		*(long*)maskptr = mask;
		len = 1 + !!(mask & 0x00800000) + !!(mask & 0x00008000)
						+ !!(mask & 0x00000080);
		if (len != 4 && ptr-maskptr > 4)
			memmove(maskptr+len,maskptr+4,ptr-maskptr-4);
		ptr += len-4;
		if (!(mask & ~0x00000180) && d->active) ptr = maskptr;
		if (!d->active) memcpy(d,&tmpent,sizeof(t_entity));
	}
	writechar(&ptr,0,&mask,0);
	writechar(&ptr,0,&mask,0);
	return ptr-dst;
}

void msg_packetentities()
{
	int l;

	t_playerinfo basepi;
	t_entity *line = baseline;
	t_playerinfo *pli = &basepi;

	discard_msg(1);
	while (parse_baseline(frame[fnindex]));

	if (previndex < 0 || isdemo == RECORD_SERVER)
	{
		memset(&basepi,0,sizeof(t_playerinfo));
	}
	else
	{
		line = frame[previndex];
		pli = &plinf[previndex];
	}
	l = write_playerinfo(&plinf[fnindex],pli,tmpblk);
	insert_msg(tmpblk,l);
	l = write_packetentities(frame[fnindex],line,tmpblk);
	insert_msg(tmpblk,l);
	previndex = fnindex;
}

void msg_deltapacketentities()
{
	error("doh! deltapacketentities msg encountered!");
}

void msg_frame()
{
	long seq1 = *(long*)(inptr+1);
	long seq2 = (isdemo != RECORD_SERVER)? *(long*)(inptr+5) : 0xffffffff;
	int i;

	copy_msg((isdemo != RECORD_SERVER)? 11+inptr[10] : 5);
	fnindex = (fnindex+1) % 16;
	if (seq2 == 0xffffffff)
	{
		memcpy(frame[fnindex],baseline,BASELINESIZE);
		memset(&plinf[fnindex],0,sizeof(t_playerinfo));
	}

	for (i = 0; i < 16; i++)
		if (framenum[i] == seq2)
		{
			memcpy(frame[fnindex],frame[i],BASELINESIZE);
			memcpy(&plinf[fnindex],&plinf[i],sizeof(t_playerinfo));
		}

	framenum[fnindex] = seq1;
	for (i = 1; i < MAX_EDICTS; i++)
		if (frame[fnindex][i].removed)
		{
			frame[fnindex][i].removed = 0;
			frame[fnindex][i].active = 1;
		}
}

void (*handle_message[])()
	= { msg_bad, msg_muzzleflash, msg_muzzlflash2, msg_temp_entity,
	    msg_layout, msg_inventory, msg_nop, msg_disconnect,
	    msg_reconnect, msg_sound, msg_print, msg_stufftext,
	    msg_serverdata, msg_configstring, msg_spawnbaseline,
	    msg_centerprint, msg_download, msg_playerinfo,
	    msg_packetentities, msg_deltapacketentities, msg_frame };

void process_demo(char *src, char *dst)
{
	FILE *infile, *outfile;
	signed long inlen;
	int i;

	infile = fopen(src,"rb");
	if (!infile) error("couldn't open %s",src);

	outfile = fopen(dst,"wb");
	if (!outfile) error("could not create temporary file %s",dst);

	fnindex = 0;
	previndex = -1;
	memset(baseline,0,BASELINESIZE);
	for (i = 0; i < 16; i++) memset(frame[i],0,BASELINESIZE);
	for (i = 0; i < 16; i++) framenum[i] = 0;
	memset(plinf,0,sizeof(t_playerinfo)*16);

	for (;;)
	{
		fread(&inlen,4,1,infile);
		if (inlen == -1) break;
		if (inlen > p_blocksize) error("block too long in %s",src);
		fread(inblk,inlen,1,infile);
		if (feof(infile) || ferror(infile))
			error("read error in %s",src);

		outlen = 0;
		inptr = inblk;

		while (inptr < inblk + inlen)
		{
			if (*inptr <= SVC_FRAME)
				handle_message[(int)(*inptr)]();
			else error("unknown message type in %s",src);
		}

		for (i = 0; i < MAX_EDICTS; i++) frame[fnindex][i].event = 0;

		if (!outlen) continue;
		if (outlen > p_blocksize) error("block too large in output");
		fwrite(&outlen,4,1,outfile);
		fwrite(outblk,outlen,1,outfile);
	}

	fwrite(&inlen,4,1,outfile);

	fclose(infile);
	fclose(outfile);
}

void parse_switch(char *arg)
{
	int i;
	char *c;

	if ((c = strchr(arg,'=')))
	{
		arg = strdup(arg);
		*(c = strchr(arg,'=')) = 0;;
	}

	for (i = 0; optname[i].name; i++)
		if (!strcmp(optname[i].name,arg))
		{
			if (optname[i].flag >=0) flag[optname[i].flag]++;
			if (optname[i].type == FL_WRITE) rewriting++;
			if (optname[i].type == FL_PRINT) printing++;
			if (c)
			{
				if (!optname[i].parm)
					error("switch %s needs no args",arg);
				*optname[i].parm = strdup(c+1);
				free(arg);
			}
			else if (optname[i].need_arg)
				error("switch %s needs arg",optname[i].name);
			return;
		}
	error("unknown switch %s",arg);
}

void usage()
{
    fprintf(stderr,"put your comments here\n");
    exit(1);
}

int main(int argc, char **argv)
{
	int i, j, oflag = 0;
	int fileargs = 0;
	char *newfile, *optr = NULL;
	char **files;

	if (argc < 2) usage();
 	pfile = stdout;
	for (i = 0; i < NUM_SWITCHES; flag[i++] = 0);

	for (i = 1; i < argc; i++)
		if (!strcmp(argv[i],"-o"))
			oflag = 1;
		else if (*argv[i] == '-')
			parse_switch(argv[i]);
		else if (oflag && oflag--)
			optr = argv[i];
		else
			fileargs++;

	if (!rewriting && !printing) error("nothing to do");
	if (!fileargs) error("no input files");
	if (oflag) error("-o without argument");
	if (rewriting && printing) error("incompatible switches");
	if (rewriting && optr && fileargs > 1)
		error("output file specified with more than one input file");
	sscanf(p_cblocksize,"%u",&p_blocksize);
	if (p_blocksize <= 0) error("blocksize must be non-negative");

	inblk = malloc(p_blocksize);
	tmpblk = malloc(p_blocksize);
	outblk = malloc(p_blocksize);

	baseline = malloc(BASELINESIZE);
	for (i = 0; i < 16; i++) frame[i] = malloc(BASELINESIZE);

	if (printing && optr)
	{
		pfile = fopen(optr,"w");
		if (!pfile) error("could not create %s",optr);
	}

	files = malloc(fileargs * sizeof(char*));

	for (i = 1, j = 0; i < argc; i++)
	{
		if (*argv[i] != '-') files[j++] = argv[i];
		if (!strcmp(argv[i],"-o")) i++;
	}

	for (i = 0; i < fileargs; i++)
	{
		if (rewriting && fileargs > 1)
			fprintf(pfile,"processing %s\n",files[i]);
		newfile = strdup(tempnam(".",NULL));
		process_demo(files[i],newfile);
		if (rewriting)
			if (optr)
			{
				if (rename(newfile,optr))
					error("could not create %s",optr);
			}
			else
			{
				if (rename(newfile,files[i]))
					error("warning: could not "
					      "overwrite %s",files[i]);
			}
		else
			remove(newfile);

		free(newfile);
	}

	free(baseline);
	for (i = 0; i < 16; i++) free(frame[i]);

	free(files);
	if (printing && optr) fclose(pfile);
}
