// Copyright (C) 1999-2000 Id Software, Inc.
//
// Quake III Radio Version 1.1
// by Sven Jacobs (Sven.Jacobs@gmx.de) 
//
// cg_radio.c -- client side radio stuff


#include "cg_local.h"

cgRadio_t		cg_radio[MAX_RADIO_DEF-1];
int				radioDefCount=0;
int				radioHeadClient;
qboolean		radioMenu;
int				radioMenuMode;
int				radioMenuPos;


//////////
// MAIN //
//////////


/*
===========================================
CG_IsEmpty

Checks if a string only consits of
non-alphabetical characters
===========================================
*/
qboolean CG_IsEmpty( const char *str ) {
	int			len, i;
	qboolean	empty=qtrue;

	len = strlen(str);
	
	if ( len == 0 )
		return qtrue;

	for (i = 0; i < len; i++ ) {
		if ( (str[i] >= 'a' && str[i] <= 'z') || (str[i] >= 'A' && str[i] <= 'Z') || (str[i] >= '0' && str[i] <= '9') || str[i] == '_' || str[i] == '/' || str[i] == '\\' || str[i] == '.' || str[i] == '!' || str[i] == '?' || str[i] == '.' || str[i] == ',' || str[i] == '-' ) {
			empty = qfalse;
			break;
		}
	}

	return empty;
}


/*
===================
CG_LoadRadioConfig
===================
*/
qboolean CG_LoadRadioConfig( const char *filename ) {
	char			*text_p;
	char			*token;
	char			text[20000];
	int				len;
	int				i;
	fileHandle_t	f;

	// load the file
	len = trap_FS_FOpenFile( filename, &f, FS_READ );
	if ( len <= 0 ) {
		CG_Printf( "Could not open file %s\n", filename );
		return qfalse;
	}
	if ( len >= sizeof( text ) - 1 ) {
		CG_Printf( "File %s too long\n", filename );
		return qfalse;
	}
	trap_FS_Read( text, len, f );
	text[len] = 0;
	trap_FS_FCloseFile( f );

	// parse the text
	text_p = text;
	
	// read config
	for ( i = 0; i < MAX_RADIO_DEF; i++ ) {
		token = COM_Parse( &text_p );
		if ( !token || CG_IsEmpty(token) ) {
			break;
		}
		Q_strncpyz( cg_radio[i].command, token, sizeof( cg_radio[i].command ) );

		token = COM_Parse( &text_p );
		if ( !token || CG_IsEmpty(token) ) {
			break;
		}
		Q_strncpyz( cg_radio[i].soundfile, token, sizeof( cg_radio[i].soundfile ) );

		token = COM_Parse( &text_p );
		if ( !token || CG_IsEmpty(token) ) {
			break;
		}
		Q_strncpyz( cg_radio[i].txtmsg, token, sizeof( cg_radio[i].txtmsg ) );

		//CG_Printf( "DEBUG %i: command: %s soundfile: %s txtmsg: %s\n", i, cg_radio[i].command, cg_radio[i].soundfile, cg_radio[i].txtmsg );
	}

	radioDefCount = i;
	return qtrue;
}


/*
=======================
CG_RegisterRadioSounds
=======================
*/
void CG_RegisterRadioSounds( void ) {
	int	i;

	for ( i = 0; i < radioDefCount; i++ ) {
		if ( CG_IsEmpty(cg_radio[i].command) )
			break;
		cg_radio[i].handle = trap_S_RegisterSound( cg_radio[i].soundfile );
	}
}


/*
==================================
CG_FindRadioCmd

Searches for a radio command,
returns the array number if found
==================================
*/
int CG_FindRadioCmd( const char *cmd ) {
	int j;
	int foundat; 

	foundat = -1; // return -1 if cmd not found
	for ( j = 0; j < MAX_RADIO_DEF; j++ ) {
		if ( !Q_stricmp( cg_radio[j].command, cmd ) ) {
			foundat = j;
			break;		
		}
	}
	return foundat;
}


/*
======================
CG_Radio

Play sound, draw head
======================
*/
void CG_Radio( int clientNum, const char *message ) {
	int soundnr;
	int y;

	y = 0;

	soundnr = CG_FindRadioCmd( message );
	if ( soundnr >= 0 ) {
		trap_S_StartLocalSound( cg_radio[soundnr].handle, 0 );
	}

	// show player's head and name in corner
	radioHeadClient = clientNum;
	cg.radioHeadTime = cg.time;
}


/*
======================================================
CG_SendRadioTxtMsg

Sends client's alternative radio txt msg to the
server which will then broadcast it to all
appropriate clients.
======================================================
*/
void CG_SendRadioTxtMsg( const char *message ) {
	int		txtmsgnr;
	char	command[128];
	
	txtmsgnr = CG_FindRadioCmd( message );
	if ( txtmsgnr == -1 ) 	// Client does not have this radio message?!
		Com_sprintf( command, 256, "radiotxtmsg %i %s", cg.snap->ps.clientNum, message );
	else
		Com_sprintf( command, 256, "radiotxtmsg %i %s", cg.snap->ps.clientNum, cg_radio[txtmsgnr].txtmsg );
	
	trap_SendClientCommand( command );
}


/*
===================
CG_RadioMenuSelect
===================
*/
void CG_RadioMenuSelect( int select ) {
	int		clientNum;
	char	command[256];
	char	message[128];

	if ( select < 0 || select > 9 ) {
		radioMenu = qfalse;
		return;
	}

	// prev
	if ( select == 9 ) {
		if ( radioMenuPos == 0 ) 
			return;
		radioMenuPos -= 7;
		return;
	}
	
	// next
	if ( select == 0 ) {
		if ( ( radioMenuPos + 7 ) >= radioDefCount ) 
			return;
		radioMenuPos += 7;
		return;
	}

	if ( ( radioMenuPos + (select-1) ) >= radioDefCount ) {
		radioMenuPos = 0;
		radioMenu = qfalse;
		return;
	}


	Q_strncpyz( message, cg_radio[radioMenuPos+(select-1)].command, sizeof( message ) );

	if ( radioMenuMode == 0 ) {
		Com_sprintf( command, 256, "radio %s", message );
	}
	else if ( radioMenuMode == 1 ) {
		Com_sprintf( command, 256, "radio_team %s", message );
	}
	else if ( radioMenuMode == 2 ) {
		clientNum = CG_CrosshairPlayer();
		if ( clientNum == -1 ) {
			radioMenuPos = 0;
			radioMenu = qfalse;
			return;
		}
		Com_sprintf( command, 256, "radio_tell %i %s", clientNum, message );
	}
	else {	// this should never happen!
		CG_Printf("CG_RadioMenuSelect: UNKNOWN radioMenuMode");
	}

	CG_SendRadioTxtMsg( message );
	trap_SendClientCommand( command );

	// resetting menu
	radioMenuPos = 0;
	radioMenu = qfalse;
}



//////////
// DRAW //
//////////

/*
=================
CG_DrawRadioHead
=================
*/
float CG_DrawRadioHead( float y, int clientNum ) {
	int			t;
	float		size;
	vec3_t		angles;
	const char	*info;
	const char	*name;

	if ( cg.predictedPlayerState.stats[STAT_HEALTH] <= 0 ) {
		return y;
	}

	if ( !cg.radioHeadTime ) {
		return y;
	}

	t = cg.time - cg.radioHeadTime;
	if ( t > RADIO_HEAD_TIME ) {
		cg.radioHeadTime = 0;
		return y;
	}

	if ( clientNum < 0 || clientNum >= MAX_CLIENTS ) {
		return y;
	}

	size = ICON_SIZE * 1.25;

	// draw exclamation mark next to head
	CG_DrawPic( ((640 - size) - 32), y, 32, 64, cgs.media.radioExclamation );

	angles[PITCH] = 0;
	angles[YAW] = 180;
	angles[ROLL] = 0;
	CG_DrawHead( 640 - size, y, size, size, clientNum, angles );
		
	info = CG_ConfigString( CS_PLAYERS + clientNum );
	name = Info_ValueForKey(  info, "n" );
	y += size;
	CG_DrawBigString( 640 - ( Q_PrintStrlen( name ) * BIGCHAR_WIDTH), y, name, 0.5 );

	return y + BIGCHAR_HEIGHT + 2;
}


/*
=================
CG_DrawRadioMenu
=================
*/
void CG_DrawRadioMenu(void) {
	char	*s;
	char	*mode;
	int		i,j,x,y;
	
	if (radioMenu == qfalse)
		return;
	
	x = 4;
	y = 58;

	if ( radioMenuMode == 0 )
		mode = "Radio All\0";
	else if ( radioMenuMode == 1 )
		mode = "Radio Team\0";
	else if ( radioMenuMode == 2 )
		mode = "Radio Tell\0";
	else
		mode = "UNKNOWN MODE\0";

	CG_DrawSmallString( x, y, mode, 1.0F );
	y += 2*SMALLCHAR_HEIGHT;

	for ( i = 1; i < 9; i++ ) {
	
		if ((radioMenuPos + (i-1)) >= radioDefCount)
			break;
				
		s = va("%i) %s", i, cg_radio[radioMenuPos+(i-1)].txtmsg );
		CG_DrawSmallString( 2*x, y, s, 1.0F );
		y += SMALLCHAR_HEIGHT;
	}
	
	
	for ( j = i; j < 10; j++ )
		y += SMALLCHAR_HEIGHT;	
	
	// disable button
	if ( radioMenuPos == 0 ) {
		CG_DrawSmallString( 2*x, y, "9) Previous", 0.5F );
		y += SMALLCHAR_HEIGHT;
	}
	// show button
	else {
		CG_DrawSmallString( 2*x, y, "9) Previous", 1.0F );
		y += SMALLCHAR_HEIGHT;
	}
	
	// disable button
	if ( ( radioMenuPos + 7 ) >= radioDefCount ) {
		CG_DrawSmallString( 2*x, y, "0) Next", 0.5F );
		y += SMALLCHAR_HEIGHT;		
	}
	// show button
	else {
		CG_DrawSmallString( 2*x, y, "0) Next", 1.0F );
		y += SMALLCHAR_HEIGHT;
	}
}


//////////////////////
// CONSOLE COMMANDS //
//////////////////////

// Prepare radio txt msg before passing command to server
void CG_Radio_f( void ) {
	char	command[256];
	char	message[128];

	trap_Args( message, 128 );
	message[strlen(message)-1]='\0';	// clean the string

	// convert to lower cases
	Q_strlwr( message );

	Com_sprintf( command, 256, "radio %s", message );
	CG_SendRadioTxtMsg( message );
	trap_SendClientCommand( command );
}


// Prepare radio txt msg before passing command to server
void CG_RadioTeam_f( void ) {
	char	command[128];
	char	message[128];

	trap_Args( message, 128 );
	message[strlen(message)-1]='\0';	// clean the string

	// convert to lower cases
	Q_strlwr( message );
	
	Com_sprintf( command, 256, "radio_team %s", message );
	CG_SendRadioTxtMsg( message );
	trap_SendClientCommand( command );
}


void CG_RadioTellTarget_f( void ) {
	int		clientNum;
	char	command[128];
	char	message[128];

	clientNum = CG_CrosshairPlayer();
	if ( clientNum == -1 ) {
		return;
	}

	trap_Args( message, 128 );
	message[strlen(message)-1]='\0';	// clean the string

	// convert to lower cases
	Q_strlwr( message );

	Com_sprintf( command, 256, "radio_tell %i %s", clientNum, message );
	CG_SendRadioTxtMsg( message );
	trap_SendClientCommand( command );
}


void CG_ReloadRadioConfig_f( void ) {
	qboolean loaded;

	// resetting menu
	radioMenuPos = 0;

	// clear all
	memset( cg_radio, 0, sizeof(cg_radio) );

	// load config, cache sounds
	loaded = CG_LoadRadioConfig( cg_radioCfg.string );
	if ( loaded )	// only cache sounds if file could be loaded...
		CG_RegisterRadioSounds();
}


void CG_RadioMenu_f( void ) {
	char	mode[128];
	int		oldmode;
	
	trap_Args( mode, 128 );
	mode[strlen(mode)-1]='\0';	// clean the string

	// convert to lower cases
	Q_strlwr( mode );
	
	oldmode = radioMenuMode;

	if ( !Q_stricmp( mode, "all" ) )
		radioMenuMode = 0;
	else if ( !Q_stricmp( mode, "team" ) )
		radioMenuMode = 1;
	else if ( !Q_stricmp( mode, "tell" ) )
		radioMenuMode = 2;
	else
		return;


	if ( radioMenu == qfalse ) {
		radioMenu = qtrue;
		return;
	}
	else {
		radioMenuPos = 0;
		if ( oldmode == radioMenuMode )
			radioMenu = qfalse;
	}
}
