#ifndef GAMEX86_IMPORT_HEADER
#define GAMEX86_IMPORT_HEADER


#include <stdlib.h>	// for rand()
#include <math.h>	// for fabs()
#include "Defines.h"

#define DLLIMPORT __declspec(dllimport)

// shared.h

typedef float vec_t;
typedef vec_t vec3_t[3];
typedef int qboolean;
typedef unsigned char byte;

enum damage_t
{
	DAMAGE_NO,
	DAMAGE_YES,			// will take damage if hit
	DAMAGE_AIM			// auto targeting recognizes this
};

enum movetype_t
{
MOVETYPE_NONE,			// never moves
MOVETYPE_NOCLIP,		// origin and angles change with no interaction
MOVETYPE_PUSH,			// no clip to world, push on box contact
MOVETYPE_STOP,			// no clip to world, stops on box contact

MOVETYPE_WALK,			// gravity
MOVETYPE_STEP,			// gravity, special edge handling
MOVETYPE_FLY,
MOVETYPE_TOSS,			// gravity
MOVETYPE_FLYMISSILE,	// extra size to monsters
MOVETYPE_BOUNCE
};

class CPlane
{
public:
	vec3_t	normal;
	float	dist;
	byte	type;			// for fast side tests
	byte	signbits;		// signx + (signy<<1) + (signz<<1)
	byte	pad[2];
};

class CUserCommand
{
public:
	byte	msec;
	byte	buttons;
	short	angles[3];
	short	forwardmove, sidemove, upmove;
	byte	impulse;		// remove?
	byte	lightlevel;		// light level the player is standing on
};
class CLink { public: CLink	*prev, *next; };

class CEntityState
{
public:

	int		number;			// edict index
	vec3_t		origin;
	vec3_t		angles;
	vec3_t		old_origin;		// for lerping
	int		modelindex;
	int		modelindex2, modelindex3, modelindex4;	// weapons, CTF flags, etc
	int		frame;
	int		skinnum;
	int		effects;
	int		renderfx;
	int		solid;			// for client side prediction, 8*(bits 0-4) is x/y radius
						// 8*(bits 5-9) is z down distance, 8(bits10-15) is z up
						// gi.linkentity sets this properly
	int		sound;			// for looping sounds, to guarantee shutoff
	int		event;			// impulse events -- muzzle flashes, footsteps, etc
						// events only go out for a single frame, they
						// are automatically cleared each frame
};

enum pmtype_t
{
	// can accelerate and turn
	PM_NORMAL,
	PM_SPECTATOR,
	// no acceleration or turning
	PM_DEAD,
	PM_GIB,		// different bounding box
	PM_FREEZE
};
class CPerdictionMoveState
{
public:

	pmtype_t	pm_type;
	short		origin[3];		// 12.3
	short		velocity[3];		// 12.3
	byte		pm_flags;		// ducked, jump_held, etc
	byte		pm_time;		// each unit = 8 ms
	short		gravity;
	short		delta_angles[3];	// add to command angles to get view direction
						// changed by spawns, rotating objects, and teleporters
};
#define	MAX_STATS				32
class CPlayerState
{
public:

	CPerdictionMoveState	pmove;			// for prediction
	// these fields do not need to be communicated bit-precise
	vec3_t					viewangles;		// for fixed views
	vec3_t					viewoffset;		// add to pmovestate->origin
	vec3_t					kick_angles;	// add to view direction to get render angles
											// set by weapon kicks, pain effects, etc
	vec3_t					gunangles;
	vec3_t					gunoffset;
	int						gunindex;
	int						gunframe;
	float					blend[4];		// rgba full screen effect
	float					fov;			// horizontal field of view
	int						rdflags;		// refdef flags
	short					stats[MAX_STATS];	// fast status bar updates
};
class CEdict;
class CGameItem
{
public:

	char		*classname;	// spawning name
	qboolean	(*pickup)(CEdict *ent, CEdict *other);
	void		(*use)(CEdict *ent, CGameItem *item);
	void		(*drop)(CEdict *ent, CGameItem *item);
	void		(*weaponthink)(CEdict *ent);
	char		*pickup_sound;
	char		*world_model;
	int			world_model_flags;
	char		*view_model;
	// client side info
	char		*icon;
	char		*pickup_name;			// for printing on pickup
	int			count_width;			// number of digits to display by icon
	int			quantity;				// for ammo how much, for weapons how much is used per shot
	char		*ammo;					// for weapons
	int			flags;					// IT_* flags
	void		*info;
	int			tag;
	char		*precaches;				// string of all models, sounds, and images this item will use
};
#define	MAX_INFO_STRING		512
#define	MAX_ITEMS			256
class CGameClientPersistant
{
public:

	char		userinfo[MAX_INFO_STRING];
	char		netname[16];
	int			hand;
	qboolean	connected;			// a loadgame will leave valid entities that
							// just don't have a connection yet
	// values saved and restored from edicts when changing levels
	int		health;
	int		max_health;
	qboolean	powerArmorActive;
	int		selected_item;
	int		inventory[MAX_ITEMS];
	// ammo capacities
	int		max_bullets;
	int		max_shells;
	int		max_rockets;
	int		max_grenades;
	int		max_cells;
	int		max_slugs;
	CGameItem		*weapon;
	CGameItem		*lastweapon;
	int		power_cubes;			// used for tracking the cubes in coop games
	int		score;				// for calculating total unit score in coop games
};
class CGameClientRespawn
{
public:

	CGameClientPersistant	coop_respawn;		// what to set client->pers to on a respawn
	int						enterframe;			// level.framenum the client entered the game
	int						score;				// frags, etc
	vec3_t					cmd_angles;			// angles sent over in the last command
	int						game_helpchanged;
	int						helpchanged;
};

class CGameItem;

enum weaponstate_t
{
	WEAPON_READY, 
	WEAPON_ACTIVATING,
	WEAPON_DROPPING,
	WEAPON_FIRING
};

class CGameClient
{
public:

	// known to server
	CPlayerState			ps;		// communicated by server to clients
	int						ping;
	// private to game
	CGameClientPersistant	pers;
	CGameClientRespawn		resp;
	CPerdictionMoveState	old_pmove;	// for detecting out-of-pmove changes
	qboolean				showscores;	// set layout stat
	qboolean				showinventory;	// set layout stat
	qboolean				showhelp;
	qboolean				showhelpicon;
	int						ammo_index;
	int						buttons;
	int						oldbuttons;
	int						latched_buttons;
	qboolean				weapon_thunk;
	CGameItem				*newweapon;
	// sum up damage over an entire frame, so
	// shotgun blasts give a single big kick
	int						damage_armor;		// damage absorbed by armor
	int						damage_parmor;		// damage absorbed by power armor
	int						damage_blood;		// damage taken out of health
	int						damage_knockback;	// impact damage
	vec3_t					damage_from;		// origin for vector calculation
	float					killer_yaw;		// when dead, look at killer
	weaponstate_t			weaponstate;
	vec3_t					kick_angles;		// weapon kicks
	vec3_t					kick_origin;
	float					v_dmg_roll, v_dmg_pitch, v_dmg_time;	// damage kicks
	float					fall_time, fall_value;	// for view drop on fall
	float					damage_alpha;
	float					bonus_alpha;
	vec3_t					damage_blend;
	vec3_t					v_angle;		// aiming direction
	float					bobtime;		// so off-ground doesn't change it
	vec3_t					oldviewangles;
	vec3_t					oldvelocity;
	float					next_drown_time;
	int						old_waterlevel;
	int						breather_sound;
	int						machinegun_shots;	// for weapon raising
	// animation vars
	int						anim_end;
	int						anim_priority;
	qboolean				anim_duck;
	qboolean				anim_run;
	// powerup timers
	float					quad_framenum;
	float					invincible_framenum;
	float					breather_framenum;
	float					enviro_framenum;
	qboolean				grenade_blew_up;
	float					grenade_time;
	int						silencer_shots;
	int						weapon_sound;
	float					pickup_msg_time;
	float					respawn_time;		// can respawn when time > this
};
class CGameLocals
{
public:

	char		helpmessage1[512];
	char		helpmessage2[512];
	int		helpchanged;		// flash F1 icon if non 0, play sound
						// and increment only if 1, 2, or 3
	CGameClient	*clients;		// [maxclients]
	// can't store spawnpoint in level, because
	// it would get overwritten by the savegame restore
	char		spawnpoint[512];	// needed for coop respawns
	// store latched cvars here that we want to get at often
	int		maxclients;
	int		maxentities;
	// cross level triggers
	int		serverflags;
	// items
	int		num_items;
	qboolean	autosaved;
};
class CConsoleVariable
{
public:

	char			*name;
	char			*string;
	char			*latched_string;	// for CVAR_LATCH vars
	int			flags;
	qboolean		modified;		// set each time the cvar is changed
	float			value;
	CConsoleVariable  	*next;
};
class CSurface
{
public:

	char	name[16];
	int		flags;
	int		value;
};
class CTrace
{
public:

	qboolean	allsolid;	// if true, plane is not valid
	qboolean	startsolid;	// if true, the initial point was in a solid area
	float		fraction;	// time completed, 1.0 = didn't hit anything
	vec3_t		endpos;		// final position
	CPlane		plane;		// surface normal at impact
	CSurface	*surface;	// surface hit
	int			contents;	// contents on other side of surface hit
	CEdict		*ent;		// not set by CM_*() functions
};
#define	MAXTOUCH	32
class CPerdictionMove
{
public:

	// state (in / out)
	CPerdictionMoveState		s;
	// command (in)
	CUserCommand				cmd;
	qboolean					snapinitial;	// if s has been changed outside pmove
	// results (out)
	int							numtouch;
	CEdict						*touchents[MAXTOUCH];
	vec3_t						viewangles;			// clamped
	float						viewheight;
	vec3_t						mins, maxs;			// bounding box size
	CEdict						*groundentity;
	int							watertype;
	int							waterlevel;
	// callbacks to test the world
	CTrace						(*trace) (vec3_t start, vec3_t mins, vec3_t maxs, vec3_t end);
	int							(*pointcontents) (vec3_t point);
};
enum multicast_t
{
MULTICAST_ALL,
MULTICAST_PHS,
MULTICAST_PVS,
MULTICAST_ALL_R,
MULTICAST_PHS_R,
MULTICAST_PVS_R
};
class CGameImport
{
public:

	// special messages
	void	(*bprintf) (int printlevel, char *fmt, ...);
	void	(*dprintf) (char *fmt, ...);
	void	(*cprintf) (CEdict *ent, int printlevel, char *fmt, ...);
	void	(*centerprintf) (CEdict *ent, char *fmt, ...);
	void	(*sound) (CEdict *ent, int channel, int soundindex, float volume, float attenuation, float timeofs);
	void	(*positioned_sound) (vec3_t origin, CEdict *ent, int channel, int soundinedex, float volume, float attenuation, float timeofs);
	// config strings hold all the index strings, the lightstyles,
	// and misc data like the sky definition and cdtrack.
	// All of the current configstrings are sent to clients when
	// they connect, and changes are sent to all connected clients.
	void	(*configstring) (int num, char *string);
	void	(*error) (char *fmt, ...);
	// the *index functions create configstrings and some internal server state
	int	(*modelindex) (char *name);
	int	(*soundindex) (char *name);
	int	(*imageindex) (char *name);
	void	(*setmodel) (CEdict *ent, char *name);
	// collision detection
	CTrace		(*trace) (vec3_t start, vec3_t mins, vec3_t maxs, vec3_t end, CEdict *passent, int contentmask);
	int		(*pointcontents) (vec3_t point);
	qboolean	(*inPVS) (vec3_t p1, vec3_t p2);
	qboolean	(*inPHS) (vec3_t p1, vec3_t p2);
	void		(*SetAreaPortalState) (int portalnum, qboolean open);
	qboolean	(*AreasConnected) (int area1, int area2);
	// an entity will never be sent to a client or used for collision
	// if it is not passed to linkentity.  If the size, position, or
	// solidity changes, it must be relinked.
	void	(*linkentity) (CEdict *ent);
	void	(*unlinkentity) (CEdict *ent);		// call before removing an interactive edict
	int		(*BoxEdicts) (vec3_t mins, vec3_t maxs, CEdict **list,	int maxcount, int areatype);
	void	(*Pmove) (CPerdictionMove *pmove);		// player movement code common with client prediction
	// network messaging
	void	(*multicast) (vec3_t origin, multicast_t to);
	void	(*unicast) (CEdict *ent, qboolean reliable);
	void	(*WriteChar) (int c);
	void	(*WriteByte) (int c);
	void	(*WriteShort) (int c);
	void	(*WriteLong) (int c);
	void	(*WriteFloat) (float f);
	void	(*WriteString) (char *s);
	void	(*WritePosition) (vec3_t pos);	// some fractional bits
	void	(*WriteDir) (vec3_t pos);		// single byte encoded, very coarse
	void	(*WriteAngle) (float f);
	// managed memory allocation
	void	*(*TagMalloc) (int size, int tag);
	void	(*TagFree) (void *block);
	void	(*FreeTags) (int tag);
	// console variable interaction
	CConsoleVariable	*(*cvar) (char *var_name, char *value, int flags);
	CConsoleVariable	*(*cvar_set) (char *var_name, char *value);
	CConsoleVariable	*(*cvar_forceset) (char *var_name, char *value);
	// ClientCommand and ServerCommand parameter access
	int	(*argc) (void);
	char	*(*argv) (int n);
	char	*(*args) (void);	// concatenation of all argv >= 1
	// add commands to the server console as if they were typed in
	// for map changing, etc
	void	(*AddCommandString) (char *text);
	void	(*DebugGraph) (float value, int color);
};
class CMonsterFrame
{
public:
	void	(*aifunc)(CEdict *self, float dist);
	float	dist;
	void	(*thinkfunc)(CEdict *self);
};

#define	MAX_ENT_CLUSTERS	16
enum solid_t
{
SOLID_NOT,			// no interaction with other objects
SOLID_TRIGGER,		// only touch when inside, after moving
SOLID_BBOX,			// touch on edge
SOLID_BSP			// bsp clip, touch on edge
};

class CMoveInfo
{
public:

	// fixed data
	vec3_t		start_origin;
	vec3_t		start_angles;
	vec3_t		end_origin;
	vec3_t		end_angles;
	int			sound_start;
	int			sound_middle;
	int			sound_end;
	float		accel;
	float		speed;
	float		decel;
	float		distance;
	float		wait;
	// state data
	int			state;
	vec3_t		dir;
	float		current_speed;
	float		move_speed;
	float		next_speed;
	float		remaining_distance;
	float		decel_distance;
	void		(*endfunc)(CEdict *);
};
class CMonsterMove
{
public:

	int				firstframe;
	int				lastframe;
	CMonsterFrame 	*frame;
	void			(*endfunc)(CEdict *self);
};
class CMonsterInfo
{
public:

	CMonsterMove	*currentmove;
	int				aiflags;
	int				nextframe;
	float			scale;
	void			(*stand)(CEdict *self);
	void			(*idle)(CEdict *self);
	void			(*search)(CEdict *self);
	void			(*walk)(CEdict *self);
	void			(*run)(CEdict *self);
	void			(*dodge)(CEdict *self, CEdict *other, float eta);
	void			(*attack)(CEdict *self);
	void			(*melee)(CEdict *self);
	void			(*sight)(CEdict *self, CEdict *other);
	qboolean		(*checkattack)(CEdict *self);
	float			pausetime;
	float			attack_finished;
	vec3_t			saved_goal;
	float			search_time;
	float			trail_time;
	vec3_t			last_sighting;
	int				attack_state;
	int				lefty;
	float			idle_time;
	int				linkcount;
	int				power_armor_type;
	int				power_armor_power;
};

#define	MAX_QPATH			64		// max length of a quake game pathname
#define	MAX_OSPATH			128		// max length of a filesystem pathname
class CLevelLocals
{
public:

	int			framenum;
	float		time;
	char		level_name[MAX_QPATH];	// the descriptive name (Outer Base, etc)
	char		mapname[MAX_QPATH];	// the server name (base1, etc)
	char		nextmap[MAX_QPATH];	// go here when fraglimit is hit
	// intermission state
	float		intermissiontime;	// time the intermission was started
	char		*changemap;
	int			exitintermission;
	vec3_t		intermission_origin;
	vec3_t		intermission_angle;
	CEdict		*sight_client;		// changed once each frame for coop games
	CEdict		*sight_entity;
	int			sight_entity_framenum;
	CEdict		*sound_entity;
	int			sound_entity_framenum;
	CEdict		*sound2_entity;
	int			sound2_entity_framenum;
	int			pic_health;
	int			total_secrets;
	int			found_secrets;
	int			total_goals;
	int			found_goals;
	int			total_monsters;
	int			killed_monsters;
	CEdict		*current_entity;	// entity running from G_RunFrame
	int			body_que;		// dead bodies
	int			power_cubes;		// ugly necessity for coop
};
class CEdict
{

	public:

	CEntityState	s;
	CGameClient		*client;	// NULL if not a player
									// the server expects the first part
									// of CGameClient to be a player_state_t
									// but the rest of it is opaque

	qboolean	inuse;
	int			linkcount;

	// FIXME: move these fields to a server private sv_entity_t
	CLink		area;				// linked to a division node or leaf
	
	int			num_clusters;		// if -1, use headnode instead
	int			clusternums[MAX_ENT_CLUSTERS];
	int			headnode;			// unused if num_clusters != -1
	int			areanum, areanum2;

	//================================

	int			svflags;
	vec3_t		mins, maxs;
	vec3_t		absmin, absmax, size;
	solid_t		solid;
	int			clipmask;
	CEdict		*owner;


	// DO NOT MODIFY ANYTHING ABOVE THIS, THE SERVER
	// EXPECTS THE FIELDS IN THAT ORDER!

	//================================
	int			movetype;
	int			flags;

	char		*model;
	float		freetime;			// sv.time when the object was freed
	
	//
	// only used locally in game, not by server
	//
	char		*message;
	char		*classname;
	int			spawnflags;

	float		timestamp;

	float		angle;			// set in qe3, -1 = up, -2 = down
	char		*target;
	char		*targetname;
	char		*killtarget;
	char		*team;
	char		*pathtarget;
	char		*deathtarget;
	char		*combattarget;
	CEdict		*target_ent;

	float		speed, accel, decel;
	vec3_t		movedir;
	vec3_t		pos1, pos2;

	vec3_t		velocity;
	vec3_t		avelocity;
	int			mass;
	float		air_finished;
	float		gravity;		// per entity gravity multiplier (1.0 is normal)
								// use for lowgrav artifact, flares

	CEdict		*goalentity;
	CEdict		*movetarget;
	float		yaw_speed;
	float		ideal_yaw;

	float		nextthink;
	void		(*prethink) (CEdict *ent);
	void		(*think)(CEdict *self);
	void		(*blocked)(CEdict *self, CEdict *other);	//move to moveinfo?
	void		(*touch)(CEdict *self, CEdict *other, CPlane *plane, CSurface *surf);
	void		(*use)(CEdict *self, CEdict *other, CEdict *activator);
	void		(*pain)(CEdict *self, CEdict *other, float kick, int damage);
	void		(*die)(CEdict *self, CEdict *inflictor, CEdict *attacker, int damage, vec3_t point);

	float		touch_debounce_time;		// are all these legit?  do we need more/less of them?
	float		pain_debounce_time;
	float		damage_debounce_time;
	float		fly_sound_debounce_time;	//move to clientinfo
	float		last_move_time;

	int			health;
	int			max_health;
	int			gib_health;
	int			deadflag;
	qboolean	show_hostile;

	float		powerarmor_time;

	char		*map;			// target_changelevel

	int			viewheight;		// height above origin where eyesight is determined
	int			takedamage;
	int			dmg;
	int			radius_dmg;
	float		dmg_radius;
	int			sounds;			//make this a spawntemp var?
	int			count;

	CEdict		*chain;
	CEdict		*enemy;
	CEdict		*oldenemy;
	CEdict		*activator;
	CEdict		*groundentity;
	int			groundentity_linkcount;
	CEdict		*teamchain;
	CEdict		*teammaster;

	CEdict		*mynoise;		// can go in client only
	CEdict		*mynoise2;

	int			noise_index;
	int			noise_index2;
	float		volume;
	float		attenuation;

	// timing variables
	float		wait;
	float		delay;			// before firing targets
	float		random;

	float		teleport_time;

	int			watertype;
	int			waterlevel;

	vec3_t		move_origin;
	vec3_t		move_angles;

	// move this to clientinfo?
	int			light_level;

	int			style;			// also used as areaportal number

	CGameItem		*item;			// for bonus items

	// common data blocks
	CMoveInfo		moveinfo;
	CMonsterInfo	monsterinfo;
};
class CGameExport
{
public:

	int		apiversion;
	// the init function will only be called when a game starts,
	// not each time a level is loaded.  Persistant data for clients
	// and the server can be allocated in init
	void		(*Init) (void);
	void		(*Shutdown) (void);
	// each new level entered will cause a call to SpawnEntities
	void		(*SpawnEntities) (char *mapname, char *entstring, char *spawnpoint);
	// Read/Write Game is for storing persistant cross level information
	// about the world state and the clients.
	// WriteGame is called every time a level is exited.
	// ReadGame is called on a loadgame.
	void		(*WriteGame) (char *filename, qboolean autosave);
	void		(*ReadGame) (char *filename);
	// ReadLevel is called after the default map information has been
	// loaded with SpawnEntities
	void		(*WriteLevel) (char *filename);
	void		(*ReadLevel) (char *filename);
	qboolean	(*ClientConnect) (CEdict *ent, char *userinfo);
	void		(*ClientBegin) (CEdict *ent);
	void		(*ClientUserinfoChanged) (CEdict *ent, char *userinfo);
	void		(*ClientDisconnect) (CEdict *ent);
	void		(*ClientCommand) (CEdict *ent);
	void		(*ClientThink) (CEdict *ent, CUserCommand *cmd);
	void		(*RunFrame) (void);
	// ServerCommand will be called when an "sv <command>" command is issued on the
	// server console.
	// The game can issue gi.argc() / gi.argv() commands to get the rest
	// of the parameters
	void		(*ServerCommand) (void);
	//
	// global variables shared between game and server
	//
	// The edict array is allocated in the game dll so it
	// can vary in size from one game to another.
	// 
	// The size will be fixed when ge->Init() is called
	CEdict		*edicts;
	int			edict_size;
	int			num_edicts;		// current number, <= max_edicts
	int			max_edicts;
};
class CSpawn
{
	public:
		char	*name;
		void	(*spawn)(CEdict *ent);
};
class CSpawnTemp
{
public:

	// world vars
	char		*sky;
	float		skyrotate;
	vec3_t		skyaxis;
	char		*nextmap;
	int		lip;
	int		distance;
	int		height;
	char		*noise;
	float		pausetime;
	char		*item;
	char		*gravity;
	float		minyaw;
	float		maxyaw;
	float		minpitch;
	float		maxpitch;
};
////////////////////////////////////////////////////////
// IMPORTS FROM SHARED.H

void	DLLIMPORT	VectorMA (vec3_t veca, float scale, vec3_t vecb, vec3_t vecc);

vec_t	DLLIMPORT	_DotProduct (vec3_t v1, vec3_t v2);
void	DLLIMPORT	_VectorSubtract (vec3_t veca, vec3_t vecb, vec3_t out);
void	DLLIMPORT	_VectorAdd (vec3_t veca, vec3_t vecb, vec3_t out);
void	DLLIMPORT	_VectorCopy (vec3_t in, vec3_t out);

void	DLLIMPORT	ClearBounds (vec3_t mins, vec3_t maxs);
void	DLLIMPORT	AddPointToBounds (vec3_t v, vec3_t mins, vec3_t maxs);
int		DLLIMPORT	VectorCompare (vec3_t v1, vec3_t v2);
vec_t	DLLIMPORT	VectorLength (vec3_t v);
void	DLLIMPORT	CrossProduct (vec3_t v1, vec3_t v2, vec3_t cross);
vec_t	DLLIMPORT	VectorNormalize (vec3_t v);		// returns vector length
vec_t	DLLIMPORT	VectorNormalize2 (vec3_t v, vec3_t out);
void	DLLIMPORT	VectorInverse (vec3_t v);
void	DLLIMPORT	VectorScale (vec3_t in, vec_t scale, vec3_t out);
int		DLLIMPORT	Q_log2(int val);

void	DLLIMPORT	R_ConcatRotations (float in1[3][3], float in2[3][3], float out[3][3]);
void	DLLIMPORT	R_ConcatTransforms (float in1[3][4], float in2[3][4], float out[3][4]);

void	DLLIMPORT	AngleVectors (vec3_t angles, vec3_t forward, vec3_t right, vec3_t up);
int		DLLIMPORT	BoxOnPlaneSide (vec3_t emins, vec3_t emaxs, CPlane *plane);
float	DLLIMPORT	anglemod(float a);
float	DLLIMPORT	LerpAngle (float a1, float a2, float frac);

void	DLLIMPORT	ProjectPointOnPlane( vec3_t dst, const vec3_t p, const vec3_t normal );
void	DLLIMPORT	PerpendicularVector( vec3_t dst, const vec3_t src );
void	DLLIMPORT	RotatePointAroundVector( vec3_t dst, const vec3_t dir, const vec3_t point, float degrees );


//=============================================

char*	DLLIMPORT	COM_SkipPath (char *pathname);
void	DLLIMPORT	COM_StripExtension (char *in, char *out);
void	DLLIMPORT	COM_FileBase (char *in, char *out);
void	DLLIMPORT	COM_FilePath (char *in, char *out);
void	DLLIMPORT	COM_DefaultExtension (char *path, char *extension);

char*	DLLIMPORT	COM_Parse (char **data_p);

void	DLLIMPORT	Com_sprintf (char *dest, int size, char *fmt, ...);

void	DLLIMPORT	Com_PageInMemory (byte *buffer, int size);

int		DLLIMPORT	Q_stricmp (char *s1, char *s2);
int		DLLIMPORT	Q_strcasecmp (char *s1, char *s2);
int		DLLIMPORT	Q_strncasecmp (char *s1, char *s2, int n);

short	DLLIMPORT	BigShort(short l);
short	DLLIMPORT	LittleShort(short l);
int		DLLIMPORT	BigLong (int l);
int		DLLIMPORT	LittleLong (int l);
float	DLLIMPORT	BigFloat (float l);
float	DLLIMPORT	LittleFloat (float l);

void	DLLIMPORT	Swap_Init (void);
char*	DLLIMPORT	va(char *format, ...);

char*	DLLIMPORT	Info_ValueForKey (char *s, char *key);
void	DLLIMPORT	Info_RemoveKey (char *s, char *key);
void	DLLIMPORT	Info_SetValueForKey (char *s, char *key, char *value);
qboolean DLLIMPORT	Info_Validate (char *s);

void	DLLIMPORT	Sys_Error (char *error, ...);
void	DLLIMPORT	Com_Printf (char *msg, ...);

///////////////////////////////////////////////////////////
// g_ai.c

void	DLLIMPORT	AI_SetSightClient (void);

void	DLLIMPORT	ai_stand (CEdict *self, float dist);
void	DLLIMPORT	ai_move (CEdict *self, float dist);
void	DLLIMPORT	ai_walk (CEdict *self, float dist);
void	DLLIMPORT	ai_turn (CEdict *self, float dist);
void	DLLIMPORT	ai_run (CEdict *self, float dist);
void	DLLIMPORT	ai_charge (CEdict *self, float dist);
int		DLLIMPORT	range (CEdict *self, CEdict *other);

void	DLLIMPORT	FoundTarget (CEdict *self);
qboolean DLLIMPORT	infront (CEdict *self, CEdict *other);
qboolean DLLIMPORT	visible (CEdict *self, CEdict *other);
qboolean DLLIMPORT	FacingIdeal(CEdict *self);

///////////////////////////////////////////////////////////
// g_utils.c
void	DLLIMPORT	G_ProjectSource (vec3_t point, vec3_t distance, vec3_t forward, vec3_t right, vec3_t result);
qboolean DLLIMPORT	KillBox (CEdict *ent);
CEdict*	DLLIMPORT	G_Find (CEdict *from, int fieldofs, char *match);
CEdict*	DLLIMPORT	findradius (CEdict *from, vec3_t org, float rad);
CEdict*	DLLIMPORT	G_PickTarget (char *targetname);
void	DLLIMPORT	G_UseTargets (CEdict *ent, CEdict *activator);
void	DLLIMPORT	G_SetMovedir (vec3_t angles, vec3_t movedir);
void	DLLIMPORT	G_InitEdict (CEdict *e);
CEdict*	DLLIMPORT	G_Spawn (void);
void	DLLIMPORT	G_FreeEdict (CEdict *e);
void	DLLIMPORT	G_TouchTriggers (CEdict *ent);
void	DLLIMPORT	G_TouchSolids (CEdict *ent);
char*	DLLIMPORT	G_CopyString (char *in);
float*	DLLIMPORT	tv (float x, float y, float z);
char*	DLLIMPORT	vtos (vec3_t v);
float	DLLIMPORT	vectoyaw (vec3_t vec);
void	DLLIMPORT	vectoangles (vec3_t vec, vec3_t angles);

///////////////////////////////////////////////////////////////
// g_items.c

void	DLLIMPORT	PrecacheItem (CGameItem *it);
void	DLLIMPORT	InitItems (void);
void	DLLIMPORT	SetItemNames (void);
CGameItem*	DLLIMPORT	FindItem (char *pickup_name);
CGameItem*	DLLIMPORT	FindItemByClassname (char *classname);
CEdict*	DLLIMPORT	Drop_Item (CEdict *ent, CGameItem *item);
void	DLLIMPORT	SetRespawn (CEdict *ent, float delay);
void	DLLIMPORT	ChangeWeapon (CEdict *ent);
void	DLLIMPORT	SpawnItem (CEdict *ent, CGameItem *item);
void	DLLIMPORT	Think_Weapon (CEdict *ent);
int		DLLIMPORT	ArmorIndex (CEdict *ent);
int		DLLIMPORT	PowerArmorType (CEdict *ent);
CGameItem*	DLLIMPORT	GetItemByIndex (int index);
qboolean	DLLIMPORT	Add_Ammo (CEdict *ent, CGameItem *item, int count);
void	DLLIMPORT	Touch_Item (CEdict *ent, CEdict *other, CPlane *plane, CSurface *surf);

//////////////////////////////////////////////////
// g_cmds.c

void	DLLIMPORT	Cmd_Help_f (CEdict *ent);
void	DLLIMPORT	Cmd_Score_f (CEdict *ent);

//////////////////////////////////////////////////////
// g_combat.c

qboolean	DLLIMPORT	OnSameTeam (CEdict *ent1, CEdict *ent2);
qboolean	DLLIMPORT	CanDamage (CEdict *targ, CEdict *inflictor);
void	DLLIMPORT	T_Damage (CEdict *targ, CEdict *inflictor, CEdict *attacker, vec3_t dir, vec3_t point, vec3_t normal, int damage, int knockback, int dflags, int mod);
void	DLLIMPORT	T_RadiusDamage (CEdict *inflictor, CEdict *attacker, float damage, CEdict *ignore, float radius, int mod);

///////////////////////////////////////////////////////
// g_monster.c

void	DLLIMPORT	monster_fire_bullet (CEdict *self, vec3_t start, vec3_t dir, int damage, int kick, int hspread, int vspread, int flashtype);
void	DLLIMPORT	monster_fire_shotgun (CEdict *self, vec3_t start, vec3_t aimdir, int damage, int kick, int hspread, int vspread, int count, int flashtype);
void	DLLIMPORT	monster_fire_blaster (CEdict *self, vec3_t start, vec3_t dir, int damage, int speed, int flashtype, int effect);
void	DLLIMPORT	monster_fire_grenade (CEdict *self, vec3_t start, vec3_t aimdir, int damage, int speed, int flashtype);
void	DLLIMPORT	monster_fire_rocket (CEdict *self, vec3_t start, vec3_t dir, int damage, int speed, int flashtype);
void	DLLIMPORT	monster_fire_railgun (CEdict *self, vec3_t start, vec3_t aimdir, int damage, int kick, int flashtype);
void	DLLIMPORT	monster_fire_bfg (CEdict *self, vec3_t start, vec3_t aimdir, int damage, int speed, int kick, float damage_radius, int flashtype);
void	DLLIMPORT	M_droptofloor (CEdict *ent);
void	DLLIMPORT	monster_think (CEdict *self);
void	DLLIMPORT	walkmonster_start (CEdict *self);
void	DLLIMPORT	swimmonster_start (CEdict *self);
void	DLLIMPORT	flymonster_start (CEdict *self);
void	DLLIMPORT	AttackFinished (CEdict *self, float time);
void	DLLIMPORT	monster_death_use (CEdict *self);
void	DLLIMPORT	M_CatagorizePosition (CEdict *ent);
qboolean	DLLIMPORT	M_CheckAttack (CEdict *self);
void	DLLIMPORT	M_FlyCheck (CEdict *self);
void	DLLIMPORT	M_CheckGround (CEdict *ent);

/////////////////////////////////////////////////
// g_misc.c
//
void	DLLIMPORT	ThrowHead (CEdict *self, char *gibname, int damage, int type);
void	DLLIMPORT	ThrowClientHead (CEdict *self, int damage);
void	DLLIMPORT	ThrowGib (CEdict *self, char *gibname, int damage, int type);
void	DLLIMPORT	BecomeExplosion1(CEdict *self);

///////////////////////////////////////////////////
// g_weapon.c

void	DLLIMPORT	ThrowDebris (CEdict *self, char *modelname, float speed, vec3_t origin);
qboolean	DLLIMPORT	fire_hit (CEdict *self, vec3_t aim, int damage, int kick);
void	DLLIMPORT	fire_bullet (CEdict *self, vec3_t start, vec3_t aimdir, int damage, int kick, int hspread, int vspread, int mod);
void	DLLIMPORT	fire_shotgun (CEdict *self, vec3_t start, vec3_t aimdir, int damage, int kick, int hspread, int vspread, int count, int mod);
void	DLLIMPORT	fire_blaster (CEdict *self, vec3_t start, vec3_t aimdir, int damage, int speed, int effect, qboolean hyper);
void	DLLIMPORT	fire_grenade (CEdict *self, vec3_t start, vec3_t aimdir, int damage, int speed, float timer, float damage_radius);
void	DLLIMPORT	fire_grenade2 (CEdict *self, vec3_t start, vec3_t aimdir, int damage, int speed, float timer, float damage_radius, qboolean held);
void	DLLIMPORT	fire_rocket (CEdict *self, vec3_t start, vec3_t dir, int damage, int speed, float damage_radius, int radius_damage);
void	DLLIMPORT	fire_rail (CEdict *self, vec3_t start, vec3_t aimdir, int damage, int kick);
void	DLLIMPORT	fire_bfg (CEdict *self, vec3_t start, vec3_t dir, int damage, int speed, float damage_radius);

///////////////////////////////////////////////////
// g_ptrail.c

void	DLLIMPORT	PlayerTrail_Init (void);
void	DLLIMPORT	PlayerTrail_Add (vec3_t spot);
void	DLLIMPORT	PlayerTrail_New (vec3_t spot);
CEdict*	DLLIMPORT	PlayerTrail_PickFirst (CEdict *self);
CEdict*	DLLIMPORT	PlayerTrail_PickNext (CEdict *self);
CEdict*	DLLIMPORT	PlayerTrail_LastSpot (void);


///////////////////////////////////////////////////
// g_client.c

void	DLLIMPORT	respawn (CEdict *ent);
void	DLLIMPORT	BeginIntermission (CEdict *targ);
void	DLLIMPORT	PutClientInServer (CEdict *ent);
void	DLLIMPORT	InitClientPersistant (CGameClient *client);
void	DLLIMPORT	InitClientResp (CGameClient *client);
void	DLLIMPORT	InitBodyQue (void);
void	DLLIMPORT	ClientBeginServerFrame (CEdict *ent);

///////////////////////////////////////////////////
// g_player.c

void	DLLIMPORT	player_pain (CEdict *self, CEdict *other, float kick, int damage);
void	DLLIMPORT	player_die (CEdict *self, CEdict *inflictor, CEdict *attacker, int damage, vec3_t point);

///////////////////////////////////////////////////
// g_svcmds.c

void	ServerCommand (void);

///////////////////////////////////////////////////
// p_view.c

void	DLLIMPORT	ClientEndServerFrame (CEdict *ent);

///////////////////////////////////////////////////
// p_hud.c

void	DLLIMPORT	MoveClientToIntermission (CEdict *client);
void	DLLIMPORT	G_SetStats (CEdict *ent);
void	DLLIMPORT	ValidateSelectedItem (CEdict *ent);
void	DLLIMPORT	DeathmatchScoreboardMessage (CEdict *client, CEdict *killer);

///////////////////////////////////////////////////
// g_pweapon.c

void	DLLIMPORT	PlayerNoise(CEdict *who, vec3_t where, int type);

///////////////////////////////////////////////////
// m_move.c

qboolean	DLLIMPORT	M_CheckBottom (CEdict *ent);
qboolean	DLLIMPORT	M_walkmove (CEdict *ent, float yaw, float dist);
void	DLLIMPORT	M_MoveToGoal (CEdict *ent, float dist);
void	DLLIMPORT	M_ChangeYaw (CEdict *ent);

///////////////////////////////////////////////////
// g_phys.c

void	DLLIMPORT	G_RunEntity (CEdict *ent);

///////////////////////////////////////////////////
// g_main.c

void	DLLIMPORT	SaveClientData (void);
void	DLLIMPORT	FetchClientEntData (CEdict *ent);

////////////////////////////////
// global vars
vec3_t			DLLIMPORT	monster_flash_offset [];
CGameImport		DLLIMPORT	gi;
CLevelLocals	DLLIMPORT	level;
CGameLocals		DLLIMPORT	game;
CGameExport		DLLIMPORT	globals;
CSpawnTemp		DLLIMPORT	st;
int				DLLIMPORT	sm_meat_index;
int				DLLIMPORT	snd_fry;
int				DLLIMPORT	meansOfDeath;
//CEdict*			DLLIMPORT	g_edicts;


CConsoleVariable	DLLIMPORT	*deathmatch;
CConsoleVariable	DLLIMPORT	*coop;
CConsoleVariable	DLLIMPORT	*dmflags;
CConsoleVariable	DLLIMPORT	*skill;
CConsoleVariable	DLLIMPORT	*fraglimit;
CConsoleVariable	DLLIMPORT	*timelimit;
CConsoleVariable	DLLIMPORT	*password;
CConsoleVariable	DLLIMPORT	*maxclients;
CConsoleVariable	DLLIMPORT	*maxentities;
CConsoleVariable	DLLIMPORT	*g_select_empty;
CConsoleVariable	DLLIMPORT	*dedicated;

CConsoleVariable	DLLIMPORT	*sv_maxvelocity;
CConsoleVariable	DLLIMPORT	*sv_gravity;

CConsoleVariable	DLLIMPORT	*sv_rollspeed;
CConsoleVariable	DLLIMPORT	*sv_rollangle;
CConsoleVariable	DLLIMPORT	*gun_x;
CConsoleVariable	DLLIMPORT	*gun_y;
CConsoleVariable	DLLIMPORT	*gun_z;

CConsoleVariable	DLLIMPORT	*run_pitch;
CConsoleVariable	DLLIMPORT	*run_roll;
CConsoleVariable	DLLIMPORT	*bob_up;
CConsoleVariable	DLLIMPORT	*bob_pitch;
CConsoleVariable	DLLIMPORT	*bob_roll;

CConsoleVariable	DLLIMPORT	*sv_cheats;


////////////////////////////////////////////
// Macros we need.  Make into funcs an export???
#define random()	((rand () & 0x7fff) / ((float)0x7fff))
#define crandom()	(2.0 * (random() - 0.5))
#define	ITEM_INDEX(x) ((x)-itemlist)	// g_items.h
#define VectorSet(v, x, y, z)	(v[0]=(x), v[1]=(y), v[2]=(z)) // shared.h

////////////////////////////////////
// defines we need?

// sound channels
// channel 0 never willingly overrides
// other channels (1-7) allways override a playing sound on that channel
#define	CHAN_AUTO               0
#define	CHAN_WEAPON             1
#define	CHAN_VOICE              2
#define	CHAN_ITEM               3
#define	CHAN_BODY               4
// modifier flags
#define	CHAN_NO_PHS_ADD			8	// send to all clients, not just ones in PHS (ATTN 0 will also do this)
#define	CHAN_RELIABLE			16	// send by reliable message, not datagram

// sound attenuation values
#define	ATTN_NONE               0	// full volume the entire level
#define	ATTN_NORM               1
#define	ATTN_IDLE               2
#define	ATTN_STATIC             3	// diminish very rapidly with distance

//monster ai flags
#define AI_STAND_GROUND			0x00000001
#define AI_TEMP_STAND_GROUND	0x00000002
#define AI_SOUND_TARGET			0x00000004
#define AI_LOST_SIGHT			0x00000008
#define AI_PURSUIT_LAST_SEEN	0x00000010
#define AI_PURSUE_NEXT			0x00000020
#define AI_PURSUE_TEMP			0x00000040
#define AI_HOLD_FRAME			0x00000080
#define AI_GOOD_GUY				0x00000100
#define AI_BRUTAL				0x00000200
#define AI_NOSTEP				0x00000400
#define AI_DUCKED				0x00000800
#define AI_COMBAT_POINT			0x00001000
#define AI_MEDIC				0x00002000
#define AI_RESURRECTING			0x00004000

//
// monster muzzle flashes
//
#define MZ2_TANK_BLASTER_1				1
#define MZ2_TANK_BLASTER_2				2
#define MZ2_TANK_BLASTER_3				3
#define MZ2_TANK_MACHINEGUN_1			4
#define MZ2_TANK_MACHINEGUN_2			5
#define MZ2_TANK_MACHINEGUN_3			6
#define MZ2_TANK_MACHINEGUN_4			7
#define MZ2_TANK_MACHINEGUN_5			8
#define MZ2_TANK_MACHINEGUN_6			9
#define MZ2_TANK_MACHINEGUN_7			10
#define MZ2_TANK_MACHINEGUN_8			11
#define MZ2_TANK_MACHINEGUN_9			12
#define MZ2_TANK_MACHINEGUN_10			13
#define MZ2_TANK_MACHINEGUN_11			14
#define MZ2_TANK_MACHINEGUN_12			15
#define MZ2_TANK_MACHINEGUN_13			16
#define MZ2_TANK_MACHINEGUN_14			17
#define MZ2_TANK_MACHINEGUN_15			18
#define MZ2_TANK_MACHINEGUN_16			19
#define MZ2_TANK_MACHINEGUN_17			20
#define MZ2_TANK_MACHINEGUN_18			21
#define MZ2_TANK_MACHINEGUN_19			22
#define MZ2_TANK_ROCKET_1				23
#define MZ2_TANK_ROCKET_2				24
#define MZ2_TANK_ROCKET_3				25

#define MZ2_INFANTRY_MACHINEGUN_1		26
#define MZ2_INFANTRY_MACHINEGUN_2		27
#define MZ2_INFANTRY_MACHINEGUN_3		28
#define MZ2_INFANTRY_MACHINEGUN_4		29
#define MZ2_INFANTRY_MACHINEGUN_5		30
#define MZ2_INFANTRY_MACHINEGUN_6		31
#define MZ2_INFANTRY_MACHINEGUN_7		32
#define MZ2_INFANTRY_MACHINEGUN_8		33
#define MZ2_INFANTRY_MACHINEGUN_9		34
#define MZ2_INFANTRY_MACHINEGUN_10		35
#define MZ2_INFANTRY_MACHINEGUN_11		36
#define MZ2_INFANTRY_MACHINEGUN_12		37
#define MZ2_INFANTRY_MACHINEGUN_13		38

#define MZ2_SOLDIER_BLASTER_1			39
#define MZ2_SOLDIER_BLASTER_2			40
#define MZ2_SOLDIER_SHOTGUN_1			41
#define MZ2_SOLDIER_SHOTGUN_2			42
#define MZ2_SOLDIER_MACHINEGUN_1		43
#define MZ2_SOLDIER_MACHINEGUN_2		44

#define MZ2_GUNNER_MACHINEGUN_1			45
#define MZ2_GUNNER_MACHINEGUN_2			46
#define MZ2_GUNNER_MACHINEGUN_3			47
#define MZ2_GUNNER_MACHINEGUN_4			48
#define MZ2_GUNNER_MACHINEGUN_5			49
#define MZ2_GUNNER_MACHINEGUN_6			50
#define MZ2_GUNNER_MACHINEGUN_7			51
#define MZ2_GUNNER_MACHINEGUN_8			52
#define MZ2_GUNNER_GRENADE_1			53
#define MZ2_GUNNER_GRENADE_2			54
#define MZ2_GUNNER_GRENADE_3			55
#define MZ2_GUNNER_GRENADE_4			56

#define MZ2_CHICK_ROCKET_1				57

#define MZ2_FLYER_BLASTER_1				58
#define MZ2_FLYER_BLASTER_2				59

#define MZ2_MEDIC_BLASTER_1				60

#define MZ2_GLADIATOR_RAILGUN_1			61

#define MZ2_HOVER_BLASTER_1				62

#define MZ2_ACTOR_MACHINEGUN_1			63

#define MZ2_SUPERTANK_MACHINEGUN_1		64
#define MZ2_SUPERTANK_MACHINEGUN_2		65
#define MZ2_SUPERTANK_MACHINEGUN_3		66
#define MZ2_SUPERTANK_MACHINEGUN_4		67
#define MZ2_SUPERTANK_MACHINEGUN_5		68
#define MZ2_SUPERTANK_MACHINEGUN_6		69
#define MZ2_SUPERTANK_ROCKET_1			70
#define MZ2_SUPERTANK_ROCKET_2			71
#define MZ2_SUPERTANK_ROCKET_3			72

#define MZ2_BOSS2_MACHINEGUN_L1			73
#define MZ2_BOSS2_MACHINEGUN_L2			74
#define MZ2_BOSS2_MACHINEGUN_L3			75
#define MZ2_BOSS2_MACHINEGUN_L4			76
#define MZ2_BOSS2_MACHINEGUN_L5			77
#define MZ2_BOSS2_ROCKET_1				78
#define MZ2_BOSS2_ROCKET_2				79
#define MZ2_BOSS2_ROCKET_3				80
#define MZ2_BOSS2_ROCKET_4				81

#define MZ2_FLOAT_BLASTER_1				82

#define MZ2_SOLDIER_BLASTER_3			83
#define MZ2_SOLDIER_SHOTGUN_3			84
#define MZ2_SOLDIER_MACHINEGUN_3		85
#define MZ2_SOLDIER_BLASTER_4			86
#define MZ2_SOLDIER_SHOTGUN_4			87
#define MZ2_SOLDIER_MACHINEGUN_4		88
#define MZ2_SOLDIER_BLASTER_5			89
#define MZ2_SOLDIER_SHOTGUN_5			90
#define MZ2_SOLDIER_MACHINEGUN_5		91
#define MZ2_SOLDIER_BLASTER_6			92
#define MZ2_SOLDIER_SHOTGUN_6			93
#define MZ2_SOLDIER_MACHINEGUN_6		94
#define MZ2_SOLDIER_BLASTER_7			95
#define MZ2_SOLDIER_SHOTGUN_7			96
#define MZ2_SOLDIER_MACHINEGUN_7		97
#define MZ2_SOLDIER_BLASTER_8			98
#define MZ2_SOLDIER_SHOTGUN_8			99
#define MZ2_SOLDIER_MACHINEGUN_8		100

// --- Xian shit below ---
#define	MZ2_MAKRON_BFG					101
#define MZ2_MAKRON_BLASTER_1			102
#define MZ2_MAKRON_BLASTER_2			103
#define MZ2_MAKRON_BLASTER_3			104
#define MZ2_MAKRON_BLASTER_4			105
#define MZ2_MAKRON_BLASTER_5			106
#define MZ2_MAKRON_BLASTER_6			107
#define MZ2_MAKRON_BLASTER_7			108
#define MZ2_MAKRON_BLASTER_8			109
#define MZ2_MAKRON_BLASTER_9			110
#define MZ2_MAKRON_BLASTER_10			111
#define MZ2_MAKRON_BLASTER_11			112
#define MZ2_MAKRON_BLASTER_12			113
#define MZ2_MAKRON_BLASTER_13			114
#define MZ2_MAKRON_BLASTER_14			115
#define MZ2_MAKRON_BLASTER_15			116
#define MZ2_MAKRON_BLASTER_16			117
#define MZ2_MAKRON_BLASTER_17			118
#define MZ2_MAKRON_RAILGUN_1			119
#define	MZ2_JORG_MACHINEGUN_L1			120
#define	MZ2_JORG_MACHINEGUN_L2			121
#define	MZ2_JORG_MACHINEGUN_L3			122
#define	MZ2_JORG_MACHINEGUN_L4			123
#define	MZ2_JORG_MACHINEGUN_L5			124
#define	MZ2_JORG_MACHINEGUN_L6			125
#define	MZ2_JORG_MACHINEGUN_R1			126
#define	MZ2_JORG_MACHINEGUN_R2			127
#define	MZ2_JORG_MACHINEGUN_R3			128
#define	MZ2_JORG_MACHINEGUN_R4			129
#define MZ2_JORG_MACHINEGUN_R5			130
#define	MZ2_JORG_MACHINEGUN_R6			131
#define MZ2_JORG_BFG_1					132
#define MZ2_BOSS2_MACHINEGUN_R1			133
#define MZ2_BOSS2_MACHINEGUN_R2			134
#define MZ2_BOSS2_MACHINEGUN_R3			135
#define MZ2_BOSS2_MACHINEGUN_R4			136
#define MZ2_BOSS2_MACHINEGUN_R5			137





#endif // GAMEX86_IMPORT_HEADER