/*
 * Copyright (c) 1999 Thomas Grey All Rights Reserved.
 *
 * Thomas Grey grants you ("Licensee") a non-exclusive, royalty
 * free, license to use, modify and redistribute this software in
 * source and binary code form, provided that the following conditions 
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * This software is provided "AS IS," without a warranty of any
 * kind. ALL EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND
 * WARRANTIES, INCLUDING ANY IMPLIED WARRANTY OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON-INFRINGEMENT, ARE HEREBY
 * EXCLUDED.  THOMAS GREY AND HIS LICENSORS SHALL NOT BE LIABLE
 * FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
 * MODIFYING OR DISTRIBUTING THE SOFTWARE OR ITS DERIVATIVES. IN NO
 * EVENT WILL THOMAS GREY OR HIS LICENSORS BE LIABLE FOR ANY
 * LOST REVENUE, PROFIT OR DATA, OR FOR DIRECT, INDIRECT, SPECIAL,
 * CONSEQUENTIAL, INCIDENTAL OR PUNITIVE DAMAGES, HOWEVER CAUSED AND
 * REGARDLESS OF THE THEORY OF LIABILITY, ARISING OUT OF THE USE OF OR
 * INABILITY TO USE SOFTWARE, EVEN IF THOMAS GREY HAS BEEN
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 *
 * This software is not designed or intended for use in on-line
 * control of aircraft, air traffic, aircraft navigation or aircraft
 * communications; or in the design, construction, operation or
 * maintenance of any nuclear facility. Licensee represents and
 * warrants that it will not use or redistribute the Software for such
 * purposes.  
 */

import java.net.URL;
import java.net.MalformedURLException;
import java.io.InputStream;
import java.io.BufferedInputStream;
import java.io.IOException;
import java.util.zip.ZipInputStream;

public final class MDLquakeModel extends MDLModel{

//	private InputStream modelStream;
	private int version = 102;
	private int currentSkin = 0;
	private int currentSkinGroup = 0;
	private int currentFrame = 0;
	private int currentSimpleFrame = 0;
	private int totalFrames;
	private int totalSkins;
	private String framesString;
	private String skinsString;
	int shiftY;
// Header
//	private String filetype;
	private int modelVersion;
	private float xscale,yscale,zscale;        // scale model
	private float xoffset,yoffset,zoffset;     // scale origin
	private float radius;                      // bounding radius
//	private float eyex,eyey,eyez;              // eye position
	private int num_skins;                // num skins
	private int skinw;                    // skin width
	private int skinwOver2;
	private int skinh;                    // skin height
	private int num_vertices; 
	private int num_triangles;
	private int num_frames;
//	private int sync_type;
//	private int flags;
//	private float ave_size;
	private int k=0;
	private int j=0;
// Content
//	private MDLSkinGroup[] skins;
	private byte[][][] skins;
	private int[] skins_numPics;
//	private MDLSkinVertex[] sVerts;
	private short[] sVertsS;
	private short[] sVertsT;
	private boolean[] sVertsOnseam;
//        private MDLTriangle[] triangles;
        private short[][] triangles;
	private boolean[] trianglesFace;
	private MDLFrame[] frames;
	
	private int percentageDone = 0;
	
	public int getVersion(){
	    return version;
	}
		
	public short getTriangleVertexIndex(int triangle, int vertex){
	    return triangles[triangle][vertex];
	}
	
	public short getSkinVertexT(int triangle, int vertexIndex){
	    return sVertsT[(triangles[triangle][vertexIndex])];
	}
	
	public short getSkinVertexS(int triangle, int vertexIndex){
	// Shift if vertex is onseam and poly faces back
	   k = (triangles[triangle][vertexIndex]);
	    if (!trianglesFace[triangle]){
		if (sVertsOnseam[k]){
			return (short)(sVertsS[k] + skinwOver2);
		}
	    }
	    return sVertsS[k];
	}
    	
	public void setCurrentSkinIndex(int skin){
	    currentSkinGroup = skin;
	}
	
	public void extraInfo(String extra){
	    if (extra != null){
		System.out.println("MDLquakeModel: No EXTRA parameters avalible with this class '"+extra+"' will be ignored");
	    }
	}
	
	public int getCurrentSkinIndex(){
	    return currentSkinGroup;
	}
		
	public int getCurrentFrameIndex(){
	    return currentFrame;
	}
	
	public void setCurrentFrameIndex(int fi){
	    currentFrame = fi;
	}
	
	public int getNumberOfAnimationFrames(int start, int finish){
	    int number=0;
	    for (int i = start; i<=finish; i++){
		number += frames[i].num_SimpleFrames;
	    }
	    return number;
	}
	
	public int getFrameDelayMillis(){
	    if(frames[currentFrame].isGroup){
		if (currentSimpleFrame ==0){
		    return (int)(1000*frames[currentFrame].time[0]);
		}else{
		    return (int)(1000*(frames[currentFrame].time[currentSimpleFrame]-frames[currentFrame].time[currentSimpleFrame-1]) );
		}
	    }else{
		return 100;
	    }
	}	    
		
	    
	public void nextSkin(){
	    currentSkin++;
	    if (currentSkin >= skins_numPics[currentSkinGroup]){
		currentSkinGroup++;
		currentSkin = 0;
		if (currentSkinGroup >= num_skins){
		    currentSkinGroup = 0;
		}
	    }
	}
	
	public void lastSkin(){
	    currentSkin--;
	    if (currentSkin < 0){
		currentSkinGroup--;
		if (currentSkinGroup < 0){
		    currentSkinGroup = (short)(num_skins- 1);
		}
		currentSkin = (short)(skins_numPics[currentSkinGroup] - 1);
	    }
	}
	
	public void lastFrame(){
	    currentSimpleFrame--;
	    if (currentSimpleFrame < 0){
		currentFrame--;
		if (currentFrame < 0){
		    currentFrame = (short)(num_frames - 1);
		}
		currentSimpleFrame = (short)(frames[currentFrame].num_SimpleFrames - 1);
	    }
	}

	public void nextFrame(){
	    currentSimpleFrame++;
	    if (currentSimpleFrame >= frames[currentFrame].num_SimpleFrames){
		currentFrame++;
		currentSimpleFrame = 0;
		if (currentFrame >= num_frames){
		    currentFrame = 0;
		}
	    }
	}
	
	public byte getTexel(short x, short y){
	    if (num_skins > 0){
		return skins[currentSkinGroup][x][y];
	    }else{
		return (byte)0;
	    }
	}
	
	public int getTotalFrames(){
	    return totalFrames;
	}
	
	public int getTotalSkins(){
	    return totalSkins;
	}
	
	public int getYShift(){
	    return shiftY;
	}
		
	public String getFramesInfo(){
	    return framesString;
	}
	
	public String getFrameName(){
	    return frames[currentFrame].simple_Frames[currentSimpleFrame].name;
	}
	
	public String getSkinsInfo(){
	    return skinsString;
	}
	
	public int getVertsInfo(){
	    return num_vertices;
	}
	
	public int getTrianglesInfo(){
	    return num_triangles;
	}
			
	public float getBoundingRadius(){
	    return radius;
	}
	
	public int getNumberOfSkins(){
	    return num_skins;
	}
	
	public int getSkinWidth(){
	    return skinw;
	}
	
	public int getSkinHeight(){
	    return skinh;
	}
	

	public float getVertexX(int vertex){
	    return frames[currentFrame].simple_Frames[currentSimpleFrame].frameVertsX[vertex];
	}
	
	public float getVertexY(int vertex){
	    return frames[currentFrame].simple_Frames[currentSimpleFrame].frameVertsY[vertex];
	}
	
	public float getVertexZ(int vertex){
	    return frames[currentFrame].simple_Frames[currentSimpleFrame].frameVertsZ[vertex];
	}
	
	public boolean getClockwiseCull(){
	    return true;
	}
	
	public int getNumberOfVertices(){
	    return num_vertices;
	}
	
	public int getNumberOfTriangles(){
	    return num_triangles;
	}
	
	public int getNumberOfFrames(){
	    return num_frames;
	}
	
	public String getModelFormat(){
	    return ("quake");
	}

	public String getAuthorInfo(){
	    return("Copyright T.J.Grey 1998");
	}
	
	public String getAuthorName(){
	    return("Tom Grey");
	}

	private void skip(int nb, InputStream modelStream) throws IOException{
		int x = 0;
		while (x < nb){
			modelStream.read();
			yield();
			x++;
		}
	}

	protected void loadFrameVertex(InputStream stream, MDLSimpleFrame frame, int vertex) throws IOException{
	
		    frame.frameVertsX[vertex] = xscale * stream.read();
		    frame.frameVertsY[vertex] = yscale * stream.read();
		    frame.frameVertsZ[vertex] = zscale * stream.read();
		    frame.frameVertsX[vertex] += xoffset;
		    frame.frameVertsY[vertex] += yoffset;
		    frame.frameVertsZ[vertex] += zoffset;
		    frame.frameVertsZ[vertex] += shiftY;
		    stream.read();		    
	}
	
	protected void readModelThread(int[] loadframes, int[] loadskins, int shift){
	
	    shiftY = shift;
	    showStatus("Quake model class for MDLView by T.J.Grey");
	    String filename = getParameter("MODEL");
	    try{
		
		if (filename.endsWith(".zip")){
		    showStatus("MDLView: Attempting opening of zip file!");
		    ZipInputStream zipModelStream = new ZipInputStream(new BufferedInputStream(getStreamForFile(filename)) );
		    zipModelStream.getNextEntry();
		    modelMethod(zipModelStream,loadskins,loadframes);
		    showStatus("MDLView: Reading model... done");
//		    zipModelStream.closeEntry();
		    zipModelStream.close();
		    showStatus("MDLView: Closing data stream");
		    zipModelStream = null;               
		}else{	    
		    showStatus("MDLView: Attempting opening of mdl file");
		    BufferedInputStream modelStream = new BufferedInputStream(getStreamForFile(filename));
		    modelMethod(modelStream,loadskins,loadframes);
		    showStatus("MDLView: Reading model... done");
		    modelStream.close();
		    showStatus("MDLView: Closing data stream");
		    modelStream = null;               
		}
	    }catch(IOException e){
		System.out.println("MDLView: Error opening data Stream for model- <"+e+">");
		showStatus("MDLView: Error opening data stream model- <"+e+">");
		setModelError(true);
		throw new RuntimeException("MDLquakeModel: Error opening data stream model");
	    }
	    return;
	}
	
	private void modelMethod(InputStream modelStream, int[] loadskins, int[] loadframes) throws IOException{
		readModelHeader(modelStream);
	// now the skin stuff
		readModelSkins(loadskins, modelStream);
		percentageDone = 50;
		showPercentage(percentageDone);
        // now read the 2d vertices
		readSkinVertices(modelStream);
        // now read triange definitions
		readModelTriangles(modelStream);
        // now read the frames
		readModelFrames(loadframes, modelStream);
		percentageDone = 100;
		showPercentage(percentageDone);
	// All done now
	    if(modelVersion!=MDL_VER){
		modelVersion=MDL_VER;
	    }
//	    modelReady();
	    framesString = (num_frames+"/"+totalFrames);
	    skinsString = (num_skins+"/"+totalSkins);
	}

	protected void readModelHeader(InputStream modelStream) throws IOException{
	    // The header:
			showStatus("MDLView: Reading model header");
// Get filetype= IDPO fo us
			byte[] fourBytes = new byte[4];
			for (int i=0; i<4; i++){
				fourBytes[i]=(byte)modelStream.read();       // magic nums
			}
			String filetype=new String(fourBytes);
			if (!filetype.equals("IDPO")){
				System.out.println("MDLView: Warning this file may not be a quake model file (non IDPO identifier)");
				showStatus("MDLView: Warning may not be quake model file");
			}
// Version
			this.modelVersion = this.readInt(modelStream);
			checkNeg(false,modelVersion,"Model Version");
// x,y and z scales
			this.xscale = this.readFloat(modelStream);	// scale = (max - min)/255.9
			this.yscale= this.readFloat(modelStream);
			this.zscale= this.readFloat(modelStream);
// Origin position
			this.xoffset = this.readFloat(modelStream);	// min
			this.yoffset = this.readFloat(modelStream);
			this.zoffset = this.readFloat(modelStream);
// model radius
			this.radius = this.readFloat(modelStream);
			checkNeg(false,radius,"Model Radius");
// Eye position (?)
			skip(12,modelStream);     // eye position
// Number of skins
			this.num_skins = this.readInt(modelStream);      // num skins
			if (checkNeg(true,num_skins,"Number of Skins") || checkShort(num_skins,"Number of Skins") ){
			    return;
			}
			totalSkins = num_skins;
// Skin texture width and height, should/must be multiple of four (?)
			this.skinw = this.readInt(modelStream);        // skin width
			skinwOver2 = skinw/2;
			this.skinh = this.readInt(modelStream);        // skin height
			if ((this.skinw%4)!=0){
				System.out.println("MDLView: Warning skin width not multiple of four");
			}
			if (checkNeg(true,skinw,"Skin Width") || checkShort(skinw,"Skin Width") ){
			    return;
			}
			if (checkNeg(true,skinh,"Skin Height") || checkShort(skinh,"Skin Height") ){
			    return;
			}
// Number of verticies
			this.num_vertices = this.readInt(modelStream);
			if (checkNeg(true,num_vertices,"Number of Vertices") || checkShort(num_vertices,"Number of Vertices") ){
			    return;
			}
// Number of triangles (polygons)
			this.num_triangles = this.readInt(modelStream);
			if (checkNeg(true,num_triangles,"Number of Triangles") || checkShort(num_triangles,"Number of Triangles") ){
			    return;
			}
// Number of frames
			this.num_frames = this.readInt(modelStream);
			totalFrames = num_frames;
// 0= synchron, 1= random
			if(this.modelVersion == MDL_VER){
				skip(4,modelStream);	// sync type, either 0 or random
			}
// Flags (?)
			skip(4,modelStream);          // flags
// Average size of triangles
			if(this.modelVersion == MDL_VER){
				skip(4,modelStream);       // average size  pixels/triangle?
			}
        // that ends the header
    }

    protected void readModelSkins(int[] loadSkins, InputStream modelStream) throws IOException{
    	    if(loadSkins[0] == -2){
		for(int k=0; k<this.num_skins; k++){
		    yield();
		    showStatus("MDLView: Skipping all skins....");
		    skipModelSkin(modelStream);		
		}
		num_skins = 0;
		return;
	    }
//	    this.skins = new byte[num_skins][skinw][skinh];//MDLSkinGroup[this.num_skins];
	    int loadCounter = 0;
	    showStatus("MDLView: Reading model skins");
	    if (loadSkins[0] != -1){
		skins = new byte[loadSkins.length][skinw][skinh];//MDLSkinGroup[loadSkins.length];
	    }else{
		skins = new byte[num_skins][skinw][skinh];//MDLSkinGroup[num_skins];
	    }
// Loop over skins
	    float pcPerSkin = (float)50/(float)num_skins;
	    for(int k=0; k<this.num_skins; k++){
		yield();
		showStatus("MDLView: Reading skins");
		if (loadSkins[0] == -1){
		    readModelSkin(loadCounter, modelStream,pcPerSkin);
		    loadCounter++;
		}else if(loadCounter >= loadSkins.length){
		    skipModelSkin(modelStream);
		    percentageDone += pcPerSkin;
		    showPercentage(percentageDone);
		}else if(loadSkins[loadCounter] ==k){
		    readModelSkin(loadCounter, modelStream,pcPerSkin);
		    loadCounter++;
		}else{
		    skipModelSkin(modelStream);
		    percentageDone += pcPerSkin;
		    showPercentage(percentageDone);
		}
	    }
	    num_skins = loadCounter;
    }
    
    protected void readModelSkin(int k, InputStream modelStream,float pcPerSkin)throws IOException{
			showStatus("MDLView: Reading in skin "+k);
//			System.out.print("MDLView: Reading skins.");
//			    this.skins[k] = new MDLSkinGroup();
//	System.out.println("Skin: "+k);
// is this a group of pictures- a skin group -or a single picture?
			    if (this.readInt(modelStream) > 0){
				String errorMsg = ("MDLquakeModel does not support group skins!!! Contact the author for more info");
				showStatus(errorMsg);
				System.out.println(errorMsg);
				throw new RuntimeException(errorMsg);
			    }
/* // If simple skin  then 1 picture)
			    if(!skins[k].isGroup){
				this.skins[k].numPics = 1;
			    }else{
				this.skins[k].numPics = readInt(modelStream);
			    }

// Read time intervals
			    if (this.skins[k].isGroup){
				this.skins[k].time = new float[this.skins[k].numPics];
				for (int t = 0; t < this.skins[k].numPics; t++){
					this.skins[k].time[t] = this.readFloat(modelStream);
				}
			    }
*/
// Initialise picture arrays
//			    this.skins[k].pixelData = new byte[this.skins[k].numPics][this.skinw][this.skinh];
// Initialise percentages
			    float pcDone = (float) percentageDone;
			    float pcPerLine = pcPerSkin / (float)(skinh);
// for each picture
//			    for(int pic=0; pic<this.skins[k].numPics; pic++){
//	System.out.println("PIcture: "+pic);
// For each of the skinh rows:
				for(int row=0; row<this.skinh; row++){
					pcDone += pcPerLine;
					percentageDone = (int) pcDone;
					showPercentage(percentageDone);
// For each of the skinw pixels in the row:
					for(int col = 0; col < this.skinw; col++){
						this.skins[k][col][row] = (byte)modelStream.read();
					}
					// End of loop across row
					    yield();
				}        
				// End of loop over rows
//			    }
			// End of loop over pics

	}

	protected void skipModelSkin(InputStream modelStream)throws IOException{
			boolean isGroup;
			int numPics = 0;
			showStatus("MDLView: Skipping skin +");
// is this a group of pictures- a skin group -or a single picture?
			    if (this.readInt(modelStream) == 0){
//		System.out.println("Simple skin");
				isGroup = false;
			    }else{
//		System.out.println("Group skin");
				isGroup=true;
			    }
			    if(!isGroup){
				numPics = 1;
			    }else{
				numPics = readInt(modelStream);
			    }
// Skip time intervals
			    if (isGroup){
				skip(4*numPics, modelStream);
			    }
// for each picture
			    skip(numPics*skinh*skinw, modelStream);
			showStatus("MDLView: Skipping skin x");

	}
	
	protected void readSkinVertices(InputStream modelStream) throws IOException{
		    showStatus("MDLView: Reading skin vertices");
//		    System.out.print("MDLView: Reading skin vertices");
		    sVertsS = new short[this.num_vertices];
		    sVertsT = new short[this.num_vertices];
		    sVertsOnseam = new boolean[this.num_vertices];

		    for (int i=0; i < this.num_vertices; i++){
//			.sVerts[i] = new MDLSkinVertex();
			if (readInt(modelStream)>0){
			    sVertsOnseam[i] = true;
			}else{
			    sVertsOnseam[i] = false;
			}
			sVertsS[i] = (short)readInt(modelStream);
			sVertsT[i] = (short)readInt(modelStream);
			if ( (sVertsS[i]>skinw)||(sVertsT[i]>skinh)){
			    System.out.println("MDLView: Skin vertex prob: "+sVertsS[i]+","+sVertsT[i]+" outside skin");
			    throw new RuntimeException("MDLquakeModel: Problem with model");
			}
			showStatus("MDLView: Reading skin vertices....");
		    }
//		    System.out.print("\n");
    }
    
	protected void readModelTriangles(InputStream modelStream) throws IOException{                    
	    showStatus("MDLView: Reading triangles");
	    triangles = new short[num_triangles][3];//MDLTriangle[num_triangles];
	    trianglesFace = new boolean[num_triangles];//MDLTriangle[num_triangles];

	    int count = 0;
	    int progressInt = num_triangles / 10;
		                            
	    for(int i=0; i<num_triangles; i++){

//		triangles[i] = new MDLTriangle();
		if (readInt(modelStream) > 0){
		    trianglesFace[i] = true;
		}else{
		    trianglesFace[i] = false;
		}
		
		triangles[i][0] = (short)readInt(modelStream);
		triangles[i][1] = (short)readInt(modelStream);
		triangles[i][2] = (short)readInt(modelStream);

		if (triangles[i][0] >= num_vertices || triangles[i][1] >= num_vertices || triangles[i][2] >= num_vertices){
		    System.out.println("MDLView: Prob triangle "+i+" verts: "+triangles[i][0]+", "+triangles[i][1]+", "+triangles[i][2]);
		    throw new RuntimeException("MDLquakeModel: Problem with model");
		}
		if (count % progressInt == 0){
		    showStatus("MDLView: Reading triangles.... (about " + (i/progressInt*10) + "% done)");
		//    System.out.print(".");
		    yield();
		}

	    count++;
	}
//	System.out.print("\n");
                        
    }
    
	protected void readModelFrames(int[] loadframes, InputStream modelStream) throws IOException{
    
    	    showStatus("MDLView: Reading frames");
	    yield();
	    float pcPerFrame = (float)50/(float)num_frames;
	    float pcDone = 50;
	    int loadCounter = 0;
	    if (loadframes[0] != -1){
		frames = new MDLFrame[loadframes.length];
	    }else{
		frames = new MDLFrame[num_frames];
	    }
	    for(int k=0; k<num_frames;  k++){
		if (loadframes[0] == -1){
		    frames[loadCounter] = new MDLFrame();
		    if(modelVersion == MDL_VER){
			if( readInt(modelStream) > 0){
			    frames[loadCounter].isGroup = true;
			}else{
			    frames[loadCounter].isGroup = false;
			}
		    }else{
			frames[loadCounter].isGroup = false;
		    }
    // if single frames
		    if(!frames[loadCounter].isGroup){
			readSingleFrame(loadCounter, modelStream);
    // Read group
		    }else{
			readGroupFrame(loadCounter, modelStream);
		    }
		    loadCounter++;
		}else if(loadCounter >= loadframes.length){
		    num_frames = loadCounter;
		    showStatus("MDLView: Frames read");
		    return;
//		    if(modelVersion == MDL_VER){
//			if( readInt(modelStream) > 0){
		    // Skip single fram
//			    skipGroupFrame(modelStream);
//			}else{
//			    skipSingleFrame(modelStream);
//			}
//		    }else{
//			skipSingleFrame(modelStream);
//		    }
		}else if(loadframes[loadCounter] ==k){
		    frames[loadCounter] = new MDLFrame();
		    if(modelVersion == MDL_VER){
			if( readInt(modelStream) > 0){
			    frames[loadCounter].isGroup = true;
			}else{
			    frames[loadCounter].isGroup = false;
			}
		    }else{
			frames[loadCounter].isGroup = false;
		    }
    // if single frames
		    if(!frames[loadCounter].isGroup){
			readSingleFrame(loadCounter, modelStream);
    // Read group
		    }else{
			readGroupFrame(loadCounter, modelStream);
		    }
		    loadCounter++;
		}else{
		    if(modelVersion == MDL_VER){
			if( readInt(modelStream) > 0){
		    // Skip single fram
			    skipGroupFrame(modelStream);
			}else{
			    skipSingleFrame(modelStream);
			}
		    }else{
			skipSingleFrame(modelStream);
		    }
		}
		pcDone += pcPerFrame;
		percentageDone = (int) pcDone;
		showPercentage(percentageDone);
	    }
	    num_frames = loadCounter;
	    showStatus("MDLView: Frames read");
    }
    
    protected void readSingleFrame(int frame, InputStream modelStream) throws IOException{
	showStatus("Reading frame "+frame);
  //   System.out.println("Reading single");
///	System.out.print(".");
	frames[frame].num_SimpleFrames = 1;
	frames[frame].simple_Frames = new MDLSimpleFrame[1];
	frames[frame].simple_Frames[0] = new MDLSimpleFrame(num_vertices);
    // Don't ask questions...---V
	if(modelVersion!=MDL_VER){
	    skip(4,modelStream);//loadFrameVertex(modelStream, frames[frame].min);
	}
	skip(4,modelStream);//loadFrameVertex(modelStream, frames[frame].min);
	skip(4,modelStream);//loadFrameVertex(modelStream, frames[frame].max);
	if(modelVersion==MDL_VER){
	    byte[] sixteenBytes = new byte[16];
	    int length = 16;
	    for (int i = 0; i < 16; i++){
		sixteenBytes[i] = (byte) modelStream.read();
		if (sixteenBytes[i] == 0 && i < length){
		    length = i;
		}
	    }
	    frames[frame].simple_Frames[0].name = new String(sixteenBytes,0,length);
	    showStatus("MDLView: Frame name is "+frames[frame].simple_Frames[0].name);
	}else{
	    frames[frame].simple_Frames[0].name = "noname";
	}
// Final reading in the 3D data
	for(int i=0;i<num_vertices;i++){
	    loadFrameVertex(modelStream, frames[frame].simple_Frames[0],i);
	}
    }
    
    protected void skipSingleFrame(InputStream modelStream) throws IOException{
    
    // Don't ask questions...---V
	if(modelVersion!=MDL_VER){
	    skip(4, modelStream);
	}
	else{
	    skip(16, modelStream);
	}
// Final skip the 3D data
	skip(4*num_vertices+8, modelStream);
    }
    
    protected void readGroupFrame(int k, InputStream modelStream) throws IOException{
	showStatus("MDLView: Reading frame group");
//    System.out.println("Reading group");
//	System.out.print(".");
	frames[k].num_SimpleFrames = readInt(modelStream);
	skip(4,modelStream);//loadFrameVertex(modelStream, frames[k].min);
	skip(4,modelStream);//loadFrameVertex(modelStream, frames[k].min);
// Create the time and simpleframe arrays
	frames[k].time = new float[frames[k].num_SimpleFrames];
	frames[k].simple_Frames = new MDLSimpleFrame[frames[k].num_SimpleFrames];
// Loop to read in num_SimpleFrames time values
	showStatus("MDLView: Time values for frames in group");
	for(int j=0; j<frames[k].num_SimpleFrames; j++){
	    frames[k].time[j] = readFloat(modelStream);
	}
// Loop over simple frames, actually reading in data
	for(int j=0; j<frames[k].num_SimpleFrames; j++){
	    frames[k].simple_Frames[j] = new MDLSimpleFrame(num_vertices);
	    skip(4,modelStream);//loadFrameVertex(modelStream, frames[k].simple_Frames[j].min);
	    skip(4,modelStream);//loadFrameVertex(modelStream, frames[k].simple_Frames[j].max);
	    byte[] sixteenBytes = new byte[16];
	    for (int i=0; i < 16; i++){
		sixteenBytes[i] = (byte) modelStream.read();
	    }
	    frames[k].simple_Frames[0].name = new String(sixteenBytes);
	    showStatus("MDLView: Frame name is "+frames[k].simple_Frames[0].name);
// Loop over the data for the simple frame
	    for(int i=0;i<num_vertices;i++){
		loadFrameVertex(modelStream, frames[k].simple_Frames[j],i);
	    }
	}
    }
    
    protected void skipGroupFrame(InputStream modelStream) throws IOException{
	showStatus("MDLView: Skipping frame group");
  //  System.out.println("Skipping group");
	 int numberSimpleFrames = readInt(modelStream);
	 skip(8, modelStream);
/// Loop to read in num_SimpleFrames time values
	for(int j=0; j<numberSimpleFrames; j++){
	    skip(4, modelStream);
	}
//// Loop over simple frames, actually reading in data
	skip(8*numberSimpleFrames, modelStream);
// Loop over the data for the simple frame
	skip(4*num_vertices+16, modelStream);
    }
    
    public java.awt.image.IndexColorModel getInternalPalette(){
	return null;
    }
}