/*
 * Copyright (c) 1999 Thomas Grey All Rights Reserved.
 *
 * Thomas Grey grants you ("Licensee") a non-exclusive, royalty
 * free, license to use, modify and redistribute this software in
 * source and binary code form, provided that the following conditions 
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * This software is provided "AS IS," without a warranty of any
 * kind. ALL EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND
 * WARRANTIES, INCLUDING ANY IMPLIED WARRANTY OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON-INFRINGEMENT, ARE HEREBY
 * EXCLUDED.  THOMAS GREY AND HIS LICENSORS SHALL NOT BE LIABLE
 * FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
 * MODIFYING OR DISTRIBUTING THE SOFTWARE OR ITS DERIVATIVES. IN NO
 * EVENT WILL THOMAS GREY OR HIS LICENSORS BE LIABLE FOR ANY
 * LOST REVENUE, PROFIT OR DATA, OR FOR DIRECT, INDIRECT, SPECIAL,
 * CONSEQUENTIAL, INCIDENTAL OR PUNITIVE DAMAGES, HOWEVER CAUSED AND
 * REGARDLESS OF THE THEORY OF LIABILITY, ARISING OUT OF THE USE OF OR
 * INABILITY TO USE SOFTWARE, EVEN IF THOMAS GREY HAS BEEN
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 *
 * This software is not designed or intended for use in on-line
 * control of aircraft, air traffic, aircraft navigation or aircraft
 * communications; or in the design, construction, operation or
 * maintenance of any nuclear facility. Licensee represents and
 * warrants that it will not use or redistribute the Software for such
 * purposes.  
 */

import java.net.URL;
import java.net.MalformedURLException;
import java.io.InputStream;
import java.io.BufferedInputStream;
import java.io.IOException;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipEntry;

public final class MDLquake2Model extends MDLModel{

    private int version = 99;
    private int filepos;
    private int i;
    private int skinw;
    private int skinh;
    private int totalSkins;
    private int frameSizeBytes;
    private int num_skins;
    private int num_vertices;
    private int num_st_vertices;
    private int num_triangles;
    private int num_gl_cmds;
    private int num_frames;
    private int totalFrames;
    private float radius;
    
//    private MDLSkinVertex[] sVerts;
    private short[] sVertsS;
    private short[] sVertsT;
    private short[][] tris;
    private short[][] stTris;
    private byte[][][] skins;
    
    private MDLSimpleFrame[] frames;
    private byte[][] cSkin;

    public String weaponName;
    public int weaponSkinw;
    public int weaponSkinh;
    public int weaponFrameSizeBytes;
    public int weaponNum_skins;
    public int weaponNum_vertices;
    public int weaponNum_st_vertices;
    public int weaponNum_triangles;
    public int weaponNum_gl_cmds;
    public int weaponNum_frames;
    public int weaponTotalFrames;
    
    public short[] weaponSVertsS;
    public short[] weaponSVertsT;
    public short[][] weaponTris;
    public short[][] weaponStTris;
    public MDLSimpleFrame[] weaponFrames;
    public byte[][] weaponSkin;
    
    private int currentFrame = 0;
    private int currentSkin = 0;
    private boolean haveWeapon = false;
    private boolean loadWeapon = false;
    private int max_vertices;
    private int max_triangles;
    private int max_skinh;
    private int shift;
    
    private int percentageDone;
    private int percentagePerFile = 1;
    
    private int[] loadframes;
    private boolean haveWepSkin = false;
    private int loadingSkin;
    private int skinsSeen;
    private int[] loadskins;
    
    private boolean specialWepSkin = false;
	    
	public int getVersion(){
	    return version;
	}
	
	public short getTriangleVertexIndex(int triangle, int vertex){
	    if (!haveWeapon){
		return tris[triangle][vertex];
	    }else{
		if (triangle < num_triangles){
		    return tris[triangle][vertex];
		}else{
		    i = triangle - num_triangles;
		    if (currentFrame < weaponNum_frames){
			return weaponTris[i][vertex];
		    }else{
			return tris[0][vertex];
		    }
		}
	    }
	}
	
	public short getSkinVertexT(int triangle, int vertexIndex){
	    if (!haveWeapon){
		return sVertsT[(stTris[triangle][vertexIndex])];
	    }else{
		if (triangle < num_triangles){
		    return sVertsT[(stTris[triangle][vertexIndex])];
		}else{
		    i = triangle - num_triangles;
		    if (currentFrame < weaponNum_frames){
			return weaponSVertsT[(weaponStTris[i][vertexIndex])];
		    }else{
			return sVertsT[(stTris[0][vertexIndex])];
		    }
		}
	    }
	}
	
	public short getSkinVertexS(int triangle, int vertexIndex){
	    if (!haveWeapon){
		return sVertsS[(stTris[triangle][vertexIndex])];
	    }else{
		if (triangle < num_triangles){
		    return sVertsS[(stTris[triangle][vertexIndex])];
		}else{
		    i = triangle - num_triangles;
		    if (currentFrame < weaponNum_frames){
			return weaponSVertsS[(weaponStTris[i][vertexIndex])];
		    }else{
			return sVertsS[(stTris[0][vertexIndex])];
		    }
		}
	    }
	}
    	
	public void setCurrentSkinIndex(int skin){
	    currentSkin = skin;
	}
	
	public int getCurrentSkinIndex(){
	    return currentSkin;
	}
		
	public int getCurrentFrameIndex(){
	    return currentFrame;
	}
	
	public void setCurrentFrameIndex(int fi){
	    currentFrame = fi;
	}
	
	public int getNumberOfAnimationFrames(int start, int finish){
		return (finish-start+1);
	}
	
	public int getFrameDelayMillis(){
	    return 100;
	}	    
		
	public void nextSkin(){
	    currentSkin++;
	    currentSkin %= num_skins;
	    cSkin = skins[currentSkin];
	}
	
	public void lastSkin(){
	    currentSkin--;
	    if (currentSkin <0){
		currentSkin = num_skins -1;
	    }
	    cSkin = skins[currentSkin];
	}
	
	public void lastFrame(){
	    currentFrame--;
	    if (currentFrame <0){
		currentFrame = num_frames -1;
	    }
	}

	public void nextFrame(){
	    currentFrame++;
	    currentFrame %= num_frames;
	}
	
	public byte getTexel(short x, short y){
	    if (num_skins <1){
		return (byte)0;
	    }
	    if (!haveWeapon){
		return cSkin[x][y];
	    }else if (y > -1 && y < skinh){
		return cSkin[x][y];
	    }else{
		i = y - skinh;
		if(!specialWepSkin){
		    return weaponSkin[x][i];
		}else{
		    return cSkin[x][i];
		}
	    }
	}
	
	public int getTotalFrames(){
	    return totalFrames;
	}
	
	public int getTotalSkins(){
	    return totalSkins;
	}
		
	public String getFrameName(){
	    return frames[currentFrame].name;
	}
	
	public float getBoundingRadius(){
	    return radius;
	}
	
	public int getNumberOfSkins(){
	    return num_skins;
	}
	
	public int getSkinWidth(){
	    return skinw;
	}
	
	public int getSkinHeight(){
	    return skinh;
	}
	

	public float getVertexX(int vertex){
	    if (!haveWeapon){
		return frames[currentFrame].frameVertsX[vertex];
	    }else{
		if (vertex < num_vertices){
		    return frames[currentFrame].frameVertsX[vertex];
		}else{
		    i = vertex - num_vertices;
		    if (currentFrame < weaponNum_frames){
			return weaponFrames[currentFrame].frameVertsX[i];
		    }else{
			return frames[currentFrame].frameVertsX[0];
		    }
		}
	    }
	}
	
	public float getVertexY(int vertex){
	    if (!haveWeapon){
		return frames[currentFrame].frameVertsY[vertex];
	    }else{
		if (vertex < num_vertices){
		    return frames[currentFrame].frameVertsY[vertex];
		}else{
		    i = vertex - num_vertices;
		    if (currentFrame < weaponNum_frames){
			return weaponFrames[currentFrame].frameVertsY[i];
		    }else{
			return frames[currentFrame].frameVertsY[0];
		    }
		}
	    }
	}
	
	public float getVertexZ(int vertex){
	    if (!haveWeapon){
		return frames[currentFrame].frameVertsZ[vertex];
	    }else{
		if (vertex < num_vertices){
		    return frames[currentFrame].frameVertsZ[vertex];
		}else{
		    i = vertex - num_vertices;
		    if (currentFrame < weaponNum_frames){
			return weaponFrames[currentFrame].frameVertsZ[i];
		    }else{
			return frames[currentFrame].frameVertsZ[0];
		    }
		}
	    }
	}
	
	public boolean getClockwiseCull(){
	    return true;
	}
	
	public int getNumberOfVertices(){
	    if (haveWeapon){
		return max_vertices;
	    }else{
		return num_vertices;
	    }
	}
	
	public int getNumberOfTriangles(){
	    if (haveWeapon){
		return max_triangles;
	    }else{
		return num_triangles;
	    }
	}
	
	public int getNumberOfFrames(){
	    return num_frames;
	}
	
	public String getModelFormat(){
	    return "quake2";
	}

	public String getAuthorInfo(){
	    return "Copyright T.J.Grey 1998";
	}
	
	public String getAuthorName(){
	    return "Tom Grey";
	}
	
	private void loadExtraInfo(){
	    loadWeapon = true;
	    String[] extras = getExtraInfo();
	    if (extras == null){
		showStatus("No extra info");
		return;
	    }
	    int num_extras = extras.length;
	    for (int k=0; k < num_extras; k++){
		if (extras[k].startsWith("LOADWEAPONS")){
		    String wepStr = extras[k].substring( (extras[k].indexOf("=")+1) );
		    wepStr = wepStr.toLowerCase();
		    if (wepStr.startsWith("0") || wepStr.startsWith("f")){
			loadWeapon = false;
		    }
		}
		if (extras[k].startsWith("NOWEPSKIN")){
		    String wepSkinStr = extras[k].substring( (extras[k].indexOf("=")+1) );
		    wepSkinStr = wepSkinStr.toLowerCase();
		    if (wepSkinStr.startsWith("t")){
			specialWepSkin = true;
			showStatus("MDLquake2Model: Weapon skin will be sought on main skin");
		    }
		}
		showStatus("EXTRA "+k+": "+extras[k]);
	    }
	}

	
	protected void readModelThread(int[] loadf, int[] loads, int shifty){
	    
	    showStatus("MDLquake2Model: Quake2 model format class copyright T.J.Grey 1998");
	    loadExtraInfo();
	    radius = 0;
	    loadframes = loadf;
	    loadskins = loads;
	    shift = shifty;
	    if (loadframes[0] == -2){
		System.out.println("MDLquake2Model: Hey! You say load no frames but we have to load at least one!");
		showStatus("MDLquake2Model: Hey! You say load no frames but we have to load at least one!");
		loadframes[0] = 0;
	    }	
	    String modelName = getParameter("MODEL");
	    String modelNameLC = modelName.toLowerCase();
	    if (modelName.startsWith("!*!")){
		modelName = modelName.substring(3);
		readInfoModel(modelName, false);
	    }else if (modelNameLC.endsWith(".zip")){
		readSingleZip(modelName);
	    }else if (modelNameLC.endsWith(".txt")){
		readInfoModel(modelName, true);
	    }else{
		error("Unknown model specifier- should be either .zip or .txt");
	    }
	    if (num_skins > 0){
		cSkin = skins[currentSkin];
	    }
    // Calculate the bounding radius
	    calculateRadius();
    // Ensure get 100% at end!
	    if (haveWeapon){
		for (i=0; i < weaponNum_st_vertices; i++){
		    weaponSVertsT[i] = (short)(weaponSVertsT[i]+skinh);
		}
	    }
	    showPercentage(100);
	    max_vertices += num_vertices;
	    max_triangles += num_triangles;
	    max_skinh += skinh;
	    System.gc();
	    System.gc();

	}

	private void readInfoModel(String modelFilename, boolean fromFile){
	    showStatus("MDLquake2Model: Preparing to read model info file");
	    num_skins = loadskins.length;
	    if (loadskins[0] < 0){
		    num_skins = 0;
	    }//else{
	//	skins = new byte[num_skins][skinw][skinh];//MDLSkinGroup[num_skins];
	 //   }
	    int totalEntries = 100;
	    int[] numentries = null;
	    int numfiles = 1;
	    String[] files = null;
	    int fnumber =0;
	    if (fromFile){
		BufferedInputStream bi = null;
		try{
		    bi = new BufferedInputStream(getStreamForFile(modelFilename) );
		    java.io.StreamTokenizer st = new java.io.StreamTokenizer(bi);
		    st.eolIsSignificant(false);
		    int type;
		    numfiles = 0;
		    int tt = st.nextToken();
		    if (tt != st.TT_NUMBER){
			error("First entry in txt file must be number of files");
		    }else{
			numfiles = (int)st.nval;
		    }
		    files = new String[numfiles];
		    totalEntries = 0;
		    numentries = new int[numfiles];
		    fnumber = 0;
		    int temp;
		    while ( (type = st.nextToken()) != st.TT_EOF){
			files[fnumber] = st.sval;
			showStatus("MDLquake2Model: filename "+fnumber+" is "+files[fnumber]);
			if (!(files[fnumber].endsWith(".zip") || files[fnumber].endsWith(".ZIP"))){
			    System.out.println("MDLquake2Model: Warning- file "+files[fnumber]+" may not be a zip file");
			}
			temp = st.nextToken();
			if (temp != st.TT_NUMBER){
			    showStatus("MDLquake2Model: Assuming 1 entry in zipfile "+files[fnumber]);
			    numentries[fnumber] = 1;
			    totalEntries++;
			}else{
			    numentries[fnumber] = (int)st.nval;
			    totalEntries += numentries[fnumber];
			}
			fnumber++;
		    }
		    bi.close();
		    bi = null;
		    numfiles = fnumber;
		}catch (IOException e){
		    bi = null;
		    error("This IOException occured reading infofile: "+e);
		}
	    }else{
    //!fromFile- ie MODEL param list
		totalEntries = 0;
		java.util.StringTokenizer st = new java.util.StringTokenizer(modelFilename,";");
		String temp = st.nextToken();
		numfiles = (Integer.decode(temp)).intValue();
		files = new String[numfiles];
		numentries = new int[numfiles];
		fnumber = 0;
		while(fnumber < numfiles){
		    files[fnumber] = st.nextToken();
		    temp = st.nextToken();
		    numentries[fnumber] = (Integer.decode(temp)).intValue();
		    totalEntries += numentries[fnumber];
		    fnumber++;
		}
	    }
	    percentagePerFile = (int)(100/totalEntries);
	    int pcFromEntries = 0;
// read the files in
	    boolean haveTris = false;
//Loop over files
	    try{
		for (int f = 0; f < numfiles; f++){
		    String token = files[f];
		    showStatus("MDLquake2Model: Opening connection for "+token);
		    BufferedInputStream bis = null;
		    try{
			bis = new BufferedInputStream(getStreamForFile(token));
		    }catch(MalformedURLException e){
			error("Got MalformedURLException : "+e+" try checking MODELPATH");
		    }
		    int entriesSeen = 0;
		    ZipInputStream zipModelStream = new ZipInputStream(bis);
		    ZipEntry cfile = zipModelStream.getNextEntry();
		    String cname = cfile.getName();
		    cname = cname.toLowerCase();
		    showStatus("MDLquake2Model: Current file is "+cname);
		    if (cfile == null){
			error("Error, zipfile "+token+" is empty or corrupt!");
		    }
    // check if tris.md2.. if so then read, if not have we seen it?
		    if (cname.endsWith(".md2") && !haveTris){// && !cname.startsWith("w_") && !cname.startsWith("weapon")){
			md2Method(zipModelStream,loadframes, shift, false);
			haveTris = true;
			pcFromEntries = percentagePerFile;
			percentageDone = pcFromEntries;
			showPercentage(percentageDone);
			entriesSeen++;
			if (entriesSeen < numentries[f]){
			    cfile = zipModelStream.getNextEntry();
			}
//			skins = new byte[num_skins][skinw][skinh];
		    }
		    if (!haveTris){
			error("Sorry but tris.md2 must be the first file in the first archive in the info file");
		    }
    // Loop over entries
		    while(entriesSeen < numentries[f]){
			readEntry(cfile, zipModelStream);
			entriesSeen++;
			pcFromEntries += percentagePerFile;
			percentageDone = pcFromEntries;
			showPercentage(percentageDone);
			if (entriesSeen < numentries[f]){
			    cfile = zipModelStream.getNextEntry();
			}
		    }
		    showStatus("MDLquake2Model: Closing connection");
		    zipModelStream.close();
		    bis.close();
		}
	    }catch (IOException e){
		error("Got IOException: "+e);
	    }
	    if (haveWeapon && !haveWepSkin && !specialWepSkin){
		System.out.println("MDLquake2Model: Error! No skin found for weapon disabling texturemapping");
		showStatus("MDLquake2Model: Error! No skin found for weapon disabling texturemapping");
		num_skins = 0;
	    }
	    
	}
	    
	private void readSingleZip(String modelFilename){
	    max_skinh = 0;
//	    String filename = modelURL.getFile();
	    num_skins = loadskins.length;
	    if (loadskins[0] < 0){
	        num_skins = 0;
	    }
	    if (loadWeapon){
		percentagePerFile = (int)(100 / (3+num_skins));
	    }else{
		percentagePerFile = (int)(100 / (1+num_skins));
	    }
	    int pcFromEntries = 0;
	    try{
		showStatus("MDLquakeModel: Opening connection for zipped quake2 model..");
		ZipEntry cfile; //current zip entry
		String cname = null; //name of current zip entry file
		ZipInputStream zipModelStream = new ZipInputStream(new BufferedInputStream(getStreamForFile(modelFilename)) );
		cfile = zipModelStream.getNextEntry();
		if (cfile == null){
		    error("Error, zipfile is empty or corrupt!");
		}
		cname = cfile.getName();
		cname = cname.toLowerCase();
		showStatus("MDLquake2Model: Current file is "+cname);
		if (!cname.endsWith(".md2")){
		    error("Sorry but tris.md2 must be the first file in the archive");
		}
		md2Method(zipModelStream,loadframes, shift, false);
	//Since we now have this
//	        skins = new byte[num_skins][skinw][skinh];//MDLSkinGroup[num_skins];
		pcFromEntries += percentagePerFile;
		percentageDone = pcFromEntries;
		showPercentage(percentageDone);
		cfile = zipModelStream.getNextEntry();
		boolean keepReading = true;
		if (cfile == null){
		    keepReading = false;
		}
    // Read the other gubbings
		while(keepReading){
		    if (readEntry(cfile, zipModelStream)){
			pcFromEntries += percentagePerFile;
			percentageDone = pcFromEntries;
			showPercentage(percentageDone);
		    }
		    cfile = zipModelStream.getNextEntry();
	    // Only read on if this was not last entry and don't have all the bits!
		    keepReading = ( (cfile!=null) && !( haveWeapon&&haveWepSkin&&(loadingSkin>=num_skins) ) );
		}
		showStatus("MDLquake2Model: Reading model... done... Closing data stream");
		zipModelStream.closeEntry();
		zipModelStream.close();
		showStatus("MDLquake2Model: Data stream closed ok");
		zipModelStream = null;
		num_skins = loadingSkin;
		if (haveWeapon && !haveWepSkin && !specialWepSkin){
		    System.out.println("MDLquake2Model: Error! No skin found for weapon, disabling texturemapping");
		    num_skins = 0;
		}
	    }catch(IOException e){
		error("Error opening data Stream for model- <"+e+">");
	    }
	    return;
	}
	
	private boolean readEntry(ZipEntry cfile, InputStream zipModelStream) throws IOException{
	    String cname = cfile.getName();
	    cname = cname.toLowerCase();
	    cname = cfile.getName();
	    cname = cname.toLowerCase();
	    showStatus("MDLquake2Model: Current file is "+cname);
	    if (cname.endsWith(".md2")){
		if(loadWeapon && !haveWeapon){
		    weaponName = cname.substring(0,cname.lastIndexOf("."));
		    showStatus("MDLquake2Model: Reading weapon md2 file "+weaponName);
		    md2Method(zipModelStream,loadframes, shift, true);
		    haveWeapon = true;
		    return true;
		}else{
		    showStatus("MDLquake2Model: Skipping weapon "+cname);
		    return false;
		}
	    }else if (cname.endsWith("_i.pcx")){
		showStatus("MDLquake2Model: Skipping icon pcx");
		System.out.println("MDLquake2Model: Skipping icon pcx");
		return false;
	    }else if (cname.endsWith(".pcx")){
//	System.out.println("!!!!!!!"+cname+weaponName);
		if (cname.startsWith("weapon") || cname.startsWith("w_")){
		    if (!specialWepSkin && haveWeapon && cname.equals(weaponName+".pcx")){// || cname.startsWith("w_") ){
			loadPCX(zipModelStream,loadingSkin, true);
			haveWepSkin = true;
			return true;
		    }else{
			return false;
		    }
		}else{
		    if (loadingSkin < num_skins && loadskins[loadingSkin] == skinsSeen){
			loadPCX(zipModelStream,loadingSkin, false);
			loadingSkin++;
			skinsSeen++;
			return true;
		    }else{
			skinsSeen++;
			return false;
		    }
		}
	    }else{
		System.out.println("MDLquake2Model: Unknown file type: "+cname+"; skipping");
		    return false;
	    }
    }
		
	private void calculateRadius(){
	    radius = (float) Math.sqrt(radius);
	    radius *= 2;
	    showStatus("MDLquake2Model: Radius calculated as "+radius);
	}

	private void md2Method(InputStream zipModelStream,int[] loadframes, int shift, boolean isWeapon) throws IOException{
	    showStatus("MDLquake2Model: Preparing to read md2 file");
	    filepos = 0;
	    showStatus("MDLView: Reading model header");
// Get filetype= IDP2 for us
	    byte[] fourBytes = new byte[4];
	    for (int i=0; i<4; i++){
		fourBytes[i]=(byte)zipModelStream.read();       // magic nums
	    }
	    String filetype=new String(fourBytes);
	    showStatus("File type: "+filetype);
	    if (!filetype.equals("IDP2")){
		    System.out.println("MDLView: Warning this file may not be a quake2 model file (non IDP2 identifier)");
		    showStatus("MDLView: Warning may not be quake2 model file");
	    }
// Version
	    int version = this.readInt(zipModelStream);
	    checkNeg(false,version,"Model Version");
// Skin texture width and height, should/must be multiple of four (?)
	    if (!isWeapon){
		skinw = this.readInt(zipModelStream);        // skin width
		skinh = this.readInt(zipModelStream);        // skin height
		
//		if (sw !
// framesize
		frameSizeBytes = this.readInt(zipModelStream);
		if (checkNeg(false,frameSizeBytes,"frame size in bytes")){
		    return;
		}
// Number of skins
		totalSkins = readInt(zipModelStream);      // num skins
// Number of vertices
		this.num_vertices = this.readInt(zipModelStream);
		if (checkNeg(true,num_vertices,"Number of Vertices") || checkShort(num_vertices,"Number of Vertices") ){
		    return;
		}
// Number of skin vertices- > above due to seams
		this.num_st_vertices = this.readInt(zipModelStream);
		if (checkNeg(true,num_st_vertices,"Number of skin vertices") || checkShort(num_st_vertices,"Number of skin Vertices") ){
		    return;
		}
// Number of triangles
		this.num_triangles = this.readInt(zipModelStream);
		if (checkNeg(true,num_triangles,"Number of Triangles") || checkShort(num_triangles,"Number of Triangles") ){
		    return;
		}
//NUmber gl_cmds
		this.num_gl_cmds = readInt(zipModelStream);
// Number of frames
		this.totalFrames = this.readInt(zipModelStream);
		if (loadframes[0] == -1){
		    num_frames = totalFrames;
		}else{
		    num_frames = loadframes.length;
		}
    // Check for screw ups- probably invalid loadframes
		if (num_frames > totalFrames){
		    error("You requested more frames than the model contains! May not load correctly");
		}
		if (loadframes[loadframes.length-1] > totalFrames){
		    error("Request for frame beyond number of frames in model");
		}
	    }else{
		weaponSkinw = this.readInt(zipModelStream);
		weaponSkinh = this.readInt(zipModelStream);
		max_skinh = weaponSkinh;
// framesize
		weaponFrameSizeBytes = this.readInt(zipModelStream);
// Number of skins
		readInt(zipModelStream);      // num skins for weapon ignored!
// Number of vertices
		weaponNum_vertices = this.readInt(zipModelStream);
		max_vertices = weaponNum_vertices;
// Number of skin vertices- > above due to seams
		weaponNum_st_vertices = this.readInt(zipModelStream);
// Number of triangles
		weaponNum_triangles = this.readInt(zipModelStream);
		max_triangles = weaponNum_triangles;
//NUmber gl_cmds
		readInt(zipModelStream);
// Number of frames
		weaponTotalFrames = readInt(zipModelStream);

	    }
// ****OFFSETS NOTE these are overwritten by each succesive md2 file
	    int ofs_skins = this.readInt(zipModelStream);
	    int ofs_st = this.readInt(zipModelStream);
	    int ofs_tris = this.readInt(zipModelStream);
	    int ofs_frames = this.readInt(zipModelStream);
	    int ofs_glcmds = this.readInt(zipModelStream);
	    int ofs_end = this.readInt(zipModelStream);
// Advance the file position counter to count header
	    filepos = 68;
// Sort chunk order
	    int[] chunks = new int[5];
	    int[] offsets = new int[5];
	    chunks[0] = 0;
	    chunks[1] = 1;
	    chunks[2] = 2;
	    chunks[3] = 3;
	    chunks[4] = 4;
	    offsets[0] = ofs_skins;
	    offsets[1] = ofs_st;
	    offsets[2] = ofs_tris;
	    offsets[3] = ofs_frames;
	    offsets[4] = ofs_glcmds;
//bubblesort
	    int end= 4;
	    int start = -1;
	    int temp;
	    while(start < end){
		start++;
		end--;
		for(i=start; i<end; i++){              // Bubble the next largest one up
		    if (offsets[chunks[i]] > offsets[chunks[i+1]]){
			temp = chunks[i];
			chunks[i] = chunks[i+1];
			chunks[i+1] = temp;
		    }
		}
	    }


// load chunks
	    for (int x=0; x < 5; x++){
		switch (chunks[x]){
// Skin names
		case 0:
		    if (!isWeapon){
			skipTill(ofs_skins, zipModelStream);
			readSkinNames(zipModelStream);
		    }
		    break;
// Skin verts
		case 1:
		    skipTill(ofs_st, zipModelStream);
		    readSkinVerts(zipModelStream, isWeapon);
		    break;
// Triangles
		case 2:
		    skipTill(ofs_tris, zipModelStream);
		    readTriangles(zipModelStream, isWeapon);
		    break;
// Frames
		case 3:
		    skipTill(ofs_frames, zipModelStream);
		    readFrames(zipModelStream, loadframes, shift, isWeapon);
		    break;
		case 4:
	    // gl_cmds will be skipped by next chuck anyhow
		    showStatus("MDLquake2Model: Skipping gl_cmds");
		    break;
		}
	    }
	    showStatus("MDLquake2Model: md2 read... advancing to next read");
	}
	
	private void loadPCX(InputStream zipModelStream,int loadingSkin, boolean isWepSkin) throws IOException{
	    showStatus("MDLquake2Model: Preparing to read pcx file");
// Read only size from header	    
	    dodgySkip(4, zipModelStream);
	    short xmin = readShort(zipModelStream);
	    short ymin = readShort(zipModelStream);
	    short xmax = readShort(zipModelStream);
	    short ymax = readShort(zipModelStream);
	    dodgySkip(116, zipModelStream);
	    int sw = (int)(xmax-xmin+1);
	    int sh = (int)(ymax-ymin+1);
    	    byte[][] theSkin;
	    if (!isWepSkin){
		skinh = sh;
		skinw = sw;
		if (skins == null){
		    skins = new byte[num_skins][sw][sh];
		}
		theSkin = skins[loadingSkin];
	    }else{
		if (weaponSkin == null){
		    weaponSkin = new byte[sw][sh];
		}
		weaponSkinh = sh;
		weaponSkinw = sw;
		theSkin = weaponSkin;
	    }
	    int x = 0;
	    int y = 0;
	    int count = 0;
	    int data;
	    byte bdata;
	    float skinPC = 0;
	    int startPC = percentageDone;
	    float overallPCPerLine = 0;
	    overallPCPerLine = ((float)percentagePerFile/(float)sh);
	    while (y < sh){
		showStatus("MDLquake2Model: Reading skin line: "+y+" of "+skinh);
		skinPC += overallPCPerLine;
		percentageDone = startPC + (int)skinPC;
		showPercentage(percentageDone);
		while (x<sw){
		    data = 255 & zipModelStream.read();
// Is repeated or single?
		    if ( data > 192){
		// repeater
			count = data -192;
			bdata = (byte)zipModelStream.read();
			for (i=0; i < count; i++){
			    theSkin[x][y] = bdata;
			    x++;
			    if (x >= sw){
				showStatus("MDLquake2Model: Reading skin line: "+y+" of "+skinh);
				skinPC += overallPCPerLine;
				percentageDone = startPC + (int)skinPC;
				showPercentage(percentageDone);
				x -= sw;
				y++;
				if (y >= sh){
				    return;
				}
			    }
			}
		    }else{
		// Byte IS the data (like a shock unmasking of guilty party)
			theSkin[x][y] = (byte)data;
			x++;
		    }
		}
		y++;
		x = 0;
		yield();
	    }
	}
	
	private void skipTill(int position, InputStream dataStream) throws IOException{
	    int i = position - filepos;
	    if (i < 0){
		error("Unable to skip- file has advanced beyond requested position");
	    }
	    if (i == 0){
		return;
	    }
	    while (filepos < position){
		dataStream.read();
		filepos++;
		yield();
	    }
	}
	
	private final void readSkinNames(InputStream modelStream) throws IOException{
	    showStatus("MDLquake2Model: Reading skin names");
	    for (i = 0; i < totalSkins; i++){
		byte[] sixfourBytes = new byte[64];
		int length = 64;
		for (int j=0; j<64; j++){
		    sixfourBytes[j]=(byte)modelStream.read();       // magic nums
		    if (sixfourBytes[j] == 0 && j < length){
			length = j;
		    }
		}
		String skinName=new String(sixfourBytes,0,length);
	    }
	    filepos += (64*totalSkins);
	}
	
	private final void readSkinVerts(InputStream modelStream, boolean isWeapon) throws IOException{
	    showStatus("MDLquake2Model: Reading skin vertices");
	    if (!isWeapon){
		sVertsS = new short[num_st_vertices];
		sVertsT = new short[num_st_vertices];
		for (i=0; i < num_st_vertices; i++){
		    sVertsS[i] = readShort(modelStream);
		    sVertsT[i] = readShort(modelStream);
		}
		filepos += (4*num_st_vertices);
	    }else{
		weaponSVertsS = new short[weaponNum_st_vertices];
		weaponSVertsT = new short[weaponNum_st_vertices];
		for (i=0; i < weaponNum_st_vertices; i++){
		    weaponSVertsS[i] = readShort(modelStream);
		    weaponSVertsT[i] = readShort(modelStream); //Now done at end
		}
		filepos += (4*weaponNum_st_vertices);
	    }
	}
	
	private final void readTriangles(InputStream modelStream, boolean isWeapon) throws IOException{
	    showStatus("MDLquake2Model: Reading triangle definitions");
	    if (!isWeapon){
		tris = new short[num_triangles][3];
		stTris = new short[num_triangles][3];
		for (i=0; i < num_triangles; i++){
		    tris[i][0] = readShort(modelStream);
		    tris[i][1] = readShort(modelStream);
		    tris[i][2] = readShort(modelStream);
		    stTris[i][0] = readShort(modelStream);
		    stTris[i][1] = readShort(modelStream);
		    stTris[i][2] = readShort(modelStream);
		}
		filepos += 12*num_triangles;
	    }else{
		weaponTris = new short[num_triangles][3];
		weaponStTris = new short[num_triangles][3];
		for (i=0; i < weaponNum_triangles; i++){
	    //Add num_vertices to indexes- used to identify as weapon vertices
		    weaponTris[i][0] = (short)(readShort(modelStream)+num_vertices);
		    weaponTris[i][1] = (short)(readShort(modelStream)+num_vertices);
		    weaponTris[i][2] = (short)(readShort(modelStream)+num_vertices);
		    weaponStTris[i][0] = readShort(modelStream);
		    weaponStTris[i][1] = readShort(modelStream);
		    weaponStTris[i][2] = readShort(modelStream);
		}
		filepos += 12*weaponNum_triangles;
	    }
	}
	
	private final void readFrames(InputStream modelStream, int[] loadframes, int shift, boolean isWeapon) throws IOException{
	    showStatus("MDLquake2Model: Reading frames");
	    int loadingFrame = 0;
	    if (!isWeapon){
		frames = new MDLSimpleFrame[num_frames];
	    }else{
		weaponFrames = new MDLSimpleFrame[(Math.min(weaponTotalFrames,num_frames))];
	    }
	    i = 0;
	    float percentagePerFrame = (float)percentagePerFile/(float)totalFrames;
	    float framesPC= 0;
	    float startPC = (int)percentageDone;
	    while(i < totalFrames){
		if (loadingFrame >= num_frames){
		    return;
		}
		if (loadframes[0] == -1 || (loadingFrame < num_frames && loadframes[loadingFrame] == i) ){
		    float temp = 0;
		    if (!isWeapon){
			int skippy = frameSizeBytes-(40+(4*num_vertices));
			frames[loadingFrame] = new MDLSimpleFrame(num_vertices);
			float xscale = 1;
			float yscale = 1;
			float zscale = 1;
			float xtrans = 0;
			float ytrans = 0;
			float ztrans = 0;
			xscale = readFloat(modelStream);
			yscale = readFloat(modelStream);
			zscale = readFloat(modelStream);
			xtrans = readFloat(modelStream);
			ytrans = readFloat(modelStream);
			ztrans = readFloat(modelStream);
			byte[] sixteenBytes = new byte[16];
			int length = 16;
			for (int k = 0; k < 16; k++){
			    sixteenBytes[k] = (byte) modelStream.read();
			    if (sixteenBytes[k] == 0 && k < length){
				length = k;
			    }
			}
			frames[loadingFrame].name = new String(sixteenBytes,0,length);
			showStatus("MDLquake2Model: Frame name is "+frames[loadingFrame].name);
			for (int j = 0; j < num_vertices; j++){
			    frames[loadingFrame].frameVertsX[j] = modelStream.read();
			    frames[loadingFrame].frameVertsY[j] = modelStream.read();
			    frames[loadingFrame].frameVertsZ[j] = modelStream.read();
// Bin the light normal
			    modelStream.read();
// Scale and translate
			    frames[loadingFrame].frameVertsX[j] = frames[loadingFrame].frameVertsX[j] * xscale + xtrans;		    
			    frames[loadingFrame].frameVertsY[j] = frames[loadingFrame].frameVertsY[j] * yscale + ytrans;
			    frames[loadingFrame].frameVertsZ[j] = frames[loadingFrame].frameVertsZ[j] * zscale + ztrans;		    
			    frames[loadingFrame].frameVertsZ[j] += shift;
			    temp = (frames[loadingFrame].frameVertsX[j]*frames[loadingFrame].frameVertsX[j]);
			    temp += (frames[loadingFrame].frameVertsY[j]*frames[loadingFrame].frameVertsY[j]);
			    temp += (frames[loadingFrame].frameVertsZ[j]*frames[loadingFrame].frameVertsZ[j]);
			    radius = Math.max(temp,radius);
			}
			dodgySkip(skippy, modelStream);
			filepos += frameSizeBytes;
		    }else if (i < weaponTotalFrames){
			int skippy = weaponFrameSizeBytes-(40+(4*weaponNum_vertices));
			weaponFrames[loadingFrame] = new MDLSimpleFrame(weaponNum_vertices);
			float xscale = 1;
			float yscale = 1;
			float zscale = 1;
			float xtrans = 0;
			float ytrans = 0;
			float ztrans = 0;
			xscale = readFloat(modelStream);
			yscale = readFloat(modelStream);
			zscale = readFloat(modelStream);
			xtrans = readFloat(modelStream);
			ytrans = readFloat(modelStream);
			ztrans = readFloat(modelStream);
			byte[] sixteenBytes = new byte[16];
			int length = 16;
			for (int k = 0; k < 16; k++){
			    sixteenBytes[k] = (byte) modelStream.read();
			    if (sixteenBytes[k] == 0 && k < length){
				length = k;
			    }
			}
			weaponFrames[loadingFrame].name = new String(sixteenBytes,0,length);
			showStatus("MDLquake2Model: Frame name is "+weaponFrames[loadingFrame].name);
			for (int j = 0; j < weaponNum_vertices; j++){
			    weaponFrames[loadingFrame].frameVertsX[j] = modelStream.read();
			    weaponFrames[loadingFrame].frameVertsY[j] = modelStream.read();
			    weaponFrames[loadingFrame].frameVertsZ[j] = modelStream.read();
// Bin the light normal
			    modelStream.read();
// Scale and translate
			    weaponFrames[loadingFrame].frameVertsX[j] = weaponFrames[loadingFrame].frameVertsX[j] * xscale + xtrans;		    
			    weaponFrames[loadingFrame].frameVertsY[j] = weaponFrames[loadingFrame].frameVertsY[j] * yscale + ytrans;
			    weaponFrames[loadingFrame].frameVertsZ[j] = weaponFrames[loadingFrame].frameVertsZ[j] * zscale + ztrans;		    
			    weaponFrames[loadingFrame].frameVertsZ[j] += shift;
			    temp = (weaponFrames[loadingFrame].frameVertsX[j]*weaponFrames[loadingFrame].frameVertsX[j]);
			    temp += (weaponFrames[loadingFrame].frameVertsY[j]*weaponFrames[loadingFrame].frameVertsY[j]);
			    temp += (weaponFrames[loadingFrame].frameVertsZ[j]*weaponFrames[loadingFrame].frameVertsZ[j]);
			    radius = Math.max(temp,radius);

			}
			dodgySkip(skippy, modelStream);
			filepos += weaponFrameSizeBytes;
			weaponNum_frames++;
		    }
		    loadingFrame++;
	// If we have loaded all the frames we need we can (touch wood) skive even looking at the rest
		    if (isWeapon){
			if(i >= weaponTotalFrames){
			    return;
			}
		    }else{
			if (loadingFrame > num_frames){
			    return;
			}
		    }
		}else{
		    showStatus("MDLquake2Model: Skipping frame "+i);
		    if (!isWeapon){
			dodgySkip(frameSizeBytes, modelStream);
			filepos += frameSizeBytes;
		    }else{
			dodgySkip(weaponFrameSizeBytes, modelStream);
			filepos += weaponFrameSizeBytes;
		    }
		}
		i++;
		framesPC += percentagePerFrame;
		percentageDone = (int)(startPC+framesPC);
		showPercentage(percentageDone);
	    }
	}

// DANGER: ignores filepos.. do not use unless advancing filepos elsewhere	
	private void dodgySkip(int nb, InputStream dataStream) throws IOException{
	    while (nb > 0){
		dataStream.read();
		nb--;
	    }
	}
	
	private final short readShort(InputStream dataStream) throws IOException{

	    byte[] twoBytes = new byte[2];
	    twoBytes[1]=(byte)dataStream.read();
	    twoBytes[0]=(byte)dataStream.read();
	    short returnShort = 0;
	    returnShort = (short)(returnShort | ( (twoBytes[0] & 255) << 8) );
	    returnShort = (short)(returnShort | (twoBytes[1] & 255) );
	    return returnShort;
	}
	
	public java.awt.image.IndexColorModel getInternalPalette(){
	    return null;
	}
	
	private void error(String errorMsg){
	    showStatus("MDLquake2Model "+errorMsg);
	    System.out.println("MDLquake2Model "+errorMsg);
	    throw new RuntimeException("MDLquake2Model "+errorMsg);
	}

}